//
// Syd: rock-solid application kernel
// src/kernel/net/bind.rs: bind(2) handler
//
// Copyright (c) 2023, 2024, 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

use std::{
    net::IpAddr,
    os::fd::{AsRawFd, OwnedFd},
};

use libseccomp::ScmpNotifResp;
use nix::{
    errno::Errno,
    sys::{
        socket::{bind, getsockname, AddressFamily, SockaddrLike, SockaddrStorage},
        stat::umask,
    },
    unistd::fchdir,
};

use crate::{fs::CanonicalPath, hook::UNotifyEventRequest, path::XPathBuf, proc::proc_umask};

#[allow(clippy::cognitive_complexity)]
pub(crate) fn handle_bind(
    fd: OwnedFd,
    addr: &SockaddrStorage,
    root: Option<&CanonicalPath>,
    request: &UNotifyEventRequest,
    allow_safe_bind: bool,
) -> Result<ScmpNotifResp, Errno> {
    if addr.as_unix_addr().and_then(|a| a.path()).is_some() {
        let fd = fd.as_raw_fd();
        let size = addr.len();
        let addr = addr.as_ptr();

        let req = request.scmpreq;
        let mask = proc_umask(req.pid())?;

        // SAFETY:
        // 1. Honour directory for too long sockets.
        //    Note, the current working directory is per-thread here.
        // 2. We cannot resolve symlinks in root or we risk TOCTOU!
        #[allow(clippy::disallowed_methods)]
        let dirfd = root.as_ref().unwrap().dir.as_ref().unwrap();
        fchdir(dirfd)?;

        // SAFETY: Honour process' umask.
        // Note, the umask is per-thread here.
        umask(mask);

        // SAFETY: bind() does not work through dangling
        // symbolic links even with SO_REUSEADDR. When called
        // with a dangling symlink as argument, bind() fails
        // with EADDRINUSE unlike creat() which is going to
        // attempt to create the symlink target. Hence basename
        // in addr here is not vulnerable to TOCTOU.
        Errno::result(unsafe { libc::bind(fd, addr, size) })?;
    } else {
        // SAFETY: addr is not a UNIX domain socket.
        bind(fd.as_raw_fd(), addr)?;
    }

    // Handle allow_safe_bind and bind_map.
    // Ignore errors as bind has already succeeded.
    let _result = (|fd: OwnedFd, request: &UNotifyEventRequest| -> Result<(), Errno> {
        let addr = match addr.family() {
            Some(AddressFamily::Unix) => {
                let addr = addr.as_unix_addr().ok_or(Errno::EINVAL)?;
                match (addr.path(), addr.as_abstract()) {
                    (Some(_), _) => {
                        // Case 1: UNIX domain socket

                        // SAFETY: addr.path()=Some asserts root is Some.
                        #[allow(clippy::disallowed_methods)]
                        let path = &root.unwrap().abs();

                        // Handle bind_map after successful bind for UNIX sockets.
                        // We ignore errors because there's nothing we can do
                        // about them.
                        let _ = request.add_bind(&fd, path);
                        drop(fd); // Close our copy of the socket.

                        if !allow_safe_bind {
                            return Ok(());
                        }

                        // Display hex encodes as necessary.
                        Some(path.to_string())
                    }
                    (_, Some(path)) => {
                        // Case 2: UNIX abstract socket

                        drop(fd); // Close our copy of the socket.

                        if !allow_safe_bind {
                            return Ok(());
                        }

                        // SAFETY: Prefix UNIX abstract sockets with `@' before access check.
                        let mut unix = XPathBuf::from("@");
                        let null = memchr::memchr(0, path).unwrap_or(path.len());
                        unix.append_bytes(&path[..null]);

                        // Display hex encodes as necessary.
                        Some(unix.to_string())
                    }
                    _ => {
                        // Case 3: unnamed UNIX socket.

                        // SAFETY: Use dummy path `!unnamed' for unnamed UNIX sockets.
                        Some("!unnamed".to_string())
                    }
                }
            }
            Some(AddressFamily::Inet) => {
                if !allow_safe_bind {
                    return Ok(());
                }

                let addr = addr.as_sockaddr_in().ok_or(Errno::EINVAL)?;
                let mut port = addr.port();

                let addr = IpAddr::V4(addr.ip());
                if port == 0 {
                    port = getsockname::<SockaddrStorage>(fd.as_raw_fd())?
                        .as_sockaddr_in()
                        .ok_or(Errno::EINVAL)?
                        .port();
                }
                drop(fd); // Close our copy of the socket.

                Some(format!("{addr}!{port}"))
            }
            Some(AddressFamily::Inet6) => {
                if !allow_safe_bind {
                    return Ok(());
                }

                let addr = addr.as_sockaddr_in6().ok_or(Errno::EINVAL)?;
                let mut port = addr.port();

                let addr = IpAddr::V6(addr.ip());
                if port == 0 {
                    port = getsockname::<SockaddrStorage>(fd.as_raw_fd())?
                        .as_sockaddr_in6()
                        .ok_or(Errno::EINVAL)?
                        .port();
                }
                drop(fd); // Close our copy of the socket.

                Some(format!("{addr}!{port}"))
            }
            _ => {
                drop(fd); // Close our copy of the socket.

                None
            }
        };

        if let Some(addr) = addr {
            // Configure sandbox, note we remove
            // and readd the address so repeated
            // binds to the same address cannot
            // overflow the vector.
            let config: &[String] = &[
                format!("allow/net/connect-{addr}"),
                format!("allow/net/connect+{addr}"),
            ];

            // TODO: Log errors!
            let mut sandbox = request.get_mut_sandbox();
            for cmd in config {
                sandbox.config(cmd)?;
            }
            drop(sandbox);
        }

        // 1. The sandbox lock will be released on drop here.
        // 2. The socket fd will be closed on drop here.
        Ok(())
    })(fd, request);

    Ok(request.return_syscall(0))
}
