/*
 * This file is part of libdcadec.
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#define SCALE(x)    ((x) * M_SQRT2 * 128)

// Annex D.8 - 32-band interpolation FIR, perfect reconstruction
static const double band_fir_perfect[512] = {
     SCALE( 1.140033200E-10),  SCALE( 7.138742100E-11),
     SCALE(-8.358679600E-09),  SCALE(-2.529296600E-08),
     SCALE(-9.130198800E-08),  SCALE(-2.771560000E-07),
     SCALE(-5.746147600E-07),  SCALE(-3.712986200E-07),
     SCALE(-4.468735700E-07),  SCALE(-5.697322600E-07),
     SCALE(-6.300390500E-07),  SCALE(-6.677818900E-07),
     SCALE(-6.770656500E-07),  SCALE(-6.601852300E-07),
     SCALE(-6.193701600E-07),  SCALE(-5.586146700E-07),
    -SCALE( 7.034745600E-07), -SCALE( 8.348606100E-07),
    -SCALE( 9.544782800E-07), -SCALE( 1.052683900E-06),
    -SCALE( 1.119829700E-06), -SCALE( 1.144180200E-06),
    -SCALE( 1.124542400E-06), -SCALE( 9.822894700E-07),
    -SCALE( 8.920065800E-07), -SCALE( 1.560941800E-06),
    -SCALE( 8.454480100E-07), -SCALE( 3.167104300E-07),
    -SCALE( 1.028149000E-07), -SCALE( 4.147967800E-08),
    -SCALE(-6.821591800E-10), -SCALE(-1.611726200E-09),
    -SCALE(-2.668096400E-09), -SCALE(-3.377455500E-09),
    -SCALE( 6.820855300E-09), -SCALE( 3.715261200E-09),
    -SCALE( 1.643020800E-08), -SCALE( 1.007547900E-07),
    -SCALE( 2.448299500E-07), -SCALE( 1.306777300E-06),
    -SCALE( 1.904890000E-06), -SCALE( 2.555774300E-06),
    -SCALE( 3.253336000E-06), -SCALE( 3.953604500E-06),
    -SCALE( 4.617880200E-06), -SCALE( 5.210775600E-06),
    -SCALE( 5.696789700E-06), -SCALE( 6.046428700E-06),
    -SCALE( 7.614387900E-06), -SCALE( 7.678809700E-06),
    -SCALE( 7.533601500E-06), -SCALE( 7.179758900E-06),
    -SCALE( 6.629955000E-06), -SCALE( 5.908209500E-06),
    -SCALE( 5.044609200E-06), -SCALE( 4.187209700E-06),
    -SCALE( 3.139397100E-06), -SCALE( 6.650809100E-07),
    -SCALE( 3.073465500E-07), -SCALE( 5.699348500E-08),
    -SCALE( 1.510238900E-08), -SCALE( 3.384827600E-08),
    -SCALE(-3.227406600E-08), -SCALE(-3.772031200E-08),
     SCALE( 8.454083600E-08),  SCALE( 6.479789100E-08),
     SCALE( 1.236415900E-06),  SCALE( 2.480143600E-06),
     SCALE( 3.694976800E-06),  SCALE( 3.742137100E-06),
     SCALE( 3.262621300E-06),  SCALE( 7.476824700E-06),
     SCALE( 9.321632700E-06),  SCALE( 1.121856000E-05),
     SCALE( 1.317522400E-05),  SCALE( 1.505747500E-05),
     SCALE( 1.676702500E-05),  SCALE( 1.819741000E-05),
     SCALE( 1.925789500E-05),  SCALE( 1.987389300E-05),
    -SCALE(-3.076839000E-05), -SCALE(-3.254459900E-05),
    -SCALE(-3.367812600E-05), -SCALE(-3.411568400E-05),
    -SCALE(-3.382472000E-05), -SCALE(-3.280414400E-05),
    -SCALE(-3.109003600E-05), -SCALE(-2.861654300E-05),
    -SCALE(-2.571454500E-05), -SCALE(-1.870056200E-05),
    -SCALE(-1.771374800E-05), -SCALE(-1.568432200E-05),
    -SCALE(-1.128458200E-05), -SCALE(-6.805568100E-06),
    -SCALE(-5.671807300E-07), -SCALE(-9.974569000E-07),
    -SCALE(-1.466421500E-06), -SCALE(-1.846174800E-06),
    -SCALE( 7.763173700E-08), -SCALE( 1.809597500E-06),
    -SCALE( 4.157326000E-06), -SCALE( 7.240269200E-06),
    -SCALE( 1.073666400E-05), -SCALE( 2.089583800E-05),
    -SCALE( 2.647159500E-05), -SCALE( 3.196094400E-05),
    -SCALE( 3.698112500E-05), -SCALE( 4.149260300E-05),
    -SCALE( 4.534151200E-05), -SCALE( 4.846834800E-05),
    -SCALE( 5.081695700E-05), -SCALE( 5.236303900E-05),
    -SCALE( 3.803557300E-06), -SCALE( 7.916183300E-06),
    -SCALE( 1.191309700E-05), -SCALE( 1.561346600E-05),
    -SCALE( 1.881671400E-05), -SCALE( 2.131957100E-05),
    -SCALE( 2.295038200E-05), -SCALE( 2.354812700E-05),
    -SCALE( 2.291622100E-05), -SCALE( 2.497457200E-05),
    -SCALE( 1.979628700E-05), -SCALE( 1.390508100E-05),
    -SCALE( 7.179248900E-06), -SCALE(-1.614022200E-07),
    -SCALE(-1.518084500E-05), -SCALE(-1.610369300E-05),
     SCALE( 1.994364800E-05),  SCALE( 1.774116500E-05),
     SCALE( 4.511232400E-05),  SCALE( 5.311715600E-05),
     SCALE( 6.144976200E-05),  SCALE( 7.052899300E-05),
     SCALE( 7.984114900E-05),  SCALE( 8.597821200E-05),
     SCALE( 9.341758200E-05),  SCALE( 1.002681400E-04),
     SCALE( 1.064814700E-04),  SCALE( 1.119841200E-04),
     SCALE( 1.165901700E-04),  SCALE( 1.202018700E-04),
     SCALE( 1.226936800E-04),  SCALE( 1.237377900E-04),
    -SCALE( 1.200453700E-04), -SCALE( 1.185602000E-04),
    -SCALE( 1.152534400E-04), -SCALE( 1.097435100E-04),
    -SCALE( 1.018237000E-04), -SCALE( 9.130172200E-05),
    -SCALE( 7.793692700E-05), -SCALE( 6.157321800E-05),
    -SCALE( 4.214289700E-05), -SCALE( 2.010055900E-05),
    -SCALE(-6.512868000E-06), -SCALE(-3.623958500E-05),
    -SCALE(-6.898332300E-05), -SCALE(-1.052143400E-04),
    -SCALE(-1.311540500E-04), -SCALE(-1.772621900E-04),
    -SCALE(-2.231129500E-04), -SCALE(-2.678985000E-04),
    -SCALE(-3.353960600E-04), -SCALE(-3.909221300E-04),
    -SCALE(-4.488403900E-04), -SCALE(-5.091327500E-04),
    -SCALE(-5.717321000E-04), -SCALE(-6.360244700E-04),
    -SCALE(-7.021067600E-04), -SCALE(-7.695597500E-04),
    -SCALE(-8.380918900E-04), -SCALE(-9.072555100E-04),
    -SCALE(-9.767158300E-04), -SCALE(-1.045985500E-03),
    -SCALE(-1.114606900E-03), -SCALE(-1.182107000E-03),
    -SCALE(-1.251459700E-03), -SCALE(-1.314813200E-03),
    -SCALE(-1.375058300E-03), -SCALE(-1.431717500E-03),
    -SCALE(-1.484159500E-03), -SCALE(-1.531686400E-03),
    -SCALE(-1.573715600E-03), -SCALE(-1.609496400E-03),
    -SCALE(-1.638393400E-03), -SCALE(-1.659751400E-03),
    -SCALE(-1.672691700E-03), -SCALE(-1.676540900E-03),
    -SCALE(-1.670887400E-03), -SCALE(-1.654649800E-03),
    -SCALE(-1.632849400E-03), -SCALE(-1.592423900E-03),
     SCALE( 1.541196600E-03),  SCALE( 1.478566700E-03),
     SCALE( 1.394017000E-03),  SCALE( 1.301623400E-03),
     SCALE( 1.194737700E-03),  SCALE( 1.072608600E-03),
     SCALE( 9.349224800E-04),  SCALE( 7.810380900E-04),
     SCALE( 6.109076600E-04),  SCALE( 4.241331700E-04),
     SCALE( 2.204804700E-04),  SCALE(-2.272228400E-07),
     SCALE(-2.380696500E-04),  SCALE(-4.930996000E-04),
     SCALE(-7.653038000E-04),  SCALE(-1.054538000E-03),
    -SCALE(-1.360519200E-03), -SCALE(-1.683383000E-03),
    -SCALE(-2.022614600E-03), -SCALE(-2.377899500E-03),
    -SCALE(-2.748797700E-03), -SCALE(-3.134797500E-03),
    -SCALE(-3.535329200E-03), -SCALE(-3.949734800E-03),
    -SCALE(-4.377291000E-03), -SCALE(-4.817122000E-03),
    -SCALE(-5.268542300E-03), -SCALE(-5.730478300E-03),
    -SCALE(-6.202005100E-03), -SCALE(-6.681936000E-03),
    -SCALE(-7.167914500E-03), -SCALE(-7.662045500E-03),
    -SCALE(-8.160839200E-03), -SCALE(-8.663190500E-03),
    -SCALE(-9.169050700E-03), -SCALE(-9.675131500E-03),
    -SCALE(-1.018101800E-02), -SCALE(-1.068536400E-02),
    -SCALE(-1.118674000E-02), -SCALE(-1.168377500E-02),
    -SCALE(-1.217496400E-02), -SCALE(-1.265891600E-02),
    -SCALE(-1.313420500E-02), -SCALE(-1.359941000E-02),
    -SCALE(-1.405313100E-02), -SCALE(-1.449398400E-02),
    -SCALE(-1.492061500E-02), -SCALE(-1.533170500E-02),
    -SCALE(-1.572581500E-02), -SCALE(-1.610200000E-02),
    -SCALE(-1.645893800E-02), -SCALE(-1.679548100E-02),
    -SCALE(-1.711054800E-02), -SCALE(-1.740312600E-02),
    -SCALE(-1.767225900E-02), -SCALE(-1.791707200E-02),
    -SCALE(-1.813675700E-02), -SCALE(-1.833061700E-02),
    -SCALE(-1.849797400E-02), -SCALE(-1.863830100E-02),
    -SCALE(-1.875108700E-02), -SCALE(-1.883604000E-02),
    -SCALE(-1.889315300E-02), -SCALE(-1.892151600E-02),
     SCALE( 1.892151600E-02),  SCALE( 1.889315300E-02),
     SCALE( 1.883604000E-02),  SCALE( 1.875108700E-02),
     SCALE( 1.863830100E-02),  SCALE( 1.849797400E-02),
     SCALE( 1.833061700E-02),  SCALE( 1.813675700E-02),
     SCALE( 1.791707200E-02),  SCALE( 1.767225900E-02),
     SCALE( 1.740312600E-02),  SCALE( 1.711054800E-02),
     SCALE( 1.679548100E-02),  SCALE( 1.645893800E-02),
     SCALE( 1.610200000E-02),  SCALE( 1.572581500E-02),
    -SCALE( 1.533170500E-02), -SCALE( 1.492061500E-02),
    -SCALE( 1.449398400E-02), -SCALE( 1.405313100E-02),
    -SCALE( 1.359941000E-02), -SCALE( 1.313420500E-02),
    -SCALE( 1.265891600E-02), -SCALE( 1.217496400E-02),
    -SCALE( 1.168377500E-02), -SCALE( 1.118674000E-02),
    -SCALE( 1.068536400E-02), -SCALE( 1.018101800E-02),
    -SCALE( 9.675131500E-03), -SCALE( 9.169050700E-03),
    -SCALE( 8.663190500E-03), -SCALE( 8.160839200E-03),
    -SCALE( 7.662045500E-03), -SCALE( 7.167914500E-03),
    -SCALE( 6.681936000E-03), -SCALE( 6.202005100E-03),
    -SCALE( 5.730478300E-03), -SCALE( 5.268542300E-03),
    -SCALE( 4.817122000E-03), -SCALE( 4.377291000E-03),
    -SCALE( 3.949734800E-03), -SCALE( 3.535329200E-03),
    -SCALE( 3.134797500E-03), -SCALE( 2.748797700E-03),
    -SCALE( 2.377899500E-03), -SCALE( 2.022614600E-03),
    -SCALE( 1.683383000E-03), -SCALE( 1.360519200E-03),
    -SCALE( 1.054538000E-03), -SCALE( 7.653038000E-04),
    -SCALE( 4.930996000E-04), -SCALE( 2.380696500E-04),
    -SCALE( 2.272228400E-07), -SCALE(-2.204804700E-04),
    -SCALE(-4.241331700E-04), -SCALE(-6.109076600E-04),
    -SCALE(-7.810380900E-04), -SCALE(-9.349224800E-04),
    -SCALE(-1.072608600E-03), -SCALE(-1.194737700E-03),
    -SCALE(-1.301623400E-03), -SCALE(-1.394017000E-03),
    -SCALE(-1.478566700E-03), -SCALE(-1.541196600E-03),
     SCALE( 1.592423900E-03),  SCALE( 1.632849400E-03),
     SCALE( 1.654649800E-03),  SCALE( 1.670887400E-03),
     SCALE( 1.676540900E-03),  SCALE( 1.672691700E-03),
     SCALE( 1.659751400E-03),  SCALE( 1.638393400E-03),
     SCALE( 1.609496400E-03),  SCALE( 1.573715600E-03),
     SCALE( 1.531686400E-03),  SCALE( 1.484159500E-03),
     SCALE( 1.431717500E-03),  SCALE( 1.375058300E-03),
     SCALE( 1.314813200E-03),  SCALE( 1.251459700E-03),
    -SCALE( 1.182107000E-03), -SCALE( 1.114606900E-03),
    -SCALE( 1.045985500E-03), -SCALE( 9.767158300E-04),
    -SCALE( 9.072555100E-04), -SCALE( 8.380918900E-04),
    -SCALE( 7.695597500E-04), -SCALE( 7.021067600E-04),
    -SCALE( 6.360244700E-04), -SCALE( 5.717321000E-04),
    -SCALE( 5.091327500E-04), -SCALE( 4.488403900E-04),
    -SCALE( 3.909221300E-04), -SCALE( 3.353960600E-04),
    -SCALE( 2.678985000E-04), -SCALE( 2.231129500E-04),
    -SCALE( 1.772621900E-04), -SCALE( 1.311540500E-04),
    -SCALE( 1.052143400E-04), -SCALE( 6.898332300E-05),
    -SCALE( 3.623958500E-05), -SCALE( 6.512868000E-06),
    -SCALE(-2.010055900E-05), -SCALE(-4.214289700E-05),
    -SCALE(-6.157321800E-05), -SCALE(-7.793692700E-05),
    -SCALE(-9.130172200E-05), -SCALE(-1.018237000E-04),
    -SCALE(-1.097435100E-04), -SCALE(-1.152534400E-04),
    -SCALE(-1.185602000E-04), -SCALE(-1.200453700E-04),
    -SCALE(-1.237377900E-04), -SCALE(-1.226936800E-04),
    -SCALE(-1.202018700E-04), -SCALE(-1.165901700E-04),
    -SCALE(-1.119841200E-04), -SCALE(-1.064814700E-04),
    -SCALE(-1.002681400E-04), -SCALE(-9.341758200E-05),
    -SCALE(-8.597821200E-05), -SCALE(-7.984114900E-05),
    -SCALE(-7.052899300E-05), -SCALE(-6.144976200E-05),
    -SCALE(-5.311715600E-05), -SCALE(-4.511232400E-05),
    -SCALE(-1.774116500E-05), -SCALE(-1.994364800E-05),
     SCALE( 1.610369300E-05),  SCALE( 1.518084500E-05),
     SCALE( 1.614022200E-07),  SCALE(-7.179248900E-06),
     SCALE(-1.390508100E-05),  SCALE(-1.979628700E-05),
     SCALE(-2.497457200E-05),  SCALE(-2.291622100E-05),
     SCALE(-2.354812700E-05),  SCALE(-2.295038200E-05),
     SCALE(-2.131957100E-05),  SCALE(-1.881671400E-05),
     SCALE(-1.561346600E-05),  SCALE(-1.191309700E-05),
     SCALE(-7.916183300E-06),  SCALE(-3.803557300E-06),
    -SCALE(-5.236303900E-05), -SCALE(-5.081695700E-05),
    -SCALE(-4.846834800E-05), -SCALE(-4.534151200E-05),
    -SCALE(-4.149260300E-05), -SCALE(-3.698112500E-05),
    -SCALE(-3.196094400E-05), -SCALE(-2.647159500E-05),
    -SCALE(-2.089583800E-05), -SCALE(-1.073666400E-05),
    -SCALE(-7.240269200E-06), -SCALE(-4.157326000E-06),
    -SCALE(-1.809597500E-06), -SCALE(-7.763173700E-08),
    -SCALE( 1.846174800E-06), -SCALE( 1.466421500E-06),
    -SCALE( 9.974569000E-07), -SCALE( 5.671807300E-07),
    -SCALE( 6.805568100E-06), -SCALE( 1.128458200E-05),
    -SCALE( 1.568432200E-05), -SCALE( 1.771374800E-05),
    -SCALE( 1.870056200E-05), -SCALE( 2.571454500E-05),
    -SCALE( 2.861654300E-05), -SCALE( 3.109003600E-05),
    -SCALE( 3.280414400E-05), -SCALE( 3.382472000E-05),
    -SCALE( 3.411568400E-05), -SCALE( 3.367812600E-05),
    -SCALE( 3.254459900E-05), -SCALE( 3.076839000E-05),
    -SCALE(-1.987389300E-05), -SCALE(-1.925789500E-05),
    -SCALE(-1.819741000E-05), -SCALE(-1.676702500E-05),
    -SCALE(-1.505747500E-05), -SCALE(-1.317522400E-05),
    -SCALE(-1.121856000E-05), -SCALE(-9.321632700E-06),
    -SCALE(-7.476824700E-06), -SCALE(-3.262621300E-06),
    -SCALE(-3.742137100E-06), -SCALE(-3.694976800E-06),
    -SCALE(-2.480143600E-06), -SCALE(-1.236415900E-06),
    -SCALE(-6.479789100E-08), -SCALE(-8.454083600E-08),
     SCALE( 3.772031200E-08),  SCALE( 3.227406600E-08),
     SCALE(-3.384827600E-08),  SCALE(-1.510238900E-08),
     SCALE(-5.699348500E-08),  SCALE(-3.073465500E-07),
     SCALE(-6.650809100E-07),  SCALE(-3.139397100E-06),
     SCALE(-4.187209700E-06),  SCALE(-5.044609200E-06),
     SCALE(-5.908209500E-06),  SCALE(-6.629955000E-06),
     SCALE(-7.179758900E-06),  SCALE(-7.533601500E-06),
     SCALE(-7.678809700E-06),  SCALE(-7.614387900E-06),
    -SCALE(-6.046428700E-06), -SCALE(-5.696789700E-06),
    -SCALE(-5.210775600E-06), -SCALE(-4.617880200E-06),
    -SCALE(-3.953604500E-06), -SCALE(-3.253336000E-06),
    -SCALE(-2.555774300E-06), -SCALE(-1.904890000E-06),
    -SCALE(-1.306777300E-06), -SCALE(-2.448299500E-07),
    -SCALE(-1.007547900E-07), -SCALE(-1.643020800E-08),
    -SCALE(-3.715261200E-09), -SCALE(-6.820855300E-09),
    -SCALE( 3.377455500E-09), -SCALE( 2.668096400E-09),
    -SCALE( 1.611726200E-09), -SCALE( 6.821591800E-10),
    -SCALE(-4.147967800E-08), -SCALE(-1.028149000E-07),
    -SCALE(-3.167104300E-07), -SCALE(-8.454480100E-07),
    -SCALE(-1.560941800E-06), -SCALE(-8.920065800E-07),
    -SCALE(-9.822894700E-07), -SCALE(-1.124542400E-06),
    -SCALE(-1.144180200E-06), -SCALE(-1.119829700E-06),
    -SCALE(-1.052683900E-06), -SCALE(-9.544782800E-07),
    -SCALE(-8.348606100E-07), -SCALE(-7.034745600E-07),
    -SCALE( 5.586146700E-07), -SCALE( 6.193701600E-07),
    -SCALE( 6.601852300E-07), -SCALE( 6.770656500E-07),
    -SCALE( 6.677818900E-07), -SCALE( 6.300390500E-07),
    -SCALE( 5.697322600E-07), -SCALE( 4.468735700E-07),
    -SCALE( 3.712986200E-07), -SCALE( 5.746147600E-07),
    -SCALE( 2.771560000E-07), -SCALE( 9.130198800E-08),
    -SCALE( 2.529296600E-08), -SCALE( 8.358679600E-09),
    -SCALE(-7.138742100E-11), -SCALE(-1.140033200E-10)
};

// Annex D.8 - 32-band interpolation FIR, non-perfect reconstruction
static const double band_fir_nonperfect[512] = {
     SCALE(-1.390191784E-07),  SCALE(-1.693738625E-07),
     SCALE(-2.030677564E-07),  SCALE(-2.404238444E-07),
     SCALE(-2.818143514E-07),  SCALE(-3.276689142E-07),
     SCALE(-3.784752209E-07),  SCALE(-4.347855338E-07),
     SCALE(-4.972276315E-07),  SCALE(-5.665120852E-07),
     SCALE(-6.434325428E-07),  SCALE(-7.288739425E-07),
     SCALE(-8.238164355E-07),  SCALE(-9.293416952E-07),
     SCALE(-1.046637067E-06),  SCALE(-1.176999604E-06),
    -SCALE(-1.321840614E-06), -SCALE(-1.482681114E-06),
    -SCALE(-1.661159786E-06), -SCALE(-1.859034001E-06),
    -SCALE(-2.078171747E-06), -SCALE(-2.320550948E-06),
    -SCALE(-2.588257530E-06), -SCALE(-2.883470643E-06),
    -SCALE(-3.208459020E-06), -SCALE(-3.565570978E-06),
    -SCALE(-3.957220997E-06), -SCALE(-4.385879038E-06),
    -SCALE(-4.854050530E-06), -SCALE(-5.364252502E-06),
    -SCALE(-5.918994248E-06), -SCALE(-6.520755960E-06),
    -SCALE(-7.171964626E-06), -SCALE(-7.874960829E-06),
    -SCALE(-8.631964192E-06), -SCALE(-9.445050637E-06),
    -SCALE(-1.031611009E-05), -SCALE(-1.124680875E-05),
    -SCALE(-1.223855270E-05), -SCALE(-1.329243969E-05),
    -SCALE(-1.440921824E-05), -SCALE(-1.558924305E-05),
    -SCALE(-1.683242772E-05), -SCALE(-1.813820381E-05),
    -SCALE(-1.950545993E-05), -SCALE(-2.093250441E-05),
    -SCALE(-2.241701623E-05), -SCALE(-2.395598858E-05),
    -SCALE(-2.554569073E-05), -SCALE(-2.718161704E-05),
    -SCALE(-2.885844333E-05), -SCALE(-3.056998685E-05),
    -SCALE(-3.230916263E-05), -SCALE(-3.406793985E-05),
    -SCALE(-3.583733633E-05), -SCALE(-3.760734762E-05),
    -SCALE(-3.936696885E-05), -SCALE(-4.110412556E-05),
    -SCALE(-4.280570283E-05), -SCALE(-4.445751256E-05),
    -SCALE(-4.604430433E-05), -SCALE(-4.754976908E-05),
    -SCALE(-4.895655002E-05), -SCALE(-5.024627535E-05),
     SCALE( 5.139957648E-05),  SCALE( 5.239612074E-05),
     SCALE( 5.321469871E-05),  SCALE( 5.383323878E-05),
     SCALE( 5.422891263E-05),  SCALE( 5.437819709E-05),
     SCALE( 5.425697600E-05),  SCALE( 5.384063843E-05),
     SCALE( 5.310418419E-05),  SCALE( 5.202236207E-05),
     SCALE( 5.056979353E-05),  SCALE( 4.872112549E-05),
     SCALE( 4.645117951E-05),  SCALE( 4.373511547E-05),
     SCALE( 4.054862075E-05),  SCALE( 3.686808850E-05),
    -SCALE( 3.267079956E-05), -SCALE( 2.793515523E-05),
    -SCALE( 2.264085742E-05), -SCALE( 1.676913780E-05),
    -SCALE( 1.030297699E-05), -SCALE( 3.227306706E-06),
    -SCALE(-4.470633485E-06), -SCALE(-1.280130618E-05),
    -SCALE(-2.177240640E-05), -SCALE(-3.138873581E-05),
    -SCALE(-4.165195787E-05), -SCALE(-5.256036457E-05),
    -SCALE(-6.410864444E-05), -SCALE(-7.628766616E-05),
    -SCALE(-8.908427117E-05), -SCALE(-1.024810626E-04),
    -SCALE(-1.164562127E-04), -SCALE(-1.309833024E-04),
    -SCALE(-1.460311323E-04), -SCALE(-1.615635992E-04),
    -SCALE(-1.775395358E-04), -SCALE(-1.939126523E-04),
    -SCALE(-2.106313768E-04), -SCALE(-2.276388550E-04),
    -SCALE(-2.448728774E-04), -SCALE(-2.622658503E-04),
    -SCALE(-2.797449124E-04), -SCALE(-2.972317743E-04),
    -SCALE(-3.146430245E-04), -SCALE(-3.318900708E-04),
    -SCALE(-3.488793736E-04), -SCALE(-3.655125911E-04),
    -SCALE(-3.816867538E-04), -SCALE(-3.972945851E-04),
    -SCALE(-4.122247046E-04), -SCALE(-4.263620067E-04),
    -SCALE(-4.395879805E-04), -SCALE(-4.517810594E-04),
    -SCALE(-4.628172028E-04), -SCALE(-4.725702747E-04),
    -SCALE(-4.809123348E-04), -SCALE(-4.877146275E-04),
    -SCALE(-4.928477574E-04), -SCALE(-4.961824161E-04),
    -SCALE(-4.975944757E-04), -SCALE(-4.969481961E-04),
    -SCALE(-4.941228544E-04), -SCALE(-4.889960401E-04),
     SCALE( 4.814492422E-04),  SCALE( 4.713678791E-04),
     SCALE( 4.586426076E-04),  SCALE( 4.431701091E-04),
     SCALE( 4.248536134E-04),  SCALE( 4.036037717E-04),
     SCALE( 3.793396754E-04),  SCALE( 3.519894381E-04),
     SCALE( 3.214911267E-04),  SCALE( 2.877934603E-04),
     SCALE( 2.508567995E-04),  SCALE( 2.106537577E-04),
     SCALE( 1.671699720E-04),  SCALE( 1.204049113E-04),
     SCALE( 7.037253090E-05),  SCALE( 1.710198012E-05),
    -SCALE(-3.936182839E-05), -SCALE(-9.895755647E-05),
    -SCALE(-1.616069785E-04), -SCALE(-2.272142592E-04),
    -SCALE(-2.956659591E-04), -SCALE(-3.668301215E-04),
    -SCALE(-4.405563814E-04), -SCALE(-5.166754709E-04),
    -SCALE(-5.949990009E-04), -SCALE(-6.753197522E-04),
    -SCALE(-7.574109477E-04), -SCALE(-8.410271257E-04),
    -SCALE(-9.259034996E-04), -SCALE(-1.011756598E-03),
    -SCALE(-1.098284614E-03), -SCALE(-1.185167348E-03),
    -SCALE(-1.272067428E-03), -SCALE(-1.358630019E-03),
    -SCALE(-1.444484224E-03), -SCALE(-1.529243193E-03),
    -SCALE(-1.612505526E-03), -SCALE(-1.693855622E-03),
    -SCALE(-1.772865304E-03), -SCALE(-1.849094522E-03),
    -SCALE(-1.922092517E-03), -SCALE(-1.991399564E-03),
    -SCALE(-2.056547208E-03), -SCALE(-2.117061289E-03),
    -SCALE(-2.172462177E-03), -SCALE(-2.222266514E-03),
    -SCALE(-2.265989315E-03), -SCALE(-2.303145360E-03),
    -SCALE(-2.333251061E-03), -SCALE(-2.355825622E-03),
    -SCALE(-2.370394068E-03), -SCALE(-2.376487479E-03),
    -SCALE(-2.373647178E-03), -SCALE(-2.361423569E-03),
    -SCALE(-2.339380793E-03), -SCALE(-2.307097195E-03),
    -SCALE(-2.264167881E-03), -SCALE(-2.210205887E-03),
    -SCALE(-2.144844970E-03), -SCALE(-2.067740774E-03),
    -SCALE(-1.978572691E-03), -SCALE(-1.877046190E-03),
    -SCALE(-1.762894331E-03), -SCALE(-1.635878929E-03),
     SCALE( 1.495792647E-03),  SCALE( 1.342460280E-03),
     SCALE( 1.175740734E-03),  SCALE( 9.955273708E-04),
     SCALE( 8.017504588E-04),  SCALE( 5.943773431E-04),
     SCALE( 3.734139318E-04),  SCALE( 1.389056415E-04),
     SCALE(-1.090620208E-04),  SCALE(-3.703625989E-04),
     SCALE(-6.448282511E-04),  SCALE(-9.322494152E-04),
     SCALE(-1.232374110E-03),  SCALE(-1.544908970E-03),
     SCALE(-1.869517611E-03),  SCALE(-2.205822384E-03),
    -SCALE(-2.553403843E-03), -SCALE(-2.911801683E-03),
    -SCALE(-3.280514618E-03), -SCALE(-3.659002949E-03),
    -SCALE(-4.046686925E-03), -SCALE(-4.442950245E-03),
    -SCALE(-4.847140983E-03), -SCALE(-5.258570891E-03),
    -SCALE(-5.676518660E-03), -SCALE(-6.100233644E-03),
    -SCALE(-6.528933067E-03), -SCALE(-6.961807609E-03),
    -SCALE(-7.398022339E-03), -SCALE(-7.836719044E-03),
    -SCALE(-8.277016692E-03), -SCALE(-8.718019351E-03),
    -SCALE(-9.158811532E-03), -SCALE(-9.598465636E-03),
    -SCALE(-1.003604382E-02), -SCALE(-1.047059800E-02),
    -SCALE(-1.090117730E-02), -SCALE(-1.132682897E-02),
    -SCALE(-1.174659748E-02), -SCALE(-1.215953380E-02),
    -SCALE(-1.256469358E-02), -SCALE(-1.296114177E-02),
    -SCALE(-1.334795821E-02), -SCALE(-1.372423489E-02),
    -SCALE(-1.408908330E-02), -SCALE(-1.444163360E-02),
    -SCALE(-1.478104480E-02), -SCALE(-1.510649733E-02),
    -SCALE(-1.541720331E-02), -SCALE(-1.571240649E-02),
    -SCALE(-1.599138230E-02), -SCALE(-1.625344716E-02),
    -SCALE(-1.649795473E-02), -SCALE(-1.672429405E-02),
    -SCALE(-1.693190821E-02), -SCALE(-1.712027565E-02),
    -SCALE(-1.728892699E-02), -SCALE(-1.743743755E-02),
    -SCALE(-1.756543480E-02), -SCALE(-1.767260395E-02),
    -SCALE(-1.775865816E-02), -SCALE(-1.782339066E-02),
    -SCALE(-1.786663756E-02), -SCALE(-1.788828894E-02),
     SCALE( 1.788828894E-02),  SCALE( 1.786663756E-02),
     SCALE( 1.782339066E-02),  SCALE( 1.775865816E-02),
     SCALE( 1.767260395E-02),  SCALE( 1.756543480E-02),
     SCALE( 1.743743755E-02),  SCALE( 1.728892699E-02),
     SCALE( 1.712027565E-02),  SCALE( 1.693190821E-02),
     SCALE( 1.672429405E-02),  SCALE( 1.649795473E-02),
     SCALE( 1.625344716E-02),  SCALE( 1.599138230E-02),
     SCALE( 1.571240649E-02),  SCALE( 1.541720331E-02),
    -SCALE( 1.510649733E-02), -SCALE( 1.478104480E-02),
    -SCALE( 1.444163360E-02), -SCALE( 1.408908330E-02),
    -SCALE( 1.372423489E-02), -SCALE( 1.334795821E-02),
    -SCALE( 1.296114177E-02), -SCALE( 1.256469358E-02),
    -SCALE( 1.215953380E-02), -SCALE( 1.174659748E-02),
    -SCALE( 1.132682897E-02), -SCALE( 1.090117730E-02),
    -SCALE( 1.047059800E-02), -SCALE( 1.003604382E-02),
    -SCALE( 9.598465636E-03), -SCALE( 9.158811532E-03),
    -SCALE( 8.718019351E-03), -SCALE( 8.277016692E-03),
    -SCALE( 7.836719044E-03), -SCALE( 7.398022339E-03),
    -SCALE( 6.961807609E-03), -SCALE( 6.528933067E-03),
    -SCALE( 6.100233644E-03), -SCALE( 5.676518660E-03),
    -SCALE( 5.258570891E-03), -SCALE( 4.847140983E-03),
    -SCALE( 4.442950245E-03), -SCALE( 4.046686925E-03),
    -SCALE( 3.659002949E-03), -SCALE( 3.280514618E-03),
    -SCALE( 2.911801683E-03), -SCALE( 2.553403843E-03),
    -SCALE( 2.205822384E-03), -SCALE( 1.869517611E-03),
    -SCALE( 1.544908970E-03), -SCALE( 1.232374110E-03),
    -SCALE( 9.322494152E-04), -SCALE( 6.448282511E-04),
    -SCALE( 3.703625989E-04), -SCALE( 1.090620208E-04),
    -SCALE(-1.389056415E-04), -SCALE(-3.734139318E-04),
    -SCALE(-5.943773431E-04), -SCALE(-8.017504588E-04),
    -SCALE(-9.955273708E-04), -SCALE(-1.175740734E-03),
    -SCALE(-1.342460280E-03), -SCALE(-1.495792647E-03),
     SCALE( 1.635878929E-03),  SCALE( 1.762894331E-03),
     SCALE( 1.877046190E-03),  SCALE( 1.978572691E-03),
     SCALE( 2.067740774E-03),  SCALE( 2.144844970E-03),
     SCALE( 2.210205887E-03),  SCALE( 2.264167881E-03),
     SCALE( 2.307097195E-03),  SCALE( 2.339380793E-03),
     SCALE( 2.361423569E-03),  SCALE( 2.373647178E-03),
     SCALE( 2.376487479E-03),  SCALE( 2.370394068E-03),
     SCALE( 2.355825622E-03),  SCALE( 2.333251061E-03),
    -SCALE( 2.303145360E-03), -SCALE( 2.265989315E-03),
    -SCALE( 2.222266514E-03), -SCALE( 2.172462177E-03),
    -SCALE( 2.117061289E-03), -SCALE( 2.056547208E-03),
    -SCALE( 1.991399564E-03), -SCALE( 1.922092517E-03),
    -SCALE( 1.849094522E-03), -SCALE( 1.772865304E-03),
    -SCALE( 1.693855622E-03), -SCALE( 1.612505526E-03),
    -SCALE( 1.529243193E-03), -SCALE( 1.444484224E-03),
    -SCALE( 1.358630019E-03), -SCALE( 1.272067428E-03),
    -SCALE( 1.185167348E-03), -SCALE( 1.098284614E-03),
    -SCALE( 1.011756598E-03), -SCALE( 9.259034996E-04),
    -SCALE( 8.410271257E-04), -SCALE( 7.574109477E-04),
    -SCALE( 6.753197522E-04), -SCALE( 5.949990009E-04),
    -SCALE( 5.166754709E-04), -SCALE( 4.405563814E-04),
    -SCALE( 3.668301215E-04), -SCALE( 2.956659591E-04),
    -SCALE( 2.272142592E-04), -SCALE( 1.616069785E-04),
    -SCALE( 9.895755647E-05), -SCALE( 3.936182839E-05),
    -SCALE(-1.710198012E-05), -SCALE(-7.037253090E-05),
    -SCALE(-1.204049113E-04), -SCALE(-1.671699720E-04),
    -SCALE(-2.106537577E-04), -SCALE(-2.508567995E-04),
    -SCALE(-2.877934603E-04), -SCALE(-3.214911267E-04),
    -SCALE(-3.519894381E-04), -SCALE(-3.793396754E-04),
    -SCALE(-4.036037717E-04), -SCALE(-4.248536134E-04),
    -SCALE(-4.431701091E-04), -SCALE(-4.586426076E-04),
    -SCALE(-4.713678791E-04), -SCALE(-4.814492422E-04),
     SCALE( 4.889960401E-04),  SCALE( 4.941228544E-04),
     SCALE( 4.969481961E-04),  SCALE( 4.975944757E-04),
     SCALE( 4.961824161E-04),  SCALE( 4.928477574E-04),
     SCALE( 4.877146275E-04),  SCALE( 4.809123348E-04),
     SCALE( 4.725702747E-04),  SCALE( 4.628172028E-04),
     SCALE( 4.517810594E-04),  SCALE( 4.395879805E-04),
     SCALE( 4.263620067E-04),  SCALE( 4.122247046E-04),
     SCALE( 3.972945851E-04),  SCALE( 3.816867538E-04),
    -SCALE( 3.655125911E-04), -SCALE( 3.488793736E-04),
    -SCALE( 3.318900708E-04), -SCALE( 3.146430245E-04),
    -SCALE( 2.972317743E-04), -SCALE( 2.797449124E-04),
    -SCALE( 2.622658503E-04), -SCALE( 2.448728774E-04),
    -SCALE( 2.276388550E-04), -SCALE( 2.106313768E-04),
    -SCALE( 1.939126523E-04), -SCALE( 1.775395358E-04),
    -SCALE( 1.615635992E-04), -SCALE( 1.460311323E-04),
    -SCALE( 1.309833024E-04), -SCALE( 1.164562127E-04),
    -SCALE( 1.024810626E-04), -SCALE( 8.908427117E-05),
    -SCALE( 7.628766616E-05), -SCALE( 6.410864444E-05),
    -SCALE( 5.256036457E-05), -SCALE( 4.165195787E-05),
    -SCALE( 3.138873581E-05), -SCALE( 2.177240640E-05),
    -SCALE( 1.280130618E-05), -SCALE( 4.470633485E-06),
    -SCALE(-3.227306706E-06), -SCALE(-1.030297699E-05),
    -SCALE(-1.676913780E-05), -SCALE(-2.264085742E-05),
    -SCALE(-2.793515523E-05), -SCALE(-3.267079956E-05),
    -SCALE(-3.686808850E-05), -SCALE(-4.054862075E-05),
    -SCALE(-4.373511547E-05), -SCALE(-4.645117951E-05),
    -SCALE(-4.872112549E-05), -SCALE(-5.056979353E-05),
    -SCALE(-5.202236207E-05), -SCALE(-5.310418419E-05),
    -SCALE(-5.384063843E-05), -SCALE(-5.425697600E-05),
    -SCALE(-5.437819709E-05), -SCALE(-5.422891263E-05),
    -SCALE(-5.383323878E-05), -SCALE(-5.321469871E-05),
    -SCALE(-5.239612074E-05), -SCALE(-5.139957648E-05),
     SCALE( 5.024627535E-05),  SCALE( 4.895655002E-05),
     SCALE( 4.754976908E-05),  SCALE( 4.604430433E-05),
     SCALE( 4.445751256E-05),  SCALE( 4.280570283E-05),
     SCALE( 4.110412556E-05),  SCALE( 3.936696885E-05),
     SCALE( 3.760734762E-05),  SCALE( 3.583733633E-05),
     SCALE( 3.406793985E-05),  SCALE( 3.230916263E-05),
     SCALE( 3.056998685E-05),  SCALE( 2.885844333E-05),
     SCALE( 2.718161704E-05),  SCALE( 2.554569073E-05),
    -SCALE( 2.395598858E-05), -SCALE( 2.241701623E-05),
    -SCALE( 2.093250441E-05), -SCALE( 1.950545993E-05),
    -SCALE( 1.813820381E-05), -SCALE( 1.683242772E-05),
    -SCALE( 1.558924305E-05), -SCALE( 1.440921824E-05),
    -SCALE( 1.329243969E-05), -SCALE( 1.223855270E-05),
    -SCALE( 1.124680875E-05), -SCALE( 1.031611009E-05),
    -SCALE( 9.445050637E-06), -SCALE( 8.631964192E-06),
    -SCALE( 7.874960829E-06), -SCALE( 7.171964626E-06),
    -SCALE( 6.520755960E-06), -SCALE( 5.918994248E-06),
    -SCALE( 5.364252502E-06), -SCALE( 4.854050530E-06),
    -SCALE( 4.385879038E-06), -SCALE( 3.957220997E-06),
    -SCALE( 3.565570978E-06), -SCALE( 3.208459020E-06),
    -SCALE( 2.883470643E-06), -SCALE( 2.588257530E-06),
    -SCALE( 2.320550948E-06), -SCALE( 2.078171747E-06),
    -SCALE( 1.859034001E-06), -SCALE( 1.661159786E-06),
    -SCALE( 1.482681114E-06), -SCALE( 1.321840614E-06),
    -SCALE( 1.176999604E-06), -SCALE( 1.046637067E-06),
    -SCALE( 9.293416952E-07), -SCALE( 8.238164355E-07),
    -SCALE( 7.288739425E-07), -SCALE( 6.434325428E-07),
    -SCALE( 5.665120852E-07), -SCALE( 4.972276315E-07),
    -SCALE( 4.347855338E-07), -SCALE( 3.784752209E-07),
    -SCALE( 3.276689142E-07), -SCALE( 2.818143514E-07),
    -SCALE( 2.404238444E-07), -SCALE( 2.030677564E-07),
    -SCALE( 1.693738625E-07), -SCALE( 1.390191784E-07)
};

#undef SCALE

// Annex D.8 - LFE interpolation FIR, 64x interpolation
static const double lfe_fir_64[256] = {
    2.658434387E-04, 9.029330686E-03, 7.939263433E-02, 2.425158769E-01,
    3.430179358E-01, 2.398228943E-01, 7.746443897E-02, 8.622321300E-03,
    8.179365250E-05, 9.450953454E-03, 8.134882897E-02, 2.451938838E-01,
    3.429597318E-01, 2.371159792E-01, 7.556436211E-02, 8.229630999E-03,
    9.439323912E-05, 9.887560271E-03, 8.333285898E-02, 2.478559017E-01,
    3.428434134E-01, 2.343961596E-01, 7.369252294E-02, 7.850865833E-03,
    1.082170274E-04, 1.033949479E-02, 8.534456789E-02, 2.505008876E-01,
    3.426689506E-01, 2.316644788E-01, 7.184901088E-02, 7.485736627E-03,
    1.233371440E-04, 1.080708485E-02, 8.738376945E-02, 2.531278133E-01,
    3.424364924E-01, 2.289219648E-01, 7.003392279E-02, 7.133882027E-03,
    1.397485757E-04, 1.129068248E-02, 8.945026249E-02, 2.557355762E-01,
    3.421461284E-01, 2.261696160E-01, 6.824731827E-02, 6.794991903E-03,
    1.575958013E-04, 1.179065090E-02, 9.154383838E-02, 2.583232224E-01,
    3.417979777E-01, 2.234084606E-01, 6.648923457E-02, 6.468691397E-03,
    1.769922383E-04, 1.230732165E-02, 9.366425127E-02, 2.608896792E-01,
    3.413922191E-01, 2.206395119E-01, 6.475970894E-02, 6.154712290E-03,
    1.981738606E-04, 1.284105983E-02, 9.581124038E-02, 2.634339035E-01,
    3.409290314E-01, 2.178637981E-01, 6.305878609E-02, 5.852684379E-03,
    2.211847313E-04, 1.339218579E-02, 9.798453748E-02, 2.659549415E-01,
    3.404086530E-01, 2.150822729E-01, 6.138643622E-02, 5.562345497E-03,
    2.460231190E-04, 1.396108977E-02, 1.001838669E-01, 2.684516609E-01,
    3.398312926E-01, 2.122959495E-01, 5.974265561E-02, 5.283284001E-03,
    2.726115927E-04, 1.454808749E-02, 1.024089083E-01, 2.709231377E-01,
    3.391972482E-01, 2.095058411E-01, 5.812742189E-02, 5.015311297E-03,
    3.013863170E-04, 1.515355054E-02, 1.046593264E-01, 2.733682692E-01,
    3.385068178E-01, 2.067128718E-01, 5.654069409E-02, 4.758012015E-03,
    3.328395542E-04, 1.577781141E-02, 1.069347933E-01, 2.757860720E-01,
    3.377602994E-01, 2.039180547E-01, 5.498242006E-02, 4.511159845E-03,
    3.658991191E-04, 1.642123051E-02, 1.092349365E-01, 2.781755328E-01,
    3.369580209E-01, 2.011223286E-01, 5.345252529E-02, 4.274417181E-03,
    4.018281470E-04, 1.708412915E-02, 1.115593687E-01, 2.805356979E-01,
    3.361004293E-01, 1.983266175E-01, 5.195093155E-02, 4.047499038E-03,
    4.401875485E-04, 1.776690222E-02, 1.139076948E-01, 2.828655839E-01,
    3.351879120E-01, 1.955319196E-01, 5.047753453E-02, 3.830091329E-03,
    4.812776169E-04, 1.846982725E-02, 1.162794977E-01, 2.851640880E-01,
    3.342207968E-01, 1.927391142E-01, 4.903224111E-02, 3.621967277E-03,
    5.252459669E-04, 1.919330470E-02, 1.186743453E-01, 2.874303460E-01,
    3.331996202E-01, 1.899491698E-01, 4.761491716E-02, 3.422776936E-03,
    5.721592461E-04, 1.993762329E-02, 1.210917681E-01, 2.896633744E-01,
    3.321248591E-01, 1.871629506E-01, 4.622544348E-02, 3.232272575E-03,
    6.222130032E-04, 2.070316114E-02, 1.235313043E-01, 2.918621898E-01,
    3.309969604E-01, 1.843813360E-01, 4.486365616E-02, 3.050152911E-03,
    6.755515351E-04, 2.149021253E-02, 1.259924471E-01, 2.940258980E-01,
    3.298164308E-01, 1.816052496E-01, 4.352942482E-02, 2.876190469E-03,
    7.324148901E-04, 2.229913883E-02, 1.284746826E-01, 2.961534858E-01,
    3.285838962E-01, 1.788355410E-01, 4.222255200E-02, 2.710093278E-03,
    7.928516716E-04, 2.313023806E-02, 1.309774816E-01, 2.982441187E-01,
    3.272998929E-01, 1.760730892E-01, 4.094288871E-02, 2.551567042E-03,
    8.570110658E-04, 2.398385666E-02, 1.335003078E-01, 3.002967536E-01,
    3.259649575E-01, 1.733186990E-01, 3.969023004E-02, 2.400433412E-03,
    9.251192096E-04, 2.486028522E-02, 1.360425949E-01, 3.023106754E-01,
    3.245797157E-01, 1.705732346E-01, 3.846437484E-02, 2.256359672E-03,
    9.974770946E-04, 2.575986087E-02, 1.386037618E-01, 3.042849004E-01,
    3.231448531E-01, 1.678375006E-01, 3.726511076E-02, 2.119151875E-03,
    1.073930296E-03, 2.668286115E-02, 1.411831975E-01, 3.062185347E-01,
    3.216609657E-01, 1.651122719E-01, 3.609224036E-02, 1.988604199E-03,
    1.155023579E-03, 2.762960829E-02, 1.437802613E-01, 3.081108034E-01,
    3.201287389E-01, 1.623983532E-01, 3.494550660E-02, 1.864377526E-03,
    1.240676851E-03, 2.860039286E-02, 1.463943720E-01, 3.099608123E-01,
    3.185488880E-01, 1.596965194E-01, 3.382468969E-02, 1.746327500E-03,
    1.331258914E-03, 2.959549613E-02, 1.490248144E-01, 3.117676973E-01,
    3.169221282E-01, 1.570075154E-01, 3.272953629E-02, 1.634211512E-03,
    1.426893868E-03, 3.061520495E-02, 1.516709626E-01, 3.135308027E-01,
    3.152491748E-01, 1.543320864E-01, 3.165979683E-02, 1.527829794E-03
};

// Annex D.8 - LFE interpolation FIR, 128x interpolation
static const double lfe_fir_128[256] = {
    5.316857100E-04, 1.587849557E-01, 6.860344410E-01, 1.549285650E-01,
    1.635869100E-04, 1.626973301E-01, 6.859180331E-01, 1.511284113E-01,
    1.887860900E-04, 1.666653752E-01, 6.856853962E-01, 1.473847479E-01,
    2.164336300E-04, 1.706887931E-01, 6.853365302E-01, 1.436977387E-01,
    2.466738200E-04, 1.747671962E-01, 6.848715544E-01, 1.400675476E-01,
    2.794966000E-04, 1.789001823E-01, 6.842908263E-01, 1.364943385E-01,
    3.151909600E-04, 1.830873191E-01, 6.835945249E-01, 1.329781860E-01,
    3.539837500E-04, 1.873281151E-01, 6.827830076E-01, 1.295191795E-01,
    3.963469100E-04, 1.916220933E-01, 6.818566918E-01, 1.261173040E-01,
    4.423685900E-04, 1.959686577E-01, 6.808158755E-01, 1.227726117E-01,
    4.920452500E-04, 2.003673166E-01, 6.796611548E-01, 1.194850579E-01,
    5.452220800E-04, 2.048173845E-01, 6.783930659E-01, 1.162546203E-01,
    6.027714100E-04, 2.093182206E-01, 6.770122051E-01, 1.130811572E-01,
    6.656776500E-04, 2.138691545E-01, 6.755192280E-01, 1.099646092E-01,
    7.317967800E-04, 2.184694260E-01, 6.739146709E-01, 1.069048345E-01,
    8.036546600E-04, 2.231182903E-01, 6.721994877E-01, 1.039016470E-01,
    8.803732300E-04, 2.278149277E-01, 6.703743935E-01, 1.009548605E-01,
    9.625531400E-04, 2.325585187E-01, 6.684402227E-01, 9.806428105E-02,
    1.050489840E-03, 2.373482138E-01, 6.663978696E-01, 9.522963315E-02,
    1.144316160E-03, 2.421830446E-01, 6.642482877E-01, 9.245070815E-02,
    1.244423330E-03, 2.470620573E-01, 6.619924903E-01, 8.972713351E-02,
    1.351100280E-03, 2.519843280E-01, 6.596315503E-01, 8.705867827E-02,
    1.464826870E-03, 2.569487989E-01, 6.571664810E-01, 8.444493264E-02,
    1.585700080E-03, 2.619544268E-01, 6.545983553E-01, 8.188561350E-02,
    1.714018640E-03, 2.670000792E-01, 6.519285440E-01, 7.938029617E-02,
    1.850234690E-03, 2.720846236E-01, 6.491580606E-01, 7.692859322E-02,
    1.994950230E-03, 2.772069275E-01, 6.462883353E-01, 7.453006506E-02,
    2.147856400E-03, 2.823657692E-01, 6.433205605E-01, 7.218432426E-02,
    2.310042500E-03, 2.875599265E-01, 6.402561665E-01, 6.989086419E-02,
    2.481348810E-03, 2.927881181E-01, 6.370964646E-01, 6.764923781E-02,
    2.662512240E-03, 2.980490029E-01, 6.338429451E-01, 6.545893103E-02,
    2.853781920E-03, 3.033412695E-01, 6.304970384E-01, 6.331945211E-02,
    3.055653300E-03, 3.086635172E-01, 6.270602942E-01, 6.123027951E-02,
    3.268416510E-03, 3.140144050E-01, 6.235341430E-01, 5.919086933E-02,
    3.492647550E-03, 3.193923831E-01, 6.199202538E-01, 5.720067024E-02,
    3.728747140E-03, 3.247960210E-01, 6.162202954E-01, 5.525910854E-02,
    3.977200480E-03, 3.302238286E-01, 6.124358177E-01, 5.336561054E-02,
    4.238294900E-03, 3.356742859E-01, 6.085684896E-01, 5.151961371E-02,
    4.512710030E-03, 3.411457539E-01, 6.046201587E-01, 4.972046614E-02,
    4.800856580E-03, 3.466366828E-01, 6.005923152E-01, 4.796761274E-02,
    5.103122910E-03, 3.521454632E-01, 5.964869261E-01, 4.626038298E-02,
    5.420174920E-03, 3.576703668E-01, 5.923057795E-01, 4.459818453E-02,
    5.752369300E-03, 3.632097244E-01, 5.880505443E-01, 4.298033938E-02,
    6.100293250E-03, 3.687619269E-01, 5.837231875E-01, 4.140623659E-02,
    6.464532110E-03, 3.743250966E-01, 5.793255568E-01, 3.987516090E-02,
    6.845539900E-03, 3.798975349E-01, 5.748594403E-01, 3.838652745E-02,
    7.243919190E-03, 3.854774535E-01, 5.703269839E-01, 3.693958372E-02,
    7.660165890E-03, 3.910630047E-01, 5.657299161E-01, 3.553372994E-02,
    8.094980380E-03, 3.966524303E-01, 5.610702634E-01, 3.416819125E-02,
    8.548815730E-03, 4.022437930E-01, 5.563499928E-01, 3.284239396E-02,
    9.022301060E-03, 4.078352153E-01, 5.515710115E-01, 3.155555204E-02,
    9.516004470E-03, 4.134248793E-01, 5.467353463E-01, 3.030703776E-02,
    1.003060210E-02, 4.190107882E-01, 5.418450832E-01, 2.909611352E-02,
    1.056654565E-02, 4.245910645E-01, 5.369022489E-01, 2.792212367E-02,
    1.112466771E-02, 4.301636219E-01, 5.319088101E-01, 2.678431384E-02,
    1.170534454E-02, 4.357266724E-01, 5.268667936E-01, 2.568206564E-02,
    1.230939943E-02, 4.412781000E-01, 5.217782855E-01, 2.461459488E-02,
    1.293735672E-02, 4.468160272E-01, 5.166453719E-01, 2.358125709E-02,
    1.358995494E-02, 4.523383081E-01, 5.114701390E-01, 2.258131653E-02,
    1.426773332E-02, 4.578429461E-01, 5.062545538E-01, 2.161412500E-02,
    1.497144438E-02, 4.633280039E-01, 5.010007620E-01, 2.067894675E-02,
    1.570170000E-02, 4.687913656E-01, 4.957108200E-01, 1.977507770E-02,
    1.645922661E-02, 4.742309451E-01, 4.903867543E-01, 1.890186779E-02,
    1.724460535E-02, 4.796448052E-01, 4.850307405E-01, 1.805862412E-02
};

#define SCALE(x)    ((x) * M_SQRT2 * 256)

// Annex D.9 - 1024 tap FIR for X96 synthesis QMF
static const double band_fir_x96[1024] = {
     SCALE(-7.127938986604168E-08),  SCALE(-7.095090315087499E-08),
     SCALE(-7.952503432137510E-08),  SCALE(-8.932687128137479E-08),
     SCALE(-9.748319049487465E-08),  SCALE(-1.058665951072195E-07),
     SCALE(-1.152736341186576E-07),  SCALE(-1.252310964561535E-07),
     SCALE(-1.355563613797138E-07),  SCALE(-1.464093182119164E-07),
     SCALE(-1.578742499812371E-07),  SCALE(-1.699459292657122E-07),
     SCALE(-1.826418431101815E-07),  SCALE(-1.960059458775785E-07),
     SCALE(-2.100789970600680E-07),  SCALE(-2.248975904755362E-07),
     SCALE(-2.405024587792340E-07),  SCALE(-2.569387461577766E-07),
     SCALE(-2.742537030854958E-07),  SCALE(-2.924966254175760E-07),
     SCALE(-3.117198196826890E-07),  SCALE(-3.319786404047318E-07),
     SCALE(-3.533313458884360E-07),  SCALE(-3.758391046541218E-07),
     SCALE(-3.995662812166919E-07),  SCALE(-4.245804529615933E-07),
     SCALE(-4.509524956369838E-07),  SCALE(-4.787565911260676E-07),
     SCALE(-5.080703129463105E-07),  SCALE(-5.389746422333555E-07),
     SCALE(-5.715542338879300E-07),  SCALE(-6.058974119201980E-07),
    -SCALE(-6.420958008785305E-07), -SCALE(-6.802443284999140E-07),
    -SCALE(-7.204416525901345E-07), -SCALE(-7.627901817207720E-07),
    -SCALE(-8.073961036828060E-07), -SCALE(-8.543693702029365E-07),
    -SCALE(-9.038233322489550E-07), -SCALE(-9.558747254049829E-07),
    -SCALE(-1.010643712821376E-06), -SCALE(-1.068253890614633E-06),
    -SCALE(-1.128832363715543E-06), -SCALE(-1.192509718464610E-06),
    -SCALE(-1.259419343863326E-06), -SCALE(-1.329697405230052E-06),
    -SCALE(-1.403482943378026E-06), -SCALE(-1.480917863497994E-06),
    -SCALE(-1.562146591398006E-06), -SCALE(-1.647316037754536E-06),
    -SCALE(-1.736575155397134E-06), -SCALE(-1.830074910452612E-06),
    -SCALE(-1.927968090232205E-06), -SCALE(-2.030409266099076E-06),
    -SCALE(-2.137554168666232E-06), -SCALE(-2.249559625377646E-06),
    -SCALE(-2.366582812095056E-06), -SCALE(-2.488781197085654E-06),
    -SCALE(-2.616312057164482E-06), -SCALE(-2.749332426639114E-06),
    -SCALE(-2.887998457950138E-06), -SCALE(-3.032465363920388E-06),
    -SCALE(-3.182886786597464E-06), -SCALE(-3.339414724085582E-06),
    -SCALE(-3.502198552003754E-06), -SCALE(-3.671384926229468E-06),
    -SCALE(-3.847116621801583E-06), -SCALE(-4.029532465769980E-06),
    -SCALE(-4.218767020985225E-06), -SCALE(-4.414950504331232E-06),
    -SCALE(-4.618207304054390E-06), -SCALE(-4.828655865889468E-06),
    -SCALE(-5.046407670500340E-06), -SCALE(-5.271567145540585E-06),
    -SCALE(-5.504230753545810E-06), -SCALE(-5.744486878502875E-06),
    -SCALE(-5.992414242480480E-06), -SCALE(-6.248081786307280E-06),
    -SCALE(-6.511547627860400E-06), -SCALE(-6.782858956580495E-06),
    -SCALE(-7.062050754515410E-06), -SCALE(-7.349145669714490E-06),
    -SCALE(-7.644152509859200E-06), -SCALE(-7.947066136436549E-06),
    -SCALE(-8.257866642922776E-06), -SCALE(-8.576519215476810E-06),
    -SCALE(-8.902971890012200E-06), -SCALE(-9.237155419546841E-06),
    -SCALE(-9.578982598825311E-06), -SCALE(-9.928348166726454E-06),
    -SCALE(-1.028512733460845E-05), -SCALE(-1.064917563432859E-05),
    -SCALE(-1.102032704633465E-05), -SCALE(-1.139839388088106E-05),
    -SCALE(-1.178316603669356E-05), -SCALE(-1.217441088925653E-05),
    -SCALE(-1.257187157446922E-05), -SCALE(-1.297526685803031E-05),
    -SCALE(-1.338428997824206E-05), -SCALE(-1.379860854096596E-05),
    -SCALE(-1.421786336584378E-05), -SCALE(-1.464166838349645E-05),
    -SCALE(-1.506960952767971E-05), -SCALE(-1.550124459778260E-05),
    -SCALE(-1.593610134164050E-05), -SCALE(-1.637367742646690E-05),
    -SCALE(-1.681344171653956E-05), -SCALE(-1.725483414306857E-05),
    -SCALE(-1.769726156360750E-05), -SCALE(-1.814009771461820E-05),
    -SCALE(-1.858268630848355E-05), -SCALE(-1.902434097009924E-05),
    -SCALE(-1.946434074488017E-05), -SCALE(-1.990193003014108E-05),
    -SCALE(-2.033632155746451E-05), -SCALE(-2.076669641777394E-05),
    -SCALE(-2.119220163057428E-05), -SCALE(-2.161195008423191E-05),
    -SCALE(-2.202502165289463E-05), -SCALE(-2.243046323951717E-05),
    -SCALE(-2.282728860550347E-05), -SCALE(-2.321447837191817E-05),
    -SCALE(-2.359098021649775E-05), -SCALE(-2.395570893976328E-05),
    -SCALE(-2.430754772248991E-05), -SCALE(-2.464534820090372E-05),
    -SCALE(-2.496793086354262E-05), -SCALE(-2.527408508232496E-05),
     SCALE( 2.556256939919902E-05),  SCALE( 2.583211172456204E-05),
     SCALE( 2.608141341583320E-05),  SCALE( 2.630914939052064E-05),
     SCALE( 2.651396653512446E-05),  SCALE( 2.669448385046480E-05),
     SCALE( 2.684929723999712E-05),  SCALE( 2.697697975324517E-05),
     SCALE( 2.707608215301207E-05),  SCALE( 2.714513310366389E-05),
     SCALE( 2.718264274615772E-05),  SCALE( 2.718710298053676E-05),
     SCALE( 2.715699010579455E-05),  SCALE( 2.709076506473594E-05),
     SCALE( 2.698687619105470E-05),  SCALE( 2.684375953903512E-05),
     SCALE( 2.665984338990853E-05),  SCALE( 2.643354859248296E-05),
     SCALE( 2.616329155063932E-05),  SCALE( 2.584748457743932E-05),
     SCALE( 2.548454069800296E-05),  SCALE( 2.507287405244726E-05),
     SCALE( 2.461090395766135E-05),  SCALE( 2.409705530058412E-05),
     SCALE( 2.352976312853915E-05),  SCALE( 2.290747311740374E-05),
     SCALE( 2.222864728099456E-05),  SCALE( 2.149176446183214E-05),
     SCALE( 2.069532541857637E-05),  SCALE( 1.983785326550204E-05),
     SCALE( 1.891789805352808E-05),  SCALE( 1.793403733553470E-05),
    -SCALE( 1.688488402231131E-05), -SCALE( 1.576908692157790E-05),
    -SCALE( 1.458533474050792E-05), -SCALE( 1.333235659026308E-05),
    -SCALE( 1.200892908323512E-05), -SCALE( 1.061387697354165E-05),
    -SCALE( 9.146080150692231E-06), -SCALE( 7.604474075726550E-06),
    -SCALE( 5.988052382589700E-06), -SCALE( 4.295867784770884E-06),
    -SCALE( 2.527049419980207E-06), -SCALE( 6.808031538695990E-07),
    -SCALE(-1.243599082783966E-06), -SCALE(-3.246818706694593E-06),
    -SCALE(-5.329426896805280E-06), -SCALE(-7.491903655942790E-06),
    -SCALE(-9.734640752487110E-06), -SCALE(-1.205794140863581E-05),
    -SCALE(-1.446201050055097E-05), -SCALE(-1.694695384810771E-05),
    -SCALE(-1.951277238921524E-05), -SCALE(-2.215936159729271E-05),
    -SCALE(-2.488650449141616E-05), -SCALE(-2.769387104186206E-05),
    -SCALE(-3.058101208191696E-05), -SCALE(-3.354735876525904E-05),
    -SCALE(-3.659221671654097E-05), -SCALE(-3.971476548553926E-05),
    -SCALE(-4.291405238785664E-05), -SCALE(-4.618899196172456E-05),
    -SCALE(-4.953836017695434E-05), -SCALE(-5.296079400014060E-05),
    -SCALE(-5.645478762018535E-05), -SCALE(-6.001869197279105E-05),
    -SCALE(-6.365070805378855E-05), -SCALE(-6.734888652588270E-05),
    -SCALE(-7.111112575372286E-05), -SCALE(-7.493517141117804E-05),
    -SCALE(-7.881860980632000E-05), -SCALE(-8.275886762862380E-05),
    -SCALE(-8.675321306240340E-05), -SCALE(-9.079875548213924E-05),
    -SCALE(-9.489243763625400E-05), -SCALE(-9.903103541547551E-05),
    -SCALE(-1.032111605730492E-04), -SCALE(-1.074292607067401E-04),
    -SCALE(-1.116816161426427E-04), -SCALE(-1.159643396677203E-04),
    -SCALE(-1.202733737618488E-04), -SCALE(-1.246044908903475E-04),
    -SCALE(-1.289533027076185E-04), -SCALE(-1.333152598388584E-04),
    -SCALE(-1.376856378740834E-04), -SCALE(-1.420595376057812E-04),
    -SCALE(-1.464319042632354E-04), -SCALE(-1.507975279169229E-04),
    -SCALE(-1.551510331409437E-04), -SCALE(-1.594868790905897E-04),
    -SCALE(-1.637993715464274E-04), -SCALE(-1.680826635644682E-04),
    -SCALE(-1.723307577358468E-04), -SCALE(-1.765375064656002E-04),
    -SCALE(-1.806966158461229E-04), -SCALE(-1.848016463895374E-04),
    -SCALE(-1.888460252663805E-04), -SCALE(-1.928230469653129E-04),
    -SCALE(-1.967258755680427E-04), -SCALE(-2.005475454710861E-04),
    -SCALE(-2.042809749895736E-04), -SCALE(-2.079189673565927E-04),
    -SCALE(-2.114542191049128E-04), -SCALE(-2.148793208048870E-04),
    -SCALE(-2.181867649165566E-04), -SCALE(-2.213689470691719E-04),
    -SCALE(-2.244181863585484E-04), -SCALE(-2.273267267668629E-04),
    -SCALE(-2.300867481008122E-04), -SCALE(-2.326903665352515E-04),
    -SCALE(-2.351296356350780E-04), -SCALE(-2.373965479929124E-04),
    -SCALE(-2.394830702362197E-04), -SCALE(-2.413811459337238E-04),
    -SCALE(-2.430827245294182E-04), -SCALE(-2.445797571768798E-04),
    -SCALE(-2.458640761601700E-04), -SCALE(-2.469274156993822E-04),
    -SCALE(-2.477619902517768E-04), -SCALE(-2.483604609263728E-04),
    -SCALE(-2.487146183038166E-04), -SCALE(-2.488154045724704E-04),
    -SCALE(-2.486547177024881E-04), -SCALE(-2.482254333360894E-04),
    -SCALE(-2.475200821995122E-04), -SCALE(-2.465308179417657E-04),
    -SCALE(-2.452500321752444E-04), -SCALE(-2.436703743734102E-04),
     SCALE( 2.417845745791980E-04),  SCALE( 2.395854398583434E-04),
     SCALE( 2.370659535642138E-04),  SCALE( 2.342192800869662E-04),
     SCALE( 2.310387700701966E-04),  SCALE( 2.275179613297294E-04),
     SCALE( 2.236505938503120E-04),  SCALE( 2.194306115956788E-04),
     SCALE( 2.148521873332429E-04),  SCALE( 2.099097253875555E-04),
     SCALE( 2.045978973917164E-04),  SCALE( 1.989116442431618E-04),
     SCALE( 1.928461833795790E-04),  SCALE( 1.863970107085474E-04),
     SCALE( 1.795599357837175E-04),  SCALE( 1.723310839515549E-04),
     SCALE( 1.647069084035198E-04),  SCALE( 1.566841923938267E-04),
     SCALE( 1.482600859779846E-04),  SCALE( 1.394321081840759E-04),
     SCALE( 1.301981580423545E-04),  SCALE( 1.205565161870574E-04),
     SCALE( 1.105058690666598E-04),  SCALE( 1.000453111637885E-04),
     SCALE( 8.917436962303755E-05),  SCALE( 7.789300610544491E-05),
     SCALE( 6.620163295915880E-05),  SCALE( 5.410111500896320E-05),
     SCALE( 4.159279275517104E-05),  SCALE( 2.867848400338988E-05),
     SCALE( 1.536049652046955E-05),  SCALE( 1.641629701930447E-06),
    -SCALE(-1.247480140056488E-05), -SCALE(-2.698496475767416E-05),
    -SCALE(-4.188450631602256E-05), -SCALE(-5.716854816951505E-05),
    -SCALE(-7.283165337456365E-05), -SCALE(-8.886782533520320E-05),
    -SCALE(-1.052705294604104E-04), -SCALE(-1.220326918528753E-04),
    -SCALE(-1.391466488046540E-04), -SCALE(-1.566041455665768E-04),
    -SCALE(-1.743963597795672E-04), -SCALE(-1.925139014261300E-04),
    -SCALE(-2.109467883317648E-04), -SCALE(-2.296844445765594E-04),
    -SCALE(-2.487156900488424E-04), -SCALE(-2.680287412777888E-04),
    -SCALE(-2.876112401970844E-04), -SCALE(-3.074502529716603E-04),
    -SCALE(-3.275322154221950E-04), -SCALE(-3.478429334738513E-04),
    -SCALE(-3.683676476039790E-04), -SCALE(-3.890910333571532E-04),
    -SCALE(-4.099971482282953E-04), -SCALE(-4.310694309740984E-04),
    -SCALE(-4.522907395812775E-04), -SCALE(-4.736433535042661E-04),
    -SCALE(-4.951089849262823E-04), -SCALE(-5.166687782268060E-04),
    -SCALE(-5.383032870049990E-04), -SCALE(-5.599924762541345E-04),
    -SCALE(-5.817157931755955E-04), -SCALE(-6.034521681004050E-04),
    -SCALE(-6.251799639488680E-04), -SCALE(-6.468769776333065E-04),
    -SCALE(-6.685205214586190E-04), -SCALE(-6.900874252648055E-04),
    -SCALE(-7.115540021619675E-04), -SCALE(-7.328960503543460E-04),
    -SCALE(-7.540889275341360E-04), -SCALE(-7.751075532553095E-04),
    -SCALE(-7.959263867639895E-04), -SCALE(-8.165194293666030E-04),
    -SCALE(-8.368602986989685E-04), -SCALE(-8.569222323017155E-04),
    -SCALE(-8.766780920104490E-04), -SCALE(-8.961003654734095E-04),
    -SCALE(-9.151611951498600E-04), -SCALE(-9.338323832499584E-04),
    -SCALE(-9.520854714135470E-04), -SCALE(-9.698917421439440E-04),
    -SCALE(-9.872221706725139E-04), -SCALE(-1.004047429744579E-03),
    -SCALE(-1.020338043044835E-03), -SCALE(-1.036064390096491E-03),
    -SCALE(-1.051196660616898E-03), -SCALE(-1.065704855603833E-03),
    -SCALE(-1.079558856878201E-03), -SCALE(-1.092728433285052E-03),
    -SCALE(-1.105183307573424E-03), -SCALE(-1.116893160529398E-03),
    -SCALE(-1.127827655023416E-03), -SCALE(-1.137956440566105E-03),
    -SCALE(-1.147249229489164E-03), -SCALE(-1.155675800049931E-03),
    -SCALE(-1.163205988551180E-03), -SCALE(-1.169809696015420E-03),
    -SCALE(-1.175457042899868E-03), -SCALE(-1.180118372021201E-03),
    -SCALE(-1.183764158185287E-03), -SCALE(-1.186365015306024E-03),
    -SCALE(-1.187891943390233E-03), -SCALE(-1.188316331684390E-03),
    -SCALE(-1.187609780917844E-03), -SCALE(-1.185744107237668E-03),
    -SCALE(-1.182691606532144E-03), -SCALE(-1.178425062841502E-03),
    -SCALE(-1.172917669070769E-03), -SCALE(-1.166143029435388E-03),
    -SCALE(-1.158075292551968E-03), -SCALE(-1.148689156870073E-03),
    -SCALE(-1.137959857080735E-03), -SCALE(-1.125863169053070E-03),
    -SCALE(-1.112375532031342E-03), -SCALE(-1.097474054394210E-03),
    -SCALE(-1.081136518161085E-03), -SCALE(-1.063341381932592E-03),
    -SCALE(-1.044067841074319E-03), -SCALE(-1.023295833844173E-03),
    -SCALE(-1.001006116010387E-03), -SCALE(-9.771802661360279E-04),
    -SCALE(-9.518007272200706E-04), -SCALE(-9.248508094554660E-04),
    -SCALE(-8.963147092655810E-04), -SCALE(-8.661775139092965E-04),
    -SCALE(-8.344252837566670E-04), -SCALE(-8.010450565557860E-04),
     SCALE( 7.660248590280630E-04),  SCALE( 7.293537115784970E-04),
     SCALE( 6.910217203263935E-04),  SCALE( 6.510200794896010E-04),
     SCALE( 6.093410318069875E-04),  SCALE( 5.659778724459110E-04),
     SCALE( 5.209250745440945E-04),  SCALE( 4.741782930361297E-04),
     SCALE( 4.257343232947606E-04),  SCALE( 3.755911023931640E-04),
     SCALE( 3.237477782338944E-04),  SCALE( 2.702047136138704E-04),
     SCALE( 2.149635065251304E-04),  SCALE( 1.580269916310419E-04),
     SCALE( 9.939925244214647E-05),  SCALE( 3.908562327914961E-05),
     SCALE(-2.290727971715371E-05),  SCALE(-8.657155665857780E-05),
     SCALE(-1.518980125188441E-04),  SCALE(-2.188761566878072E-04),
     SCALE(-2.874942045551261E-04),  SCALE(-3.577390751948810E-04),
     SCALE(-4.295963388434640E-04),  SCALE(-5.030502185326690E-04),
     SCALE(-5.780836785397680E-04),  SCALE(-6.546784227970995E-04),
     SCALE(-7.328147714529330E-04),  SCALE(-8.124716609148709E-04),
     SCALE(-8.936267682438125E-04),  SCALE(-9.762565135043539E-04),
     SCALE(-1.060335987079630E-03),  SCALE(-1.145838949770837E-03),
    -SCALE(-1.232737907672045E-03), -SCALE(-1.321004112591985E-03),
    -SCALE(-1.410607496458753E-03), -SCALE(-1.501516675908098E-03),
    -SCALE(-1.593699118821068E-03), -SCALE(-1.687121144658436E-03),
    -SCALE(-1.781747765280413E-03), -SCALE(-1.877542686278472E-03),
    -SCALE(-1.974468495455718E-03), -SCALE(-2.072486671719865E-03),
    -SCALE(-2.171557592248277E-03), -SCALE(-2.271640530170194E-03),
    -SCALE(-2.372693596410548E-03), -SCALE(-2.474673741849360E-03),
    -SCALE(-2.577536862984530E-03), -SCALE(-2.681237815643306E-03),
    -SCALE(-2.785730610971646E-03), -SCALE(-2.890968412999295E-03),
    -SCALE(-2.996903289082014E-03), -SCALE(-3.103486213203266E-03),
    -SCALE(-3.210667388169048E-03), -SCALE(-3.318396256656389E-03),
    -SCALE(-3.426621422085545E-03), -SCALE(-3.535290652053989E-03),
    -SCALE(-3.644351033012211E-03), -SCALE(-3.753748970963593E-03),
    -SCALE(-3.863430052185924E-03), -SCALE(-3.973339058810391E-03),
    -SCALE(-4.083420450837847E-03), -SCALE(-4.193618368607346E-03),
    -SCALE(-4.303876205087690E-03), -SCALE(-4.414136605908662E-03),
    -SCALE(-4.524341897756710E-03), -SCALE(-4.634434107578317E-03),
    -SCALE(-4.744354956658913E-03), -SCALE(-4.854045856535885E-03),
    -SCALE(-4.963447825006078E-03), -SCALE(-5.072501497871895E-03),
    -SCALE(-5.181147471348025E-03), -SCALE(-5.289326317382795E-03),
    -SCALE(-5.396978578320565E-03), -SCALE(-5.504044759237560E-03),
    -SCALE(-5.610465164666585E-03), -SCALE(-5.716179914026345E-03),
    -SCALE(-5.821129444341225E-03), -SCALE(-5.925254521610190E-03),
    -SCALE(-6.028495988202895E-03), -SCALE(-6.130794760236025E-03),
    -SCALE(-6.232092022456570E-03), -SCALE(-6.332329249242580E-03),
    -SCALE(-6.431448471736440E-03), -SCALE(-6.529392276231145E-03),
    -SCALE(-6.626103501535170E-03), -SCALE(-6.721525246692210E-03),
    -SCALE(-6.815601343450995E-03), -SCALE(-6.908276363015645E-03),
    -SCALE(-6.999495292082720E-03), -SCALE(-7.089203549057705E-03),
    -SCALE(-7.177347664780630E-03), -SCALE(-7.263875283956655E-03),
    -SCALE(-7.348734515904140E-03), -SCALE(-7.431873934100050E-03),
    -SCALE(-7.513243215430925E-03), -SCALE(-7.592793162768140E-03),
    -SCALE(-7.670475562372150E-03), -SCALE(-7.746243173811765E-03),
    -SCALE(-7.820049650783600E-03), -SCALE(-7.891849558781685E-03),
    -SCALE(-7.961598843009079E-03), -SCALE(-8.029254840905250E-03),
    -SCALE(-8.094776089836320E-03), -SCALE(-8.158122302753560E-03),
    -SCALE(-8.219254024989381E-03), -SCALE(-8.278132675268024E-03),
    -SCALE(-8.334721791143661E-03), -SCALE(-8.388987020627474E-03),
    -SCALE(-8.440894692545369E-03), -SCALE(-8.490411808784576E-03),
    -SCALE(-8.537507303362359E-03), -SCALE(-8.582152067202975E-03),
    -SCALE(-8.624318234161430E-03), -SCALE(-8.663979162716030E-03),
    -SCALE(-8.701109747179411E-03), -SCALE(-8.735686465917280E-03),
    -SCALE(-8.767688130949044E-03), -SCALE(-8.797095815255481E-03),
    -SCALE(-8.823890503946285E-03), -SCALE(-8.848053147748315E-03),
    -SCALE(-8.869568188578319E-03), -SCALE(-8.888423592735640E-03),
    -SCALE(-8.904608055568560E-03), -SCALE(-8.918110931856005E-03),
    -SCALE(-8.928923499538055E-03), -SCALE(-8.937039001090346E-03),
    -SCALE(-8.942452290029211E-03), -SCALE(-8.945159822376361E-03),
     SCALE( 8.945159822376361E-03),  SCALE( 8.942452290029211E-03),
     SCALE( 8.937039001090346E-03),  SCALE( 8.928923499538055E-03),
     SCALE( 8.918110931856005E-03),  SCALE( 8.904608055568560E-03),
     SCALE( 8.888423592735640E-03),  SCALE( 8.869568188578319E-03),
     SCALE( 8.848053147748315E-03),  SCALE( 8.823890503946285E-03),
     SCALE( 8.797095815255481E-03),  SCALE( 8.767688130949044E-03),
     SCALE( 8.735686465917280E-03),  SCALE( 8.701109747179411E-03),
     SCALE( 8.663979162716030E-03),  SCALE( 8.624318234161430E-03),
     SCALE( 8.582152067202975E-03),  SCALE( 8.537507303362359E-03),
     SCALE( 8.490411808784576E-03),  SCALE( 8.440894692545369E-03),
     SCALE( 8.388987020627474E-03),  SCALE( 8.334721791143661E-03),
     SCALE( 8.278132675268024E-03),  SCALE( 8.219254024989381E-03),
     SCALE( 8.158122302753560E-03),  SCALE( 8.094776089836320E-03),
     SCALE( 8.029254840905250E-03),  SCALE( 7.961598843009079E-03),
     SCALE( 7.891849558781685E-03),  SCALE( 7.820049650783600E-03),
     SCALE( 7.746243173811765E-03),  SCALE( 7.670475562372150E-03),
    -SCALE( 7.592793162768140E-03), -SCALE( 7.513243215430925E-03),
    -SCALE( 7.431873934100050E-03), -SCALE( 7.348734515904140E-03),
    -SCALE( 7.263875283956655E-03), -SCALE( 7.177347664780630E-03),
    -SCALE( 7.089203549057705E-03), -SCALE( 6.999495292082720E-03),
    -SCALE( 6.908276363015645E-03), -SCALE( 6.815601343450995E-03),
    -SCALE( 6.721525246692210E-03), -SCALE( 6.626103501535170E-03),
    -SCALE( 6.529392276231145E-03), -SCALE( 6.431448471736440E-03),
    -SCALE( 6.332329249242580E-03), -SCALE( 6.232092022456570E-03),
    -SCALE( 6.130794760236025E-03), -SCALE( 6.028495988202895E-03),
    -SCALE( 5.925254521610190E-03), -SCALE( 5.821129444341225E-03),
    -SCALE( 5.716179914026345E-03), -SCALE( 5.610465164666585E-03),
    -SCALE( 5.504044759237560E-03), -SCALE( 5.396978578320565E-03),
    -SCALE( 5.289326317382795E-03), -SCALE( 5.181147471348025E-03),
    -SCALE( 5.072501497871895E-03), -SCALE( 4.963447825006078E-03),
    -SCALE( 4.854045856535885E-03), -SCALE( 4.744354956658913E-03),
    -SCALE( 4.634434107578317E-03), -SCALE( 4.524341897756710E-03),
    -SCALE( 4.414136605908662E-03), -SCALE( 4.303876205087690E-03),
    -SCALE( 4.193618368607346E-03), -SCALE( 4.083420450837847E-03),
    -SCALE( 3.973339058810391E-03), -SCALE( 3.863430052185924E-03),
    -SCALE( 3.753748970963593E-03), -SCALE( 3.644351033012211E-03),
    -SCALE( 3.535290652053989E-03), -SCALE( 3.426621422085545E-03),
    -SCALE( 3.318396256656389E-03), -SCALE( 3.210667388169048E-03),
    -SCALE( 3.103486213203266E-03), -SCALE( 2.996903289082014E-03),
    -SCALE( 2.890968412999295E-03), -SCALE( 2.785730610971646E-03),
    -SCALE( 2.681237815643306E-03), -SCALE( 2.577536862984530E-03),
    -SCALE( 2.474673741849360E-03), -SCALE( 2.372693596410548E-03),
    -SCALE( 2.271640530170194E-03), -SCALE( 2.171557592248277E-03),
    -SCALE( 2.072486671719865E-03), -SCALE( 1.974468495455718E-03),
    -SCALE( 1.877542686278472E-03), -SCALE( 1.781747765280413E-03),
    -SCALE( 1.687121144658436E-03), -SCALE( 1.593699118821068E-03),
    -SCALE( 1.501516675908098E-03), -SCALE( 1.410607496458753E-03),
    -SCALE( 1.321004112591985E-03), -SCALE( 1.232737907672045E-03),
    -SCALE( 1.145838949770837E-03), -SCALE( 1.060335987079630E-03),
    -SCALE( 9.762565135043539E-04), -SCALE( 8.936267682438125E-04),
    -SCALE( 8.124716609148709E-04), -SCALE( 7.328147714529330E-04),
    -SCALE( 6.546784227970995E-04), -SCALE( 5.780836785397680E-04),
    -SCALE( 5.030502185326690E-04), -SCALE( 4.295963388434640E-04),
    -SCALE( 3.577390751948810E-04), -SCALE( 2.874942045551261E-04),
    -SCALE( 2.188761566878072E-04), -SCALE( 1.518980125188441E-04),
    -SCALE( 8.657155665857780E-05), -SCALE( 2.290727971715371E-05),
    -SCALE(-3.908562327914961E-05), -SCALE(-9.939925244214647E-05),
    -SCALE(-1.580269916310419E-04), -SCALE(-2.149635065251304E-04),
    -SCALE(-2.702047136138704E-04), -SCALE(-3.237477782338944E-04),
    -SCALE(-3.755911023931640E-04), -SCALE(-4.257343232947606E-04),
    -SCALE(-4.741782930361297E-04), -SCALE(-5.209250745440945E-04),
    -SCALE(-5.659778724459110E-04), -SCALE(-6.093410318069875E-04),
    -SCALE(-6.510200794896010E-04), -SCALE(-6.910217203263935E-04),
    -SCALE(-7.293537115784970E-04), -SCALE(-7.660248590280630E-04),
     SCALE( 8.010450565557860E-04),  SCALE( 8.344252837566670E-04),
     SCALE( 8.661775139092965E-04),  SCALE( 8.963147092655810E-04),
     SCALE( 9.248508094554660E-04),  SCALE( 9.518007272200706E-04),
     SCALE( 9.771802661360279E-04),  SCALE( 1.001006116010387E-03),
     SCALE( 1.023295833844173E-03),  SCALE( 1.044067841074319E-03),
     SCALE( 1.063341381932592E-03),  SCALE( 1.081136518161085E-03),
     SCALE( 1.097474054394210E-03),  SCALE( 1.112375532031342E-03),
     SCALE( 1.125863169053070E-03),  SCALE( 1.137959857080735E-03),
     SCALE( 1.148689156870073E-03),  SCALE( 1.158075292551968E-03),
     SCALE( 1.166143029435388E-03),  SCALE( 1.172917669070769E-03),
     SCALE( 1.178425062841502E-03),  SCALE( 1.182691606532144E-03),
     SCALE( 1.185744107237668E-03),  SCALE( 1.187609780917844E-03),
     SCALE( 1.188316331684390E-03),  SCALE( 1.187891943390233E-03),
     SCALE( 1.186365015306024E-03),  SCALE( 1.183764158185287E-03),
     SCALE( 1.180118372021201E-03),  SCALE( 1.175457042899868E-03),
     SCALE( 1.169809696015420E-03),  SCALE( 1.163205988551180E-03),
    -SCALE( 1.155675800049931E-03), -SCALE( 1.147249229489164E-03),
    -SCALE( 1.137956440566105E-03), -SCALE( 1.127827655023416E-03),
    -SCALE( 1.116893160529398E-03), -SCALE( 1.105183307573424E-03),
    -SCALE( 1.092728433285052E-03), -SCALE( 1.079558856878201E-03),
    -SCALE( 1.065704855603833E-03), -SCALE( 1.051196660616898E-03),
    -SCALE( 1.036064390096491E-03), -SCALE( 1.020338043044835E-03),
    -SCALE( 1.004047429744579E-03), -SCALE( 9.872221706725139E-04),
    -SCALE( 9.698917421439440E-04), -SCALE( 9.520854714135470E-04),
    -SCALE( 9.338323832499584E-04), -SCALE( 9.151611951498600E-04),
    -SCALE( 8.961003654734095E-04), -SCALE( 8.766780920104490E-04),
    -SCALE( 8.569222323017155E-04), -SCALE( 8.368602986989685E-04),
    -SCALE( 8.165194293666030E-04), -SCALE( 7.959263867639895E-04),
    -SCALE( 7.751075532553095E-04), -SCALE( 7.540889275341360E-04),
    -SCALE( 7.328960503543460E-04), -SCALE( 7.115540021619675E-04),
    -SCALE( 6.900874252648055E-04), -SCALE( 6.685205214586190E-04),
    -SCALE( 6.468769776333065E-04), -SCALE( 6.251799639488680E-04),
    -SCALE( 6.034521681004050E-04), -SCALE( 5.817157931755955E-04),
    -SCALE( 5.599924762541345E-04), -SCALE( 5.383032870049990E-04),
    -SCALE( 5.166687782268060E-04), -SCALE( 4.951089849262823E-04),
    -SCALE( 4.736433535042661E-04), -SCALE( 4.522907395812775E-04),
    -SCALE( 4.310694309740984E-04), -SCALE( 4.099971482282953E-04),
    -SCALE( 3.890910333571532E-04), -SCALE( 3.683676476039790E-04),
    -SCALE( 3.478429334738513E-04), -SCALE( 3.275322154221950E-04),
    -SCALE( 3.074502529716603E-04), -SCALE( 2.876112401970844E-04),
    -SCALE( 2.680287412777888E-04), -SCALE( 2.487156900488424E-04),
    -SCALE( 2.296844445765594E-04), -SCALE( 2.109467883317648E-04),
    -SCALE( 1.925139014261300E-04), -SCALE( 1.743963597795672E-04),
    -SCALE( 1.566041455665768E-04), -SCALE( 1.391466488046540E-04),
    -SCALE( 1.220326918528753E-04), -SCALE( 1.052705294604104E-04),
    -SCALE( 8.886782533520320E-05), -SCALE( 7.283165337456365E-05),
    -SCALE( 5.716854816951505E-05), -SCALE( 4.188450631602256E-05),
    -SCALE( 2.698496475767416E-05), -SCALE( 1.247480140056488E-05),
    -SCALE(-1.641629701930447E-06), -SCALE(-1.536049652046955E-05),
    -SCALE(-2.867848400338988E-05), -SCALE(-4.159279275517104E-05),
    -SCALE(-5.410111500896320E-05), -SCALE(-6.620163295915880E-05),
    -SCALE(-7.789300610544491E-05), -SCALE(-8.917436962303755E-05),
    -SCALE(-1.000453111637885E-04), -SCALE(-1.105058690666598E-04),
    -SCALE(-1.205565161870574E-04), -SCALE(-1.301981580423545E-04),
    -SCALE(-1.394321081840759E-04), -SCALE(-1.482600859779846E-04),
    -SCALE(-1.566841923938267E-04), -SCALE(-1.647069084035198E-04),
    -SCALE(-1.723310839515549E-04), -SCALE(-1.795599357837175E-04),
    -SCALE(-1.863970107085474E-04), -SCALE(-1.928461833795790E-04),
    -SCALE(-1.989116442431618E-04), -SCALE(-2.045978973917164E-04),
    -SCALE(-2.099097253875555E-04), -SCALE(-2.148521873332429E-04),
    -SCALE(-2.194306115956788E-04), -SCALE(-2.236505938503120E-04),
    -SCALE(-2.275179613297294E-04), -SCALE(-2.310387700701966E-04),
    -SCALE(-2.342192800869662E-04), -SCALE(-2.370659535642138E-04),
    -SCALE(-2.395854398583434E-04), -SCALE(-2.417845745791980E-04),
     SCALE( 2.436703743734102E-04),  SCALE( 2.452500321752444E-04),
     SCALE( 2.465308179417657E-04),  SCALE( 2.475200821995122E-04),
     SCALE( 2.482254333360894E-04),  SCALE( 2.486547177024881E-04),
     SCALE( 2.488154045724704E-04),  SCALE( 2.487146183038166E-04),
     SCALE( 2.483604609263728E-04),  SCALE( 2.477619902517768E-04),
     SCALE( 2.469274156993822E-04),  SCALE( 2.458640761601700E-04),
     SCALE( 2.445797571768798E-04),  SCALE( 2.430827245294182E-04),
     SCALE( 2.413811459337238E-04),  SCALE( 2.394830702362197E-04),
     SCALE( 2.373965479929124E-04),  SCALE( 2.351296356350780E-04),
     SCALE( 2.326903665352515E-04),  SCALE( 2.300867481008122E-04),
     SCALE( 2.273267267668629E-04),  SCALE( 2.244181863585484E-04),
     SCALE( 2.213689470691719E-04),  SCALE( 2.181867649165566E-04),
     SCALE( 2.148793208048870E-04),  SCALE( 2.114542191049128E-04),
     SCALE( 2.079189673565927E-04),  SCALE( 2.042809749895736E-04),
     SCALE( 2.005475454710861E-04),  SCALE( 1.967258755680427E-04),
     SCALE( 1.928230469653129E-04),  SCALE( 1.888460252663805E-04),
    -SCALE( 1.848016463895374E-04), -SCALE( 1.806966158461229E-04),
    -SCALE( 1.765375064656002E-04), -SCALE( 1.723307577358468E-04),
    -SCALE( 1.680826635644682E-04), -SCALE( 1.637993715464274E-04),
    -SCALE( 1.594868790905897E-04), -SCALE( 1.551510331409437E-04),
    -SCALE( 1.507975279169229E-04), -SCALE( 1.464319042632354E-04),
    -SCALE( 1.420595376057812E-04), -SCALE( 1.376856378740834E-04),
    -SCALE( 1.333152598388584E-04), -SCALE( 1.289533027076185E-04),
    -SCALE( 1.246044908903475E-04), -SCALE( 1.202733737618488E-04),
    -SCALE( 1.159643396677203E-04), -SCALE( 1.116816161426427E-04),
    -SCALE( 1.074292607067401E-04), -SCALE( 1.032111605730492E-04),
    -SCALE( 9.903103541547551E-05), -SCALE( 9.489243763625400E-05),
    -SCALE( 9.079875548213924E-05), -SCALE( 8.675321306240340E-05),
    -SCALE( 8.275886762862380E-05), -SCALE( 7.881860980632000E-05),
    -SCALE( 7.493517141117804E-05), -SCALE( 7.111112575372286E-05),
    -SCALE( 6.734888652588270E-05), -SCALE( 6.365070805378855E-05),
    -SCALE( 6.001869197279105E-05), -SCALE( 5.645478762018535E-05),
    -SCALE( 5.296079400014060E-05), -SCALE( 4.953836017695434E-05),
    -SCALE( 4.618899196172456E-05), -SCALE( 4.291405238785664E-05),
    -SCALE( 3.971476548553926E-05), -SCALE( 3.659221671654097E-05),
    -SCALE( 3.354735876525904E-05), -SCALE( 3.058101208191696E-05),
    -SCALE( 2.769387104186206E-05), -SCALE( 2.488650449141616E-05),
    -SCALE( 2.215936159729271E-05), -SCALE( 1.951277238921524E-05),
    -SCALE( 1.694695384810771E-05), -SCALE( 1.446201050055097E-05),
    -SCALE( 1.205794140863581E-05), -SCALE( 9.734640752487110E-06),
    -SCALE( 7.491903655942790E-06), -SCALE( 5.329426896805280E-06),
    -SCALE( 3.246818706694593E-06), -SCALE( 1.243599082783966E-06),
    -SCALE(-6.808031538695990E-07), -SCALE(-2.527049419980207E-06),
    -SCALE(-4.295867784770884E-06), -SCALE(-5.988052382589700E-06),
    -SCALE(-7.604474075726550E-06), -SCALE(-9.146080150692231E-06),
    -SCALE(-1.061387697354165E-05), -SCALE(-1.200892908323512E-05),
    -SCALE(-1.333235659026308E-05), -SCALE(-1.458533474050792E-05),
    -SCALE(-1.576908692157790E-05), -SCALE(-1.688488402231131E-05),
    -SCALE(-1.793403733553470E-05), -SCALE(-1.891789805352808E-05),
    -SCALE(-1.983785326550204E-05), -SCALE(-2.069532541857637E-05),
    -SCALE(-2.149176446183214E-05), -SCALE(-2.222864728099456E-05),
    -SCALE(-2.290747311740374E-05), -SCALE(-2.352976312853915E-05),
    -SCALE(-2.409705530058412E-05), -SCALE(-2.461090395766135E-05),
    -SCALE(-2.507287405244726E-05), -SCALE(-2.548454069800296E-05),
    -SCALE(-2.584748457743932E-05), -SCALE(-2.616329155063932E-05),
    -SCALE(-2.643354859248296E-05), -SCALE(-2.665984338990853E-05),
    -SCALE(-2.684375953903512E-05), -SCALE(-2.698687619105470E-05),
    -SCALE(-2.709076506473594E-05), -SCALE(-2.715699010579455E-05),
    -SCALE(-2.718710298053676E-05), -SCALE(-2.718264274615772E-05),
    -SCALE(-2.714513310366389E-05), -SCALE(-2.707608215301207E-05),
    -SCALE(-2.697697975324517E-05), -SCALE(-2.684929723999712E-05),
    -SCALE(-2.669448385046480E-05), -SCALE(-2.651396653512446E-05),
    -SCALE(-2.630914939052064E-05), -SCALE(-2.608141341583320E-05),
    -SCALE(-2.583211172456204E-05), -SCALE(-2.556256939919902E-05),
     SCALE( 2.527408508232496E-05),  SCALE( 2.496793086354262E-05),
     SCALE( 2.464534820090372E-05),  SCALE( 2.430754772248991E-05),
     SCALE( 2.395570893976328E-05),  SCALE( 2.359098021649775E-05),
     SCALE( 2.321447837191817E-05),  SCALE( 2.282728860550347E-05),
     SCALE( 2.243046323951717E-05),  SCALE( 2.202502165289463E-05),
     SCALE( 2.161195008423191E-05),  SCALE( 2.119220163057428E-05),
     SCALE( 2.076669641777394E-05),  SCALE( 2.033632155746451E-05),
     SCALE( 1.990193003014108E-05),  SCALE( 1.946434074488017E-05),
     SCALE( 1.902434097009924E-05),  SCALE( 1.858268630848355E-05),
     SCALE( 1.814009771461820E-05),  SCALE( 1.769726156360750E-05),
     SCALE( 1.725483414306857E-05),  SCALE( 1.681344171653956E-05),
     SCALE( 1.637367742646690E-05),  SCALE( 1.593610134164050E-05),
     SCALE( 1.550124459778260E-05),  SCALE( 1.506960952767971E-05),
     SCALE( 1.464166838349645E-05),  SCALE( 1.421786336584378E-05),
     SCALE( 1.379860854096596E-05),  SCALE( 1.338428997824206E-05),
     SCALE( 1.297526685803031E-05),  SCALE( 1.257187157446922E-05),
    -SCALE( 1.217441088925653E-05), -SCALE( 1.178316603669356E-05),
    -SCALE( 1.139839388088106E-05), -SCALE( 1.102032704633465E-05),
    -SCALE( 1.064917563432859E-05), -SCALE( 1.028512733460845E-05),
    -SCALE( 9.928348166726454E-06), -SCALE( 9.578982598825311E-06),
    -SCALE( 9.237155419546841E-06), -SCALE( 8.902971890012200E-06),
    -SCALE( 8.576519215476810E-06), -SCALE( 8.257866642922776E-06),
    -SCALE( 7.947066136436549E-06), -SCALE( 7.644152509859200E-06),
    -SCALE( 7.349145669714490E-06), -SCALE( 7.062050754515410E-06),
    -SCALE( 6.782858956580495E-06), -SCALE( 6.511547627860400E-06),
    -SCALE( 6.248081786307280E-06), -SCALE( 5.992414242480480E-06),
    -SCALE( 5.744486878502875E-06), -SCALE( 5.504230753545810E-06),
    -SCALE( 5.271567145540585E-06), -SCALE( 5.046407670500340E-06),
    -SCALE( 4.828655865889468E-06), -SCALE( 4.618207304054390E-06),
    -SCALE( 4.414950504331232E-06), -SCALE( 4.218767020985225E-06),
    -SCALE( 4.029532465769980E-06), -SCALE( 3.847116621801583E-06),
    -SCALE( 3.671384926229468E-06), -SCALE( 3.502198552003754E-06),
    -SCALE( 3.339414724085582E-06), -SCALE( 3.182886786597464E-06),
    -SCALE( 3.032465363920388E-06), -SCALE( 2.887998457950138E-06),
    -SCALE( 2.749332426639114E-06), -SCALE( 2.616312057164482E-06),
    -SCALE( 2.488781197085654E-06), -SCALE( 2.366582812095056E-06),
    -SCALE( 2.249559625377646E-06), -SCALE( 2.137554168666232E-06),
    -SCALE( 2.030409266099076E-06), -SCALE( 1.927968090232205E-06),
    -SCALE( 1.830074910452612E-06), -SCALE( 1.736575155397134E-06),
    -SCALE( 1.647316037754536E-06), -SCALE( 1.562146591398006E-06),
    -SCALE( 1.480917863497994E-06), -SCALE( 1.403482943378026E-06),
    -SCALE( 1.329697405230052E-06), -SCALE( 1.259419343863326E-06),
    -SCALE( 1.192509718464610E-06), -SCALE( 1.128832363715543E-06),
    -SCALE( 1.068253890614633E-06), -SCALE( 1.010643712821376E-06),
    -SCALE( 9.558747254049829E-07), -SCALE( 9.038233322489550E-07),
    -SCALE( 8.543693702029365E-07), -SCALE( 8.073961036828060E-07),
    -SCALE( 7.627901817207720E-07), -SCALE( 7.204416525901345E-07),
    -SCALE( 6.802443284999140E-07), -SCALE( 6.420958008785305E-07),
    -SCALE( 6.058974119201980E-07), -SCALE( 5.715542338879300E-07),
    -SCALE( 5.389746422333555E-07), -SCALE( 5.080703129463105E-07),
    -SCALE( 4.787565911260676E-07), -SCALE( 4.509524956369838E-07),
    -SCALE( 4.245804529615933E-07), -SCALE( 3.995662812166919E-07),
    -SCALE( 3.758391046541218E-07), -SCALE( 3.533313458884360E-07),
    -SCALE( 3.319786404047318E-07), -SCALE( 3.117198196826890E-07),
    -SCALE( 2.924966254175760E-07), -SCALE( 2.742537030854958E-07),
    -SCALE( 2.569387461577766E-07), -SCALE( 2.405024587792340E-07),
    -SCALE( 2.248975904755362E-07), -SCALE( 2.100789970600680E-07),
    -SCALE( 1.960059458775785E-07), -SCALE( 1.826418431101815E-07),
    -SCALE( 1.699459292657122E-07), -SCALE( 1.578742499812371E-07),
    -SCALE( 1.464093182119164E-07), -SCALE( 1.355563613797138E-07),
    -SCALE( 1.252310964561535E-07), -SCALE( 1.152736341186576E-07),
    -SCALE( 1.058665951072195E-07), -SCALE( 9.748319049487465E-08),
    -SCALE( 8.932687128137479E-08), -SCALE( 7.952503432137510E-08),
    -SCALE( 7.095090315087499E-08), -SCALE( 7.127938986604168E-08)
};

#undef SCALE
