/*
 * Endpoint - Linux SBP2 Disk Target
 *
 * Copyright (C) 2003 Oracle.  All rights reserved.
 *
 * Author: Manish Singh <manish.singh@oracle.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have recieved a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 021110-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
#include <signal.h>

#include <glib.h>

#include "app.h"
#include "cleanup.h"
#include "manager.h"
#include "process.h"
#include "util.h"
#include "wire.h"
#include "worker.h"


#define CONFIG_FILE SYSCONFDIR "/endpoint.conf"

#define MASTER_NAME  "endpoint"
#define MANAGER_NAME "endptmgr"
#define WORKER_NAME  "endptio"


/* #define DEBUG_MANAGER */
/* #define DEBUG_WORKER */


typedef struct _ProcessInfo ProcessInfo;

struct _ProcessInfo
{
  Process    *manager;
  GHashTable *workers;
};


static gboolean start_manager      (EndpointApp      *app);

static gboolean login_worker       (EndpointApp      *app,
				    WireLogin        *message);
static gboolean reconnect_worker   (EndpointApp      *app,
				    WireReconnect    *message);
static gboolean logout_worker      (EndpointApp      *app,
				    WireLogout       *message);

static gboolean worker_active      (EndpointApp      *app,
				    WireWorkerActive *message);

static void     manager_reap       (Process          *process,
				    gint              status);
static void     worker_reap        (Process          *process,
				    gint              status);

static void     kill_all_processes (EndpointApp      *app,
				    gint              signum);

static void     set_ps_name        (EndpointApp      *app,
				    const gchar      *name);


static WireMsgFuncs wire_funcs = {
  .login         = login_worker,
  .reconnect     = reconnect_worker,
  .logout        = logout_worker,
  .worker_active = worker_active
};


#ifdef DEBUG_MANAGER
#define debug_manager() debug_process ()
#else
#define debug_manager() G_STMT_START{ (void)0; }G_STMT_END
#endif

#ifdef DEBUG_WORKER
#define debug_worker() debug_process ()
#else
#define debug_worker() G_STMT_START{ (void)0; }G_STMT_END
#endif


static inline void
debug_process (void)
{
  GLogLevelFlags fatal_mask;

  fatal_mask = g_log_set_always_fatal (G_LOG_FATAL_MASK);
  fatal_mask |= G_LOG_LEVEL_WARNING | G_LOG_LEVEL_CRITICAL;
  g_log_set_always_fatal (fatal_mask);

  raise (SIGSTOP);
}


int
main (int   argc,
      char *argv[])
{
  EndpointApp *app;
  gchar       *configfile = CONFIG_FILE;
  ProcessInfo *pinfo;

  if (argc > 1)
    configfile = argv[1];

  app = endpoint_app_new (configfile);

  if (!app)
    {
      /* ERROR MESSAGE */
      exit (1);
    }

  app->ps_name = argv[0];
  app->ps_len = strlen(argv[0]);

  if (app->ps_len < strlen (MASTER_NAME))
    {
      /* ERROR MESSAGE */
      exit (1);
    }

  set_ps_name (app, MASTER_NAME);

  app->wire_funcs = &wire_funcs;

  if (!wire_init ())
    {
      /* ERROR MESSAGE */
      exit (1);
    }

  pinfo = g_new (ProcessInfo, 1);

  pinfo->manager = NULL;
  pinfo->workers = g_hash_table_new_full (g_direct_hash, g_direct_equal,
                                          NULL, g_free);

  app->user_data = pinfo;

  app->context = g_main_context_new ();

  setpgid (0, 0);

#if 0
  if (!cleanup_register (app, kill_all_processes))
    {
      /* ERROR MESSAGE */
      exit (1);
    }
#endif

  signal (SIGPIPE, SIG_IGN);

  util_run_first_add (app->context, (GSourceFunc) start_manager, app);

  while (app->context)
    {
      app->loop = g_main_loop_new (app->context, TRUE);
      g_main_context_unref (app->context);

      g_main_loop_run (app->loop);
      g_main_loop_unref (app->loop);

      if (pinfo)
	{
	  g_hash_table_destroy (pinfo->workers);
	  g_free (pinfo->manager);

	  g_free (pinfo);
	  pinfo = NULL;
	}
    }

  while (wait (NULL) > 0);

  return 0;
}

static gboolean
start_manager (EndpointApp *app)
{
  Process *process;

  g_return_val_if_fail (app != NULL, FALSE);
  g_return_val_if_fail (app->user_data != NULL, FALSE);

  process = process_start (app, manager_reap);

  if (!process)
    {
      /* ERROR MESSAGE */
      exit (1);
    }

  if (process->pid == 0)
    {
      set_ps_name (app, MANAGER_NAME);

      debug_manager ();

      manager_process (app, process);
      g_free (process);
    }
  else
    {
      ProcessInfo *pinfo;

      pinfo = app->user_data;
      pinfo->manager = process;
    }

  return FALSE;
}

static gboolean
login_worker (EndpointApp *app,
              WireLogin   *message)
{
  Process     *process;
  ProcessInfo *pinfo;
  gpointer     ret_message;

  g_return_val_if_fail (app != NULL, FALSE);
  g_return_val_if_fail (app->user_data != NULL, FALSE);
  g_return_val_if_fail (message != NULL, FALSE);

  pinfo = app->user_data;

  process = process_start (app, worker_reap);

  if (!process)
    {
      ret_message = wire_message_worker_died (message->login_ID);
      wire_message_send (pinfo->manager->writer, ret_message, app);

      return TRUE;
    }

  if (process->pid == 0)
    {
      set_ps_name (app, WORKER_NAME);

      debug_worker ();

      worker_process (app, process, message);
      g_free (process);
    }
  else
    {
      process->login_ID = message->login_ID;

      g_hash_table_insert (pinfo->workers,
	                   GUINT_TO_POINTER (message->login_ID), process);
    }
 
  return TRUE;
}

static gboolean
reconnect_worker (EndpointApp   *app,
		  WireReconnect *message)
{
  Process     *process;
  ProcessInfo *pinfo;

  g_return_val_if_fail (app != NULL, FALSE);
  g_return_val_if_fail (app->user_data != NULL, FALSE);
  g_return_val_if_fail (message != NULL, FALSE);

  pinfo = app->user_data;

  process = g_hash_table_lookup (pinfo->workers,
                                 GUINT_TO_POINTER (message->login_ID));

  if (process)
    {
      WireReconnect *copy;

      copy = wire_message_copy (message);
      wire_message_send (process->writer, copy, app);
    }

  return TRUE;
}

static gboolean
logout_worker (EndpointApp *app,
	       WireLogout  *message)
{
  Process     *process;
  ProcessInfo *pinfo;

  g_return_val_if_fail (app != NULL, FALSE);
  g_return_val_if_fail (app->user_data != NULL, FALSE);
  g_return_val_if_fail (message != NULL, FALSE);
  
  pinfo = app->user_data;

  process = g_hash_table_lookup (pinfo->workers,
                                 GUINT_TO_POINTER (message->login_ID));

  if (process)
    kill (process->pid, SIGTERM);

  return TRUE;
}

static gboolean
worker_active (EndpointApp      *app,
	       WireWorkerActive *message)
{
  ProcessInfo      *pinfo;
  WireWorkerActive *copy;

  g_return_val_if_fail (app != NULL, FALSE);
  g_return_val_if_fail (app->user_data != NULL, FALSE);
  g_return_val_if_fail (message != NULL, FALSE);

  pinfo = app->user_data;

  copy = wire_message_copy (message);
  wire_message_send (pinfo->manager->writer, copy, app);

  return TRUE;
}

static void
manager_reap (Process *process,
	      gint     status)
{             
  ProcessInfo *pinfo;

  g_return_if_fail (process != NULL);
  g_return_if_fail (process->app != NULL);
  g_return_if_fail (process->app->user_data != NULL);

  pinfo = process->app->user_data;

  if (WIFEXITED (status))
    {
      gint exit_status = WEXITSTATUS (status);

      if (exit_status != 0)
	{
	  /* ERROR MESSAGE */
	}

      exit (exit_status);
    }
  else if (WIFSIGNALED (status))
    {
      gint signum = WTERMSIG (status);

      g_printerr ("Manager exited (%s)\n", g_strsignal (signum));
      exit (5);
    }
  else
    exit (6);
}

static void
worker_reap (Process *process,
	     gint     status)
{
  ProcessInfo *pinfo;
  gpointer     message;

  g_return_if_fail (process != NULL);
  g_return_if_fail (process->app != NULL);
  g_return_if_fail (process->app->user_data != NULL);

  pinfo = process->app->user_data;

  message = wire_message_worker_died (process->login_ID);
  wire_message_send (pinfo->manager->writer, message, process->app);

  g_hash_table_steal (pinfo->workers, GUINT_TO_POINTER (process->login_ID));

  if (WIFEXITED (status))
    {
      g_printerr ("Worker exited: %d\n", WEXITSTATUS (status));
    }
  else if (WIFSIGNALED (status))
    {
      gint signum = WTERMSIG (status);

      g_printerr ("Worker exited (%s)\n", g_strsignal (signum));
    }
}

static void
kill_all_processes (EndpointApp *app,
		    gint         signum)
{
  kill (0, signum);

  app->context = NULL;
  g_main_loop_quit (app->loop);
}

static void
set_ps_name (EndpointApp *app,
	     const gchar *name)
{
  gint len;

  g_return_if_fail (app != NULL);
  g_return_if_fail (name != NULL);

  len = strlen (name);

  strcpy (app->ps_name, name);
  memset (app->ps_name + len, 0, app->ps_len - len);
}
