/*
 * Endpoint - Linux SBP2 Disk Target
 *
 * Copyright (C) 2003 Oracle.  All rights reserved.
 *
 * Author: Manish Singh <manish.singh@oracle.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have recieved a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 021110-1307, USA.
 */

#include <glib.h>


static guint attach_source (GSource      *source,
			    GMainContext *context,
			    GSourceFunc   function,
			    gpointer      data);


GIOChannel *
util_io_channel_new (gint fd)
{
  GIOChannel *channel;

  g_return_val_if_fail (fd >= 0, NULL);

  channel = g_io_channel_unix_new (fd);

  g_io_channel_set_encoding (channel, NULL, NULL);
  g_io_channel_set_buffered (channel, FALSE);
  g_io_channel_set_close_on_unref (channel, TRUE);

  return channel;
}

guint
util_run_first_add (GMainContext *context,
		    GSourceFunc   function,
		    gpointer      data)
{
  GSource *source;

  g_return_val_if_fail (function != NULL, 0);

  source = g_idle_source_new ();
  g_source_set_priority (source, G_PRIORITY_HIGH);

  return attach_source (source, context, function, data);
}

guint
util_timeout_add (GMainContext *context,
		  guint         interval,
		  GSourceFunc   function,
		  gpointer      data)
{
  GSource *source;

  g_return_val_if_fail (function != NULL, 0);

  source = g_timeout_source_new (interval);

  return attach_source (source, context, function, data);
}

guint
util_io_add_watch (GMainContext *context,
		   GIOChannel   *channel,
		   GIOCondition  condition,
		   GIOFunc       function,
		   gpointer      data)
{
  GSource *source;

  g_return_val_if_fail (channel != NULL, 0);
  g_return_val_if_fail (function != NULL, 0);

  source = g_io_create_watch (channel, condition);
  g_io_channel_unref (channel);

  return attach_source (source, context, (GSourceFunc) function, data);
}

static guint
attach_source (GSource      *source,
               GMainContext *context,
	       GSourceFunc   function,
	       gpointer      data)
{
  guint id;

  g_return_val_if_fail (source != NULL, 0);
  g_return_val_if_fail (function != NULL, 0);

  g_source_set_callback (source, function, data, NULL);

  id = g_source_attach (source, context);
  g_source_unref (source);

  return id;
}

gboolean
util_source_remove (GMainContext *context,
                    guint         tag)
{
  GSource *source;

  g_return_val_if_fail (context != NULL, FALSE);

  source = g_main_context_find_source_by_id (context, tag);

  if (source)
    g_source_destroy (source);

  return source != NULL;
}
