-- | This module defines the 'Configuration' type, which is just a
--   wrapper around all of the configuration options we accept on the
--   command line.
--
module Configuration (
  Configuration(..),
  merge_optional )
where

import System.Console.CmdArgs.Default ( Default( def ) )

import MxList ( MxList( get_mxs ) )
import qualified OptionalConfiguration as OC (
  OptionalConfiguration(..),
  merge_maybes )

-- | The main configuration data type. This will be passed to most of
--   the important functions once it has been created.
--
data Configuration =
  Configuration {
    database :: Maybe String,
    domain_query :: String,
    exclude_mx :: MxList,
    forward_query :: String,
    host :: Maybe String,
    password :: Maybe String,
    port :: Maybe Int,
    username :: Maybe String }
  deriving (Show)


-- | A Configuration with all of its fields set to their default
--   values.
--
instance Default Configuration where
  def = Configuration {
          database = def,
          domain_query = def_domain_query,
          exclude_mx = def,
          forward_query = def_forward_query,
          host = def,
          password = def,
          port = def,
          username = def }
    where
      def_domain_query = "SELECT domain " ++
                         "FROM domain " ++
                         "WHERE domain <> 'ALL' " ++
                         "ORDER BY domain;"

      def_forward_query = "SELECT address,goto " ++
                          "FROM alias " ++
                          "ORDER BY address;"

-- | Merge a 'Configuration' with an 'OptionalConfiguration'. This is
--   more or less the Monoid instance for 'OptionalConfiguration', but
--   since the two types are different, we have to repeat ourselves.
--
merge_optional :: Configuration
               -> OC.OptionalConfiguration
               -> Configuration
merge_optional cfg opt_cfg =
  Configuration
    (OC.merge_maybes (database cfg) (OC.database opt_cfg))
    (merge (domain_query cfg) (OC.domain_query opt_cfg))
    all_excluded_mx
    (merge (forward_query cfg) (OC.forward_query opt_cfg))
    (OC.merge_maybes (host cfg) (OC.host opt_cfg))
    (OC.merge_maybes (password cfg) (OC.password opt_cfg))
    (OC.merge_maybes (port cfg) (OC.port opt_cfg))
    (OC.merge_maybes (username cfg) (OC.username opt_cfg))
  where
    -- | If the thing on the right is Just something, return that
    --   something, otherwise return the thing on the left.
    merge :: a -> Maybe a -> a
    merge x Nothing  = x
    merge _ (Just y) = y

    -- If there are any optional usernames, use only those.
    all_excluded_mx = if (null (get_mxs (OC.exclude_mx opt_cfg)))
                      then (exclude_mx cfg)
                      else (OC.exclude_mx opt_cfg)
