--
--  Copyright (C) 2022-2023, AdaCore
--
--  SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
--

--  Tables for fast string to float conversion

pragma Ada_2022;

private package VSS.JSON.Implementation.Numbers.Tables is

   pragma Preelaborate;

   use type Interfaces.Integer_32;

   Exact_Powers_Of_10 : constant
     array (Interfaces.Integer_32 range 0 .. 22) of Interfaces.IEEE_Float_64 :=
     [1.0,                      --  10^0
      10.0,
      100.0,
      1000.0,
      10000.0,
      100000.0,
      1000000.0,
      10000000.0,
      100000000.0,
      1000000000.0,
      10000000000.0,            -- 10^10
      100000000000.0,
      1000000000000.0,
      10000000000000.0,
      100000000000000.0,
      1000000000000000.0,
      10000000000000000.0,
      100000000000000000.0,
      1000000000000000000.0,
      10000000000000000000.0,
      100000000000000000000.0,  --  10^20
      1000000000000000000000.0,
      10000000000000000000000.0];

   --   When mapping numbers from decimal to binary,
   --   we go from w * 10^q to m * 2^p but we have
   --   10^q = 5^q * 2^q, so effectively
   --   we are trying to match
   --   w * 2^q * 5^q to m * 2^p. Thus the powers of two
   --   are not a concern since they can be represented
   --   exactly using the binary notation, only the powers of five
   --   affect the binary significand.
   --
   --   The smallest non-zero float (binary64) is 2^−1074.
   --   We take as input numbers of the form w x 10^q where w < 2^64.
   --   We have that w * 10^-343  <  2^(64-344) 5^-343 < 2^-1076.
   --   However, we have that
   --   (2^64-1) * 10^-342 = (2^64-1) * 2^-342 * 5^-342 > 2^−1074.
   --   Thus it is possible for a number of the form w * 10^-342 where
   --   w is a 64-bit value to be a non-zero floating-point number.
   --   ********
   --   Any number of form w * 10^309 where w>= 1 is going to be
   --   infinite in binary64 so we never need to worry about powers
   --   of 5 greater than 308.

   type Powers_Of_Five_Record is record
      L : Interfaces.Unsigned_64;
      H : Interfaces.Unsigned_64;
   end record;

   Powers_Of_Five : constant
     array (Interfaces.Integer_32 range -342 .. 308)
       of Powers_Of_Five_Record :=
       [(16#eef453d6923bd65a#, 16#113faa2906a13b3f#),
        (16#9558b4661b6565f8#, 16#4ac7ca59a424c507#),
        (16#baaee17fa23ebf76#, 16#5d79bcf00d2df649#),
        (16#e95a99df8ace6f53#, 16#f4d82c2c107973dc#),
        (16#91d8a02bb6c10594#, 16#79071b9b8a4be869#),
        (16#b64ec836a47146f9#, 16#9748e2826cdee284#),
        (16#e3e27a444d8d98b7#, 16#fd1b1b2308169b25#),
        (16#8e6d8c6ab0787f72#, 16#fe30f0f5e50e20f7#),
        (16#b208ef855c969f4f#, 16#bdbd2d335e51a935#),
        (16#de8b2b66b3bc4723#, 16#ad2c788035e61382#),
        (16#8b16fb203055ac76#, 16#4c3bcb5021afcc31#),
        (16#addcb9e83c6b1793#, 16#df4abe242a1bbf3d#),
        (16#d953e8624b85dd78#, 16#d71d6dad34a2af0d#),
        (16#87d4713d6f33aa6b#, 16#8672648c40e5ad68#),
        (16#a9c98d8ccb009506#, 16#680efdaf511f18c2#),
        (16#d43bf0effdc0ba48#, 16#0212bd1b2566def2#),
        (16#84a57695fe98746d#, 16#014bb630f7604b57#),
        (16#a5ced43b7e3e9188#, 16#419ea3bd35385e2d#),
        (16#cf42894a5dce35ea#, 16#52064cac828675b9#),
        (16#818995ce7aa0e1b2#, 16#7343efebd1940993#),
        (16#a1ebfb4219491a1f#, 16#1014ebe6c5f90bf8#),
        (16#ca66fa129f9b60a6#, 16#d41a26e077774ef6#),
        (16#fd00b897478238d0#, 16#8920b098955522b4#),
        (16#9e20735e8cb16382#, 16#55b46e5f5d5535b0#),
        (16#c5a890362fddbc62#, 16#eb2189f734aa831d#),
        (16#f712b443bbd52b7b#, 16#a5e9ec7501d523e4#),
        (16#9a6bb0aa55653b2d#, 16#47b233c92125366e#),
        (16#c1069cd4eabe89f8#, 16#999ec0bb696e840a#),
        (16#f148440a256e2c76#, 16#c00670ea43ca250d#),
        (16#96cd2a865764dbca#, 16#380406926a5e5728#),
        (16#bc807527ed3e12bc#, 16#c605083704f5ecf2#),
        (16#eba09271e88d976b#, 16#f7864a44c633682e#),
        (16#93445b8731587ea3#, 16#7ab3ee6afbe0211d#),
        (16#b8157268fdae9e4c#, 16#5960ea05bad82964#),
        (16#e61acf033d1a45df#, 16#6fb92487298e33bd#),
        (16#8fd0c16206306bab#, 16#a5d3b6d479f8e056#),
        (16#b3c4f1ba87bc8696#, 16#8f48a4899877186c#),
        (16#e0b62e2929aba83c#, 16#331acdabfe94de87#),
        (16#8c71dcd9ba0b4925#, 16#9ff0c08b7f1d0b14#),
        (16#af8e5410288e1b6f#, 16#07ecf0ae5ee44dd9#),
        (16#db71e91432b1a24a#, 16#c9e82cd9f69d6150#),
        (16#892731ac9faf056e#, 16#be311c083a225cd2#),
        (16#ab70fe17c79ac6ca#, 16#6dbd630a48aaf406#),
        (16#d64d3d9db981787d#, 16#092cbbccdad5b108#),
        (16#85f0468293f0eb4e#, 16#25bbf56008c58ea5#),
        (16#a76c582338ed2621#, 16#af2af2b80af6f24e#),
        (16#d1476e2c07286faa#, 16#1af5af660db4aee1#),
        (16#82cca4db847945ca#, 16#50d98d9fc890ed4d#),
        (16#a37fce126597973c#, 16#e50ff107bab528a0#),
        (16#cc5fc196fefd7d0c#, 16#1e53ed49a96272c8#),
        (16#ff77b1fcbebcdc4f#, 16#25e8e89c13bb0f7a#),
        (16#9faacf3df73609b1#, 16#77b191618c54e9ac#),
        (16#c795830d75038c1d#, 16#d59df5b9ef6a2417#),
        (16#f97ae3d0d2446f25#, 16#4b0573286b44ad1d#),
        (16#9becce62836ac577#, 16#4ee367f9430aec32#),
        (16#c2e801fb244576d5#, 16#229c41f793cda73f#),
        (16#f3a20279ed56d48a#, 16#6b43527578c1110f#),
        (16#9845418c345644d6#, 16#830a13896b78aaa9#),
        (16#be5691ef416bd60c#, 16#23cc986bc656d553#),
        (16#edec366b11c6cb8f#, 16#2cbfbe86b7ec8aa8#),
        (16#94b3a202eb1c3f39#, 16#7bf7d71432f3d6a9#),
        (16#b9e08a83a5e34f07#, 16#daf5ccd93fb0cc53#),
        (16#e858ad248f5c22c9#, 16#d1b3400f8f9cff68#),
        (16#91376c36d99995be#, 16#23100809b9c21fa1#),
        (16#b58547448ffffb2d#, 16#abd40a0c2832a78a#),
        (16#e2e69915b3fff9f9#, 16#16c90c8f323f516c#),
        (16#8dd01fad907ffc3b#, 16#ae3da7d97f6792e3#),
        (16#b1442798f49ffb4a#, 16#99cd11cfdf41779c#),
        (16#dd95317f31c7fa1d#, 16#40405643d711d583#),
        (16#8a7d3eef7f1cfc52#, 16#482835ea666b2572#),
        (16#ad1c8eab5ee43b66#, 16#da3243650005eecf#),
        (16#d863b256369d4a40#, 16#90bed43e40076a82#),
        (16#873e4f75e2224e68#, 16#5a7744a6e804a291#),
        (16#a90de3535aaae202#, 16#711515d0a205cb36#),
        (16#d3515c2831559a83#, 16#0d5a5b44ca873e03#),
        (16#8412d9991ed58091#, 16#e858790afe9486c2#),
        (16#a5178fff668ae0b6#, 16#626e974dbe39a872#),
        (16#ce5d73ff402d98e3#, 16#fb0a3d212dc8128f#),
        (16#80fa687f881c7f8e#, 16#7ce66634bc9d0b99#),
        (16#a139029f6a239f72#, 16#1c1fffc1ebc44e80#),
        (16#c987434744ac874e#, 16#a327ffb266b56220#),
        (16#fbe9141915d7a922#, 16#4bf1ff9f0062baa8#),
        (16#9d71ac8fada6c9b5#, 16#6f773fc3603db4a9#),
        (16#c4ce17b399107c22#, 16#cb550fb4384d21d3#),
        (16#f6019da07f549b2b#, 16#7e2a53a146606a48#),
        (16#99c102844f94e0fb#, 16#2eda7444cbfc426d#),
        (16#c0314325637a1939#, 16#fa911155fefb5308#),
        (16#f03d93eebc589f88#, 16#793555ab7eba27ca#),
        (16#96267c7535b763b5#, 16#4bc1558b2f3458de#),
        (16#bbb01b9283253ca2#, 16#9eb1aaedfb016f16#),
        (16#ea9c227723ee8bcb#, 16#465e15a979c1cadc#),
        (16#92a1958a7675175f#, 16#0bfacd89ec191ec9#),
        (16#b749faed14125d36#, 16#cef980ec671f667b#),
        (16#e51c79a85916f484#, 16#82b7e12780e7401a#),
        (16#8f31cc0937ae58d2#, 16#d1b2ecb8b0908810#),
        (16#b2fe3f0b8599ef07#, 16#861fa7e6dcb4aa15#),
        (16#dfbdcece67006ac9#, 16#67a791e093e1d49a#),
        (16#8bd6a141006042bd#, 16#e0c8bb2c5c6d24e0#),
        (16#aecc49914078536d#, 16#58fae9f773886e18#),
        (16#da7f5bf590966848#, 16#af39a475506a899e#),
        (16#888f99797a5e012d#, 16#6d8406c952429603#),
        (16#aab37fd7d8f58178#, 16#c8e5087ba6d33b83#),
        (16#d5605fcdcf32e1d6#, 16#fb1e4a9a90880a64#),
        (16#855c3be0a17fcd26#, 16#5cf2eea09a55067f#),
        (16#a6b34ad8c9dfc06f#, 16#f42faa48c0ea481e#),
        (16#d0601d8efc57b08b#, 16#f13b94daf124da26#),
        (16#823c12795db6ce57#, 16#76c53d08d6b70858#),
        (16#a2cb1717b52481ed#, 16#54768c4b0c64ca6e#),
        (16#cb7ddcdda26da268#, 16#a9942f5dcf7dfd09#),
        (16#fe5d54150b090b02#, 16#d3f93b35435d7c4c#),
        (16#9efa548d26e5a6e1#, 16#c47bc5014a1a6daf#),
        (16#c6b8e9b0709f109a#, 16#359ab6419ca1091b#),
        (16#f867241c8cc6d4c0#, 16#c30163d203c94b62#),
        (16#9b407691d7fc44f8#, 16#79e0de63425dcf1d#),
        (16#c21094364dfb5636#, 16#985915fc12f542e4#),
        (16#f294b943e17a2bc4#, 16#3e6f5b7b17b2939d#),
        (16#979cf3ca6cec5b5a#, 16#a705992ceecf9c42#),
        (16#bd8430bd08277231#, 16#50c6ff782a838353#),
        (16#ece53cec4a314ebd#, 16#a4f8bf5635246428#),
        (16#940f4613ae5ed136#, 16#871b7795e136be99#),
        (16#b913179899f68584#, 16#28e2557b59846e3f#),
        (16#e757dd7ec07426e5#, 16#331aeada2fe589cf#),
        (16#9096ea6f3848984f#, 16#3ff0d2c85def7621#),
        (16#b4bca50b065abe63#, 16#0fed077a756b53a9#),
        (16#e1ebce4dc7f16dfb#, 16#d3e8495912c62894#),
        (16#8d3360f09cf6e4bd#, 16#64712dd7abbbd95c#),
        (16#b080392cc4349dec#, 16#bd8d794d96aacfb3#),
        (16#dca04777f541c567#, 16#ecf0d7a0fc5583a0#),
        (16#89e42caaf9491b60#, 16#f41686c49db57244#),
        (16#ac5d37d5b79b6239#, 16#311c2875c522ced5#),
        (16#d77485cb25823ac7#, 16#7d633293366b828b#),
        (16#86a8d39ef77164bc#, 16#ae5dff9c02033197#),
        (16#a8530886b54dbdeb#, 16#d9f57f830283fdfc#),
        (16#d267caa862a12d66#, 16#d072df63c324fd7b#),
        (16#8380dea93da4bc60#, 16#4247cb9e59f71e6d#),
        (16#a46116538d0deb78#, 16#52d9be85f074e608#),
        (16#cd795be870516656#, 16#67902e276c921f8b#),
        (16#806bd9714632dff6#, 16#00ba1cd8a3db53b6#),
        (16#a086cfcd97bf97f3#, 16#80e8a40eccd228a4#),
        (16#c8a883c0fdaf7df0#, 16#6122cd128006b2cd#),
        (16#fad2a4b13d1b5d6c#, 16#796b805720085f81#),
        (16#9cc3a6eec6311a63#, 16#cbe3303674053bb0#),
        (16#c3f490aa77bd60fc#, 16#bedbfc4411068a9c#),
        (16#f4f1b4d515acb93b#, 16#ee92fb5515482d44#),
        (16#991711052d8bf3c5#, 16#751bdd152d4d1c4a#),
        (16#bf5cd54678eef0b6#, 16#d262d45a78a0635d#),
        (16#ef340a98172aace4#, 16#86fb897116c87c34#),
        (16#9580869f0e7aac0e#, 16#d45d35e6ae3d4da0#),
        (16#bae0a846d2195712#, 16#8974836059cca109#),
        (16#e998d258869facd7#, 16#2bd1a438703fc94b#),
        (16#91ff83775423cc06#, 16#7b6306a34627ddcf#),
        (16#b67f6455292cbf08#, 16#1a3bc84c17b1d542#),
        (16#e41f3d6a7377eeca#, 16#20caba5f1d9e4a93#),
        (16#8e938662882af53e#, 16#547eb47b7282ee9c#),
        (16#b23867fb2a35b28d#, 16#e99e619a4f23aa43#),
        (16#dec681f9f4c31f31#, 16#6405fa00e2ec94d4#),
        (16#8b3c113c38f9f37e#, 16#de83bc408dd3dd04#),
        (16#ae0b158b4738705e#, 16#9624ab50b148d445#),
        (16#d98ddaee19068c76#, 16#3badd624dd9b0957#),
        (16#87f8a8d4cfa417c9#, 16#e54ca5d70a80e5d6#),
        (16#a9f6d30a038d1dbc#, 16#5e9fcf4ccd211f4c#),
        (16#d47487cc8470652b#, 16#7647c3200069671f#),
        (16#84c8d4dfd2c63f3b#, 16#29ecd9f40041e073#),
        (16#a5fb0a17c777cf09#, 16#f468107100525890#),
        (16#cf79cc9db955c2cc#, 16#7182148d4066eeb4#),
        (16#81ac1fe293d599bf#, 16#c6f14cd848405530#),
        (16#a21727db38cb002f#, 16#b8ada00e5a506a7c#),
        (16#ca9cf1d206fdc03b#, 16#a6d90811f0e4851c#),
        (16#fd442e4688bd304a#, 16#908f4a166d1da663#),
        (16#9e4a9cec15763e2e#, 16#9a598e4e043287fe#),
        (16#c5dd44271ad3cdba#, 16#40eff1e1853f29fd#),
        (16#f7549530e188c128#, 16#d12bee59e68ef47c#),
        (16#9a94dd3e8cf578b9#, 16#82bb74f8301958ce#),
        (16#c13a148e3032d6e7#, 16#e36a52363c1faf01#),
        (16#f18899b1bc3f8ca1#, 16#dc44e6c3cb279ac1#),
        (16#96f5600f15a7b7e5#, 16#29ab103a5ef8c0b9#),
        (16#bcb2b812db11a5de#, 16#7415d448f6b6f0e7#),
        (16#ebdf661791d60f56#, 16#111b495b3464ad21#),
        (16#936b9fcebb25c995#, 16#cab10dd900beec34#),
        (16#b84687c269ef3bfb#, 16#3d5d514f40eea742#),
        (16#e65829b3046b0afa#, 16#0cb4a5a3112a5112#),
        (16#8ff71a0fe2c2e6dc#, 16#47f0e785eaba72ab#),
        (16#b3f4e093db73a093#, 16#59ed216765690f56#),
        (16#e0f218b8d25088b8#, 16#306869c13ec3532c#),
        (16#8c974f7383725573#, 16#1e414218c73a13fb#),
        (16#afbd2350644eeacf#, 16#e5d1929ef90898fa#),
        (16#dbac6c247d62a583#, 16#df45f746b74abf39#),
        (16#894bc396ce5da772#, 16#6b8bba8c328eb783#),
        (16#ab9eb47c81f5114f#, 16#066ea92f3f326564#),
        (16#d686619ba27255a2#, 16#c80a537b0efefebd#),
        (16#8613fd0145877585#, 16#bd06742ce95f5f36#),
        (16#a798fc4196e952e7#, 16#2c48113823b73704#),
        (16#d17f3b51fca3a7a0#, 16#f75a15862ca504c5#),
        (16#82ef85133de648c4#, 16#9a984d73dbe722fb#),
        (16#a3ab66580d5fdaf5#, 16#c13e60d0d2e0ebba#),
        (16#cc963fee10b7d1b3#, 16#318df905079926a8#),
        (16#ffbbcfe994e5c61f#, 16#fdf17746497f7052#),
        (16#9fd561f1fd0f9bd3#, 16#feb6ea8bedefa633#),
        (16#c7caba6e7c5382c8#, 16#fe64a52ee96b8fc0#),
        (16#f9bd690a1b68637b#, 16#3dfdce7aa3c673b0#),
        (16#9c1661a651213e2d#, 16#06bea10ca65c084e#),
        (16#c31bfa0fe5698db8#, 16#486e494fcff30a62#),
        (16#f3e2f893dec3f126#, 16#5a89dba3c3efccfa#),
        (16#986ddb5c6b3a76b7#, 16#f89629465a75e01c#),
        (16#be89523386091465#, 16#f6bbb397f1135823#),
        (16#ee2ba6c0678b597f#, 16#746aa07ded582e2c#),
        (16#94db483840b717ef#, 16#a8c2a44eb4571cdc#),
        (16#ba121a4650e4ddeb#, 16#92f34d62616ce413#),
        (16#e896a0d7e51e1566#, 16#77b020baf9c81d17#),
        (16#915e2486ef32cd60#, 16#0ace1474dc1d122e#),
        (16#b5b5ada8aaff80b8#, 16#0d819992132456ba#),
        (16#e3231912d5bf60e6#, 16#10e1fff697ed6c69#),
        (16#8df5efabc5979c8f#, 16#ca8d3ffa1ef463c1#),
        (16#b1736b96b6fd83b3#, 16#bd308ff8a6b17cb2#),
        (16#ddd0467c64bce4a0#, 16#ac7cb3f6d05ddbde#),
        (16#8aa22c0dbef60ee4#, 16#6bcdf07a423aa96b#),
        (16#ad4ab7112eb3929d#, 16#86c16c98d2c953c6#),
        (16#d89d64d57a607744#, 16#e871c7bf077ba8b7#),
        (16#87625f056c7c4a8b#, 16#11471cd764ad4972#),
        (16#a93af6c6c79b5d2d#, 16#d598e40d3dd89bcf#),
        (16#d389b47879823479#, 16#4aff1d108d4ec2c3#),
        (16#843610cb4bf160cb#, 16#cedf722a585139ba#),
        (16#a54394fe1eedb8fe#, 16#c2974eb4ee658828#),
        (16#ce947a3da6a9273e#, 16#733d226229feea32#),
        (16#811ccc668829b887#, 16#0806357d5a3f525f#),
        (16#a163ff802a3426a8#, 16#ca07c2dcb0cf26f7#),
        (16#c9bcff6034c13052#, 16#fc89b393dd02f0b5#),
        (16#fc2c3f3841f17c67#, 16#bbac2078d443ace2#),
        (16#9d9ba7832936edc0#, 16#d54b944b84aa4c0d#),
        (16#c5029163f384a931#, 16#0a9e795e65d4df11#),
        (16#f64335bcf065d37d#, 16#4d4617b5ff4a16d5#),
        (16#99ea0196163fa42e#, 16#504bced1bf8e4e45#),
        (16#c06481fb9bcf8d39#, 16#e45ec2862f71e1d6#),
        (16#f07da27a82c37088#, 16#5d767327bb4e5a4c#),
        (16#964e858c91ba2655#, 16#3a6a07f8d510f86f#),
        (16#bbe226efb628afea#, 16#890489f70a55368b#),
        (16#eadab0aba3b2dbe5#, 16#2b45ac74ccea842e#),
        (16#92c8ae6b464fc96f#, 16#3b0b8bc90012929d#),
        (16#b77ada0617e3bbcb#, 16#09ce6ebb40173744#),
        (16#e55990879ddcaabd#, 16#cc420a6a101d0515#),
        (16#8f57fa54c2a9eab6#, 16#9fa946824a12232d#),
        (16#b32df8e9f3546564#, 16#47939822dc96abf9#),
        (16#dff9772470297ebd#, 16#59787e2b93bc56f7#),
        (16#8bfbea76c619ef36#, 16#57eb4edb3c55b65a#),
        (16#aefae51477a06b03#, 16#ede622920b6b23f1#),
        (16#dab99e59958885c4#, 16#e95fab368e45eced#),
        (16#88b402f7fd75539b#, 16#11dbcb0218ebb414#),
        (16#aae103b5fcd2a881#, 16#d652bdc29f26a119#),
        (16#d59944a37c0752a2#, 16#4be76d3346f0495f#),
        (16#857fcae62d8493a5#, 16#6f70a4400c562ddb#),
        (16#a6dfbd9fb8e5b88e#, 16#cb4ccd500f6bb952#),
        (16#d097ad07a71f26b2#, 16#7e2000a41346a7a7#),
        (16#825ecc24c873782f#, 16#8ed400668c0c28c8#),
        (16#a2f67f2dfa90563b#, 16#728900802f0f32fa#),
        (16#cbb41ef979346bca#, 16#4f2b40a03ad2ffb9#),
        (16#fea126b7d78186bc#, 16#e2f610c84987bfa8#),
        (16#9f24b832e6b0f436#, 16#0dd9ca7d2df4d7c9#),
        (16#c6ede63fa05d3143#, 16#91503d1c79720dbb#),
        (16#f8a95fcf88747d94#, 16#75a44c6397ce912a#),
        (16#9b69dbe1b548ce7c#, 16#c986afbe3ee11aba#),
        (16#c24452da229b021b#, 16#fbe85badce996168#),
        (16#f2d56790ab41c2a2#, 16#fae27299423fb9c3#),
        (16#97c560ba6b0919a5#, 16#dccd879fc967d41a#),
        (16#bdb6b8e905cb600f#, 16#5400e987bbc1c920#),
        (16#ed246723473e3813#, 16#290123e9aab23b68#),
        (16#9436c0760c86e30b#, 16#f9a0b6720aaf6521#),
        (16#b94470938fa89bce#, 16#f808e40e8d5b3e69#),
        (16#e7958cb87392c2c2#, 16#b60b1d1230b20e04#),
        (16#90bd77f3483bb9b9#, 16#b1c6f22b5e6f48c2#),
        (16#b4ecd5f01a4aa828#, 16#1e38aeb6360b1af3#),
        (16#e2280b6c20dd5232#, 16#25c6da63c38de1b0#),
        (16#8d590723948a535f#, 16#579c487e5a38ad0e#),
        (16#b0af48ec79ace837#, 16#2d835a9df0c6d851#),
        (16#dcdb1b2798182244#, 16#f8e431456cf88e65#),
        (16#8a08f0f8bf0f156b#, 16#1b8e9ecb641b58ff#),
        (16#ac8b2d36eed2dac5#, 16#e272467e3d222f3f#),
        (16#d7adf884aa879177#, 16#5b0ed81dcc6abb0f#),
        (16#86ccbb52ea94baea#, 16#98e947129fc2b4e9#),
        (16#a87fea27a539e9a5#, 16#3f2398d747b36224#),
        (16#d29fe4b18e88640e#, 16#8eec7f0d19a03aad#),
        (16#83a3eeeef9153e89#, 16#1953cf68300424ac#),
        (16#a48ceaaab75a8e2b#, 16#5fa8c3423c052dd7#),
        (16#cdb02555653131b6#, 16#3792f412cb06794d#),
        (16#808e17555f3ebf11#, 16#e2bbd88bbee40bd0#),
        (16#a0b19d2ab70e6ed6#, 16#5b6aceaeae9d0ec4#),
        (16#c8de047564d20a8b#, 16#f245825a5a445275#),
        (16#fb158592be068d2e#, 16#eed6e2f0f0d56712#),
        (16#9ced737bb6c4183d#, 16#55464dd69685606b#),
        (16#c428d05aa4751e4c#, 16#aa97e14c3c26b886#),
        (16#f53304714d9265df#, 16#d53dd99f4b3066a8#),
        (16#993fe2c6d07b7fab#, 16#e546a8038efe4029#),
        (16#bf8fdb78849a5f96#, 16#de98520472bdd033#),
        (16#ef73d256a5c0f77c#, 16#963e66858f6d4440#),
        (16#95a8637627989aad#, 16#dde7001379a44aa8#),
        (16#bb127c53b17ec159#, 16#5560c018580d5d52#),
        (16#e9d71b689dde71af#, 16#aab8f01e6e10b4a6#),
        (16#9226712162ab070d#, 16#cab3961304ca70e8#),
        (16#b6b00d69bb55c8d1#, 16#3d607b97c5fd0d22#),
        (16#e45c10c42a2b3b05#, 16#8cb89a7db77c506a#),
        (16#8eb98a7a9a5b04e3#, 16#77f3608e92adb242#),
        (16#b267ed1940f1c61c#, 16#55f038b237591ed3#),
        (16#df01e85f912e37a3#, 16#6b6c46dec52f6688#),
        (16#8b61313bbabce2c6#, 16#2323ac4b3b3da015#),
        (16#ae397d8aa96c1b77#, 16#abec975e0a0d081a#),
        (16#d9c7dced53c72255#, 16#96e7bd358c904a21#),
        (16#881cea14545c7575#, 16#7e50d64177da2e54#),
        (16#aa242499697392d2#, 16#dde50bd1d5d0b9e9#),
        (16#d4ad2dbfc3d07787#, 16#955e4ec64b44e864#),
        (16#84ec3c97da624ab4#, 16#bd5af13bef0b113e#),
        (16#a6274bbdd0fadd61#, 16#ecb1ad8aeacdd58e#),
        (16#cfb11ead453994ba#, 16#67de18eda5814af2#),
        (16#81ceb32c4b43fcf4#, 16#80eacf948770ced7#),
        (16#a2425ff75e14fc31#, 16#a1258379a94d028d#),
        (16#cad2f7f5359a3b3e#, 16#096ee45813a04330#),
        (16#fd87b5f28300ca0d#, 16#8bca9d6e188853fc#),
        (16#9e74d1b791e07e48#, 16#775ea264cf55347e#),
        (16#c612062576589dda#, 16#95364afe032a819e#),
        (16#f79687aed3eec551#, 16#3a83ddbd83f52205#),
        (16#9abe14cd44753b52#, 16#c4926a9672793543#),
        (16#c16d9a0095928a27#, 16#75b7053c0f178294#),
        (16#f1c90080baf72cb1#, 16#5324c68b12dd6339#),
        (16#971da05074da7bee#, 16#d3f6fc16ebca5e04#),
        (16#bce5086492111aea#, 16#88f4bb1ca6bcf585#),
        (16#ec1e4a7db69561a5#, 16#2b31e9e3d06c32e6#),
        (16#9392ee8e921d5d07#, 16#3aff322e62439fd0#),
        (16#b877aa3236a4b449#, 16#09befeb9fad487c3#),
        (16#e69594bec44de15b#, 16#4c2ebe687989a9b4#),
        (16#901d7cf73ab0acd9#, 16#0f9d37014bf60a11#),
        (16#b424dc35095cd80f#, 16#538484c19ef38c95#),
        (16#e12e13424bb40e13#, 16#2865a5f206b06fba#),
        (16#8cbccc096f5088cb#, 16#f93f87b7442e45d4#),
        (16#afebff0bcb24aafe#, 16#f78f69a51539d749#),
        (16#dbe6fecebdedd5be#, 16#b573440e5a884d1c#),
        (16#89705f4136b4a597#, 16#31680a88f8953031#),
        (16#abcc77118461cefc#, 16#fdc20d2b36ba7c3e#),
        (16#d6bf94d5e57a42bc#, 16#3d32907604691b4d#),
        (16#8637bd05af6c69b5#, 16#a63f9a49c2c1b110#),
        (16#a7c5ac471b478423#, 16#0fcf80dc33721d54#),
        (16#d1b71758e219652b#, 16#d3c36113404ea4a9#),
        (16#83126e978d4fdf3b#, 16#645a1cac083126ea#),
        (16#a3d70a3d70a3d70a#, 16#3d70a3d70a3d70a4#),
        (16#cccccccccccccccc#, 16#cccccccccccccccd#),
        (16#8000000000000000#, 16#0000000000000000#),
        (16#a000000000000000#, 16#0000000000000000#),
        (16#c800000000000000#, 16#0000000000000000#),
        (16#fa00000000000000#, 16#0000000000000000#),
        (16#9c40000000000000#, 16#0000000000000000#),
        (16#c350000000000000#, 16#0000000000000000#),
        (16#f424000000000000#, 16#0000000000000000#),
        (16#9896800000000000#, 16#0000000000000000#),
        (16#bebc200000000000#, 16#0000000000000000#),
        (16#ee6b280000000000#, 16#0000000000000000#),
        (16#9502f90000000000#, 16#0000000000000000#),
        (16#ba43b74000000000#, 16#0000000000000000#),
        (16#e8d4a51000000000#, 16#0000000000000000#),
        (16#9184e72a00000000#, 16#0000000000000000#),
        (16#b5e620f480000000#, 16#0000000000000000#),
        (16#e35fa931a0000000#, 16#0000000000000000#),
        (16#8e1bc9bf04000000#, 16#0000000000000000#),
        (16#b1a2bc2ec5000000#, 16#0000000000000000#),
        (16#de0b6b3a76400000#, 16#0000000000000000#),
        (16#8ac7230489e80000#, 16#0000000000000000#),
        (16#ad78ebc5ac620000#, 16#0000000000000000#),
        (16#d8d726b7177a8000#, 16#0000000000000000#),
        (16#878678326eac9000#, 16#0000000000000000#),
        (16#a968163f0a57b400#, 16#0000000000000000#),
        (16#d3c21bcecceda100#, 16#0000000000000000#),
        (16#84595161401484a0#, 16#0000000000000000#),
        (16#a56fa5b99019a5c8#, 16#0000000000000000#),
        (16#cecb8f27f4200f3a#, 16#0000000000000000#),
        (16#813f3978f8940984#, 16#4000000000000000#),
        (16#a18f07d736b90be5#, 16#5000000000000000#),
        (16#c9f2c9cd04674ede#, 16#a400000000000000#),
        (16#fc6f7c4045812296#, 16#4d00000000000000#),
        (16#9dc5ada82b70b59d#, 16#f020000000000000#),
        (16#c5371912364ce305#, 16#6c28000000000000#),
        (16#f684df56c3e01bc6#, 16#c732000000000000#),
        (16#9a130b963a6c115c#, 16#3c7f400000000000#),
        (16#c097ce7bc90715b3#, 16#4b9f100000000000#),
        (16#f0bdc21abb48db20#, 16#1e86d40000000000#),
        (16#96769950b50d88f4#, 16#1314448000000000#),
        (16#bc143fa4e250eb31#, 16#17d955a000000000#),
        (16#eb194f8e1ae525fd#, 16#5dcfab0800000000#),
        (16#92efd1b8d0cf37be#, 16#5aa1cae500000000#),
        (16#b7abc627050305ad#, 16#f14a3d9e40000000#),
        (16#e596b7b0c643c719#, 16#6d9ccd05d0000000#),
        (16#8f7e32ce7bea5c6f#, 16#e4820023a2000000#),
        (16#b35dbf821ae4f38b#, 16#dda2802c8a800000#),
        (16#e0352f62a19e306e#, 16#d50b2037ad200000#),
        (16#8c213d9da502de45#, 16#4526f422cc340000#),
        (16#af298d050e4395d6#, 16#9670b12b7f410000#),
        (16#daf3f04651d47b4c#, 16#3c0cdd765f114000#),
        (16#88d8762bf324cd0f#, 16#a5880a69fb6ac800#),
        (16#ab0e93b6efee0053#, 16#8eea0d047a457a00#),
        (16#d5d238a4abe98068#, 16#72a4904598d6d880#),
        (16#85a36366eb71f041#, 16#47a6da2b7f864750#),
        (16#a70c3c40a64e6c51#, 16#999090b65f67d924#),
        (16#d0cf4b50cfe20765#, 16#fff4b4e3f741cf6d#),
        (16#82818f1281ed449f#, 16#bff8f10e7a8921a4#),
        (16#a321f2d7226895c7#, 16#aff72d52192b6a0d#),
        (16#cbea6f8ceb02bb39#, 16#9bf4f8a69f764490#),
        (16#fee50b7025c36a08#, 16#02f236d04753d5b4#),
        (16#9f4f2726179a2245#, 16#01d762422c946590#),
        (16#c722f0ef9d80aad6#, 16#424d3ad2b7b97ef5#),
        (16#f8ebad2b84e0d58b#, 16#d2e0898765a7deb2#),
        (16#9b934c3b330c8577#, 16#63cc55f49f88eb2f#),
        (16#c2781f49ffcfa6d5#, 16#3cbf6b71c76b25fb#),
        (16#f316271c7fc3908a#, 16#8bef464e3945ef7a#),
        (16#97edd871cfda3a56#, 16#97758bf0e3cbb5ac#),
        (16#bde94e8e43d0c8ec#, 16#3d52eeed1cbea317#),
        (16#ed63a231d4c4fb27#, 16#4ca7aaa863ee4bdd#),
        (16#945e455f24fb1cf8#, 16#8fe8caa93e74ef6a#),
        (16#b975d6b6ee39e436#, 16#b3e2fd538e122b44#),
        (16#e7d34c64a9c85d44#, 16#60dbbca87196b616#),
        (16#90e40fbeea1d3a4a#, 16#bc8955e946fe31cd#),
        (16#b51d13aea4a488dd#, 16#6babab6398bdbe41#),
        (16#e264589a4dcdab14#, 16#c696963c7eed2dd1#),
        (16#8d7eb76070a08aec#, 16#fc1e1de5cf543ca2#),
        (16#b0de65388cc8ada8#, 16#3b25a55f43294bcb#),
        (16#dd15fe86affad912#, 16#49ef0eb713f39ebe#),
        (16#8a2dbf142dfcc7ab#, 16#6e3569326c784337#),
        (16#acb92ed9397bf996#, 16#49c2c37f07965404#),
        (16#d7e77a8f87daf7fb#, 16#dc33745ec97be906#),
        (16#86f0ac99b4e8dafd#, 16#69a028bb3ded71a3#),
        (16#a8acd7c0222311bc#, 16#c40832ea0d68ce0c#),
        (16#d2d80db02aabd62b#, 16#f50a3fa490c30190#),
        (16#83c7088e1aab65db#, 16#792667c6da79e0fa#),
        (16#a4b8cab1a1563f52#, 16#577001b891185938#),
        (16#cde6fd5e09abcf26#, 16#ed4c0226b55e6f86#),
        (16#80b05e5ac60b6178#, 16#544f8158315b05b4#),
        (16#a0dc75f1778e39d6#, 16#696361ae3db1c721#),
        (16#c913936dd571c84c#, 16#03bc3a19cd1e38e9#),
        (16#fb5878494ace3a5f#, 16#04ab48a04065c723#),
        (16#9d174b2dcec0e47b#, 16#62eb0d64283f9c76#),
        (16#c45d1df942711d9a#, 16#3ba5d0bd324f8394#),
        (16#f5746577930d6500#, 16#ca8f44ec7ee36479#),
        (16#9968bf6abbe85f20#, 16#7e998b13cf4e1ecb#),
        (16#bfc2ef456ae276e8#, 16#9e3fedd8c321a67e#),
        (16#efb3ab16c59b14a2#, 16#c5cfe94ef3ea101e#),
        (16#95d04aee3b80ece5#, 16#bba1f1d158724a12#),
        (16#bb445da9ca61281f#, 16#2a8a6e45ae8edc97#),
        (16#ea1575143cf97226#, 16#f52d09d71a3293bd#),
        (16#924d692ca61be758#, 16#593c2626705f9c56#),
        (16#b6e0c377cfa2e12e#, 16#6f8b2fb00c77836c#),
        (16#e498f455c38b997a#, 16#0b6dfb9c0f956447#),
        (16#8edf98b59a373fec#, 16#4724bd4189bd5eac#),
        (16#b2977ee300c50fe7#, 16#58edec91ec2cb657#),
        (16#df3d5e9bc0f653e1#, 16#2f2967b66737e3ed#),
        (16#8b865b215899f46c#, 16#bd79e0d20082ee74#),
        (16#ae67f1e9aec07187#, 16#ecd8590680a3aa11#),
        (16#da01ee641a708de9#, 16#e80e6f4820cc9495#),
        (16#884134fe908658b2#, 16#3109058d147fdcdd#),
        (16#aa51823e34a7eede#, 16#bd4b46f0599fd415#),
        (16#d4e5e2cdc1d1ea96#, 16#6c9e18ac7007c91a#),
        (16#850fadc09923329e#, 16#03e2cf6bc604ddb0#),
        (16#a6539930bf6bff45#, 16#84db8346b786151c#),
        (16#cfe87f7cef46ff16#, 16#e612641865679a63#),
        (16#81f14fae158c5f6e#, 16#4fcb7e8f3f60c07e#),
        (16#a26da3999aef7749#, 16#e3be5e330f38f09d#),
        (16#cb090c8001ab551c#, 16#5cadf5bfd3072cc5#),
        (16#fdcb4fa002162a63#, 16#73d9732fc7c8f7f6#),
        (16#9e9f11c4014dda7e#, 16#2867e7fddcdd9afa#),
        (16#c646d63501a1511d#, 16#b281e1fd541501b8#),
        (16#f7d88bc24209a565#, 16#1f225a7ca91a4226#),
        (16#9ae757596946075f#, 16#3375788de9b06958#),
        (16#c1a12d2fc3978937#, 16#0052d6b1641c83ae#),
        (16#f209787bb47d6b84#, 16#c0678c5dbd23a49a#),
        (16#9745eb4d50ce6332#, 16#f840b7ba963646e0#),
        (16#bd176620a501fbff#, 16#b650e5a93bc3d898#),
        (16#ec5d3fa8ce427aff#, 16#a3e51f138ab4cebe#),
        (16#93ba47c980e98cdf#, 16#c66f336c36b10137#),
        (16#b8a8d9bbe123f017#, 16#b80b0047445d4184#),
        (16#e6d3102ad96cec1d#, 16#a60dc059157491e5#),
        (16#9043ea1ac7e41392#, 16#87c89837ad68db2f#),
        (16#b454e4a179dd1877#, 16#29babe4598c311fb#),
        (16#e16a1dc9d8545e94#, 16#f4296dd6fef3d67a#),
        (16#8ce2529e2734bb1d#, 16#1899e4a65f58660c#),
        (16#b01ae745b101e9e4#, 16#5ec05dcff72e7f8f#),
        (16#dc21a1171d42645d#, 16#76707543f4fa1f73#),
        (16#899504ae72497eba#, 16#6a06494a791c53a8#),
        (16#abfa45da0edbde69#, 16#0487db9d17636892#),
        (16#d6f8d7509292d603#, 16#45a9d2845d3c42b6#),
        (16#865b86925b9bc5c2#, 16#0b8a2392ba45a9b2#),
        (16#a7f26836f282b732#, 16#8e6cac7768d7141e#),
        (16#d1ef0244af2364ff#, 16#3207d795430cd926#),
        (16#8335616aed761f1f#, 16#7f44e6bd49e807b8#),
        (16#a402b9c5a8d3a6e7#, 16#5f16206c9c6209a6#),
        (16#cd036837130890a1#, 16#36dba887c37a8c0f#),
        (16#802221226be55a64#, 16#c2494954da2c9789#),
        (16#a02aa96b06deb0fd#, 16#f2db9baa10b7bd6c#),
        (16#c83553c5c8965d3d#, 16#6f92829494e5acc7#),
        (16#fa42a8b73abbf48c#, 16#cb772339ba1f17f9#),
        (16#9c69a97284b578d7#, 16#ff2a760414536efb#),
        (16#c38413cf25e2d70d#, 16#fef5138519684aba#),
        (16#f46518c2ef5b8cd1#, 16#7eb258665fc25d69#),
        (16#98bf2f79d5993802#, 16#ef2f773ffbd97a61#),
        (16#beeefb584aff8603#, 16#aafb550ffacfd8fa#),
        (16#eeaaba2e5dbf6784#, 16#95ba2a53f983cf38#),
        (16#952ab45cfa97a0b2#, 16#dd945a747bf26183#),
        (16#ba756174393d88df#, 16#94f971119aeef9e4#),
        (16#e912b9d1478ceb17#, 16#7a37cd5601aab85d#),
        (16#91abb422ccb812ee#, 16#ac62e055c10ab33a#),
        (16#b616a12b7fe617aa#, 16#577b986b314d6009#),
        (16#e39c49765fdf9d94#, 16#ed5a7e85fda0b80b#),
        (16#8e41ade9fbebc27d#, 16#14588f13be847307#),
        (16#b1d219647ae6b31c#, 16#596eb2d8ae258fc8#),
        (16#de469fbd99a05fe3#, 16#6fca5f8ed9aef3bb#),
        (16#8aec23d680043bee#, 16#25de7bb9480d5854#),
        (16#ada72ccc20054ae9#, 16#af561aa79a10ae6a#),
        (16#d910f7ff28069da4#, 16#1b2ba1518094da04#),
        (16#87aa9aff79042286#, 16#90fb44d2f05d0842#),
        (16#a99541bf57452b28#, 16#353a1607ac744a53#),
        (16#d3fa922f2d1675f2#, 16#42889b8997915ce8#),
        (16#847c9b5d7c2e09b7#, 16#69956135febada11#),
        (16#a59bc234db398c25#, 16#43fab9837e699095#),
        (16#cf02b2c21207ef2e#, 16#94f967e45e03f4bb#),
        (16#8161afb94b44f57d#, 16#1d1be0eebac278f5#),
        (16#a1ba1ba79e1632dc#, 16#6462d92a69731732#),
        (16#ca28a291859bbf93#, 16#7d7b8f7503cfdcfe#),
        (16#fcb2cb35e702af78#, 16#5cda735244c3d43e#),
        (16#9defbf01b061adab#, 16#3a0888136afa64a7#),
        (16#c56baec21c7a1916#, 16#088aaa1845b8fdd0#),
        (16#f6c69a72a3989f5b#, 16#8aad549e57273d45#),
        (16#9a3c2087a63f6399#, 16#36ac54e2f678864b#),
        (16#c0cb28a98fcf3c7f#, 16#84576a1bb416a7dd#),
        (16#f0fdf2d3f3c30b9f#, 16#656d44a2a11c51d5#),
        (16#969eb7c47859e743#, 16#9f644ae5a4b1b325#),
        (16#bc4665b596706114#, 16#873d5d9f0dde1fee#),
        (16#eb57ff22fc0c7959#, 16#a90cb506d155a7ea#),
        (16#9316ff75dd87cbd8#, 16#09a7f12442d588f2#),
        (16#b7dcbf5354e9bece#, 16#0c11ed6d538aeb2f#),
        (16#e5d3ef282a242e81#, 16#8f1668c8a86da5fa#),
        (16#8fa475791a569d10#, 16#f96e017d694487bc#),
        (16#b38d92d760ec4455#, 16#37c981dcc395a9ac#),
        (16#e070f78d3927556a#, 16#85bbe253f47b1417#),
        (16#8c469ab843b89562#, 16#93956d7478ccec8e#),
        (16#af58416654a6babb#, 16#387ac8d1970027b2#),
        (16#db2e51bfe9d0696a#, 16#06997b05fcc0319e#),
        (16#88fcf317f22241e2#, 16#441fece3bdf81f03#),
        (16#ab3c2fddeeaad25a#, 16#d527e81cad7626c3#),
        (16#d60b3bd56a5586f1#, 16#8a71e223d8d3b074#),
        (16#85c7056562757456#, 16#f6872d5667844e49#),
        (16#a738c6bebb12d16c#, 16#b428f8ac016561db#),
        (16#d106f86e69d785c7#, 16#e13336d701beba52#),
        (16#82a45b450226b39c#, 16#ecc0024661173473#),
        (16#a34d721642b06084#, 16#27f002d7f95d0190#),
        (16#cc20ce9bd35c78a5#, 16#31ec038df7b441f4#),
        (16#ff290242c83396ce#, 16#7e67047175a15271#),
        (16#9f79a169bd203e41#, 16#0f0062c6e984d386#),
        (16#c75809c42c684dd1#, 16#52c07b78a3e60868#),
        (16#f92e0c3537826145#, 16#a7709a56ccdf8a82#),
        (16#9bbcc7a142b17ccb#, 16#88a66076400bb691#),
        (16#c2abf989935ddbfe#, 16#6acff893d00ea435#),
        (16#f356f7ebf83552fe#, 16#0583f6b8c4124d43#),
        (16#98165af37b2153de#, 16#c3727a337a8b704a#),
        (16#be1bf1b059e9a8d6#, 16#744f18c0592e4c5c#),
        (16#eda2ee1c7064130c#, 16#1162def06f79df73#),
        (16#9485d4d1c63e8be7#, 16#8addcb5645ac2ba8#),
        (16#b9a74a0637ce2ee1#, 16#6d953e2bd7173692#),
        (16#e8111c87c5c1ba99#, 16#c8fa8db6ccdd0437#),
        (16#910ab1d4db9914a0#, 16#1d9c9892400a22a2#),
        (16#b54d5e4a127f59c8#, 16#2503beb6d00cab4b#),
        (16#e2a0b5dc971f303a#, 16#2e44ae64840fd61d#),
        (16#8da471a9de737e24#, 16#5ceaecfed289e5d2#),
        (16#b10d8e1456105dad#, 16#7425a83e872c5f47#),
        (16#dd50f1996b947518#, 16#d12f124e28f77719#),
        (16#8a5296ffe33cc92f#, 16#82bd6b70d99aaa6f#),
        (16#ace73cbfdc0bfb7b#, 16#636cc64d1001550b#),
        (16#d8210befd30efa5a#, 16#3c47f7e05401aa4e#),
        (16#8714a775e3e95c78#, 16#65acfaec34810a71#),
        (16#a8d9d1535ce3b396#, 16#7f1839a741a14d0d#),
        (16#d31045a8341ca07c#, 16#1ede48111209a050#),
        (16#83ea2b892091e44d#, 16#934aed0aab460432#),
        (16#a4e4b66b68b65d60#, 16#f81da84d5617853f#),
        (16#ce1de40642e3f4b9#, 16#36251260ab9d668e#),
        (16#80d2ae83e9ce78f3#, 16#c1d72b7c6b426019#),
        (16#a1075a24e4421730#, 16#b24cf65b8612f81f#),
        (16#c94930ae1d529cfc#, 16#dee033f26797b627#),
        (16#fb9b7cd9a4a7443c#, 16#169840ef017da3b1#),
        (16#9d412e0806e88aa5#, 16#8e1f289560ee864e#),
        (16#c491798a08a2ad4e#, 16#f1a6f2bab92a27e2#),
        (16#f5b5d7ec8acb58a2#, 16#ae10af696774b1db#),
        (16#9991a6f3d6bf1765#, 16#acca6da1e0a8ef29#),
        (16#bff610b0cc6edd3f#, 16#17fd090a58d32af3#),
        (16#eff394dcff8a948e#, 16#ddfc4b4cef07f5b0#),
        (16#95f83d0a1fb69cd9#, 16#4abdaf101564f98e#),
        (16#bb764c4ca7a4440f#, 16#9d6d1ad41abe37f1#),
        (16#ea53df5fd18d5513#, 16#84c86189216dc5ed#),
        (16#92746b9be2f8552c#, 16#32fd3cf5b4e49bb4#),
        (16#b7118682dbb66a77#, 16#3fbc8c33221dc2a1#),
        (16#e4d5e82392a40515#, 16#0fabaf3feaa5334a#),
        (16#8f05b1163ba6832d#, 16#29cb4d87f2a7400e#),
        (16#b2c71d5bca9023f8#, 16#743e20e9ef511012#),
        (16#df78e4b2bd342cf6#, 16#914da9246b255416#),
        (16#8bab8eefb6409c1a#, 16#1ad089b6c2f7548e#),
        (16#ae9672aba3d0c320#, 16#a184ac2473b529b1#),
        (16#da3c0f568cc4f3e8#, 16#c9e5d72d90a2741e#),
        (16#8865899617fb1871#, 16#7e2fa67c7a658892#),
        (16#aa7eebfb9df9de8d#, 16#ddbb901b98feeab7#),
        (16#d51ea6fa85785631#, 16#552a74227f3ea565#),
        (16#8533285c936b35de#, 16#d53a88958f87275f#),
        (16#a67ff273b8460356#, 16#8a892abaf368f137#),
        (16#d01fef10a657842c#, 16#2d2b7569b0432d85#),
        (16#8213f56a67f6b29b#, 16#9c3b29620e29fc73#),
        (16#a298f2c501f45f42#, 16#8349f3ba91b47b8f#),
        (16#cb3f2f7642717713#, 16#241c70a936219a73#),
        (16#fe0efb53d30dd4d7#, 16#ed238cd383aa0110#),
        (16#9ec95d1463e8a506#, 16#f4363804324a40aa#),
        (16#c67bb4597ce2ce48#, 16#b143c6053edcd0d5#),
        (16#f81aa16fdc1b81da#, 16#dd94b7868e94050a#),
        (16#9b10a4e5e9913128#, 16#ca7cf2b4191c8326#),
        (16#c1d4ce1f63f57d72#, 16#fd1c2f611f63a3f0#),
        (16#f24a01a73cf2dccf#, 16#bc633b39673c8cec#),
        (16#976e41088617ca01#, 16#d5be0503e085d813#),
        (16#bd49d14aa79dbc82#, 16#4b2d8644d8a74e18#),
        (16#ec9c459d51852ba2#, 16#ddf8e7d60ed1219e#),
        (16#93e1ab8252f33b45#, 16#cabb90e5c942b503#),
        (16#b8da1662e7b00a17#, 16#3d6a751f3b936243#),
        (16#e7109bfba19c0c9d#, 16#0cc512670a783ad4#),
        (16#906a617d450187e2#, 16#27fb2b80668b24c5#),
        (16#b484f9dc9641e9da#, 16#b1f9f660802dedf6#),
        (16#e1a63853bbd26451#, 16#5e7873f8a0396973#),
        (16#8d07e33455637eb2#, 16#db0b487b6423e1e8#),
        (16#b049dc016abc5e5f#, 16#91ce1a9a3d2cda62#),
        (16#dc5c5301c56b75f7#, 16#7641a140cc7810fb#),
        (16#89b9b3e11b6329ba#, 16#a9e904c87fcb0a9d#),
        (16#ac2820d9623bf429#, 16#546345fa9fbdcd44#),
        (16#d732290fbacaf133#, 16#a97c177947ad4095#),
        (16#867f59a9d4bed6c0#, 16#49ed8eabcccc485d#),
        (16#a81f301449ee8c70#, 16#5c68f256bfff5a74#),
        (16#d226fc195c6a2f8c#, 16#73832eec6fff3111#),
        (16#83585d8fd9c25db7#, 16#c831fd53c5ff7eab#),
        (16#a42e74f3d032f525#, 16#ba3e7ca8b77f5e55#),
        (16#cd3a1230c43fb26f#, 16#28ce1bd2e55f35eb#),
        (16#80444b5e7aa7cf85#, 16#7980d163cf5b81b3#),
        (16#a0555e361951c366#, 16#d7e105bcc332621f#),
        (16#c86ab5c39fa63440#, 16#8dd9472bf3fefaa7#),
        (16#fa856334878fc150#, 16#b14f98f6f0feb951#),
        (16#9c935e00d4b9d8d2#, 16#6ed1bf9a569f33d3#),
        (16#c3b8358109e84f07#, 16#0a862f80ec4700c8#),
        (16#f4a642e14c6262c8#, 16#cd27bb612758c0fa#),
        (16#98e7e9cccfbd7dbd#, 16#8038d51cb897789c#),
        (16#bf21e44003acdd2c#, 16#e0470a63e6bd56c3#),
        (16#eeea5d5004981478#, 16#1858ccfce06cac74#),
        (16#95527a5202df0ccb#, 16#0f37801e0c43ebc8#),
        (16#baa718e68396cffd#, 16#d30560258f54e6ba#),
        (16#e950df20247c83fd#, 16#47c6b82ef32a2069#),
        (16#91d28b7416cdd27e#, 16#4cdc331d57fa5441#),
        (16#b6472e511c81471d#, 16#e0133fe4adf8e952#),
        (16#e3d8f9e563a198e5#, 16#58180fddd97723a6#),
        (16#8e679c2f5e44ff8f#, 16#570f09eaa7ea7648#)];

end VSS.JSON.Implementation.Numbers.Tables;
