# Web Console Architecture

## Current implementation

Print requests could come from different threads,
e.g.: 
- From EDT when user evaluates some expression in console 
- From Netty thread when `Runtime.consoleAPICalled` comes through Chrome Debug Protocol

### ConsolePrinter

The ConsolePrinter is responsible for:

- message
  batching ([Previously](https://jetbrains.team/p/ij/repositories/ultimate/files/d7a238bd8252f8424269b5164bcc7223c06f069b/plugins/JavaScriptDebugger/source/console/WebConsoleView.kt?tab=source&line=87&lines-count=1),
  batching was done using Alarm in WebConsoleView.)
- message deduplication (Previously was done
  in [ChromeDebugProcess](../../ChromeConnector/source/com/jetbrains/debugger/wip/ChromeDebugProcess.kt))

The implementation relies on coroutines and channels. Due to the fact that all requests are processed asynchronously
by one coroutine, it allows to minimize concurrency errors and offload EDT.

Here is a bird's-eye view on ConsolePrinter:

```mermaid
flowchart LR
  Network --> RequestQueue
  newLines["`User input
  Evaluate Expression result
  `"] --> RequestQueue

  subgraph ConsolePrinter
    RequestQueue[(request queue)]
    MessageBuffer[(message buffer)]
    WorkerCoroutine(["worker coroutine"])
    RequestQueue --> WorkerCoroutine
    WorkerCoroutine -- stores pending messages --> MessageBuffer
    WorkerCoroutine -- flushes message buffer --> MessagePrinter
  end

  MessagePrinter -- render messages --> JCEF
```

All commands sent to ConsolePrinter are put into a request queue and processed by a worker coroutine.

Batching is done at 2 levels:

- each message stores a set of commands for MessagePrinter in the form of [ConsoleMessageBuilder](ConsoleMessage.kt)
- a message buffer that stores ConsoleMessageBuilders

Sending messages for rendering is now done as a separate FlushRequest, which the ConsolePrinter sends to itself at a certain interval.

###  WebConsoleView
Consists of 2 main components: 
 - `ConsoleExecutionEditor`
   - input field (editor) for expression evaluation + history of evaluation requests 
 - `WebConsoleHistoryView`
   - Shows result of evaluation and logs. 
   - Scrollable

During construction of print request for `historyView`, MessageBuilder creates `PrintableEntity`'s. **Entities.kt** contains various
kinds of entities which console is able to print.
Each `PrintableEntity` has `getJSProps` method that returns map of properties which will be serialized to json and passed to htmlPanel.

### WebConsoleHistoryView

Contains `htmlPanel` which can be implemented differently. 
(Currently `htmlPanel` has JCEF implementation, previously it was JavaFX )

Methods that invoked by `WebConsoleView` are commonly wrapped with `runWhenPageReady` method. 
Print requests could come before html page is ready, so `runWhenPageReady` ensures that page is loaded or saves requests to be executed later.

**StylesConversion.kt** is used to convert IDE styles (like editor colors) to css. 

Before passing to `htmlPanel` all `PrintableEntity`'s passed through `prepareJSProps` method, 
which checks if `PrintableEntity` needs to be preserved in memory (e.g. if it is a debugger node), and do some preparations. 

**deferred Entities** are for those cases when we need to print something but at that moment we don't have info about how to present this entity
and actual presentation will come later.  

After preparation request is passed to JS part by invoking js method from *interop.js*

**interop.js**
contains methods for connecting Java <---> JS with each other.

Representing complex interactive objects like debugger tree requires calling Java methods and getting result back to JS.
For that there is `callJVM` method in _interop.js_.
If JS waits for result from Java it passes a _callback_ to `callJVM` method.

### JSBridge
Contains Java methods that can be called from JS

### Icons
*ResourceStaticServer* serves html page and icons.
We have 2 types of icons: 
- Static. There is a fixed, known in advance list of icons. Examples: prompt, warning, error icons
- Dynamic. Could be any icon. Icons inside debugger tree (`XDebuggerTreeNode.getIcon()`)

Static icons are listed in _icons.css_.
Dynamic icons assigned with special _id_'s in `ResourceStaticServer.cacheIconId`

## Comparison with the old implementation

The previous ConsolePrinter implementation used locks shared by multiple threads.
This led to blocking of the EDT thread if the processing of the received message was
long ([example](https://youtrack.jetbrains.com/issue/WEB-53630/Debugger-freezes)).
Also, locks limited the ConsolePrinter's throughput, and it could not cope with a large number of messages.

New implementation minimizes concurrency errors and EDT freezes because all requests are processed asynchronously
by one coroutine in background and EDT performs only rendering related tasks.

Also, previously, commands for [MessagePrinter](WebConsoleHistoryView.kt) were batched globally and flushed for rendering at a certain
interval
by [com.intellij.util.Alarm](../../../../community/platform/ide-core/src/com/intellij/util/Alarm.java). Global batching required locking to
prevent the contents of messages from being mixed.
Grouping printer commands inside the MessagePrinter and buffering messages for rendering allows to get rid of the global lock
and preserve

Grouping printer commands inside MessagePrinter and buffering messages for rendering allows us to get rid of global lock and
preserve the behavior of periodically sending a batch of messages for rendering.

### Fallback console

There is an old editor-based console implementation. It can be turned on with `js.debugger.webconsole` registry key.
It is also used as a fallback option in case JCEF is not supported by the user's runtime.