/* line ('\n') oriented socket interface. srecv the \n into \0 before
 * returning. Use non-blocking sockets.
 * 
 * After these return something other than SOK_SUCCESS or SOK_TOOLONG, you
 * cannot assume anything about the state of the stream
 */
#include "bool.h"
#include <stdlib.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>
#include <sys/socket.h>
#ifdef HAVE_SYS_SELECT_H
# include <sys/select.h>
#endif
#include <netinet/in.h>
#ifdef HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif
#include "util.h"
#include "main.h"
#include "socket.h"
#include "buffer.h"

#define SOK_BUFFER_IN_SIZE MAXLINE
#define SOK_BUFFER_OUT_SIZE MAXLINE

#define SOK_DEFAULT_IN_TIMEOUT_SECS 5
#define SOK_DEFAULT_OUT_TIMEOUT_SECS 5

enum {
	GLUE_NONE,
	GLUE_SOME,
	GLUE_ALL,
	BUFFER_TOO_SMALL
};

bool smake_buffer(struct SOK_BUFFER *buf, size_t bufsize);
void sdestroy_buffer(struct SOK_BUFFER *buf);
int glue(size_t *partial, char *buf, size_t bufsize, SOK *stream);
int postpone(char *buf, char *newline, size_t lentotal, SOK *stream);
void remove_carriage(char *buf);

bool smake_buffer(struct SOK_BUFFER *buf, size_t bufsize)
{
	buf->start = xmalloc(bufsize * sizeof(*buf->start));
	if (!buf->start)
		return TRUE;
	buf->offset = 0;
	buf->bufsize = bufsize;
	buf->timeout = SOK_DEFAULT_OUT_TIMEOUT_SECS;
	return FALSE;
}

void sdestroy_buffer(struct SOK_BUFFER *buf)
{
	        free(buf->start);
}

SOK *sdopen(int sd)
{
	SOK *stream = xmalloc(sizeof(*stream));
	if (!stream)
		return NULL;
	stream->sd = sd;
	if (smake_buffer(&stream->in, SOK_BUFFER_IN_SIZE)) {
		free(stream);
		return NULL;
	}
	if (smake_buffer(&stream->out, SOK_BUFFER_OUT_SIZE)) {
		sdestroy_buffer(&stream->in);
		free(stream);
		return NULL;
	}
	return stream;
}

bool sclose(SOK *stream)
{
	int ret;
	int errsav;
	if (!stream)
		return TRUE;
	sflush(stream);
	ret = close(stream->sd);
	errsav = errno;
	sdestroy_buffer(&stream->in);
	sdestroy_buffer(&stream->out);
	free(stream);
	errno = errsav;
	return ret;
}

int ssend(SOK *stream, const char *msg)
{
	return ssendn(stream, msg, strlen(msg));
}

/* Don't really send data until '\n' in message or buffer full. */
int ssendn(SOK *stream, const char *msg, size_t msglen)
{
	size_t bufleft = stream->out.bufsize - stream->out.offset;
	bool flush = FALSE;
	bool recur = FALSE;
	ssize_t extralen = msglen - bufleft;
	int ret;

	if (stream->out.bufsize==0)
		return SOK_TOOLONG;
	if (extralen > 0) {
		msglen = bufleft; recur = TRUE;
		flush = TRUE;
	} else if (strchr(msg, '\n')) {
		flush = TRUE;
	}
	memcpy(stream->out.start+stream->out.offset, msg, msglen);
	stream->out.offset += msglen;
	if (flush) {
		ret = sflush(stream);
		/* gcc does tail recursion with -O */
		if (ret==SOK_SUCCESS && recur)
			return ssendn(stream, msg+msglen, (size_t)extralen);
		return ret;
	}
	return SOK_SUCCESS;
}

int sflush(SOK *stream)
{
	int ret;
	struct timeval tv;
	struct timeval *tvpt = NULL;
	if (stream->out.timeout) {
		tv.tv_sec = stream->out.timeout;
		tv.tv_usec = 0;
		tvpt = &tv;
	}
	ret = sok_send(stream->sd, tvpt, stream->out.start, stream->out.offset);
	stream->out.offset = 0;
	return ret;
}

int srecv(SOK *stream, char *buf, size_t bufsize)
{
	fd_set sokset;
	struct timeval tv;
	struct timeval tvbak;
	struct timeval *tvpt = NULL;
	ssize_t ret;
	ssize_t offset = 0;
	ssize_t left = (ssize_t)bufsize;
	char *fin;

	/* FIXME: To avoid doing memcpy, we read directly into buf, then copy
	 * any leftovers into the stream buffer. If the extra stuff cannot all
	 * fit in the stream buffer we get stuffed.
	 * 
	 * As of 24/Sep/2000 we are using equal sized buffers so it doesn't
	 * matter.
	 */
	if (stream->in.bufsize < bufsize)
		notice_debug("srecv: stream buffer may be too small\n");

	switch(glue(&offset, buf, left, stream)) {
	case GLUE_ALL: remove_carriage(buf);
		       return SOK_SUCCESS;
	case BUFFER_TOO_SMALL:
		       return SOK_TOOLONG;
	case GLUE_SOME: /* offset has been changed */
	case GLUE_NONE:
			break;
	}
	
	if (stream->in.timeout) {
		tv.tv_sec = stream->in.timeout;
		tv.tv_usec = 0;
		tvpt = &tv;
		memcpy(&tvbak, tvpt, sizeof(struct timeval));
	}
	while (left > 0) {
		do {
			ret = read(stream->sd, buf+offset, left);
		} while (ret==-1 && errno==EINTR);
		if (ret==-1) {
			if (errno!=EAGAIN)
				return SOK_FAILED;
			do {
				if (tvpt)
					memcpy(tvpt, &tvbak,
							sizeof(struct timeval));
				FD_ZERO(&sokset);
				FD_SET(stream->sd, &sokset);
				ret = select(stream->sd+1, &sokset, NULL, NULL,
						tvpt);
			} while (ret==-1 && errno==EINTR);
			switch(ret) {
		   	case -1: return SOK_FAILED;
			case 0: return SOK_TIMEOUT;
			default: break;
			}
		} else if (ret==0) {
			return SOK_FAILED; /* EOF */
		} else {
			fin = memchr(buf+offset, '\n', ret);
			left -= ret;
			offset += ret;
			if (fin) {
				if (postpone(buf, fin, offset, stream)) {
					return SOK_TOOLONG;
				} else {
					remove_carriage(buf);
					return SOK_SUCCESS;
				}
			}
		}
	}
	return SOK_TOOLONG;
}

/* Telnet clients like to end with \r\n. Removes any trailing \r. */
void remove_carriage(char *buf)
{
	char *end = buf+strlen(buf)-1;
	if (*buf && *end=='\r')
		*end = '\0';
	return;
}

int postpone(char *buf, char *newline, size_t lentotal, SOK *stream)
{
	char *extra = newline + 1;
	/* XXX: stream->in.offset will always = 0 here ??? */
	size_t bufleft = stream->in.bufsize - stream->in.offset;
	size_t extralen;

	*newline = '\0';
	extralen = lentotal - (strlen(buf) + 1);
	if (extralen==0) {
		return 0;
	} else if (extralen > bufleft) {
		notice_debug("stream buffer too small. extralen=%ld "
				"bufleft=%ld\n", extralen, bufleft);
		return -1;
	}
	memcpy(stream->in.start+stream->in.offset, extra, extralen);
	stream->in.offset += extralen;
	return 0;
}

int glue(size_t *partial, char *buf, size_t bufsize, SOK *stream)
{
	char *newline;
	size_t extralen;
	size_t startlen;
	if (!stream->in.offset)
		return GLUE_NONE;
	newline = memchr(stream->in.start, '\n', stream->in.offset);
	if (!newline) {
		if (stream->in.offset > bufsize)
			return BUFFER_TOO_SMALL;
		memcpy(buf, stream->in.start, stream->in.offset);
		*partial = stream->in.offset;
		stream->in.offset = 0;
		return GLUE_SOME;
	}
	*newline = '\0';
	startlen = strlen(stream->in.start) + 1;
	memcpy(buf, stream->in.start, startlen);
	extralen = stream->in.offset - startlen;
	memmove(stream->in.start, stream->in.start+startlen, extralen);
	stream->in.offset = extralen;
	return GLUE_ALL;
}

/* bufsize must be >= minimum amount left in buffer.
 *
 * i.e. use the same buffer/size as we got the SOK_TOOLONG for */
bool discard(SOK *stream, char *buf, ssize_t bufsize)
{
	struct timeval tv;
	struct timeval *tvpt = NULL;
	int ret;
	if (stream->in.timeout) {
		tv.tv_sec = stream->in.timeout;
		tv.tv_usec = 0;
		tvpt = &tv;
	}
	do {
		if (sok_recv(stream->sd, tvpt, buf, bufsize)!=SOK_SUCCESS)
			return TRUE;
		ret = srecv(stream, buf, bufsize);
	} while (ret==SOK_TOOLONG);
	if (ret!=SOK_SUCCESS)
		return TRUE;
	return FALSE;
}

/* Like select, but also FD_SETs for streams with data in their in buffer.
 *
 * Used because we can receive data like "Hello\nWhatever\n". It is possible
 * that, having used the Hello\n, we could have Whatever\n sitting in the
 * stream buffer without breaking out of select because there is no more data
 * on the file descriptor.
 *
 * i.e. Use SOK_select when you're using select on fd's in a stream buffer.
 * Make sure you still put the sd's in the streams in readfds. */
int SOK_select(int n, fd_set *readfds, fd_set *writefds, fd_set *exceptfds, 
		struct timeval *timeout, SOK **streams)
{
	struct timeval zero;
	fd_set streamset;
	int i;
	int ret;

	if (!readfds || !streams) {
		notice_debug("bug: SOK_select called when select wanted\n");
		return select(n, readfds, writefds, exceptfds, timeout);
	}
	FD_ZERO(&streamset);
	for (i=0; streams[i]; ++i) {
		if (streams[i]->in.offset &&
				(streams[i]->sd < n) &&
				FD_ISSET(streams[i]->sd, readfds)) {
			FD_SET(streams[i]->sd, &streamset);
			/* don't bother selecting on it if we already know. */
			FD_CLR(streams[i]->sd, readfds);
			/* return immediately */
			zero.tv_sec = 0; zero.tv_usec = 0;
			timeout = &zero;
		}
	}
	ret = select(n, readfds, writefds, exceptfds, timeout);
	/* To be consistent with select(), we return even when errno==EINTR */
	if (ret==-1)
		return -1;
	for (i=0; streams[i]; ++i) {
		if (FD_ISSET(streams[i]->sd, &streamset)) {
			FD_SET(streams[i]->sd, readfds);
			++ret;
		}
	}
	return ret;
}
