#include "bool.h"
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/wait.h>
#include <sys/socket.h>
#include "util.h"
#include "network.h"
#include "socket.h"
#include "exec.h"
#include "item.h"
#include "main.h"

static struct clients *prev_item_from_pid(struct clients *start, pid_t pid);

/* By convention, if a function takes "first", it wants "start->next" */

/* Changing the clients list. These also handle things like onconnect. Items
 * are identified by their unique pid.
 *
 * item_add	- add an item to the end of the list.
 * item_rem	- remove an item.
 * item_con	- set the connected flag on or off.
 * item_auth	- set the username.
 *
 */

/* Getting information about an item from its pid or onaction pid.
 *
 * item_from_pid	- get a pointer to an item.
 * prev_item_from_pid	- get a pointer to the previous item.
 * item_from_onaction	- get a pointer to an item.
 */

/* Do something to all items.
 *
 * setall_disconnected	- like item_con, but for all items.
 * kill_remove_all	- kills, closes and frees all clients. Does *not* run
 * 			  on_disconnect.
 * close_all_items	- closes pwrite and sok for all items.
 *
 */

bool item_add(struct clients **end, pid_t pid, int sok, int pwrite, 
		char *peername) 
{
	struct clients *new = xmalloc(sizeof(*new));
	if (!new) {
		(void)kill(pid, SIGTERM);
		return TRUE;
	}
	new->pid = pid;
	new->sok = sok;
	new->connected = FALSE;
	new->pwrite = pwrite;
	new->onconnect = NOT_RUN;
	new->ondisconnect = NOT_RUN;
	term_strncpy(new->username, "%unknown", USER_LEN);
	new->peername = peername;
	new->next = NULL;
	(*end)->next = new;
	*end = new;
	return FALSE;
}

/* Removes the item with the specified pid. lastp may be NULL. See comments
 * for item_con below. */
int item_rem(struct clients *start, struct clients **end, pid_t pid,
		pid_t *lastp)
{
	struct clients *prev = prev_item_from_pid(start, pid);
	struct clients *item;
	int ret = ITEM_NORMAL;
	if (!prev)
		return ITEM_NOTFOUND;
	item = prev->next;
	if (!item->next)
		*end = prev;
	if (QIS_RUNNING(item->onconnect)) {
		/* item_rem will be called again later.
		 * The second time we wil *not* adjust pon or sokcount. */
		close(item->sok); item->sok = -1;
		close(item->pwrite); item->pwrite = -1;
		if (item->connected) {
			/* onaction_maybe_died will run ondisconnect. */
			item->connected = FALSE;
			return ITEM_DISCONNECT;
		}
		return ITEM_NORMAL;
	}
	if (item->connected) {
		(void)ondisconnect(lastp, item);
		ret = ITEM_DISCONNECT;
	}
	close(item->sok);
	close(item->pwrite);
	free(item->peername);

	prev->next = item->next;
	free(item);
	return ret;
}

/* Set connected flag in list for specified pid.
 *
 * If we are the first client to CONNECT (i.e. because of us, commandon is run)
 * firstlastp should point to &pid->first_onconnect.
 *
 * If we are the last client to DISCONNECT (i.e. because of us, commandoff
 * is run) firstlastp should point to &pid->last_ondisconnect.
 *
 * Otherwise firstlasp should be NULL.
 *
 * An example of where this is important is when using onconnect with multiple
 * providers to set which provider we will dial. It is vital that the first
 * onconnect command (the one which causes the provider to be dialed) runs
 * before commandon.
 */
bool item_con(pid_t pid, pid_t *firstlastp, struct clients *first, bool action)
{
	struct clients *item = item_from_pid(first, pid);
	if (!item) {
		notice_debug("item_con: pid not found\n");
		return TRUE;
	}
	if (action==item->connected)
		return TRUE; /* no change */
	item->connected = action;
	(void)onaction(action, firstlastp, item);
	return FALSE;
}

bool item_auth(pid_t pid, char *newusername, struct clients *first)
{
	struct clients *item = item_from_pid(first, pid);
	if (!item) {
		notice_debug("item_auth: pid not found\n");
		return TRUE;
	}
	term_strncpy(item->username, newusername, USER_LEN);
	return FALSE;
}

struct clients *item_from_pid(struct clients *first, pid_t pid)
{
	while (first && first->pid!=pid)
		first = first->next;
	return first;
}

/* returns a pointer to the item where item->next->pid==pid */
struct clients *prev_item_from_pid(struct clients *start, pid_t pid)
{
	if (!start->next)
		return NULL;
	while (start->next->pid!=pid) {
		start = start->next;
		if (!start->next)
			return NULL;
	}
	return start;
}

struct clients *item_from_onaction(struct clients *first, pid_t onaction,
		bool *onconnect)
{
	while (first) {
		if (first->onconnect==onaction) {
			*onconnect = TRUE;
			return first;
		}
		if (first->ondisconnect==onaction) {
			*onconnect = FALSE;
			return first;
		}
		first = first->next;
	}
	return NULL;
}

/* After calling this, set state->pon to 0. */
void setall_disconnected(struct clients *first, pid_t *lastp)
{
	while (first) {
		if (first->connected) {
			first->connected = FALSE;
			(void)ondisconnect(lastp, first);
		}
		first = first->next;
	}
	return;
}

/* Kills all processes in list and removes records. Doesn't do waitpid or
 * on_disconnect. Only use this when dwun is going to exit. */
void kill_remove_all(struct clients *start, struct clients **end)
{
	struct clients *next;
	struct clients *curr = start->next;

	while (curr) {
		(void)kill(curr->pid, SIGTERM);
		shutdown(curr->sok, 2); /* make sure the client is cut off */
		//close(curr->sok);
		close(curr->pwrite);
		free(curr->peername);
		next = curr->next;
		free(curr);
		curr = next;
	}
	*end = start;
	(*end)->next = NULL;
	return;
}

void close_all_items(struct clients *first)
{
	while (first) {
		close(first->sok);
		close(first->pwrite);
		first = first->next;
	}
	return;
}
