/* This is where all the action happens. */

#include "bool.h"
#include "state.h"
#include "network.h"
#include "util.h"
#include "connect.h"
#include "disconnect.h"
#include "exec.h"
#include "item.h"
#include "rets.h"
#include <errno.h>
#include <sys/wait.h>
#include <time.h>
#include <unistd.h>

/* Won't run commandon until pre_commandon has exited. */
#define WAIT_FOR_PRE_COMMANDON 1

static void childdead(pid_t deadpid, struct pid_s *pid, struct state_s *state,
		struct fail_s *fail, int *sokcount, struct clients *start,
		struct clients **end, time_t *connect_time);
static pid_t onaction_maybe_died(pid_t deadpid, struct pid_s *pid,
		struct state_s *state, struct clients *first);
static void connected(pid_t commandon_pid, int *constatus, 
		struct clients *first);
static void fatal(int *pon, struct fail_s *fail, pid_t *lastp,
		struct clients *first);
void act_on_signal(int event, struct pid_s *pid, struct state_s *state,
		struct fail_s *fail, int *sokcount, struct clients *start,
		struct clients **end, time_t *connect_time,
		int *time_to_hangup)
{
	pid_t deadpid;
	bool fatal_err;
	
	switch(event) {
	case ACT_HANGUP:
		notice_debug("signal (SIGHUP)\n");
		*time_to_hangup = QSHUTDOWN;
		break;
	case ACT_CHILDDEAD:
		while ((deadpid = waitpid_noint(0, NULL, WNOHANG)) > 0) {
			childdead(deadpid, pid, state, fail, sokcount, start,
					end, connect_time);
		}
		break;
	case ACT_FATAL:
		notice_debug("signal (SIGINT)\n");
		fatal(&state->pon, fail, &pid->last_ondisc, start->next);
		break;
	case ACT_RECONNECT:
		notice_debug("signal (SIGUSR2)\n");
		if (state->con==CONS_DISCONNECTING ||
		    state->con==CONS_CONNECTING ||
		    state->con==CONS_CONNECTED) {
			(void)disconnect(pid->commandon, &pid->commandoff);
			state->con = CONS_DISCONNECTING;
		}
		break;
	case ACT_CONNECTED:
		notice_debug("signal (SIGUSR1)\n");
		connected(pid->commandon, &state->con, start->next);
		*connect_time = time(NULL);
		fail->noconnect = fail->toosoon = 0; /* XXX: necessary? */
		break;
	case ACT_CANDIAL:
		state->candial = TRUE; /* adjust() called in network.c */
		break;
	default: 
		notice_debug("act_on_signal: invalid signal %d\n", event);
		break;
	}

	fatal_err = FALSE;
	if (max_exceeded(fail->toosoon, MAX_COMMANDON_FAIL)) {
		notice("commandon died too soon %d times\n", fail->toosoon);
		fatal_err = TRUE;
	} else if (max_exceeded(fail->noconnect, set.max_redials)) {
		notice("commandon died without us connecting %d times\n", 
				fail->noconnect);
		fatal_err = TRUE;
	}
	if (fatal_err)
		fatal(&state->pon, fail, &pid->last_ondisc, start->next);
	return;
}

void childdead(pid_t deadpid, struct pid_s *pid, struct state_s *state,
		struct fail_s *fail, int *sokcount, struct clients *start,
		struct clients **end, time_t *connect_time)
{
	pid_t p;
	if (deadpid==pid->commandon) {
		notice("disconnected\n");
		notice_debug("commandon %lu died\n", (u_long)deadpid);
		if (state->con==CONS_CONNECTED) {
			/* XXX: same comments about using state->pon==0 when
			 * we mean "did *we* mean to disconnect, or did it
			 * just drop out" as below. REF:ABCDEF */
			if(set.staydropped && state->pon) {
				/* don't redial */
				fatal(&state->pon, fail, &pid->last_ondisc,
						start->next);
			}
		}
		pid->commandon = NOT_RUN; /* XXX */
		*connect_time = -1;
		if (set.post_commandon)
			pid->post_commandon = run_fd3open(set.post_commandon);
		/* Not failing "in a row" if we stop dialing in between. */
		if (state->pon==0) {
			fail->noconnect = fail->toosoon = 0;
			state->con = CONS_DISCONNECTED;
			return;
		}
		state->con = CONS_DISCONNECTED;
		sendtousers("%con", "#Reconnecting\n", start->next);
		/* FIXME: If someone CONNECTed -after- we killed commandon,
		 * but -before- we got the SIGCHLD (so we increased pon
		 * without launching commandon) it's not right to increase
		 * noconnect.
		 *
		 * Will mean noconnect may be 1 greater than it should. Oh
		 * well.
		 *
		 * We need a better way to say "-we- killed commandon" than
		 * "does state->pon==0 ?" REF:ABCDEF
		 */
		fail->noconnect++;
		if (state->candial)
			fail->toosoon = 0; /* Not "in a row" */
		else
			fail->toosoon++;
		return;
	}

	if (deadpid==pid->commandoff) {
		pid->commandoff = HAVE_RUN; /* XXX */
		notice_debug("commandoff %lu died\n", (u_long)deadpid);
		return;
	} else if (deadpid==pid->can_disconnect) {
		pid->can_disconnect = HAVE_RUN;
		return;
	} else if (deadpid==pid->first_onconnect) {
		pid->first_onconnect = HAVE_RUN; /* fall-through */
	} else if (deadpid==pid->last_ondisc) {
		pid->last_ondisc = HAVE_RUN; /* fall-through */
	} else if (deadpid==pid->pre_commandon) {
		pid->pre_commandon = HAVE_RUN;
		return;
	} else if (deadpid==pid->post_commandon) {
		pid->post_commandon = HAVE_RUN;
		return;
	}

	switch(item_rem(start, end, deadpid, &pid->last_ondisc)) {
	case ITEM_NORMAL:
		notice_debug("client %lu died\n", (u_long)deadpid);
		*sokcount-=1;
		break;
	case ITEM_DISCONNECT:
		notice_debug("client %lu died\n", (u_long)deadpid);
		*sokcount-=1;
		state->pon--;
		break;
	case ITEM_NOTFOUND:
		p = onaction_maybe_died(deadpid, pid, state, start->next);
		if (p > 0) {
			/* We were waiting for onconnect to die so we could
			 * run ondisconnect. Now that's done, we can remove
			 * the item properly.
			 *
			 * Sokcount and pon have already been handled, don't
			 * do again. */
			(void)item_rem(start, end, p, &pid->last_ondisc);
		}
		break;
	}
	return;
}

pid_t onaction_maybe_died(pid_t deadpid, struct pid_s *pid,
		struct state_s *state, struct clients *first)
{
	bool onconnect;
	struct clients *item;
	pid_t *firstlastp = NULL;

	item = item_from_onaction(first, deadpid, &onconnect);
	if (!item) {
		notice_debug("exit of unknown child %lu\n", (u_long)deadpid);
		return 0;
	}
	/* An onconnect or ondisconnect died. */
	if (onconnect) {
		notice_debug("onconnect %lu of %lu died\n", 
				(u_long)item->onconnect, (u_long)item->pid);
		item->onconnect = HAVE_RUN;
		if (item->connected) {
			if (set.onconnect_wait && state->con==CONS_CONNECTED)
				sendalert("#Connected\n", item->sok);
			return 0;
		}
		/* Was waiting for onconnect to exit. */
		if (state->pon==1)
			firstlastp = &pid->last_ondisc;
		(void)ondisconnect(firstlastp, item);
		if (item->sok==-1)
			return item->pid; /* need to clean up */
		return 0;
	}
	notice_debug("ondisconnect %lu of %lu died\n",
			(u_long)item->ondisconnect, (u_long)item->pid);
	item->ondisconnect = HAVE_RUN;
	if (item->connected) {
		/* Was waiting for ondisconnect to exit. */
		if (state->pon==0)
			firstlastp = &pid->first_onconnect;
		(void)onconnect(firstlastp, item);
	}
	return 0;
}

void fatal(int *pon, struct fail_s *fail, pid_t *lastp, struct clients *first)
{
	notice("connecting failed\n");
	sendtousers("%con", "#Connect failed\n", first);
	setall_disconnected(first, lastp);
	*pon = 0;
	fail->noconnect = fail->toosoon = 0;
	return;
}

void connected(pid_t commandon_pid, int *constatus, struct clients *first)
{
	if (*constatus==CONS_CONNECTED) {
		notice_debug("ignoring SIGUSR1; already connected\n");
	} else if (!QIS_RUNNING(commandon_pid)) {
		notice_debug("ignoring SIGUSR1; commandon isn't running\n");
	} else {
		notice("connected\n");
		sendconnectedtousers("%con", first);
		*constatus = CONS_CONNECTED;
	}
	return;
}

/* Adjust connection state if necessary. */
bool adjust(struct pid_s *pid, struct state_s *state, struct clients *first)
{
	if (state->con==CONS_DISCONNECTING)
		return FALSE;

	/* Things which can mean we can now dial:
	 *
	 * state->con changes from anything(?) to CONS_DISCONNECTED
	 * state->pon changes from 0 to > 0.
	 * start->candial changes from FALSE to TRUE.
	 * first_onconnect, or commandoff die.
	 * pre_commandon dies. (for either of two reasons).
	 */

/* Dial if necessary. */
	if (state->con==CONS_DISCONNECTED) {
		if (		state->pon==0				||
				!state->candial				||
/* The default rcfile settings with dwunlog means that many systems will be
 * setup so pre_commandon will not exit. This means we cannot wait for
 * pre_commandon to exit before running commandon. */
#ifdef WAIT_FOR_PRE_COMMANDON 
				QIS_RUNNING(pid->pre_commandon)		||
#endif
				QIS_RUNNING(pid->first_onconnect)	||
				QIS_RUNNING(pid->commandoff)) {
			return FALSE;
		}
		alarm(MIN_TIME_BETWEEN_DIALS);
		state->candial = FALSE;
		if (set.pre_commandon && !QHAVE_RUN(pid->pre_commandon)) {
			pid->pre_commandon = run_fd3open(set.pre_commandon);
#ifdef WAIT_FOR_PRE_COMMANDON
			return FALSE;
#endif
		}
		pid->pre_commandon = NOT_RUN; /* Rather than HAVE_RUN. */
		pid->commandon = dial();
		if (-1==pid->commandon)
			return TRUE;
		sendtousers("%con", "#Connecting\n", first);
		state->con = CONS_CONNECTING;
		return FALSE;
	}

	/* Things which mean we can do something with disconnecting:
	 *
	 * state->pon changes from 0 to > 0 and pid->can_disconnect is running.
	 * 	(cancel disconnecting).
	 * commandoff, last_ondisc or post_commandon die.
	 * can_disconnect dies.
	 */
	   
/* Disconnect if necessary. */
	if (state->pon > 0) {
		if (QIS_RUNNING(pid->can_disconnect)) {
			/* no longer need to disconnect */
			kill(pid->can_disconnect, SIGTERM);
			/* childdead() will see as "unknown" */
			pid->can_disconnect = NOT_RUN;
		}
		return FALSE;
	}
	if (		QIS_RUNNING(pid->commandoff) ||
			QIS_RUNNING(pid->last_ondisc) ||
			QIS_RUNNING(pid->post_commandon)) {
		return FALSE;
	}
	if (set.disconnect_wait > 0) {
		switch (pid->can_disconnect) {
		case NOT_RUN:
			pid->can_disconnect = disconnect_timer();
			return FALSE; /* disconnect later */
		case HAVE_RUN:
			pid->can_disconnect = NOT_RUN;
			break; /* timer finished */
		default:
			return FALSE; /* timer is running */
		}
	}
	if (disconnect(pid->commandon, &pid->commandoff)) {
		/* Don't panic, just hope commandon happened to exit just
		 * before we tried to disconnect(). */
		return FALSE;
	}
	state->con = CONS_DISCONNECTING;
	return FALSE;
}
