package com.intellij.thymeleaf.spring.security;

import com.intellij.java.library.JavaLibraryModificationTracker;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.module.ModuleUtilCore;
import com.intellij.psi.*;
import com.intellij.psi.search.GlobalSearchScope;
import com.intellij.psi.util.CachedValueProvider.Result;
import com.intellij.psi.util.CachedValuesManager;
import com.intellij.psi.util.PropertyUtilBase;
import com.intellij.psi.util.PsiTypesUtil;
import com.intellij.psi.xml.XmlAttribute;
import com.intellij.spring.security.constants.SpringSecurityClassesConstants;
import com.intellij.spring.security.model.xml.converters.SecurityExpressionRootMethodsUtil;
import com.intellij.thymeleaf.constants.ThymeleafNamespaceConstants;
import com.intellij.thymeleaf.lang.support.ThymeleafContextVariablesProvider;
import com.intellij.thymeleaf.lang.support.beans.ThymeleafVariable;
import com.intellij.thymeleaf.lang.support.utils.ThymeleafCommonUtil;
import com.intellij.uast.UastModificationTracker;
import com.intellij.util.ArrayUtil;
import org.jetbrains.annotations.NotNull;

import java.util.*;

import static java.util.Collections.emptyList;

final class ThymeleafSpringSecurityContextVariableProvider extends ThymeleafContextVariablesProvider {
  private static final Map<String, String> myImplicitSecurityVariablesMap = new HashMap<>();

  static {
    myImplicitSecurityVariablesMap.put("_csrf", SpringSecurityClassesConstants.CSRF_TOKEN);
  }

  @Override
  public @NotNull Collection<? extends PsiVariable> getContextVariables(@NotNull PsiElement contextElement) {
    Module module = ModuleUtilCore.findModuleForPsiElement(contextElement);
    if (module == null) return emptyList();

    return getContextVariables(module);
  }

  private static Collection<PsiVariable> getContextVariables(Module module) {
    return CachedValuesManager.getManager(module.getProject()).getCachedValue(module, () -> {
      return Result.create(findContextVariables(module),
                           UastModificationTracker.getInstance(module.getProject()),
                           JavaLibraryModificationTracker.getInstance(module.getProject()));
    });
  }

  private static @NotNull Collection<PsiVariable> findContextVariables(@NotNull Module module) {
    Collection<PsiVariable> variables = new LinkedHashSet<>();

    JavaPsiFacade psiFacade = JavaPsiFacade.getInstance(module.getProject());
    GlobalSearchScope scope = GlobalSearchScope.moduleWithDependenciesAndLibrariesScope(module);
    PsiClass psiClass = psiFacade.findClass(SpringSecurityClassesConstants.AUTHENTICATION, scope);

    if (psiClass != null) {
      Map<String, PsiMethod> properties = PropertyUtilBase.getAllProperties(psiClass, false, true);
      for (Map.Entry<String, PsiMethod> entry : properties.entrySet()) {
        PsiMethod method = entry.getValue();
        PsiType propertyType = PropertyUtilBase.getPropertyType(method);
        if (propertyType != null) {
          String name = entry.getKey();
          if (name.equals("principal")) {
            PsiClass userDetailsClass = psiFacade.findClass(SpringSecurityClassesConstants.USER_DETAILS, scope);
            if (userDetailsClass != null) propertyType = PsiTypesUtil.getClassType(userDetailsClass);
          }
          variables.add(new ThymeleafVariable(name, propertyType, method));
        }
      }
    }
    for (Map.Entry<String, String> entry : myImplicitSecurityVariablesMap.entrySet()) {
      PsiClass aClass = psiFacade.findClass(entry.getValue(), scope);
      if (aClass != null) {
        variables.add(new ThymeleafVariable(entry.getKey(), PsiTypesUtil.getClassType(aClass), aClass));
      }
    }
    return List.copyOf(variables);
  }

  @Override
  public Collection<PsiMethod> getContextMethods(@NotNull PsiElement contextElement) {
    if (isSpringSecurityDialect(contextElement)) {
      final Module module = ModuleUtilCore.findModuleForPsiElement(contextElement);
      if (module != null) {
        return SecurityExpressionRootMethodsUtil
          .getExpressionRootMethods(module, SpringSecurityClassesConstants.WEB_SECURITY_EXPRESSION_ROOT);
      }
    }

    return super.getContextMethods(contextElement);
  }

  private static boolean isSpringSecurityDialect(PsiElement contextElement) {
    XmlAttribute contextAttribute = ThymeleafCommonUtil.getContextAttribute(contextElement);

    if (contextAttribute != null) {
      String ns = contextAttribute.getNamespace();

      for (String sec_ns : collectUris()) {
        if (sec_ns.equals(ns)) return true;
      }
    }
    return false;
  }

  private static String[] collectUris() {
    Set<String> all = new HashSet<>();

    Collections.addAll(all, ThymeleafNamespaceConstants.THYMELEAF_SPRING_SECURITY_6_URIS);
    Collections.addAll(all, ThymeleafNamespaceConstants.THYMELEAF_SPRING_SECURITY_5_URIS);
    Collections.addAll(all, ThymeleafNamespaceConstants.THYMELEAF_SPRING_SECURITY_4_URIS);
    Collections.addAll(all, ThymeleafNamespaceConstants.THYMELEAF_SPRING_SECURITY_3_URIS);

    return ArrayUtil.toStringArray(all);
  }
}
