/*  Pasang Emas. Enjoy a unique traditional game of Brunei.
    Copyright (C) 2015  Nor Jaidi Tuah

    This file is part of Pasang Emas.
      
    Pasang Emas is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
namespace Pasang {

class RandomPattern : Pattern {
    /**
     * A random pattern is specified using 4 letters. e.g., "OfRt".
     * The first pair ("Of" in the example above) specifies the symmetry of the outer part.
     * The second pair ("Rt" in the example above) specifies the symmetry of the inner part.
     * 
     * In each pair, the first letter ("O" and "R" in the above example) is for the first quadrant.
     * The second letter ("f" and "t" in the above example) is for the second quadrant.
     * The 3rd and the 4th quadrants are copied from the 1st and the 2nd.
     *
     * Codes for symmetry type in a quadrant: 
     *      R : totally random
     *      T : Reflection on x=3
     *      E : Reflection on y=3
     *      L : Reflection on y=6-x 
     *      J : Reflection on y=x
     *      S : 180 rotation about (3,3)
     *      H : T and E
     *      X : L and J
     *      F : 90 rotation about (3,3)
     *      O : H and X and F
     *      * : Wild card (see read_code)
     *
     * Codes for the second quadrant:
     *      If the pattern in the second quadrant is independent of the first,
     *      then the same capital letters as above are used.
     *      If the pattern in the second quadrant is a copy of the first,
     *      then use small letters (see figure below).
     *
     *          0 +----- 
     *          1 |\ :f/      Effect
     *          2 |i\:/t  i : translation
     *          3 |--:--  t : reflection about axes
     *          4 | /:\s  f : rotation about (0,0)
     *          5 |/ : \  s : translation of the 3rd quadrant
     *            012345  ? : wild card (see read_code)
     *
     * Short forms:
     *      Only one pair specified. e.g., "JL" for "JLJL", "RR" for "RRRR", "Rf" for "RfRf"
     *      Only one letter specified. e.g., "R" for "RfRf"
     *
     * Special code:
     *      "d" stands for "day and night" pattern.
     */
    private string spec;

    public RandomPattern (string name, string spec) {
        assert (spec[0] == ' ');
        base.empty (name);
        this.spec = spec;
        next_random ();
        temperature = 0;
    }

    /**
     * temperature == 0   ==>   cool green
     * temperature == 1   ==>   red hot
     */
    private double _temperature;
    public double temperature {
        get { return _temperature; }
        set {
            _temperature = value;
            create_pixbuf ();
        }
    }

    protected override void background (int n, int x, int y) {
        pixbuf_data[n + 0] = (uint8)((255 - 100 / (3 - (x + y) % 3)) * _temperature);
        pixbuf_data[n + 1] = (uint8)((150 - 100 / (3 - (x + y) % 3)) * (1 - _temperature / 2));
        pixbuf_data[n + 2] = 0;
        pixbuf_data[n + 3] = 255;
    }

    /**
     * Select a char randomly from s.
     */
    private char rand_char (string s) {
        return (char) (s[Random.int_range (0, (int) s.length)]);
    }

    /**
     * Read a character from spec. A wild card (*,?) is replaced with
     * an appropriate instance. A space and eos is translated into default_code.
     */
    private char read_spec (ref int index, char default_code) {
        // ensures ("RTELJSHXFOifts".contains(result.to_string ())) { TODO : this clause commented out due to vala bug
        if (index >= spec.length || spec[index] == ' ') 
          return default_code;
        char code = (char) (spec[index]);
        index++;
        return code == '*' ? rand_char ("RTELJSHXFO") :
               code == '?' ? rand_char ("ifts") :
                             code ;
    }

    /**
     * Build a pattern where the background and the foreground form the same pattern.
     *
     * Side effect: this.dots
     */
    private void next_random_day_and_night () {
        int seed_num = Random.int_range (0,2);
        string seed = seed_num == 0 ?
                          "PPPPPHHHHHP HPPPPHHHHPP HHPPPHHHPPP HHHPPHHPPPP HHHHPHPPPPP HHHHH" :
                          "PPPPPHHHPPP PPPPPHHHPPP PPPPPHHHPPP HHHHHHHHPPP HHHHHHHHPPP HHHHH";
        char[] dots = new char[NUM_WHITES + NUM_BLACKS + 1];
        int j = 0;
        for (int i=0; i < seed.length; i++) {
            if (seed[i] == ' ') continue;
            dots[j] = dots[120 - j] = (char) seed[i];
            j++;
        }
        dots[dots.length / 2] = '-';

        // Generate n pairs of points p[0] and p[1].
        // Swap p[0]<-->p[1] (and their images p[2]<-->p[3]).
        for (int n=Random.int_range(0, 20); n > 0; n--) {
            Point[] p = new Point[4];
            p[0] = Point.xy (Random.int_range (1, 6), Random.int_range (0, 6));
            p[1] = Point.xy (seed_num == 0 ? 6 - p[0].x : p[0].x, 5 - p[0].y);
            p[2] = Point.xy (-p[0].y, p[0].x);
            p[3] = Point.xy (-p[1].y, p[1].x);
            foreach (var t in p) {
                int xy = (t.x + 5) + 11 * (t.y + 5);
                dots[xy] = 'H' + 'P' - dots[xy];
                dots[120 - xy] = 'H' + 'P' - dots[120 - xy];
            }
        }

        var b = new StringBuilder ();
        for (int i=0; i < 60; i++) b.append_unichar (dots[i]);
        this.dots = b.str;
    }

    /**
     * From a random pattern specification such as " Rt Rt Rt *t RtJf JfRt",
     * select (randomly) a symmetry type such as "*t". Then construct a random
     * pattern based on the chosen symmetry type.
     *
     * Side effect: this.dots
     */
    public void next_random () {
        next_random_aux ();
        create_pixbuf ();
    }

    private void next_random_aux () {
        var index = 0;    // A random position containing a space
        do {
            index = Random.int_range (0, (int) spec.length);
        } while (spec[index] != ' ');
        index++;          // Skip the space    

        if (spec[index] == 'd') {
            next_random_day_and_night ();
            return;
        }
        
        // Construct implicit code. e.g., O => OfOf,  Sft => SfSt
        var code1 = read_spec (ref index, ' ');
        var code2 = read_spec (ref index, 'f');
        var code3 = read_spec (ref index, code1);
        var code4 = read_spec (ref index, code2);
        char[] dots = new char[NUM_WHITES + NUM_BLACKS + 1];

        for (int attempt=0; ; attempt++) { // break when we get the required number of blacks and whites
            // Fill with a background of black pieces
            for (int i=0; i < dots.length; i++) {
                dots[i] = 'H';
            }
            dots[dots.length / 2] = '-';
            
            // Overlay with a pattern using white pieces
            int center_range = Random.int_range (2*2, 4*4);
            int num_whites = 0;
            while (num_whites < NUM_WHITES) {
                // Generate a point on which to place a white piece. The centre of the board is (0,0).
                // Initially generate points inside the quarter x>=0, y>=0 (x and y axes included)
                int x = Random.int_range (0, 6);
                int y = Random.int_range (0, 6);
                if (x == 0 && y == 0) continue;
                bool center =  x * x + y * y <= center_range;
                char orig_type = (char) (center ? code3 : code1);
                char copy_type = (char) (center ? code4 : code2);
                bool duplicate_tile = copy_type >= 'a' && copy_type <= 'z';
                bool second_quarter_only = !duplicate_tile && (Random.int_range (0, 2) == 1);
                if (second_quarter_only) {
                    orig_type = copy_type;
                    copy_type = 'i';
                }
                
                // Tentative places for white pieces
                var s = orig_type.to_string ();
                Point[] grays = {Point.xy (x, y)};
                if ("JXO".contains (s))   grays += Point.xy (y, x);         // Reflect on y = x
                if ("FO".contains (s))    grays += Point.xy (6 - y, x);     // Reflect J on x = 3
                if ("THO".contains (s))   grays += Point.xy (6 - x, y);     // Reflect on x = 3
                if ("SXHFO".contains (s)) grays += Point.xy (6 - x, 6 - y); // Rotate about (3,3)
                if ("LXO".contains (s))   grays += Point.xy (6 - y, 6 - x); // Rotate J about (3,3)
                if ("FO".contains (s))    grays += Point.xy (y, 6 - x);     // Reflect J on y = 3
                if ("EHO".contains (s))   grays += Point.xy (x, 6 - y);     // Reflect on y = 3

                // Confirmed places for white pieces
                Point[] whites = {};
                foreach (var p in grays) {
                    assert (p.x >= 0 && p.y >= 0);
                    if (p.x > 5 || p.y > 5) continue;   // This may occur when the axis is rotated or reflected
                    if (!second_quarter_only) {
                        whites += p;
                        if (!duplicate_tile) continue;
                    }
                    Point p2 =
                        copy_type == 'f' ? (p.y == 0 || p.x == 0) ?
                                                Point.xy (p.y, p.x) :     // Reflect on y = x
                                                Point.xy (6 - p.y, p.x) : // Reflect J on x = 3
                        copy_type == 't' ? Point.xy (6 - p.x, p.y) :      // Reflect on x = 3
                        copy_type == 's' ? Point.xy (6 - p.x, 6 - p.y) :  // Rotate about (3,3)
                                           Point.xy (p.x, p.y);
                    int cut_off = second_quarter_only ? 1 : 0;
                    if (p2.x >= cut_off && p2.y >= cut_off && p2.x <= 5 && p2.y <= 5) {
                        // Translate onto the quarter x<=0, y>=0
                        if (p2.x != 0 && p2.y != 0) p2.x -= 6;
                        whites += p2;
                    }
                }
                
                foreach (var p in whites) {  
                    int xy = (p.x + 5) + 11 * (p.y + 5);
                    if (dots[xy] == 'P') continue;
                    if (center && p.x * p.x + p.y * p.y <= center_range || 
                        !center && p.x * p.x + p.y * p.y > center_range  || 
                        code1 == code3 && code2 == code4) 
                    {
                        assert (dots[xy] == 'H');
                        assert (dots[120 - xy] == 'H');
                        dots[xy] =
                        dots[120 - xy] = 'P';
                        num_whites += 2;
                    }
                }
            }//endwhile num_whites
            
            if (num_whites > NUM_WHITES) {
                // Invalid arrangement. This is not fatal.
                continue;
            }
            else {
                // debug ("%c%c%c%c", code1, code2, code3, code4);
                break;
            }
        }//endwhile attempt  

        var b = new StringBuilder ();
        for (int i=0; i < 60; i++) b.append_unichar (dots[i]);
        this.dots = b.str;
    }
}//class RandomPattern

}//namespace
// vim: tabstop=4: expandtab: textwidth=100: autoindent:
