package v5

type mockStore struct {
	data map[string]map[string][]Vulnerability
}

func newMockStore() *mockStore {
	d := mockStore{
		data: make(map[string]map[string][]Vulnerability),
	}
	d.stub()
	return &d
}

func (d *mockStore) stub() {
	d.data["debian:distro:debian:8"] = map[string][]Vulnerability{
		"neutron": {
			{
				PackageName:       "neutron",
				Namespace:         "debian:distro:debian:8",
				VersionConstraint: "< 2014.1.3-6",
				ID:                "CVE-2014-fake-1",
				VersionFormat:     "deb",
			},
			{
				PackageName:       "neutron",
				Namespace:         "debian:distro:debian:8",
				VersionConstraint: "< 2013.0.2-1",
				ID:                "CVE-2013-fake-2",
				VersionFormat:     "deb",
			},
			// poison the well! this is not a valid entry, but we want the matching process to survive and find other good results...
			{
				PackageName:       "neutron",
				Namespace:         "debian:distro:debian:8",
				VersionConstraint: "< 70.3.0-rc0", // intentionally bad value
				ID:                "CVE-2014-fake-3",
				VersionFormat:     "apk",
			},
		},
	}
	d.data["nvd:cpe"] = map[string][]Vulnerability{
		"activerecord": {
			{
				PackageName:       "activerecord",
				Namespace:         "nvd:cpe",
				VersionConstraint: "< 3.7.6",
				ID:                "CVE-2014-fake-3",
				VersionFormat:     "unknown",
				CPEs: []string{
					"cpe:2.3:*:activerecord:activerecord:*:*:*:*:*:rails:*:*",
				},
			},
			{
				PackageName:       "activerecord",
				Namespace:         "nvd:cpe",
				VersionConstraint: "< 3.7.4",
				ID:                "CVE-2014-fake-4",
				VersionFormat:     "unknown",
				CPEs: []string{
					"cpe:2.3:*:activerecord:activerecord:*:*:something:*:*:ruby:*:*",
				},
			},
			{
				PackageName:       "activerecord",
				Namespace:         "nvd:cpe",
				VersionConstraint: "= 4.0.1",
				ID:                "CVE-2014-fake-5",
				VersionFormat:     "unknown",
				CPEs: []string{
					"cpe:2.3:*:couldntgetthisrightcouldyou:activerecord:4.0.1:*:*:*:*:*:*:*", // shouldn't match on this
				},
			},
			{
				PackageName:       "activerecord",
				Namespace:         "nvd:cpe",
				VersionConstraint: "< 98SP3",
				ID:                "CVE-2014-fake-6",
				VersionFormat:     "unknown",
				CPEs: []string{
					"cpe:2.3:*:awesome:awesome:*:*:*:*:*:*:*:*", // shouldn't match on this
				},
			},
			// poison the well! this is not a valid entry, but we want the matching process to survive and find other good results...
			{
				PackageName:       "activerecord",
				Namespace:         "nvd:cpe",
				VersionConstraint: "< 70.3.0-rc0", // intentionally bad value
				ID:                "CVE-2014-fake-7",
				VersionFormat:     "apk",
				CPEs: []string{
					"cpe:2.3:*:activerecord:activerecord:*:*:*:*:*:rails:*:*",
				},
			},
		},
	}
}

func (d *mockStore) GetVulnerability(namespace, id string) ([]Vulnerability, error) {
	var results []Vulnerability
	for _, vulns := range d.data[namespace] {
		for _, vuln := range vulns {
			if vuln.ID == id {
				results = append(results, vuln)
			}
		}
	}
	return results, nil
}

func (d *mockStore) SearchForVulnerabilities(namespace, name string) ([]Vulnerability, error) {
	return d.data[namespace][name], nil
}

func (d *mockStore) GetAllVulnerabilities() (*[]Vulnerability, error) {
	return nil, nil
}

func (d *mockStore) GetVulnerabilityNamespaces() ([]string, error) {
	keys := make([]string, 0, len(d.data))
	for k := range d.data {
		keys = append(keys, k)
	}

	return keys, nil
}
