
/* File:      justify.P
** Author(s): Abhik Roychoudhury
** Modified:  C.R.Ramakrishnan, Haifeng Guo
** Contact:   lmc@cs.sunysb.edu
** 
** Copyright (C) SUNY at Stony Brook, 1998-2001
** 
** XMC is free software; you can redistribute it and/or modify it under the** terms of the GNU Library General Public License as published by the Free
** Software Foundation; either version 2 of the License, or (at your option)
** any later version.
** 
** XMC is distributed in the hope that it will be useful, but WITHOUT ANY
** WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
** FOR A PARTICULAR PURPOSE.  See the GNU Library General Public License for
** more details.
** 
** You should have received a copy of the GNU Library General Public License
** along with XMC; if not, write to the Free Software Foundation,
** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
** $Id: justify.P,v 1.7 2010-08-19 15:03:39 spyrosh Exp $
** 
*/

% Generic justifier for XSB programs; allows justification of T/F/U values
% NOTE : Any program predicate that depends recursively 
%        on itself through negation must be tabled.
%        This ensures that the delay list of any undefined 
%        atom can be accessed (say via get_residual/2).


% justify(G, TV, R)
% --- Given a Goal G and its truth value inherited from evaluation
% --- constructs a justification R

justify(G, true, R) :- justify_true(G, R).
justify(G, false, R) :- justify_false(G, R).
justify(G, undef, R) :- justify_undef(G, R). 

justify1(G, true, R, T) :- justify_true1(G, R, T).
justify1(G, false, R, T) :- justify_false1(G, R, T).
justify1(G, undef, R, T) :- justify_undef1(G, R, T).                     

% justify_true(G, R)
% --- Given a Goal G, construct a true-justification R

justify_true(G, (G, true, nil, R, nil)) :- 
        initialize_all(T),
	just(G, true, 0, -1, [], (G, true, nil, R, nil), T).
%        nb_asserted(seen(Node, G, _), T).

justify_true1(G, (G, true, nil, Child, nil), T) :-
        just(G, true, 0, -1, [], (G, true, nil, Child, nil), T).  

% justify_false(G, R)
% --- Given a Goal G, construct a false-justification R

justify_false(G, (G, false, nil, R, nil)) :-
        initialize_all(T),
	just(G, false, 0, -1, [], (G, false, nil, R, nil), T).
%        nb_asserted(seen(Node, G, _), T).

justify_false1(G, (G, false, nil, Child, nil), T) :-
        just(G, false, 0, -1, [], (G, false, nil, Child, nil), T).       

% justify_undef(G, R)
% --- Given a Goal G, construct a undef-justification R

justify_undef(G, (G, undef, nil, R, nil)) :-
        initialize_all(T),
	just(G, undef, 0, -1, [], (G, undef, nil, R, nil), T).
%        nb_asserted(seen(Node, G, _), T).

complete_call(G) :- call(G); true.

% just(L)
% ---  Given a literal L, (ancestor list Anc is implicit)
% ---  constructs the justification of L.

just(G, TV, DFS, LastNeg, Ancs, Root, T) :-
	proved(G, TV, DFS, LastNeg, Ancs, Root, T).

%----------------------------------------------------------------
:- dynamic '$justifier_trie'/1.

initialize_all(T) :-
	conset(nodenum, 0),
        ('$justifier_trie'(T)
                -> true
                ;  new_trie(T),
                   assert('$justifier_trie'(T)),
		   nb_assert(seen((G, _TV, _P, R, _C), G, R), T)
        ).

%----------------------------------------------------------------
negated_goal(\+(G), G).
negated_goal(not(G), G).
negated_goal(tnot(G), G).
negated_goal(sk_not(G), G).
negated_goal('t not'(G), G).

%----------------------------------------------------------------

getNode(Node) :-
	coninc(nodenum),
        conget(nodenum, Nodenum),
        Node = num(Nodenum).

% proved(G, TruthValue)
% ---  Finding truth value of a single literal.

proved(neg(G), TV, DFS, _LastNeg, Ancs, (neg(G), TV, _P, (G, TV1, nil, Child, nil), _Sibling), T) :-
	NDFS is DFS+1,
	add_to_ancestor(neg(G), DFS, Ancs, NAncs),
	toggle_tv(TV1, TV), !,
	proved(G, TV1, NDFS, DFS, NAncs, (G, TV1, nil, Child, _), T).

proved(leaf(G), TV, _DFS, _LastNeg, _Ancs, (leaf(G), TV, _P, nil, _sibling), _) :- 
	(TV = true -> call(G)
		    ; \+((call(G), close_open_tables, !))
	).

proved(G, TV, DFS, LastNeg, Ancs, Root, T) :- 
	NDFS is DFS + 1,
	(tabled_goal(G)
		-> proved_tab(G, TV),
		   (TV==true
	    		-> (ancestorchk(G, _AncDFS, Ancs)
		   		-> fail 	% A cycle can never be a valid justification for "t"
                   		;  (nb_asserted(seen(num(_), G, _), T) 
					-> Root = (G, TV, _P, tabled, _Sibling)
					;  add_to_ancestor(G, DFS, Ancs, NAncs),
					   findonerule(G, NDFS, LastNeg, NAncs, Child, T),
					   Root = (G, TV, _P, Child, _Sibling)
				   )
	       		   )
			;  TV == undef ->
            			(ancestorchk(G, _AncDFS, Ancs)
	           			->
% Any cycle of "u"s would be a valid justification of a "u"
% so, no need to check
			 		    Root = (G, undef, _P, tabled, nil)
	          			 ;  (nb_asserted(seen(num(_), G, _), T)
						-> Root = (G, TV, _P, tabled, _Sibling)
						;
					    add_to_ancestor(G, DFS, Ancs, NAncs),
			 		    findfailsandrules(G, NDFS, LastNeg, NAncs, Child, T),
					    Root = (G, TV, _P, Child, _Sibling)
					    )
				   )
	    		 ; % TV=false
	       			  (ancestorchk(G, AncDFS, Ancs)
	           			-> LastNeg < AncDFS,
% A cycle may be a valid justification of "f" as long as
% it *doesnt* contain a negation
					   Root = (G, TV, _P, tabled, nil)
	           			;  (nb_asserted(seen(num(_), G, _), T)
						-> Root = (G, TV, _P, tabled, _Sibling)
						; 
					    add_to_ancestor(G, DFS, Ancs, NAncs),
					    findfails(G, NDFS, LastNeg, NAncs, Child, T),
					    Root = (G, TV, _P, Child, _Sibling)
		      	      		   ) 
				  )
	       	   )
        	;  proved_nt(G, TV, DFS, LastNeg, Ancs, Root, T)
        ).

proved_nt(G, true, DFS, LastNeg, Ancs, (G, true, nil, Child, Sibling), T) :-
	just_true_nt(G, DFS, LastNeg, Ancs, (G, true, nil, Child, Sibling), Tc, T),
	justall_true(Tc, DFS, LastNeg, Ancs, T). 

proved_nt(G, false, DFS, LastNeg, Ancs, (G, false, P, Child, Sibling), T) :- 
	just_false_nt(G, DFS, LastNeg, Ancs, (G, false, P, Child, Sibling), Tc, T),
	justall_false(Tc, DFS, LastNeg, Ancs, T).

proved_nt(G, undef, DFS, LastNeg, Ancs, (G, undef, P, Child, Sibling), T) :-
	just_failundef_nt(G, DFS, LastNeg, Ancs, (G, undef, P, Child, Sibling), Tc, T),
	justall(Tc, DFS, LastNeg, Ancs, T).

proved_tab(G, TV) :-
	\+ get_calls(G, _, _), !, 
        complete_call(G),
        proved_tab(G, TV).
proved_tab(G,true) :-
        setof(G, get_residual(G, []), List), !, member(G, List).
        % this is required; do not change setof to findall
        % illustration r(b) in p80. of wfs_tests
proved_tab(G, undef) :-
        setof(G, get_residual(G, [_|_]), List), !, member(G, List).
        % need to make sure we do not return mutiple ans.
        % for the same undef. atom with diff. delay lists.
proved_tab(G, false) :- \+ get_residual(G, _).
                                                                                                          

%-----------------------------------------------------------------------
%
% findonerule(G, B)  for table call G
% ---- finds a clause of G (with body B) which can be used to prove G

findonerule(Goal, DFS, LastNeg, Ancs, Root, T) :-
	get_clause(Goal,Body), 
	(Body = [] 	-> Root = (Goal, true, nil, fact, nil)
			; alltrue(Body, DFS, LastNeg, Ancs, Root, TcList, T),
			  justall_true(TcList, DFS, LastNeg, Ancs, T)
	),
	(nb_asserted(seen(num(_), Goal, _), T)
		-> true
		; getNode(Node),
		nb_assert(seen(Node, Goal, Root), T)
	).
	
% alltrue(ClauseBody, ListofLiterals)
% --------  Checking the truth of a clause body.

alltrue([], _, _, _, nil, [], _).
alltrue([G1|Gs], DFS, LastNeg, Ancs, (G1,true,nil,Child,Sibling), Tc1, T) :- 
	(tabled_goal(G1)
		-> tabled_goal_truth(G1, TV), 
		   (TV = true 
			  -> (ancestorchk(G1, _AncDFS, Ancs)
					-> fail
					;  Child = tabled
			  )
		    	  ; fail
		   ),
		   (nb_asserted(seen(num(_), G1,  _), T) 
				-> Tc3 = []
				; Tc3 = [(G1, TV)]
		   )
		 ;  just_true_nt(G1, DFS, LastNeg, Ancs, (G1, true, nil, Child, Sibling), Tc3, T)
        ),
	alltrue(Gs, DFS, LastNeg, Ancs, Sibling, Tc2, T),
        checkappend(Tc3, Tc2, Tc1).

findonerule_nt(G, DFS, LastNeg, Ancs, Root, TcList, T) :- 
	get_clause(G, Body),
        (Body = [] ->  Root = fact
                    ; alltrue(Body, DFS, LastNeg, Ancs, Root, TcList, T)
        ).

justall_true([], _, _, _, _).
justall_true([(neg(G), true) | R], DFS, LastNeg, Ancs, T) :- !,
        add_to_ancestor(neg(G), DFS, Ancs, NAncs),
        just(G, false, DFS, LastNeg, NAncs, _, T),
        justall_true(R, DFS, LastNeg, Ancs, T).        
justall_true([(G, true) | R], DFS, LastNeg, Ancs, T) :-
        add_to_ancestor(G, DFS, Ancs, NAncs),
	findonerule(G, DFS, LastNeg, NAncs, _, T),
	justall_true(R, DFS, LastNeg, Ancs, T). 


just_true_nt(neg(G), DFS, _LastNeg, Ancs, (neg(G), true, nil, (G, false, nil, Child, nil), _), Tc, T) :- !,
	NDFS is DFS + 1,
	just_false_nt(G, NDFS, DFS, Ancs, (G, false, nil, Child, nil), Tc, T).

just_true_nt(leaf(G), _, _, _, (leaf(G), true, nil, nil, _), [], _) :- !,
	getleafcall(G, G1), call(G1).
just_true_nt(G, DFS, LastNeg, Ancs, (G, true, nil, Child, _), Tc, T) :- 
	NDFS is DFS + 1,
	findonerule_nt(G, NDFS, LastNeg, Ancs, Child, Tc, T). 
%	(Tc == [] -> (Child1 == (fact, true) 
%				-> Child = Child1
%				;  Child = (true, true)
%		    )	
%		 ;  Child = Child1
%	).

tabled_goal_truth(neg(G), TV) :- !, tabled_goal_truth(G, TV1), toggle_tv(TV1, TV).
tabled_goal_truth(G, TV) :- proved_tab(G, TV).

%tabled_call_true(neg(G), TV) :- !, tabled_call_false(G, TV1), toggle_tv(TV1, TV).
%tabled_call_true(G, TV) :- tabled_goal(G), proved_tab(G, TV).

tabled_call_false(neg(G), TV, []) :-
	!, toggle_tv(TV1, TV), tabled_goal_truth(G, TV1).
tabled_call_false(G, TV, TcList) :-
	tabled_goal(G), 
	findall(G, proved_tab(G, true), TcList),
	close_open_tables,
	(TcList = [] ->
		TV = false
		; TV = true
	).

tabled_call_undef(neg(G), TV, TcList) :- !,
	tabled_call_undef(G, TV1, TcList), toggle_tv(TV1, TV).
tabled_call_undef(G, TV, TcList) :-
	tabled_goal(G),
	findall(G, proved_tab(G, _), TcList1),
	proved_tab(G, TV),
	(TV = true
		-> TcList = TcList1
		;  TcList = []
	).

leaf_call(neg(G)) :- leaf_call(G).	
leaf_call(leaf(_G)).

getleafcall(neg(G), \+(G1)) :- !, getleafcall(G, G1).
getleafcall(leaf(G), G1) :- !, getleafcall(G, G1).
getleafcall(G, G).

just_false_nt(neg(G), DFS, _LastNeg, Ancs, (neg(G), false, _P, (G, true, nil, Child, nil), _), Tc, T) :-
	!, NDFS is DFS + 1,
	just_true_nt(G, NDFS, DFS, Ancs, (G, true, nil, Child, nil), Tc, T).

just_false_nt(leaf(G), _DFS, _LastNeg, _, (leaf(G), false, _P, nil, _), [], _) :-
	!, getleafcall(G, G1), findall(G1, G1, GL), !, GL = [].
just_false_nt(G, DFS, LastNeg, Ancs, (G, false, _P, Child, _), Tc, T) :-
	NDFS is DFS + 1,
	findfails_nt(G, NDFS, LastNeg, Ancs, Child, Tc, T).
%	(Tc = [] -> (Child1 = (fact, false)
%				-> Child = Child1
%				;  Child = (false, false)
%		    )
%		 ;  Child = Child1
%	). 

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% generate all false instances of all the clauses.
% for each instance return "any" failed body literal.
% findfails/5 for tabled call

findfails(Goal, DFS, LastNeg, Ancs, Root, T) :-
	findall(Body, get_clause(Goal,Body), BodyList),
	(BodyList = [] -> Root = (Goal, false, _P, fact, nil)
			;  failedlits(BodyList, DFS, LastNeg, Ancs, Root, TcList, T),
			   justall_false(TcList, DFS, LastNeg, Ancs, T)
	),
        (nb_asserted(seen(num(_), Goal, _), T)
		-> true
		;  getNode(Node),
		   nb_assert(seen(Node, Goal, Root), T)
	), !.
        
findfails_nt(Goal, DFS, LastNeg, Ancs, Root, TcList, T) :- 
	findall(Body, get_clause(Goal, Body), BodyList),
	(BodyList = [] -> Root = fact, TcList = []
			; failedlits(BodyList, DFS, LastNeg, Ancs, Root, TcList, T)
	).			  

justall_false([], _, _, _, _).
justall_false([(neg(G), false) | R], DFS, LastNeg, Ancs, T) :-
        add_to_ancestor(neg(G), DFS, Ancs, NAncs),
	NDFS is DFS + 1,
	just(G, true, NDFS, DFS, NAncs, _, T),
	justall_false(R, DFS, LastNeg, Ancs, T).
justall_false([(G, false) | R], DFS, LastNeg, Ancs, T) :- 
	add_to_ancestor(G, DFS, Ancs, NAncs),
	findfails(G, DFS, LastNeg, NAncs, _, T),
	justall_false(R, DFS, LastNeg, Ancs, T).


falseinst([G1|Gs]) :- (  proved(G1, false) ;
                         (true_or_undef(G1), falseinst(Gs))
                      ).
% falseinst([]) will always fail

failedlits([], _, _, _, nil, [], _).
failedlits([Bd|Rest], DFS, LastNeg, Ancs, (Goal, TV, P, Child, Sibling), TcList, T) :-
	findfalselit(Bd, DFS, LastNeg, Ancs, (Goal, TV, P, Child, Sibling), TcList1, Sibling1, T), 
	failedlits(Rest, DFS, LastNeg, Ancs, Sibling1, TcList2, T),
	checkappend(TcList1, TcList2, TcList).

% Nondeterministically find a false literal.
% Can be backtracked to another choice if the current choice results
% in a cyclic justification of a true literal.

findfalselit1(_, [], _, _, _, _, Sibling, [], Sibling, _) :- !.
findfalselit1(G1, [Gf|Gr], Gs, DFS, LastNeg, Ancs, (G, false, [Gf|P], Child, Sibling), TcList, Sibling1, T) :-
	findall(Gs, G1 = Gf, [Gs1|_]), !,
	findfalselit(Gs1, DFS, LastNeg, Ancs, (G, false, P, Child, Sibling), TcList1, Sibling, T),
	findfalselit1(G1, Gr, Gs, DFS, LastNeg, Ancs, Sibling, TcList2, Sibling1, T),
	checkappend(TcList1, TcList2, TcList).

findfalselit([], _, _, _, _, _, _, _) :- !, fail.
findfalselit([G1|Gs], DFS, LastNeg, Ancs, (G, false, P, Child, Sibling), TcList, Sibling1, T) :-
	(tabled_goal(G1)
		-> tabled_call_false(G1, TV, Tc1),
		   (TV = true
			-> (Tc1 = []
				-> findfalselit(Gs, DFS, LastNeg, Ancs, (G, false, P1, Child, Sibling), TcList, Sibling1, T),
				   P = [G1 | P1]
				;  findfalselit1(G1, Tc1, Gs, DFS, LastNeg, Ancs, (G, false, P, Child, Sibling), TcList, Sibling1, T)
			   )
			;  (ancestorchk(G1, _AncDFS, Ancs)
				-> (TcList = [], Child = tabled)
				;  Child = tabled,
				   (nb_asserted(seen(num(_), G1, _), T)
		 				-> TcList = []
						; TcList = [(G1, false)]
				   )
			   ),
                           G = G1, Sibling = Sibling1, P = nil
		   )
		;  (leaf_call(G1)
			-> (getleafcall(G1, G2), findall(G2, G2, GL),
			    ( GL = []
				-> G = G1, TcList = [], Sibling = Sibling1, Child = nil, P = nil
				;  sort(GL, GL1), !,
				   findfalselit1(G2, GL1, Gs, DFS, LastNeg, Ancs, (G, false, P, Child, Sibling), TcList, Sibling1, T)
			    )
			   )
			;  (just_false_nt(G1, DFS, LastNeg, Ancs, (G, false, nil, Child, Sibling), TcList, T)
				-> G = G1, Sibling = Sibling1, P = nil
				; (G1 = neg(_) -> GL1 = [G1]
                                         ; findall(G1, G1, GL), sort(GL, GL1), !
                                  ),
			   	findfalselit1(G1, GL1, Gs, DFS, LastNeg, Ancs, (G, false, P, Child, Sibling), TcList, Sibling1, T)
		   	   )
		   )
	).
	


%---------------------------------------------------------------------------

% find all instances of all the clauses.
% for the false instances return any false lit.
% for the other instances return the entire body, since all 
% of the body lits have to be shown as true/undef.

findfailsandrules(Goal, DFS, LastNeg, Ancs, Root, T) :-
	findall(Body, get_clause(Goal, Body), BodyList),
	(BodyList == []
		-> fail
		;  failsandrules(BodyList, DFS, LastNeg, Ancs, Root, TcList, undef, T),
		   justall(TcList, DFS, LastNeg, Ancs, T),
                   (nb_asserted(seen(num(_), Goal, _), T)
			-> true
			;  getNode(Node),
		   	   nb_assert(seen(Node, Goal, Root), T)
		   )
	), !.


ufinst(Body) :- (falseinst(Body), TV=false) ; (undefinst(Body), TV=undef).

failsandrules([], _, _, _, nil, [], false, _) :- !.
failsandrules([Bd|Rest], DFS, LastNeg, Ancs, (Goal, TV, P, Child, Sibling), TcList, TF, T) :-
	failorrule(Bd, DFS, LastNeg, Ancs, (Goal, TV, P, Child, Sibling), TcList1, Sibling1, TF2, T),
        failsandrules(Rest, DFS, LastNeg, Ancs, Sibling1, TcList2, TF1, T),
        checkappend(TcList1, TcList2, TcList),
	(TF2 = undef
	    -> TF = undef
	    ;  (TF1 = undef
		  -> TF = undef
	 	  ;  TF = false
	       )
	).

failorrule([G1|Gs], DFS, LastNeg, Ancs, (G, TV, P, Child, Sibling), TcList, Sibling1, TF, T) :-
	(tabled_call_undef(G1, TV1, Tc1) 
		-> (TV1 = true
			-> (Tc1 = []
				-> failorrule(Gs, DFS, LastNeg, Ancs, (G, TV, P1, Child, Sibling), TcList, Sibling1, TF, T),
				   P = [G1 | P1]
				;  failorrule1(G1, Tc1, Gs, DFS, LastNeg, Ancs, (G, TV, P, Child, Sibling), TcList, Sibling1, TF, T)
			   )
                        ;  (ancestorchk(G1, _AncDFS, Ancs)
                                -> TcList = [], Child = tabled
                                ;  TcList = [(G1, TV)], Child = tabled
                           ),
                           TV = TV1, G = G1, Sibling = Sibling1, TF = TV, P = nil
                   )
                ;  (leaf_call(G1)
                        -> (getleafcall(G1, G2), findall(G2, G2, GL), 
			    ( GL = []
                                -> G = G1, TV = TV1, TF = TV, TcList = [], Sibling = Sibling1, Child = nil, P = nil
				;  sort(GL, GL1),
				   failorrule1(G2, GL1, Gs, DFS, LastNeg, Ancs, (G, TV, P, Child, Sibling), TcList, Sibling1, TF, T)
			    )
                           )
                        ;  (just_failundef_nt(G1, DFS, LastNeg, Ancs, (G, TV, nil, Child, Sibling), TcList, T)
                                -> Sibling1 = Sibling, TF = TV, P = nil
                                ;  setof(G1, G1, GL),
                                   failorrule1(G1, GL, Gs, DFS, LastNeg, Ancs, (G, TV, P, Child, Sibling), TcList, Sibling1, TF, T)
                           )
                   )
        ).			    

failorrule1(_, [], _, _, _, _, Sibling, [], Sibling, false, _) :- !.
failorrule1(G1, [Gf|Gr], Gs, DFS, LastNeg, Ancs, (G, TV, [Gf|P], Child, Sibling), TcList, Sibling1, TF, T) :-
        findall(Gs, G1 = Gf, [Gs1|_]),
        failorrule(Gs1, DFS, LastNeg, Ancs, (G, TV, P, Child, Sibling), TcList1, Sibling2, TF2, T),
        failorrule1(G1, Gr, Gs, DFS, LastNeg, Ancs, Sibling2, TcList2, Sibling1, TF1, T),
        checkappend(TcList1, TcList2, TcList),
        (TF2 = undef
	    -> TF = undef
	    ;  (TF1 = undef
		  -> TF = undef
	 	  ;  TF = false
	       )
	).


just_failundef_nt(G, DFS, LastNeg, Ancs, (G, TV, _P, Child, _), Tc, T) :-
        NDFS is DFS + 1,
        findfailundef_nt(G, NDFS, LastNeg, Ancs, Child, Tc, TV, T).
%        (Tc = [] -> (Child1 = (fact, false)
%                                -> Child = Child1
%                                ;  Child = (false, false)
%                    ),
%		    TV = false
%                 ;  Child = Child1
%        ).
 
findfailundef_nt(Goal, DFS, LastNeg, Ancs, Root, TcList, TV, T) :-
        findall(Body, get_clause(Goal, Body), BodyList),
        (BodyList = [] -> Root = fact, TcList = [], TV = false
                        ; failsandrules(BodyList, DFS, LastNeg, Ancs, Root, TcList, TV, T)
        ). 
                                                                                                                                                           
justall([], _, _, _, _).
justall([(G, true) | R], DFS, LastNeg, Ancs, T) :-
	justall_true([(G, true)], DFS, LastNeg, Ancs, T),
	justall(R, DFS, LastNeg, Ancs, T).
justall([(G, false) | R], DFS, LastNeg, Ancs, T) :-
	justall_false([(G, false)], DFS, LastNeg, Ancs, T),
	justall(R, DFS, LastNeg, Ancs, T).
justall([(neg(G), undef) | R], DFS, LastNeg, Ancs, T) :-
	justall([(G, undef) | R], DFS, LastNeg, Ancs, T).
justall([(G, undef) | R], DFS, LastNeg, Ancs, T) :-
	findfailsandrules(G, DFS, LastNeg, Ancs, _, T),
        justall(R, DFS, LastNeg, Ancs, T).


joinSibling((_, _, _, Sib1), Sib2) :- !, joinSibling(Sib1, Sib2).
joinSibling(Sibling, Sibling).
 

 
%-----------------------------------------------------------------

insert(X, L, L) :- member(X, L), !.
insert(X, L, [X|L]).

add_to_ancestor(G, DFS, Ancs, [anc(G, DFS)|Ancs]) :- !.
add_to_ancestor(_,_,Ancs, Ancs).

ancestorchk(G, DFS, Ancs) :-
	member(anc(G,DFS), Ancs).

true_or_undef(G, DFS, LastNeg, Ancs, Root) :- proved(G, true, DFS, LastNeg, Ancs, Root).
true_or_undef(G, DFS, LastNeg, Ancs, Root) :- proved(G, undef, DFS, LastNeg, Ancs, Root).

toggle_tv(true, false).
toggle_tv(false, true).
toggle_tv(undef, undef). 

tabled_goal(neg(G)) :- !, tabled_goal(G).
tabled_goal(G) :- predicate_property(G, tabled).

builtin(G) :- predicate_property(G, built_in).

get_clause(G, Conj) :- rule(G, B), member(Conj, B).

checkappend([], A, A) :- !.
checkappend([F|R], L1, L2) :-
	(ismember(F, L1) -> checkappend(R, L1, L2)
		       ;  checkappend(R, [F|L1], L2)
	). 

ismember(_, []) :- !, fail.
ismember((G, TV), [(G1, TV) | _]) :- G == G1, !.
ismember(M, [_|R]) :- ismember(M, R).

%%%%%%%%%%%%%%%%
% Code to initialize justifier with appropriate program rules
%%%%%%%%%%%%%%%%

just_init(PgmFiles, AnnotFile) :-
	load_dyn(AnnotFile),
	just_cleanup,
	seeing(OF),
	load_pgms(PgmFiles),
	load_builtin_preds,
	see(OF).

load_pgms([]) :- !.
load_pgms([P|Ps]) :- !, 
	load_pgms(P),
	load_pgms(Ps).
load_pgms(PgmFile) :- 
	see(PgmFile),
	load_pgm,
	seen.

load_pgm :-
	repeat,
	read(Rule),
	(Rule = end_of_file
		-> true
		; (Rule = ':-'(Directive)
			-> ( Directive = table(PredSpec)
				-> assert(tabled_pred(PredSpec))
				;  true	% skip other directives
			   )
			;  ( Rule = ':-'(Head, Body)
				-> (builtin_pred(Head)
				      -> true
				      ; (convert((Head;Body),[[CH]|CB]),
					assert(rule(CH, CB)))
				   )
				;  assert(rule(Rule, [[]]))
			   )
		   ), fail
	).


convert(G, [[G]]) :- var(G), !.
convert(true, [[]]) :- !.
convert((G1 '->' G2 ; G3), C) :- !,
	convert(((G1, G2); (not(G1), G3)), C).
convert((G1 '->' G2), C) :- !,
	convert((G1, G2), C).
convert((G1; G2), C) :- !,
	convert(G1, C1),
	convert(G2, C2),
	append(C1, C2, C).
convert((G1, G2), C) :- !,
	convert(G1, C1),
	convert(G2, C2),
	product(C1, C2, C).
convert(G, C) :- 
	negated_goal(G, NG),!,
	convert(NG, C1),
	(C1 = [[NC]]	-> C=[[neg(NC)]]
			; writeln('++ERROR: conjunction/disjunction in scope of not!'(NG)),
				C=[[]]
	).
convert(G, [[leaf(G)]]) :-
	(builtin(G); leaf(G)), !.
convert(G, [[G]]).

product(L1, L2, L) :-
	bagof(C, choose_from(L1, L2, C), L).
choose_from(L1, L2, C) :-
	member(C1, L1), member(C2, L2), append(C1, C2, C).

builtin_pred(forall(_,_,_)).
load_builtin_preds :-
	assert(rule(forall(_,A,C), [[neg(exists_ce(A,C))]])),
	assert(rule(exists_ce(A,C), [[A, neg(C)]])),
	assert((exists_ce(A,C) :- A, not(C))).

just_cleanup :-
	retractall(rule(_,_)),
	retractall(tabled_pred(_)).

nb_assert(T, H) :-
	trie_intern(T, H, _L, _F).

nb_asserted(T, H) :- 
	trie_interned(T, H, _, _).
