/*
  Copyright (C) 2004-2005 Tommi Tervonen, Petteri Klemola, Pasi Orovuo

  This file is part of Kajaani Kombat.

  Kajaani Kombat is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  Kajaani Kombat is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with Kajaani Kombat; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "animation.h"

animation::animation(int num_frames, SDL_Surface *first_frame)
{
  frames = vector<SDL_Surface *>(num_frames);
  
  for (int i=0;i<num_frames;i++)
    {
      SDL_Surface *t = deep_copy_surface(first_frame);
      SDL_SetColorKey (t, SDL_SRCCOLORKEY | SDL_RLEACCEL, 0);
      frames[i] = SDL_DisplayFormat(t);
      SDL_FreeSurface(t);
    }
}

animation::~animation()
{
  vector<SDL_Surface *>::iterator it = frames.begin();

  while (it != frames.end())
    {
      SDL_Surface *s = *it++;
      delete s;
    }
}

void animation::set_frame (int findex, SDL_Surface *fram)
{
  if (frames[findex])
    SDL_FreeSurface (frames[findex]);
  SDL_Surface *t = deep_copy_surface(fram);
  SDL_SetColorKey (t, SDL_SRCCOLORKEY | SDL_RLEACCEL, 0);
  frames[findex] = SDL_DisplayFormat(t);
  SDL_FreeSurface(t);
}

const SDL_Surface *animation::get_frame (int findex)
{
  return frames[findex];
}

ammo_anim_container::~ammo_anim_container()
{
  for (int i=0;i<MAX_PLAYERS;i++)
    delete anims[i];
}

exp_anim_container::~exp_anim_container()
{
  delete sanim;
  delete manim;
  delete banim;
}

exp_anim_container::exp_anim_container(int fs)
{
  SDL_Surface *t;
  frames = fs;
  real_explosion_efect::create_colormaps();

  //create a new surface
  t = SDL_CreateRGBSurface (SDL_SWSURFACE, SANIM_SIZE, SANIM_SIZE, 32, 0xFF0000, 0xFF00, 0xFF, 0xFF000000);
  SDL_SetColorKey (t, SDL_SRCCOLORKEY | SDL_RLEACCEL, 0);
  SDL_Surface *s = SDL_DisplayFormat(t);
  SDL_FreeSurface(t);
  t = NULL;

  t = SDL_CreateRGBSurface (SDL_SWSURFACE, MANIM_SIZE, MANIM_SIZE, 32, 0xFF0000, 0xFF00, 0xFF, 0xFF000000);
  SDL_SetColorKey (t, SDL_SRCCOLORKEY | SDL_RLEACCEL, 0);
  SDL_Surface *m = SDL_DisplayFormat(t);
  SDL_FreeSurface(t);
  t = NULL;

  t = SDL_CreateRGBSurface (SDL_SWSURFACE, BANIM_SIZE, BANIM_SIZE, 32, 0xFF0000, 0xFF00, 0xFF, 0xFF000000);
  SDL_SetColorKey (t, SDL_SRCCOLORKEY | SDL_RLEACCEL, 0);
  SDL_Surface *b = SDL_DisplayFormat(t);
  SDL_FreeSurface(t);
  t = NULL;

  sanim = new animation(frames, s);
  manim = new animation(frames, m);
  banim = new animation(frames, b);
  
  coord sc = coord::pc(SANIM_SIZE / 2, SANIM_SIZE / 2);
  coord mc = coord::pc(MANIM_SIZE / 2, MANIM_SIZE / 2);
  coord bc = coord::pc(BANIM_SIZE / 2, BANIM_SIZE / 2);
  real_explosion_efect se (sc, 200, 20.0);
  real_explosion_efect me (mc, 500, 50.0);
  real_explosion_efect be (bc, 1000, 100.0);

  se.set_colormap_yellowred();
  me.set_colormap_yellowredgray();
  be.set_colormap_yellowredgray();

  Uint32 adv_ams = SANIM_LEN / frames;
  Uint32 adv_amm = MANIM_LEN / frames;
  Uint32 adv_amb = BANIM_LEN / frames;
  for (int i=0;i<frames;i++)
    {
      //      printf ("gen %d frame\n", i);
      se.draw(s);
      sanim->set_frame(i, s);
      
      me.draw(m);
      manim->set_frame(i, m);

      be.draw(b);
      banim->set_frame(i, b);

      // finally, advance all
      se.move(adv_ams);
      me.move(adv_amm);
      be.move(adv_amb);
    }

  SDL_FreeSurface (s);
  SDL_FreeSurface (m);
  SDL_FreeSurface (b);

  //  exit(0);
}

ammo_anim_container::ammo_anim_container(SDL_Surface **ammogfxs, int fs)
{
  SDL_Surface *t;
  frames = fs;

  SDL_Surface *surfs[MAX_PLAYERS];

  for (int i=0;i<MAX_PLAYERS;i++)
    {
      t = SDL_CreateRGBSurface (SDL_SWSURFACE, ammogfxs[i]->w, ammogfxs[i]->h, 32, 0xFF0000, 0xFF00, 0xFF, 0xFF000000);
      SDL_SetColorKey (t, SDL_SRCCOLORKEY | SDL_RLEACCEL, 0);
      surfs[i] = SDL_DisplayFormat(t);
      SDL_FreeSurface(t);
      t = NULL;

      anims[i] = new animation(frames, surfs[i]);
    }

  double tim = 0.0;
  double adv = (double) MAX_ALT / (double) fs;
  lookup lut;
  rotozoom rz(&lut);
  for (int i=0;i<frames;i++)
    {
      tim += adv;
      //      printf ("gen %d frame\n", i);
      float scale = tim / 50.0f;
      for (int j=0;j<MAX_PLAYERS;j++)
	{
	  coord cent = coord::pc(surfs[j]->w / 2, surfs[j]->h / 2);
	  rz.rotatescale_blit(scale, (int) tim, surfs[j], ammogfxs[j], cent);
	  anims[j]->set_frame(i, surfs[j]);
	}
      //      printf ("created with tim %f\n", tim);
    }

  for (int i=0;i<MAX_PLAYERS;i++)
    SDL_FreeSurface (surfs[i]);

}

anim_explosion_obj::anim_explosion_obj(const coord &pos, exp_anim_container::type _t)
  : game_obj (pos, OWNER_NOONE)
{
  t = _t;
}

bool anim_explosion_obj::is_finished() const
{
  if (t == exp_anim_container::EXP_SMALL)
    return age > SANIM_LEN;
  if (t == exp_anim_container::EXP_MED)
    return age > MANIM_LEN;
  if (t == exp_anim_container::EXP_BIG)
    return age > BANIM_LEN;

  assert(0);
  return false;
}

const SDL_Surface *ammo_anim_container::get_frame(int pindex, int alt) const
{
  if (alt >= MAX_ALT) alt = MAX_ALT;
  alt--;
  if (alt <0) alt = 0;

  int ind = (int)(((double) alt / (double) MAX_ALT) * (double) frames);

  return anims[pindex]->get_frame(ind);
}

const SDL_Surface *exp_anim_container::get_frame (type t, int ind) const
{
  if (ind >= frames) ind = frames-1; 

  switch (t)
    {
    case EXP_SMALL:
      return sanim->get_frame(ind);
      break;
    case EXP_MED:
      return manim->get_frame(ind);
      break;
    case EXP_BIG:
      return banim->get_frame(ind);
      break;
    }
  assert(0);
  return NULL;
}
