/*
  Copyright (C) 2004-2005 Tommi Tervonen, Petteri Klemola, Pasi Orovuo

  This file is part of Kajaani Kombat.

  Kajaani Kombat is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  Kajaani Kombat is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with Kajaani Kombat; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef __GAME_OBJ_H_
#define __GAME_OBJ_H_

#include "SDL.h"
#include "coord.h"
#include "globals.h"
#include <string>

using namespace std;

#define MAX_ALT 50
#define DEFAULT_AMMO_SPEED 0.1

/**
 * A super-class for all game-objects.
 *
 * \author Tommi Tervonen <tommi@medusa.tutka.net>
 */
class game_obj
{
 public:
  /**
   * Constructor.
   * \param position Current object position.
   * \param Object owner.
   */
  game_obj (const coord &position, int owner);

  /**
   * Gives object owner.
   * \return owner of the object.
   */
  inline int get_owner() const
	{ return owner; }
  /**
   * Gives current position of the object.
   * \return Current position.
   */
  inline const coord & get_position() const
	{ return static_cast<const coord &>(position); }
  /**
   * Gives objects age.
   * \return Objects age.
   */
  inline Uint32 get_age() const
    { return age; }
  /**
   * Updates age-counter on object.
   */
  virtual void move(Uint32 time);

  virtual bool is_finished() const = 0;

 protected:
  /** Object position **/
  coord position;
  /** Object owner **/
  int owner;
  /** Age of the object. */
  Uint32 age;
  
  /**
   * Default constructor, protected from outside usage.
   */
  game_obj ();
};

/**
 * A cannon interface
 * 
 * \author Petteri Klemola <petteri@medusa.tutka.fi>
 */
class cannon_in: public game_obj
{
 public:
  cannon_in(const coord & center, int owner);
  virtual bool is_finished() const=0;  

  /**
   * Tells whether the cannon is ready or not.
   *
   * \return True is it is ready, false otherwise.
   */
  virtual bool is_ready() const=0;

  /**
   * Sets the ready-state of the cannon.
   *
   * \param ready Ready or not .. :)
   */
  virtual void set_ready(bool rd)=0;
  
 protected:
  cannon_in(){ready=true;}
  
  /** Whether this cannon is ready to shoot or not. */
  bool ready;  
};

/**
 * A cannon class.
 *
 * \author Tommi Tervonen <tommi@medusa.tutka.net>
 */
class cannon : public cannon_in
{
 public:
  /**
   * Constructor.
   * 
   * \param center Pixel-coordinate of the center of the cannon.
   * \param owner Owner index of the owner.
   */
  cannon (const coord & center, int owner);

  /**
   * Sets the ready-state of the cannon.
   *
   * \param ready Ready or not .. :)
   */
  virtual void set_ready(bool rd);

  /**
   * Tells whether the cannon is ready or not.
   *
   * \return True is it is ready, false otherwise.
   */
  virtual bool is_ready() const;

  virtual bool is_finished() const;
 protected:
  /**
   * Default constructor.
   */
  cannon();

  /** Whether this cannon is ready to shoot or not. */
  bool ready;

};

/**
 * A big cannon class.
 */
class big_cannon : public cannon_in
{
 public:
  /**
   * Constructor.
   * 
   * \param center Pixel-coordinate of the center of the big cannon.
   * \param owner Owner index of the owner.
   */
  big_cannon (const coord & center, int owner);

  /**
   * Sets the ready-state of the big cannon.
   *
   * \param ready Ready or not .. :)
   */
  virtual void set_ready(bool rd);

  /**
   * Tells whether the cannon is ready or not.
   *
   * \return True is it is ready, false otherwise.
   */
  virtual bool is_ready() const;

  virtual bool is_finished() const;
 protected:
  /**
   * Default constructor.
   */
  big_cannon();
  /** Whether this cannon is ready to shoot or not. */
  bool ready;
};

/**
 * A super-class for all "moving" game objects.
 *
 * \author Tommi Tervonen <tommi@medusa.tutka.net>
 */
class moving_game_obj : public game_obj
{
 public:
  /**
   * Constructor for outside usage.
   * \param start_position Starting coordinate of the object.
   * \param dest_position Destination coordinate of the object.
   * \param owner Owner index of the object.
   * \param alt Altitude of the object (from 0 to MAX_ALT).
   * \param speed Speed of the object (>0).
   */
  moving_game_obj(const coord &start_position, const coord &dest_position, int owner, int alt, double speed);

  /**
   * Gives objects altitude.
   * \return Objects altitude.
   */
  inline int get_alt() const
    { return alt; }
  /**
   * Gives objects speed.
   * \return Objects speed.
   */
  inline double get_speed() const
    { return speed; }
  /**
   * Moves the object.
   * \param time Amount of time to move the object.
   */
  virtual void move(Uint32 time);
  /**
   * Gives landed-status of the object.
   * \return True if the object has landed, otherwise false.
   */
  virtual bool is_finished() const;
  /**
   * Gives the starting position of the object.
   * \return Starting position of the object.
   */
  const coord & get_start_position() const;
  /**
   * Assignment operator.
   * \param other Object where to take the assignment values.
   * \return Reference to this object.
   */
  moving_game_obj & operator= (const moving_game_obj & other);


 protected:
  /**
   * Default constructor. Protected from outside usage.
   */
  moving_game_obj();

  /** Starting coordinates of the object. */
  coord start_pos;
  /** Destination coordinates of the object. */
  coord dest_pos;
  /** Objects altitude. */
  int alt;
  /** Status of objects landing. */
  bool landed;
  /** Objects speed. */
  double speed;
};

/**
 * A super-class for ammonition objects.
 *
 * \author Tommi Tervonen & Petteri Klemola <kk@medusa.tutka.net>
 */
class ammo : public moving_game_obj
{  
 public:
  /**
   * Constructor for outside usage.
   * \param start Shooting coordinates of the ammo.
   * \param dest Destination coordinates of the ammo.
   * \param owner Owner (shooter) of the ammo.
   * \param size size of the ammo
   */
  ammo (const coord &start, const coord &dest, int owner, int size);

  /**
   * returns ammos size uses definitions in globals.h such as NORMAL=1, BIG=2
   */
  int get_size();

 protected:
  /**
   * Default constructor. Protected from outside usage.
   */
  ammo();

  int size;
  
};

class text_object : public game_obj
{
 public:
  text_object(const coord & pos, int owner, const string &text, Uint32 duration);
  text_object(const coord & pos, int owner, const string &text, Uint32 duration, Uint8 red, Uint8 green, Uint8 blue);
  text_object(const coord & pos, int owner, const string &text, Uint32 duration, SDL_Color col);
  // constructor for "unknown" text position. use this if unsure!
  text_object(int owner, const string &text, Uint32 duration = ERRTEXT_DUR);
  text_object(int owner, const string &text, SDL_Color color, Uint32 duration = ERRTEXT_DUR);

  text_object (text_object &t);

  virtual ~text_object();
	
  bool is_color_set() const;
  SDL_Color & get_color();
  const string & get_text() const;
  virtual bool is_finished() const;

  bool has_unknown_position() const;
 protected:
  string text;
  Uint32 duration;
  SDL_Color color;
};

class balloon_object : public text_object
{
 public:
  balloon_object(const coord & pos, int owner, const string &text, Uint32 duration, SDL_Color col) : text_object(pos, owner, text, duration, col){}
  
};
#endif
