/**
 * @mainpage iedera
 * @version 1.05
 * @section Introduction
 *   Iedera is a program to select and design subset seeds and vectorized 
 *   subset seeds. This program heavily relies on automaton theory to compute the
 *   sensitivity of seed that are represented by the subset seed model, or the
 *   vectorized subset seed model.
 *
 *
 *
 * @section Download
 * Download the iedera source code and binaries on the following web page:\n
 * <A HREF="http://bioinfo.lifl.fr/yass/iedera.php">http://bioinfo.lifl.fr/yass/iedera.php</A>\n
 * A small tutorial to use iedera is also provided on this web page.\n
 *
 * @author <A HREF="http://www.lifl.fr/~noe" TARGET="_top">Laurent Noe</A>
 * @author Mikhail Roytberg 
 * @author <A HREF="http://www.lifl.fr/~kucherov" TARGET="_top">Gregory Kucherov</A>
 */



/** @defgroup main */
// @{

//C stuff

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/time.h>
#include <limits>

#if defined(WIN32) || defined(WIN64)
/* windows part */
/// SIGNAL terminal flag (disable terminal handler)
#undef SIGNAL
#include <windows.h>
/// sleep function (seconds)
#define SLEEP(x) Sleep(x)
/// get process id (used to init pseudo-random function)
#define GETPID() GetCurrentProcessId()
#else

/* unix part */
/// SIGNAL terminal flag (enable terminal handler)
#define SIGNAL
#include <unistd.h>
#include <pthread.h>
/// sleep function (seconds)
#define SLEEP(x) sleep(x)
/// get process id (used to init pseudo-random function)
#define GETPID() getpid()
#endif

//STL
#include <vector>
#include <list>
#include <utility>
#include <algorithm>
#include <iterator>
//STD
#include <iostream>
#include <iomanip>
#include <fstream>
#include <sstream>
#include <cstdio>
#include <cstdlib>
#include <cmath>
#include <string>
//STR
#include <errno.h>
using namespace std;
#include "macro.h"
#include "automaton.h"
#include "seed.h"

#ifdef SIGNAL
#include <signal.h>
#endif

#ifndef PACKAGE
/// iedera program name
#define PACKAGE "iedera"
#endif

#ifndef VERSION
/// iedera program version
#define VERSION "1.05"
#endif

/// maximal number of seeds being optimized together
#define MAXSEEDS 32


#ifdef WIN32
/// windows getmstime (milliseconds) function (used to init pseudo-random function)
typedef unsigned __int64 QWORD;
QWORD GETMSTIME() {
  FILETIME filetime;
  GetSystemTimeAsFileTime(&filetime);
  return filetime.dwLowDateTime + filetime.dwHighDateTime * 1000000;
}
#else
/// unix getmstime (milliseconds) function (used to init pseudo-random function)
unsigned long long GETMSTIME() {
  struct timeval tv;
  gettimeofday(&tv, 0);
  return tv.tv_usec + tv.tv_sec * 1000000;
}
#endif



/// choose between a subset seed and a vectorized subset seed
#define SEEDAUTOMATON(autom,seed,nomerge)		    \
  ( gv_vectorized_flag ?                                    \
    ( gv_lossless_flag ?                                    \
       ((autom)->Automaton_SeedScoreCost(                   \
               *(seed),                                     \
               gv_subsetseed_matching_matrix,               \
               gv_vectorizedsubsetseed_scoring_matrix,      \
               gv_vectorizedsubsetseed_scoring_threshold,   \
               nomerge,                                     \
               gv_lossless_costs_vector,                    \
               gv_lossless_cost_threshold                   \
        ))                                                  \
        :                                                   \
       ((autom)->Automaton_SeedScore(                       \
               *(seed),                                     \
               gv_subsetseed_matching_matrix,               \
               gv_vectorizedsubsetseed_scoring_matrix,      \
               gv_vectorizedsubsetseed_scoring_threshold,   \
               nomerge                                      \
        ))                                                  \
    )                                                       \
        :                                                   \
    ( gv_lossless_flag ?                                    \
        ((autom)->Automaton_SeedPrefixesMatchingCost(       \
               *(seed),                                     \
               gv_subsetseed_matching_matrix,               \
               nomerge,                                     \
               gv_lossless_costs_vector,                    \
               gv_lossless_cost_threshold                   \
        ))                                                  \
      :                                                     \
        ((autom)->Automaton_SeedPrefixesMatching(           \
               *(seed),                                     \
               gv_subsetseed_matching_matrix,               \
               nomerge                                      \
        ))                                                  \
     )                                                      \
    )


/** @name uninitialized parameters
 *  @brief must be initialized in the main function
 */
// @{
/// the boolean matching matrix
vector< vector<int> >  gv_subsetseed_matching_matrix;
/// the vector scoring matrix when vectorized subset seeds are activated
/// @see gv_vectorized_flag
/// @see gv_vectorizedsubsetseed_scoring_threshold
vector< vector<int> >  gv_vectorizedsubsetseed_scoring_matrix;
/// the lossless cost for A when lossless flag is activated
/// @see gv_lossless_flag
/// @see gv_lossless_cost_threshold
vector<int>            gv_lossless_costs_vector;


/// foreground sensitivities
vector< double >       gv_bsens;
/// k order model when Markov is activated (0 for Bernoulli)
/// @see gv_bsens_k
int                    gv_bsens_k = 0;
/// probabilistic automaton associated with the sensitivity computation
/// @see gv_bsens
automaton   *          gv_bsens_automaton = NULL;

/// background seletivities
/// @see gv_bsel_k
vector< double >       gv_bsel;
/// k order model when Markov is activated (0 for Bernoulli)
/// @see gv_bsel
int                    gv_bsel_k = 0;
// @}

/** @name initialized parameters
 *  @brief may be modified in the main function
 */
// @{
/// number of seeds being optimized
int      gv_nbseeds = 1;
/// length of the alignment on which seeds are optimized
int      gv_alignment_length = 16;

/// Hopcroft minimization applied on each deterministic automaton
bool     gv_minimize_flag      = true;
/// Hillclimbing local optimization for each seed and set of positions
bool     gv_hillclimbing_flag  = false;
/// Hillclimbing local optimization frequency
double   gv_hillclimbing_alpha = 5e-2;
/// Flag set when a seed is given on command line
int      gv_motif              = 0;
/// Set of seeds being computed or optimized
seed  *  gv_seeds[MAXSEEDS];
/// Maximal difference of weight (given as a ratio) between seeds being opimized together
double   gv_jive      = 0.0;

/** @name cycle parameters
 *  @brief generates position restricted seeds of a cycle
 */
// @{
/// cycle flag
/// @see gv_cycles
bool                    gv_cycles_flag = false;
/// cycle size (one for each seed)
/// @see gv_cycle_flag
vector< int >           gv_cycles;
/// cycle pos (number of pos per cycle)
/// @see gv_cycles_pos_list
/// @see gv_cycle_flag
vector< int >           gv_cycles_pos_nb;
/// cycle pos (pos for each cycle)
/// @see gv_cycle_flag
/// @see gv_cycle_pos_nb
vector< vector <int>  > gv_cycles_pos_list;
// @}



/* @name multihit criterion
 * @brief consider a hit if at least $m$-hits (of any of the seeds) have been counted
 */
// @{
/// flag that activates the mhits
/// @see gv_multihit_nb
bool gv_multihit_flag = false;
/// number of mhits (must be >= 1 if the flag is activated)
/// @see gv_multihit_flag
int gv_multihit_nb    = 0;
// @}

/** @name excluded seeds
 *  @brief exclude seeds motif from alignments generated 
 */
// @{
/// excluded seeds
/// @see gv_xnbseeds
seed *   gv_xseeds[MAXSEEDS];;
/// number of excluded seeds
/// @see gv_xseeds
int      gv_xnbseeds  = 0;
// @}

/** @name homogeneous alignments
 *  @brief generates homogeneous alignments according to scoring parameters
 */
// @{
/// score for each letter of the alignment alphabet
vector< int >  gv_homogeneous_scores;
/// flag that activates computation on homogeneous alignments
bool           gv_homogeneous_flag = false;
// @}


/** @name vectorized seeds
 *  @brief generate subset seeds with an additional scoring constraint
 */
// @{
/// scoring threshold for a vectorized subset seed
/// @see gv_vectorizedsubsetseed_scoring_matrix
int                    gv_vectorizedsubsetseed_scoring_threshold = 0;
/// flag that activates vectorized subset seeds (default subset seed)
bool                   gv_vectorized_flag = false;
// @}


/** @name lossless seeds
 */
// @{
/// lossless cost for an alignment (must be at most equal to this value)
/// @see gv_lossless_costs_vector
int                    gv_lossless_cost_threshold = 0;
/// flag that activates lossless computation (default lossy computation)
bool                   gv_lossless_flag   = false;
// @}



/** @name data management
 *  @brief pareto input/output files
 */
// @{
/// pareto output filename
/// @see outputPareto
char   * gv_output_filename = NULL;
/// pareto input filenames
/// @see inputPareto
char   * gv_input_filenames[256] = {NULL};
/// pareto input filenames (number of files)
/// @see gv_input_filenames
int      gv_nb_input_filenames =   0;
// @}


/** @name init functions
 *  @brief must be called at least once in the main function
 */
// @{
/// build an initial matrix gv_subsetseed_matching_matrix
/// @see gv_subsetseed_matching_matrix
int build_default_subsetseed_matching_matrix();
/// build an initial matrix gv_vectorizedsubsetseed_scoring_matrix
/// @see gv_vectorizedsubsetseed_scoring_matrix
int build_default_vectorizedsubsetseed_scoring_matrix();
/// build probabilities
/// @see gv_bsens,gv_bsens_k,gv_bsel,gv_bsel_k
int build_default_probabilities();
// @}


/** @name command line functions
 *  @brief set of command line management functions
 */

// @{

/// display a 2d matrix (a vector of vectors of int)
void DISPLAYMATRIX( std::ostream & os, vector< vector<int> > & matrix );

/// describe the command line parameters
void USAGE() {
  cerr << "* USAGE:" << endl;
  cerr << "  " << PACKAGE << " [options]" << endl;
  cerr << "      -h       display this Help screen" << endl ;
  cerr << "      -v       display program Version" << endl ;
  cerr << "                                                                                      " << endl;
  cerr << "   1) Seed Model:" << endl;
  cerr << "     a) Data "<< endl;
  cerr << "      -A <int>                : align alphabet size \'A\' (default = " << gv_align_alphabet_size << ")" << endl;
  cerr << "      -B <int>                :  seed alphabet size \'B\' (default = " <<  gv_seed_alphabet_size << ")" << endl;
  cerr << "      -M {{<bool>}} table     : subset seed Matching matrix" << endl;
  cerr << "         (default = "; DISPLAYMATRIX(cerr,gv_subsetseed_matching_matrix); cerr << " )" << endl;
  cerr << "         * note    : parameter is a C-like matrix of 'A'x'B' cols x rows " << endl;
  cerr << "         * example : -M \"{{1,0,1},{0,1,1}}\"    (for A=3 and B=2) " << endl;
  cerr << "      -MF <filename>          : gives the matrix inside a file (when it is too large for command line)" << endl;
  cerr << "                                                                               " << endl;
  cerr << "     b) SubsetSeed/VectorizedSubsetSeed "<< endl;
  cerr << "      -V                      : select the Vectorized subset seed model (default = disabled)" << endl;
  cerr << "      -S {{<int>}} table      : vectorized subset seed Scoring table " << endl;
  cerr << "      -SF <filename>          : gives the table inside a file (when it is too large for command line)" << endl;
  cerr << "      -T <int>                : vectorized subset seed scoring minimal Threshold " << endl;
  cerr << "                                                                               " << endl;
  cerr << "     c) SubsetSeed/LosslessSubsetSeed "<< endl;
  cerr << "      -L <int>,<int>,...      : set the lossless alignment alphabet costs (and select the lossless model)" << endl;
  cerr << "      -X <int>                : set the alignment maX cost to be considered in the alignment set" << endl;
  cerr << "                                                                               " << endl;
  cerr << "   2) Alignments:" << endl;
  cerr << "      -b <dbl>,<dbl>,...      : fix the Bernoulli/Markov background distribution (size |'A'|^k)" << endl;
  cerr << "      -f <dbl>,<dbl>,...      : fix the Bernoulli/Markov foreground distribution (size |'A'|^k)" << endl;
  cerr << "      -fF <filename>          : fix the foreground model (as a probabilistic NFA)" << endl;
  cerr << "                                for more details, see http://bioinfo.lifl.fr/yass/iedera.php#fFFormat" << endl;
  cerr << "      -l <int>                : fix the alignment length (default = " <<  gv_alignment_length << ") " << endl;
  cerr << "      -u <int>,<int>,...      : set the homogeneous alignment scoring system (default = disabled)" << endl;
  cerr << "                                                                               " << endl;
  cerr << "   3) Seed Enumeration:" << endl;
  cerr << "      -s <int>,<int>          : seed span (default min = " << gv_minspan << ",max = " << gv_maxspan << ")" << endl;
  cerr << "      -n <int>                : number of consecutive seeds to design  (default = " << gv_nbseeds << ", max = " << MAXSEEDS <<")" << endl;
  cerr << "      -r <int>                : random enumeration  (default = " << gv_nbruns << " is complete enumeration)" << endl;
  cerr << "      -k                      : hill climbing heuristic (check all permutations of 2 elements inside one seed)" << endl;
  cerr << "      -a <dbl>                : hill climbing heuristic frequency (good seeds are more likely to be optimized)" << endl;
  cerr << "      -w <dbl>,<dbl>          : minimal/maximal weight (default min = " << gv_minweight << ", max = " << gv_maxweight << ")" << endl;
  cerr << "         * symbol # (if defined, otherwise the max of all symbols) is of weight 1.0 " << endl;
  cerr << "         * symbol _ is of weight 0.0 " << endl;
  cerr << "         * symbol e weight is given by 'log_#(e matches background distribution)'" << endl;
  cerr << "      -i <int>,<int>,...      : signature id (number of '0','1',...'B-1' elements inside a subset seed)" << endl;
  cerr << "      -j <dbl>                : difference of weight (given as a ratio) allowed between consecutive seeds (between 0 and 1, default = " << gv_jive << ")" << endl;
  cerr << "      -x                      : only symetric seeds are selected [beta]" << endl;
  cerr << "      -y <int>                : consider sensitivity when y hits are required (multihit sensitivity) [beta]" << endl;
  cerr << "      -c <int>,<int>,...      : consider sensitivity when each seed is indexed on 1/c of its positions" << endl;
  cerr << "         * note    : the position is enumerated or choosen randomly depending on -r parameter" << endl;
  cerr << "         * example : \"##-#:1/5\"  means that the seed \"##-#\" will be placed at 1st,6th,11th... positions" << endl;
  cerr << "      -q <int>,<int>,...      : consider sensitivity when each seed is indexed on q/c of its positions" << endl; 
  cerr << "         * note    : parameter -c must be specified before, positions are enumerate or randomly drawn (-r)"<< endl;
  cerr << "         * example : \"##-#:2,5/5\"  means that the seed \"##-#\" will be placed at 2nd,5th,7th,10th... positions" << endl;
  cerr << "      -d                      : disable Hopcroft minimization (default is enabled) " << endl;
  cerr << "      -m <seeds>              : check the given seed selectivity/sensitivity (possibility to add cycle content : see -c -q examples)" << endl;
  cerr << "      -mx <seeds>             : generate alignments that exclude the given seed pattern " << endl;
  cerr << "                                                                               " << endl;
  cerr << "   4) Miscellaneous:" << endl;
  cerr << "      -BSymbols '<char>'      : chain of seed alphabet symbols (default is '01..9AB..Zab..z')" << endl;
  cerr << "      -e <filename>           : input an initial \"pareto set\" file before optimizing" << endl;
  cerr << "      -o <filename>           : output the \"pareto set\" to this file after optimizing" << endl;
  cerr << "      -t                      : sleep during working hours [8h to 20h from monday to friday]" << endl;
  cerr << "      -p <int>                : sleep when the number of processes is >= than <int> " << endl;
  cerr << "                                (someone logged in ~ 120, nobody logged in ~ 100)" << endl;
  cerr << "                                                                               " << endl;
  cerr << "   5) Nucleic spaced seeds: (shortcuts that simulate hedera parameters)" << endl;
  cerr << "      -transitive             : \"-A 3 -B 3 -f .15,.15,.70 -b 0.5,0.25,0.25 -BSymbols '-@#' -l 64 -w 9,9 -s 10,15 -i 0,2,8\"" << endl;
  cerr << "      -spaced                 : \"-A 2 -B 2 -f .30,.70     -b 0.75,0.25     -BSymbols '-#'  -l 64 -w 9,9 -s 9,15\"" << endl;
  exit(-1);
}
/// display a 2d matrix (a vector of vectors of int) 
void DISPLAYMATRIX( std::ostream & os, vector< vector<int> > & matrix ) {
  os << "{";
  for (int b=0; b<gv_seed_alphabet_size; b++) {
    os << "{";
    for (int a=0; a<gv_align_alphabet_size; a++) {
      os << (matrix[a][b]);
      if (a < (gv_align_alphabet_size-1)) os << ",";
    }
    os << "}";
    if (b < (gv_seed_alphabet_size-1)) os << ",";
  }
  os << "}";
}

/// display a 1d vector (vector of int)
void DISPLAYTABLE( std::ostream & os, vector<double> & table ) {
  os << "{";
  for (int a=0; a<(int)table.size(); a++) {
    os << (table[a]);
    if (a < (int)(table.size()-1))
      os << ",";
  }
  os << "}";
}

/// check if the sum of probabilities is ~ 1.0
void CHECKPROB(vector<double> & table) {
  double result = 0.0;
  for(int j=0 ; j<(int)table.size() ; j++)
    result += (table)[j];
  if(result < 0.99 || result > 1.01)
    ERROR("CHECKPROB"," sum of " << (table)[0] << " + ... + " << (table)[table.size()-1] << " must be 1.0 and not " << result);
}

/// display the iedera version
void VERS() {
  cerr << PACKAGE << " version " << VERSION << endl;
  exit(-1);
}

/// parse one flag
void PARSEFLAG(int & i, char ** argv, int argc, bool & var, bool value) {
  var = value;
}

/// parse and check one integer
void PARSEINT(int & i,  char ** argv, int argc, int & var, int min, int max) {
  i++;
  if (i >= argc)
    ERROR("PARSEINT","\"" << argv[i-1] << "\" found without argument");
  char * next = argv[i];
  int i_tmp = strtol(argv[i],&next,10);
  if (errno != 0 || ((*next) != '\0'))
    ERROR("PARSEINT","\"" << argv[i-1] << "\" is not followed by an integer");
  if (i_tmp>max || i_tmp <min)
    ERROR("PARSEINT","\"" << argv[i-1] << "\" is followed by an integer outside the valid range");
  var = i_tmp;
}

/// parse and check a pair of integers
void PARSEDINT(int & i, char ** argv, int argc,
               int & var1, int min1, int max1,
               int & var2, int min2, int max2) {
  i++;
  if (i >= argc)
    ERROR("PARSEDINT","\"" << argv[i-1] << "\" found without argument");
  int i_tmp = sscanf(argv[i],"%i,%i",&var1,&var2);
  if (i_tmp != 2 )
    ERROR("PARSEDINT","\"" << argv[i-1] << "\" is not followed by an <int>,<int>");
  if (var1 < min1 || var1 > max1 || var2 < min2 || var2 > max2)
    ERROR("PARSEDINT","\"" << argv[i-1] << "\" is followed by integers out of the range");
  if (var1 > var2)
    ERROR("PARSEDINT","\"" << var1 << "\" must be <= \"" << var2 << "");
}

/// parse and check a list of integers
void PARSEINTS(int & i, char ** argv, int argc,
	       vector<int> & table, int neededsize = 0, bool minmax=false, int vmin=0, int vmax=1000, bool complete=false) {
  i++;
  if (i >= argc)
    ERROR("PARSEINTS","\"" << argv[i-1] << "\" found without argument");

  /* read table */
  table.clear();
  char * pch = strtok(argv[i],",;");
  while (pch){
    int value = 0;
    int i_tmp = sscanf(pch,"%d",&value);
    if (i_tmp != 1)
      ERROR("PARSEINTS","\"" << pch << "\" is not a correct <int>");
    if (minmax) {
      if (value < vmin || value > vmax)
        ERROR("PARSEINTS","\"" << value << "\" is by an integer out of the range");
    }
    table.push_back(value);
    pch = strtok(NULL,",;");
  }

  /* check table size if needed */
  if (neededsize != 0) {
    if (complete && (int) table.size() < neededsize) {
      while ((int)table.size() < neededsize)
	table.push_back(table.back());
    }
    if ((int)table.size() != neededsize)
      ERROR("PARSEINTS", " there is not a correct number of <int> values ("<< (table.size()) <<") given by " << argv[i-1] << " when compared with the needed size " << neededsize);
  }
}

/// parse and check a double
void PARSEDOUBLE(int & i, char ** argv, int argc, double & var , double min, double max) {
  i++;
  if (i >= argc)
    ERROR("PARSEDOUBLE","\"" << argv[i-1] << "\" found without argument");
  char * next = argv[i];
  double d_tmp = strtod(argv[i],&next);
  if (errno != 0 || ((*next) != '\0'))
    ERROR("PARSEDOUBLE","\"" << argv[i-1] << "\" is not followed by a double") ;
  if (d_tmp>max || d_tmp <min)
    ERROR("PARSE","\"" << argv[i-1] << "\" is followed by a double outside the valid range");
  var = d_tmp;
}

/// parse and check a pair of double
void PARSEDDOUBLE(int & i, char ** argv, int argc,
                  double & var1, double min1, double max1,
                  double & var2, double min2, double max2) {
  i++;
  if (i >= argc)
    ERROR("PARSEDDOUBLE","\"" << argv[i-1] << "\" found without argument");
  int i_tmp = sscanf(argv[i],"%lf,%lf",&var1,&var2);
  if (i_tmp != 2 )
    ERROR("PARSEDDOUBLE","\"" << argv[i-1] << "\" is not followed by an <dbl>,<dbl>");
  if (var1 < min1 || var1 > max1 || var2 < min2 || var2 > max2)
    ERROR("PARSEDDOUBLE","\"" << argv[i-1] << "\" is followed by doubles out of the range");
  if (var1 > var2)
    ERROR("PARSEDDOUBLE","\"" << var1 << "\" must be <= \"" << var2 << "");
}

/// parse and check a list of double
void PARSEDOUBLES(int & i, char ** argv, int argc,
                  vector<double> & table, int neededsize = 0, bool positive_values=false) {
  i++;
  if (i >= argc)
    ERROR("PARSEDOUBLES","\"" << argv[i-1] << "\" found without argument");

  /* read table */
  table.clear();
  char * pch = strtok(argv[i],",;");
  while (pch){
    double value = 0.0;
    int i_tmp = sscanf(pch,"%lf",&value);
    if (i_tmp != 1)
      ERROR("PARSEDOUBLES","\"" << pch << "\" is not a correct <dbl>");
    if (positive_values) {
      if (value < 0)
        ERROR("PARSEDOUBLES","\"" << value << "\" is a negative <dbl>");
    }
    table.push_back(value);
    pch = strtok(NULL,",;");
  }

  /* check table size if needed */
  if (neededsize != 0) {
    if ((int)table.size() != neededsize)
      ERROR("PARSEDOUBLES", " there is not a correct number of <dbl> values ("<< (table.size()) <<") given by " << argv[i-1] << " when compared with the needed size " << neededsize);
  }
}


/// parse and check a signature (number of seed elements inside a seed)
void PARSESIGNATURE(int & i, char ** argv, int argc, vector<int> & table) {
  i++;
  if (i >= argc)
    ERROR("PARSESIGNATURE","\"" << argv[i-1] << "\" found without argument");

  /* read table */
  char * pch = strtok(argv[i],",;");
  table.clear();
  while (pch){
    int value = 0;
    int i_tmp = sscanf(pch,"%d",&value);
    if (i_tmp != 1)
      ERROR("PARSESIGNATURE","\"" << pch << "\" is not a correct integer");
    if (value < 0 || value >= 64)
      ERROR("PARSESIGNATURE","\"" << pch << "\" is out of the range [0..N] (N=64)");
    table.push_back(value);
    pch = strtok(NULL,",;");
  }

  /* check table size */
  if ((int)table.size() != gv_seed_alphabet_size)
    ERROR("PARSESIGNATURE", " there is not a correct number of <int> values given by " << argv[i-1] << " when compared with the current seed alphabet size B = " << gv_seed_alphabet_size);
}


/// parse and check a homogeneous scoring system
void PARSEHOMOGENEOUS(int & i, char ** argv, int argc, vector<int> & table) {
  i++;
  if (i >= argc)
    ERROR("PARSEHOMOGENEOUS","\"" << argv[i-1] << "\" found without argument");

  /* read table */
  char * pch = strtok(argv[i],",;");
  table.clear();
  while (pch){
    int value = 0;
    int i_tmp = sscanf(pch,"%d",&value);
    if (i_tmp != 1)
      ERROR("PARSEHOMOGENEOUS","\"" << pch << "\" is not a correct integer");
    if (value < -1000 || value > 1000)
      ERROR("PARSEHOMOGENEOUS","\"" << pch << "\" is out of the range [-1000..1000]");
    table.push_back(value);
    pch = strtok(NULL,",;");
  }

  /* check table size */
  if ((int)table.size() != gv_align_alphabet_size)
    ERROR("PARSEHOMOGENEOUS", " there is not a correct number of <int> values given by " << argv[i-1] << " when compared with the current align alphabet size A = " << gv_align_alphabet_size);
}

/// parse and check a set of probabilities
void PARSEPROBS(int & i, char ** argv, int argc,
                vector<double> & table, int & k) {
  i++;
  if (i >= argc)
    ERROR("PARSEPROBS","\"" << argv[i-1] << "\" found without argument");

  /* read table */
  char * pch = strtok(argv[i],",;");
  table.clear();
  while (pch){
    double value = 0.0;
    int i_tmp = sscanf(pch,"%lf",&value);
    if (i_tmp != 1)
      ERROR("PARSEPROBS","\"" << pch << "\" is not a correct <dbl>");
    if (value < 0 || value > 1 )
      ERROR("PARSEPROBS","\"" << pch << "\" is  out of the range [0,1]");
    table.push_back(value);
    pch = strtok(NULL,",;");
  }

  /* check table size */
  int a = gv_align_alphabet_size;
  k = 0;
  while(1) {
    if ((int)table.size() == a)
      break;
    else if ((int)table.size() < a)
      ERROR("PARSEPROBS", " there is not a correct number of <dbl> values ("<< (table.size()) <<") given by " << argv[i-1] << " when compared with the current alphabet size A = " << gv_align_alphabet_size);
    a *= gv_align_alphabet_size;
    k++;
  }
  CHECKPROB(table);
}

/// parse and check a set of probabilities as an automaton file
void PARSEPROBSAUTOMATONFILE(int & i, char ** argv, int argc, automaton ** p_a) {
  i++;
  if (i >= argc)
    ERROR("PARSEMATRIXFILE","\"" << argv[i-1] << "\" found without argument");

  /* read file */
  ifstream ifs_file;
  ifs_file.open(argv[i]);
  if (!ifs_file){
    cerr << "* Error : unreadable file \"" << (argv[i]) << "\" " << endl;
    exit(-1);
  }
  
  /* read the content and set the automaton */
  *p_a = new automaton();

  ifs_file >> (**p_a);
  ifs_file.close();
}

  

/// set of states on a matrix input
/// @see PARSEMATRIX
typedef enum {
  dummy,
  readfirstopenbracket,
  readsecondopenbracket,
  readint,
  readintseparator,
  readsecondclosedbracket,
  readsecondbracketseparator,
  readfirstclosedbracket,
  readfirstbracketseparator
} automatareadmatrixstates;



/// parse and check a matrix input 
void PARSEMATRIX(int & i, char ** argv, int argc, vector< vector<int> > & matrix, int min, int max, int gnbcolumns, int gnbrows) {

  i++;
  if (i >= argc)
    ERROR("PARSEMATRIX","\"" << argv[i-1] << "\" found without argument");
  char * pch = argv[i];
  int nbcolumns = 0, nbrows = 0;

  automatareadmatrixstates state = dummy;
  /* automata */
  while(*pch) {

    switch (*pch){

    case '{':
      switch (state){
      case dummy:
        nbrows = 0;
        state = readfirstopenbracket;
        break;
      case readfirstopenbracket:
      case readsecondclosedbracket:
      case readsecondbracketseparator:
        nbcolumns = 0;
        nbrows++;
        if (nbrows > gnbrows)
          ERROR("PARSEMATRIX","incorrect number of lines : " << nbrows << " != " << gnbrows << " in \"" << argv[i] << "\" string parameter ");

        state = readsecondopenbracket;
        break;
      default:
        ERROR("PARSEMATRIX","incorrect \'" << *pch << "\' in \"" << argv[i] << "\" string parameter ");
      }
      pch++;
      break;

    case '}':
      switch (state){
      case readint:
      case readintseparator:
        state = readsecondclosedbracket;
        if (nbcolumns != gnbcolumns)
          ERROR("PARSEMATRIX","incorrect number of columns :  " << nbcolumns << " != " << gnbcolumns << " in \"" << argv[i] << "\" string parameter ");
        break;
      case readsecondclosedbracket:
        state = readfirstclosedbracket;
        if (nbrows != gnbrows )
          ERROR("PARSEMATRIX","incorrect number of lines : " << nbrows << " != " << gnbrows << " in \"" << argv[i] << "\" string parameter ");
        break;
      default:
        ERROR("PARSEMATRIX","incorrect \'" << *pch << "\' in \"" << argv[i] << "\" string parameter ");
      }
      pch++;
      break;

    case ' ':
    case '\t':
    case '\n':
      pch++;
      break;

    case ';':
    case ',':
      switch (state){
      case readint:
        state = readintseparator;
        break;
      case readsecondclosedbracket:
        state = readsecondbracketseparator;
        break;
      case readfirstclosedbracket:
        state = readfirstbracketseparator;
        break;
      default:
        ERROR("PARSEMATRIX","incorrect \'" << *pch << "\' in \"" << argv[i] << "\" string parameter ");
      }
      pch++;
      break;

    default: /* read value */
      if (state == readsecondopenbracket || state == readintseparator) {
        state = readint;
        int value = strtol (pch, &pch, 10);
        if (value>max || value<min)
          ERROR("PARSEMATRIX","\"" << argv[i] << "\" contains at least one integer outside the valid range");
        if (nbcolumns >= gnbcolumns)
          ERROR("PARSEMATRIX","incorrect number of columns : " << (nbcolumns+1) << ">" <<  gnbcolumns << " in \"" << argv[i] << "\" string parameter ");
        assert(nbrows-1<gnbrows);
        matrix[nbcolumns][nbrows-1] = value;
        nbcolumns++;

      } else {
        ERROR("PARSEMATRIX","incorrect \'" << *pch << "\' in \"" << argv[i] << "\" string parameter ");
      }
    }
  } /* while */
  if (state == readfirstclosedbracket || state == readfirstbracketseparator)
    return;
  else
    ERROR("PARSEMATRIX","\" unfinished string parameter \"" << argv[i] << "\" ");
}


/// parse and check a matrix input as a file
void PARSEMATRIXFILE(int & i, char ** argv, int argc, vector< vector<int> > & matrix, int min, int max, int gnbcolumns, int gnbrows) {

  i++;
  if (i >= argc)
    ERROR("PARSEMATRIXFILE","\"" << argv[i-1] << "\" found without argument");

  /* read file */
  ifstream ifs_file;
  ifs_file.open(argv[i]);
  if (!ifs_file){
    cerr << "* Error : unreadable file \"" << (argv[i]) << "\" " << endl;
    exit(-1);
  }

  /* transform it simply into a string */
  char c;
  string s;
  while (ifs_file.get(c)) {
    s += c;
  }
  const char * sc = s.c_str();


  char * pch = (char *) sc;
  int nbcolumns = 0, nbrows = 0;

  automatareadmatrixstates state = dummy;
  /* automata */
  while(*pch) {

    switch (*pch){

    case '{':
      switch (state){
      case dummy:
        nbrows = 0;
        state = readfirstopenbracket;
        break;
      case readfirstopenbracket:
      case readsecondclosedbracket:
      case readsecondbracketseparator:
        nbcolumns = 0;
        nbrows++;
        if (nbrows > gnbrows)
          ERROR("PARSEMATRIX","incorrect number of lines : " << nbrows << " != " << gnbrows << " in \"" << argv[i] << "\" string parameter ");

        state = readsecondopenbracket;
        break;
      default:
        ERROR("PARSEMATRIX","incorrect \'" << *pch << "\' in \"" << argv[i] << "\" string parameter ");
      }
      pch++;
      break;

    case '}':
      switch (state){
      case readint:
      case readintseparator:
        state = readsecondclosedbracket;
        if (nbcolumns != gnbcolumns)
          ERROR("PARSEMATRIX","incorrect number of columns :  " << nbcolumns << " != " << gnbcolumns << " in \"" << argv[i] << "\" string parameter ");
        break;
      case readsecondclosedbracket:
        state = readfirstclosedbracket;
        if (nbrows != gnbrows )
          ERROR("PARSEMATRIX","incorrect number of lines : " << nbrows << " != " << gnbrows << " in \"" << argv[i] << "\" string parameter ");
        break;
      default:
        ERROR("PARSEMATRIX","incorrect \'" << *pch << "\' in \"" << argv[i] << "\" string parameter ");
      }
      pch++;
      break;

    case ' ':
    case '\t':
    case '\n':
      pch++;
      break;

    case ';':
    case ',':
      switch (state){
      case readint:
        state = readintseparator;
        break;
      case readsecondclosedbracket:
        state = readsecondbracketseparator;
        break;
      case readfirstclosedbracket:
        state = readfirstbracketseparator;
        break;
      default:
        ERROR("PARSEMATRIX","incorrect \'" << *pch << "\' in \"" << argv[i] << "\" string parameter ");
      }
      pch++;
      break;

    default: /* read value */
      if (state == readsecondopenbracket || state == readintseparator) {
        state = readint;
        int value = strtol (pch, &pch, 10);
        if (value>max || value<min)
          ERROR("PARSEMATRIX","\"" << argv[i] << "\" contains at least one integer outside the valid range");
        if (nbcolumns >= gnbcolumns)
          ERROR("PARSEMATRIX","incorrect number of columns : " << (nbcolumns+1) << ">" <<  gnbcolumns << " in \"" << argv[i] << "\" string parameter ");
        assert(nbrows-1<gnbrows);
        matrix[nbcolumns][nbrows-1] = value;
        nbcolumns++;

      } else {
        ERROR("PARSEMATRIX","incorrect \'" << *pch << "\' in \"" << argv[i] << "\" string parameter ");
      }
    }
  } /* while */
  if (state == readfirstclosedbracket || state == readfirstbracketseparator)
    return;
  else
    ERROR("PARSEMATRIX","\" unfinished string parameter \"" << argv[i] << "\" ");


}


/// check a matrix input as a subset seed matching set
void CHECKMATCHINGMATRIX( vector< vector<int> > & matrix ) {
  gv_matching_symbol_flag = true;

  /* check the '1' symbol existance */
  for (int b=0; b<gv_seed_alphabet_size; b++) {
    if ( matrix[gv_align_alphabet_size-1][b] == 0 ) {
      gv_matching_symbol_flag = false;
    }
  }
  /* check the '#' symbol existance */
  for(int a=0; a <gv_align_alphabet_size-1;a++) {
    if ( matrix[a][gv_seed_alphabet_size-1] == 1 ) {
      gv_matching_symbol_flag = false;
    }
  }
  if ( matrix[gv_align_alphabet_size-1][gv_seed_alphabet_size-1] == 0 ) {
    gv_matching_symbol_flag = false;
  }
}

/**
 * @brief parse new seed symbols when "overdubbed"
 * @param i is the command line argument number being processed
 * @param argv is the command line arguments being processed
 * @param argc is the command line maximal number of arguments being processed
 * @param size is the size of the seed alphabet
 * @param array is the new table where seed alphabet symbols will be stored
 * @see gv_bsymbols_flag,gv_bsymbols_array
 */
void PARSESYMBOLS(int & i, char ** argv, int argc, int size, char * &array) {
  i++;
  if (i >= argc)
    ERROR("PARSESYMBOLS","\"" << argv[i-1] << "\" found without argument");
  if (strlen(argv[i]) != (unsigned) size)
    ERROR("PARSESYMBOLS","\"" << argv[i] << "\" is not of required length " << size);
  array = new char[size];
  for(char * d = array, *s = argv[i]; *s ; d++, s++){
    if (*s == ':') {
      ERROR("PARSESYMBOLS","\'" << *s << "\' is a reserved symbol that must not be used in a seed");
    }
    *d = *s;
  }
}


/// set of states on a seed input
/// @see PARSESEEDS
typedef enum {
  readseed,
  readphase,
  readphasecycle
} automatareadseedstates;


/// parse a seed or a set of seeds (possibly with a set of restricted positions in a cycle)
void PARSESEEDS(int & i, char ** argv, int argc) {
  i++;
  if (i >= argc)
    ERROR("PARSESEEDS","" << argv[i-1] << "\" found without argument");
  gv_motif   = 1;
  gv_nbseeds = 0;

  gv_cycles          = vector< int >(0);
  gv_cycles_pos_nb   = vector< int >(0);
  gv_cycles_pos_list = vector< vector<int> >(0);

  char * pch = argv[i];
  int    num = 0;
  char * pch_seed = NULL;
  char * pch_last_seed = NULL;
  automatareadseedstates state = readseed;

  while (pch != NULL) {
    switch (state) {
      /* a) reading a seed */
    case readseed:
      if (!pch_seed)
	 pch_last_seed = pch_seed = pch;
      switch (*pch) {
      case ':':
	state = readphase;
      case ',':
      case ';':
	*pch = '\0';
	{
	  string s = string(strdup(pch_seed));
	  pch_seed = NULL;
	  gv_seeds[gv_nbseeds] = new seed(s);
	  gv_nbseeds++;
	  if ( gv_nbseeds > MAXSEEDS)
	    ERROR("PARSESEEDS","element " << gv_nbseeds << "\" is out of array bounds");		
	}
	break;
      case '\0':
	{
	  string s = string(strdup(pch_seed));
	  pch_seed = NULL;
	  gv_seeds[gv_nbseeds] = new seed(s);
	  gv_nbseeds++;
	  if ( gv_nbseeds > MAXSEEDS)
	    ERROR("PARSESEEDS","element " << gv_nbseeds << "\" is out of array bounds");
	}
	goto eowhile;

      } /* switch(*pch) */
      break;

      /* b) reading phases */
    case readphase:	
      /* cycle allowed */

      /* >> */
      gv_cycles_flag = true;
      while ((int)gv_cycles.size() < gv_nbseeds)
	gv_cycles.push_back(1);
      while ((int)gv_cycles_pos_nb.size() < gv_nbseeds)
	gv_cycles_pos_nb.push_back(0);
      while ((int)gv_cycles_pos_list.size() < gv_nbseeds)
	gv_cycles_pos_list.push_back(vector <int>(0));
      /* << */

      switch(*pch) {
      case '/':
	state = readphasecycle;
      case ',':
      case ';':
	gv_cycles_pos_nb[gv_nbseeds-1]++;
	gv_cycles_pos_list[gv_nbseeds-1].push_back(num);
	num = 0;
	break;
      case '0':	
      case '1':
      case '2':
      case '3':
      case '4':
      case '5':
      case '6':
      case '7':
      case '8':
      case '9':
	num = num * 10 + (*pch - '0');
	break;
      default:
	ERROR("PARSESEEDS","[between ':' and '/'] incorrect \'" << *pch << "\'  after the seed \"" << pch_last_seed << "\" string parameter ");
      } /* switch(*pch) */
      break;

      /* c) cycle */
    case readphasecycle:
      switch(*pch) {
      case '0':
      case '1':
      case '2':
      case '3':
      case '4':
      case '5':
      case '6':
      case '7':
      case '8':
      case '9':
	num = num * 10 + (*pch - '0');
	break;
      case ',':
      case ';':
	if (num <= 0)
	  ERROR("PARSESEEDS","[after '/'] incorrect cycle size \'" << num << "\' after the seed \"" << pch_last_seed << "\" string parameter ");
	gv_cycles[gv_nbseeds-1] = num;
	state = readseed;
	num = 0;
	break;
      case '\0':
	if (num <= 0)
	  ERROR("PARSESEEDS","[after '/'] incorrect cycle size \'" << num << "\'  after the seed \"" << pch_last_seed << "\" string parameter ");
	gv_cycles[gv_nbseeds-1] = num;
	num = 0;
	goto eowhile;
      default:
	ERROR("PARSESEEDS","[after '/'] incorrect \'" << *pch << "\'  after the seed \"" << pch_last_seed << "\" string parameter ");
      } /* switch(*pch) */
      break;
    } /* switch(state) */
    pch++;
  }
 eowhile:
  /* >> */
  if (gv_cycles_flag) {
    while ((int)gv_cycles.size() < gv_nbseeds)
      gv_cycles.push_back(1);
    while ((int)gv_cycles_pos_nb.size() < gv_nbseeds)
      gv_cycles_pos_nb.push_back(0);
    while ((int)gv_cycles_pos_list.size() < gv_nbseeds)
      gv_cycles_pos_list.push_back(vector <int>(0));


    for(int i=0;i<gv_nbseeds;i++) {
      
      if (gv_cycles_pos_nb[i] == 0) {
	gv_cycles_pos_nb[i] = 1;
	gv_cycles_pos_list[i].push_back(1);
      }

      for(int j=0;j<gv_cycles_pos_nb[i];j++) {	
	if (gv_cycles_pos_list[i][j] > gv_cycles[i]) {
	  ERROR("PARSESEEDS"," incorrect cycle position \"" << (gv_cycles_pos_list[i][j]) << "\" > cycle size \"" << (gv_cycles[i]) << "\"");
	}
	if (gv_cycles_pos_list[i][j] < 1) {
	  ERROR("PARSESEEDS"," incorrect cycle position \"" << (gv_cycles_pos_list[i][j]) << "\" < 1 \"");
	}
	gv_cycles_pos_list[i][j] =  (gv_cycles_pos_list[i][j] + gv_seeds[i]->span() - 1) % gv_cycles[i];
      }
      sort(gv_cycles_pos_list[i].begin(),gv_cycles_pos_list[i].end());
    }
  }
  /* << */
}



/// parse a seed or a set of seeds to exclude
void PARSEXSEEDS(int & i, char ** argv, int argc) {
  i++;
  if (i >= argc)
    ERROR("PARSEXSEEDS","\"" << argv[i-1] << "\" found without argument");

  gv_xnbseeds = 0;

  char * pch = strtok(argv[i],",;");
  while (pch) {
    string s = string(strdup(pch));
    gv_xseeds[gv_xnbseeds] = new seed(s);
    gv_xnbseeds++;
    if ( gv_xnbseeds > MAXSEEDS)
      ERROR("PARSEXSEEDS","\" element " << gv_xnbseeds << "\" is out of array bounds");
    pch = strtok(NULL,",;");
  }
}


/// parse a pareto input file
void PARSEINPUT(int & i, char ** argv, int argc) {
  i++;
  if (i >= argc)
    ERROR("PARSEINPUT","\"" << argv[i-1] << "\" found without argument");

  char * pch = strtok(argv[i],",;");
  while (pch) {
    char * file = strdup(pch);
    gv_input_filenames[gv_nb_input_filenames++] = file;
    pch = strtok(NULL,",;");
  }
}

/// parse the pareto output file
void PARSEOUTPUT(int & i, char ** argv, int argc) {
  i++;
  if (i >= argc)
    ERROR("PARSEINPUT","\"" << argv[i-1] << "\" found without argument");
  gv_output_filename = strdup(argv[i]);
}

/// main function that scan command line arguments
void SCANARG(int argc , char ** argv) {
  for(int i=1; i<argc; i++){

    /* 1) Seed model */
    if(!strcmp(argv[i],"-A")||!strcmp(argv[i],"--A")) {
      PARSEINT(i,argv,argc,gv_align_alphabet_size,2,1024);
      build_default_subsetseed_matching_matrix();
      build_default_vectorizedsubsetseed_scoring_matrix();
      build_default_probabilities();
      if (gv_homogeneous_flag) {
	gv_homogeneous_flag = false;
	WARNING("\"-u\" OPTION DISABLED","align alphabet size was changed \"after\" setting the \"-u\" option");
      }
      if (gv_lossless_flag) {
	gv_lossless_flag    = false;
	WARNING("\"-L\" OPTION DISABLED","align alphabet size was changed \"after\" setting the \"-L\" option");
      }
    } else if(!strcmp(argv[i],"-B")||!strcmp(argv[i],"--B")) {
      PARSEINT(i,argv,argc,gv_seed_alphabet_size,1,1024);
      build_default_subsetseed_matching_matrix();
      build_default_vectorizedsubsetseed_scoring_matrix();
      build_default_probabilities();
      if (gv_signature_flag) {
	gv_signature_flag = false;
	WARNING("\"-i\" OPTION DISABLED","seed alphabet size was changed \"after\" setting the \"-i\" option");
      }
      if (gv_bsymbols_flag) {
	gv_bsymbols_flag  = false;
	WARNING("\"-BSymbols\" OPTION DISABLED","seed alphabet size was changed \"after\" setting the \"-BSymbols\" option");
      }
      if (gv_xnbseeds) {
	gv_xnbseeds = 0;
	WARNING("\"-mx\" OPTION DISABLED","seed alphabet size was changed \"after\" setting the \"-mx\" option");
      }
      if (gv_motif) {
	gv_motif = 0;
	WARNING("\"-m\" OPTION DISABLED","seed alphabet size was changed \"after\" setting the \"-m\" option");
      }
      /* 1.1) subset seeds */
    } else if(!strcmp(argv[i],"-M")||!strcmp(argv[i],"--Matching")) {
      PARSEMATRIX(i,argv,argc,gv_subsetseed_matching_matrix,0,1,gv_align_alphabet_size,gv_seed_alphabet_size);
      CHECKMATCHINGMATRIX(gv_subsetseed_matching_matrix);
    } else if(!strcmp(argv[i],"-MF")||!strcmp(argv[i],"--MatchingFile")) {
      PARSEMATRIXFILE(i,argv,argc,gv_subsetseed_matching_matrix,0,1,gv_align_alphabet_size,gv_seed_alphabet_size);
      CHECKMATCHINGMATRIX(gv_subsetseed_matching_matrix);

      /* 1.2) vectorized subset seeds */
    } else if(!strcmp(argv[i],"-S")||!strcmp(argv[i],"--Scoring")) {
      PARSEMATRIX(i,argv,argc,gv_vectorizedsubsetseed_scoring_matrix,-10000,10000,gv_align_alphabet_size,gv_seed_alphabet_size);
    } else if(!strcmp(argv[i],"-SF")||!strcmp(argv[i],"--ScoringFile")) {
      PARSEMATRIXFILE(i,argv,argc,gv_vectorizedsubsetseed_scoring_matrix,-10000,10000,gv_align_alphabet_size,gv_seed_alphabet_size);
    } else if(!strcmp(argv[i],"-T")||!strcmp(argv[i],"--Threshold")) {
      PARSEINT(i,argv,argc,gv_vectorizedsubsetseed_scoring_threshold,-10000,+10000);
    } else if(!strcmp(argv[i],"-V")||!strcmp(argv[i],"--Vectorized")) {
      PARSEFLAG(i,argv,argc,gv_vectorized_flag,true);

      /* 1.3) lossless seeds */
    } else if(!strcmp(argv[i],"-L")||!strcmp(argv[i],"--LosslessCosts")) {
      PARSEINTS(i,argv,argc,gv_lossless_costs_vector,gv_align_alphabet_size,true,0,1000);
      gv_lossless_flag = true;
    } else if(!strcmp(argv[i],"-X")||!strcmp(argv[i],"--LosslessCostThreshold")) {
      PARSEINT(i,argv,argc,gv_lossless_cost_threshold,0,1000);

    /* 2) Alignment model */
    } else if(!strcmp(argv[i],"-b")||!strcmp(argv[i],"--background")) {
      PARSEPROBS(i,argv,argc,gv_bsel,gv_bsel_k);
    } else if(!strcmp(argv[i],"-f")||!strcmp(argv[i],"--foreground")) {
      PARSEPROBS(i,argv,argc,gv_bsens,gv_bsens_k);
    } else if(!strcmp(argv[i],"-fF")||!strcmp(argv[i],"--foregroundFile")) {
      PARSEPROBSAUTOMATONFILE(i,argv,argc,&gv_bsens_automaton);
    } else if(!strcmp(argv[i],"-l")||!strcmp(argv[i],"--length")) {
      PARSEINT(i,argv,argc,gv_alignment_length,1,1000000);
    } else if(!strcmp(argv[i],"-u")||!strcmp(argv[i],"--homogeneous")) {
      PARSEHOMOGENEOUS(i,argv,argc,gv_homogeneous_scores);
      gv_homogeneous_flag = true;
    /* 3) Seed enumeration */
    } else if(!strcmp(argv[i],"-s")||!strcmp(argv[i],"--span")) {
      PARSEDINT(i,argv,argc,gv_minspan,1,1000,gv_maxspan,1,1000);
    } else if(!strcmp(argv[i],"-w")||!strcmp(argv[i],"--weight")) {
      gv_weight_interval = 1;
      PARSEDDOUBLE(i,argv,argc,gv_minweight,0,1e32,gv_maxweight,0,1e32);
    } else if(!strcmp(argv[i],"-n")||!strcmp(argv[i],"--number")) {
      PARSEINT(i,argv,argc,gv_nbseeds,1,MAXSEEDS);
      if (gv_cycles_flag) {
	ERROR(" \"-c\" cycle flag set BEFORE changing the number \"-n\" of seeds is not correct","you must provide the \"-c\"  and \"-q\" parameters AFTER the \"-n\"");
      }
    } else if(!strcmp(argv[i],"-r")||!strcmp(argv[i],"--random")) {
      PARSEINT(i,argv,argc,gv_nbruns,1,1000000000);
    } else if(!strcmp(argv[i],"-k")||!strcmp(argv[i],"--hillclimbing")) {
      if (gv_nbruns == 0) {
	ERROR(" \"-k\" hillclimbing flag set but \"-r\" has not been set before (still in full enumeration mode)","always provide the \"-k\" parameter AFTER changing the default \"-r\" behavior");
      }
      PARSEFLAG(i,argv,argc,gv_hillclimbing_flag,true);
    } else if(!strcmp(argv[i],"-a")||!strcmp(argv[i],"--hillclimbing-alpha")) {
      PARSEDOUBLE(i,argv,argc,gv_hillclimbing_alpha,0,1);
    } else if(!strcmp(argv[i],"-d")||!strcmp(argv[i],"--disable")) {
      PARSEFLAG(i,argv,argc,gv_minimize_flag,false);
    } else if(!strcmp(argv[i],"-m")||!strcmp(argv[i],"--motifs")) {
      if (gv_cycles_flag) {
	ERROR("\"-m\" pattern and \"-c\" are not compatible together"," you must provide the cycle positions on the shape (e.g  \" -m 100101001:1,3,5/6\") ");
      }
      PARSESEEDS(i,argv,argc);
    } else if(!strcmp(argv[i],"-mx")||!strcmp(argv[i],"--motifs-excluded")) {
      PARSEXSEEDS(i,argv,argc);
    } else if(!strcmp(argv[i],"-i")||!strcmp(argv[i],"--idsign")) {
      PARSESIGNATURE(i,argv,argc,gv_signature);
      gv_signature_flag = true;
    } else if(!strcmp(argv[i],"-x")||!strcmp(argv[i],"--symetric")) {
      PARSEFLAG(i,argv,argc,gv_symetric,true);
    } else if(!strcmp(argv[i],"-y")||!strcmp(argv[i],"--multihit")) {
      PARSEINT(i,argv,argc,gv_multihit_nb,1,64);
      gv_multihit_flag = true;
    } else if(!strcmp(argv[i],"-c")||!strcmp(argv[i],"--cycles")) {
      if (gv_motif) {
	ERROR("\"-m\" pattern and \"-c\" are not compatible together"," you must provide the cycle positions on the shape (e.g  \" -m 100101001:1,3,5/6\") ");
      } 
      gv_cycles_flag = true;
      PARSEINTS(i,argv,argc,gv_cycles,gv_nbseeds,true,1,1000,true);      
      /* set the init cycles positions */
      gv_cycles_pos_nb    = vector < int > (gv_nbseeds,1);
      gv_cycles_pos_list  = vector < vector < int > >(gv_nbseeds,vector<int>(1,0));
    } else if(!strcmp(argv[i],"-q")||!strcmp(argv[i],"--cyclespos")) {
      if (gv_cycles_flag) {
	if (gv_motif) {
	  ERROR("\"-m\" pattern and \"-c\" are not compatible together"," you must provide the cycle positions on the shape (e.g  \" -m 100101001:1,3,5/6\") ");
	}
	/* set the init cycles positions */
	PARSEINTS(i,argv,argc,gv_cycles_pos_nb,gv_nbseeds,true,1,1000,true);	
	gv_cycles_pos_list = vector < vector < int > >(gv_nbseeds,vector<int>(0));
	for(int i=0;i<gv_nbseeds;i++) {
	  gv_cycles_pos_nb[i] = MIN(gv_cycles_pos_nb[i],gv_cycles[i]);
	  for(int j=0;j<gv_cycles_pos_nb[i];j++)
	    gv_cycles_pos_list[i].push_back(j);
	}
      } else {
	ERROR(" -q ","\"" << argv[i] << "\" must be preceded by -c ... ");
      }
    } else if(!strcmp(argv[i],"-j")||!strcmp(argv[i],"--jive")) {
      PARSEDOUBLE(i,argv,argc,gv_jive,0,1);
      /* 4) misc */
    } else if(!strcmp(argv[i],"-e")||!strcmp(argv[i],"--enter")) {
      PARSEINPUT(i,argv,argc);
    } else if(!strcmp(argv[i],"-o")||!strcmp(argv[i],"--output")) {
      PARSEOUTPUT(i,argv,argc);
    } else if(!strcmp(argv[i],"-t")||!strcmp(argv[i],"--time")) {
      PARSEFLAG(i,argv,argc,gv_sleeping_flag,true);
    } else if(!strcmp(argv[i],"-p")||!strcmp(argv[i],"--processes")) {
      PARSEINT(i,argv,argc,gv_nb_processes,0,10000);
      gv_nb_processes_flag = true;
    } else if(!strcmp(argv[i],"-h")||!strcmp(argv[i],"--help")) {
      USAGE();
    } else if(!strcmp(argv[i],"-v")||!strcmp(argv[i],"--version")) {
      VERS();
      /* 5) */
    } else if (!strcmp(argv[i],"-BSymbols")) {
      PARSESYMBOLS(i,argv,argc,gv_seed_alphabet_size,gv_bsymbols_array);
      gv_bsymbols_flag = true;
    } else if (!strcmp(argv[i],"-transitive")) {
      /* transitives params */
      gv_align_alphabet_size = 3;
      gv_seed_alphabet_size  = 3;
      gv_xnbseeds = 0;
      gv_motif = 0;
      gv_lossless_flag = false;
      gv_signature_flag = true;
      gv_signature = vector<int>(3); gv_signature[0] = 0; gv_signature[1] = 2; gv_signature[2] = 8;
      gv_bsens = vector<double>(3); gv_bsens[0] = 0.15; gv_bsens[1] = 0.15; gv_bsens[2] = 0.7;
      gv_bsens_k = 0;
      gv_bsel  = vector<double>(3); gv_bsel[0]  = 0.50; gv_bsel[1]  = 0.25; gv_bsel[2]  = 0.25;
      gv_bsel_k = 0;
      gv_bsel_weight = vector<double>(3); gv_bsel_weight[0] = 0.0; gv_bsel_weight[1] = 0.5; gv_bsel_weight[2] = 1.0;
      gv_bsel_minprob = 0.25;
      gv_bsel_maxprob = 1.0;
      gv_minspan = 10;
      gv_maxspan = 15;
      gv_minweight = 9; gv_maxweight = 9;
      gv_weight_interval = true;
      gv_vectorized_flag = false;
      gv_bsymbols_array = strdup(string("-@#").c_str());
      gv_bsymbols_flag = true;
      build_default_subsetseed_matching_matrix();
      gv_alignment_length = 64;
    } else if (!strcmp(argv[i],"-spaced")) {
      /* spaced params */
      gv_align_alphabet_size = 2;
      gv_seed_alphabet_size  = 2;
      gv_lossless_flag = false;
      gv_signature_flag = false;
      gv_xnbseeds = 0;
      gv_motif = 0;
      gv_bsens = vector<double>(2); gv_bsens[0] = 0.30; gv_bsens[1] = 0.70;
      gv_bsens_k = 0;
      gv_bsel  = vector<double>(2); gv_bsel[0]  = 0.75; gv_bsel[1]  = 0.25;
      gv_bsel_k = 0;
      gv_bsel_weight = vector<double>(2); gv_bsel_weight[0] = 0.0; gv_bsel_weight[1] = 1.0;
      gv_bsel_minprob = 0.25;
      gv_bsel_maxprob = 1.0;
      gv_minspan = 9;
      gv_maxspan = 15;
      gv_minweight = 9; gv_maxweight = 9;
      gv_weight_interval = true;
      gv_vectorized_flag = false;
      gv_bsymbols_array = strdup(string("-#").c_str());
      gv_bsymbols_flag  = true;
      build_default_subsetseed_matching_matrix();
      gv_alignment_length = 64;
    } else ERROR("PARSE","\"" << argv[i] << "\" is not a valid argument");
  }
}
// @}



/** @name seedproperties functions
 *  @brief class and set of function that keep seed properties
 */

// @{

/** @class seedproperties
 *  @brief simple "inner class" to keep seed properties as
 *   sensitivity, selectivity ... and other properties
 */

class seedproperties {

 public:
  /** @brief build a "seedproperties" object to keep current seed properties when needed
   */
  seedproperties(double sel, double sens, double dist, std::string str, bool lossless = false){
    this->sel   = sel;
    this->sens  = sens;
    this->dist  = dist;
    this->str   = string(str);
    this->lossless = lossless;
  }
  /// seed selectivity
  double sel;
  /// seed sensitivity
  double sens;
  /// (sel,sens) euclidian distance to (1,1)
  double dist;
  /// string representing the seed
  string str;
  /// is this seed lossless
  bool   lossless;

  /** @brief comparison operator (sort on selectivity first, then on sensitivity for equal selectivity)
   */
  bool operator<(seedproperties & e) {
    return (this->sel < e.sel) || ((this->sel == e.sel) && (!(this->lossless) && e.lossless)) || ((this->sel == e.sel) && (this->lossless == e.lossless) && (this->sens < e.sens)) ;
  }

  /** @brief comparison operator (sort on selectivity first, then on sensitivity for equal selectivity)
   */

  friend std::ostream& operator<<(std::ostream& os, seedproperties& e);

};

/**
 * @brief output a seedproperties object
 * @param os is the outputstream
 * @param e is the seedproperties to output
 * @return the outputsteam
 * @see seedproperties
 */
std::ostream& operator<<(std::ostream& os, seedproperties& e){
  os.setf(ios_base::fixed,ios_base::floatfield);
  os.precision(12);
  os <<  e.str << "\t" << e.sel <<  "\t" ;
  os.precision(6);
  if (gv_lossless_flag && e.lossless)
    os << "[lossless]";
  else
    os << e.sens;
  os <<  "\t" << e.dist ;
  return os;
}


/**
 * @brief compute the euclidian distance of (sens,sel) to (1.0,1.0)
 */

double dist(double sel,double sens) {
  return sqrt((1.0 - sel)*(1.0 - sel) + (1.0 - sens)*(1.0 - sens));
}


/**
 * @brief display a list of seeds properties
 */

int display (list<seedproperties> & l) {
  cerr << "#" << "\t" << "1th column : seed motif"
              << "\t" << "2nd column : selectivity"
              << "\t" << "3rd column : sensitivity"
              << "\t" << "4th column : distance to (1,1)" ;

  l.sort();
  for(list<seedproperties>::iterator i = l.begin(); i != l.end() ; i++) {
    cout << (*i) << endl;
  }
  return 0;
}


/**
 * @brief Compute the convex hull and the convext hull 
 * @brief area for the selectivity/sensitivity of a set of seeds
 * @brief [this definition is not suitable and must NOT be used]
 *
 * @param l is a (possibly unsorted) list of seedproperties
 * @return the area under the convex hull
 */

double areaConvex(list<seedproperties> & l) {


  /* (1) push border seedproperties and sort */
  l.sort();
  l.push_front(seedproperties(0.0,1.0,1.0,string("_")));
  l.push_back( seedproperties(1.0,0.0,1.0,string("_")));

  /* (2) compute convex hull */
  list<seedproperties>::iterator i = l.begin();
  vector<seedproperties>         c = vector<seedproperties>();
  vector<double>          s = vector<double>();
  c.push_back(*(i));
  s.push_back(+0.0);
  i++;

  while(i != l.end()) {
    /* some elments can be negative */
    if ((i->sel >= 0) && (i->sens >= 0)) {
    eval:
      /* increased x-sel */
      if (i->sel > c.back().sel){
        double slope = (i->sens - c.back().sens)/(i->sel - c.back().sel);

        if (slope <= s.back()){
          c.push_back(*i);
          s.push_back(slope);
        }else{
          if (s.size() > 1){
            c.pop_back();
            s.pop_back();
            goto eval;
          }
        }

      }else{

        /* same x-sel but increased sens */
        if ( i->sens > c.back().sens){
          c.pop_back();
          s.pop_back();
          double slope = (i->sens - c.back().sens)/(i->sel - c.back().sel);

          if (slope <= s.back()){
            c.push_back(*i);
            s.push_back(slope);
          }else{
            if (s.size() > 1){
              c.pop_back();
              s.pop_back();
              goto eval;
            }
          }

        }
      }
    }
    i++;
  }

  /* (3) compute covered area */
  double area = 0.0;
  vector<seedproperties>::iterator j = c.begin();
  vector<double>::iterator  k = s.begin();
  for( vector<seedproperties>::iterator i = c.begin() ; i !=  c.end() ; i++ ){
    area += 0.5 * (i->sel - j->sel) * (i->sens + j->sens) ;
    j = i;
    //cout << (*i) << endl;
  }
  cerr << "convex hull area : " << area << endl;
  return area;
}




/**
 * @brief Select seedproperties that are on the "pareto set"
 * @brief usefull to keep best seeds only
 *
 * @param l is a (possibly unsorted) list of seedproperties
 * @return the area under the "pareto set"
 */

double selectPareto(list<seedproperties> & l) {

  /* (1) push border seedproperties and sort */
  l.sort();

  /* (2) compute border hull */
  list<seedproperties>::iterator i = l.begin();
  list<seedproperties>::iterator j = l.begin();
  i++;

  while(i != l.end()) {

    /* increased sensitivity */
    if ((i->lossless && !(j->lossless)) || ((i->lossless == j->lossless) && (i->sens > j->sens)) /* note that i->sel >= j->sel */) {
      if (j != l.begin()) {
        list<seedproperties>::iterator k = j;
        j--;
        l.erase(k);
      } else {	
        l.erase(j);
        j = i;
        i++;
      }
    } else {
      /* increased selectivity */
      if (i->sel > j->sel) {
        j = i;
        i++;
      } else {
        /* equal selectivity but decreased sensitivity */
        list<seedproperties>::iterator k = i;
        i++;
        l.erase(k);
      }
    }
  } /* while */



  /* (3) compute covered area */
  double area = 0.0;
  j = l.begin();
  for( i = l.begin() ; i !=  l.end() ; i++ ){
    area += (i->sel - j->sel) * (i->sens) ;
    j = i;
    //cerr << (*i) << endl;
  }
  return area;
}


/**
 * @brief Compute the "pareto set" and "pareto set" area 
 * @brief for the selectivity/sensitivity of a set of seeds
 * @brief [this definition is more suitable]
 *
 * @param l is a (possibly unsorted) list of seedproperties
 * @return the area under the "pareto set"
 */

double list_and_areaPareto(list<seedproperties> & l) {

  /* (1) select pareto set and compute area */
  double area = selectPareto(l);  
  cerr << "pareto area : " << area << endl;
  /* (2) display seeds */
  for( list<seedproperties>::iterator i = l.begin() ; i !=  l.end() ; i++ ){    
    cout << (*i) << endl;
  }
  return area;
}


/**
 * @brief this method check if a given sel/sens is out of the pareto curve and thus does improve the overall selectivity/sensitivity
 * @param l is the list of seedproperties where e must be inserted
 * @param e is the seedproperties element that has to be inserted
 * @return a positive value if the e improve the pareto and has thus be inserted
 */

double insert_Pareto(list<seedproperties> & l, seedproperties & e) {
  /* seach the position where to insert */
  list<seedproperties>::iterator i = l.begin();
  list<seedproperties>::iterator j = l.begin();
  while(i != l.end() && i->sel + 1e-12 < e.sel) {
    j = i;
    i++;
  }

  /* insert (front/back/between) */
  if ( i ==  l.end()) {
    l.push_back(e);
    return (e.sens - j->sens);
  } else if ( i == l.begin() ) {
    l.push_front(e);
    return (e.sens - i->sens);
  } else {
    /*  e.sel <= i->sel */
    /* same sel */
    if (i->sel == e.sel) {
      bool lossless = e.lossless && (!(i->lossless));
      double dist       = e.sens - i->sens;
      if (lossless || (dist > 0)) {
        /* replace i by e */
        l.insert(i,e);
        l.erase(i);
        return dist;
      } else {
        return dist;
      }
    } else {
      bool lossless = e.lossless && (!(i->lossless));
      double dist = e.sens - i->sens;
      /*  j->sel < e.sel < i->sel */
      if (lossless || (dist > 0)) {
        /* add e before i (so between j and i)*/
        l.insert(i,e);
        return dist;
      } else {
        return dist;
      }
    }
  }
}



/**
 * @brief read a list of initial seeds and merge them in the pareto set
 * @param l gives the list to complete the pareto part
 * @param filename gives the input file name
 * @return error code (always 0, even if the file is unreadable)
 */

int inputPareto(list<seedproperties> & l, char * filename) {
  ifstream in;
  in.open(filename, ifstream::in);
  while(in.good() && !in.eof()) {
    string shape;
    string sens_or_lossless;
    double sel = 0.0;
    double sens = 0.0;
    double dist = 0.0;
    in >> shape >> sel >> sens_or_lossless >> dist ;
    if (!(sens_or_lossless.compare("[lossless]"))) {
      l.push_back(seedproperties(sel,1.0,dist,shape,true));
    } else {
      stringstream is(stringstream::in | stringstream::out);
      is << sens_or_lossless;
      is >> sens;
      l.push_back(seedproperties(sel,sens,dist,shape));
    }
  }
  in.close();
  selectPareto(l);
  return 0;
}


/**
 * @brief write the pareto set to a file
 * @param l gives the list to write
 * @param filename gives the output file name
 * @return error code (always 0, but quit if the file in unwritable)
 */

int outputPareto(list<seedproperties> & l, char * filename) {
  // select pareto
  selectPareto(l);
  // write pareto
  ofstream out;
  out.open(filename, ifstream::out);
  if (out.good()) {
    for( list<seedproperties>::iterator i = l.begin() ; i !=  l.end() ; i++ ){
      out << (*i) << endl;
    }
  } else {
    ERROR("outputPareto"," unable to write into \"" << filename << "\"");
  }
  out.close();
  return 0;
}

// @}




/**
 * @brief this method is needed when |A| or |B| size has changed
 */

int build_default_subsetseed_matching_matrix() {
  /* build matrix */
  gv_subsetseed_matching_matrix.clear();
  for (int a = 0 ; a < gv_align_alphabet_size ; a++) {
    vector<int> * v = new vector<int>(gv_seed_alphabet_size,0);
    gv_subsetseed_matching_matrix.push_back(*v);
  }

  /* fill it in */
  for(int b = 0 ; b < gv_seed_alphabet_size ; b++ ) {
    /* sharp */
    gv_subsetseed_matching_matrix[gv_align_alphabet_size-1][b] = 1;
    for(int a = b; a < gv_align_alphabet_size ; a++ )
      gv_subsetseed_matching_matrix[a][b] = 1;
  }

  /* set the flag to signal the '#' element */
  CHECKMATCHINGMATRIX(gv_subsetseed_matching_matrix);
  return 0;
}




/**
 * @brief this method is needed when |A| or |B| size has changed
 */

int build_default_vectorizedsubsetseed_scoring_matrix() {

  /* build matrix */
  gv_vectorizedsubsetseed_scoring_matrix.clear();
  for (int a = 0 ; a < gv_align_alphabet_size ; a++) {
    vector<int> * v = new vector<int>(gv_seed_alphabet_size,-1);
    gv_vectorizedsubsetseed_scoring_matrix.push_back(*v);
  }

  /* fill it in */
  for (int a = 0 ; a < gv_align_alphabet_size ; a++) {
    for(int b = 0 ; b < gv_seed_alphabet_size ; b++ ) {
      if (gv_subsetseed_matching_matrix[a][b])
        gv_vectorizedsubsetseed_scoring_matrix[a][b] = 1;
    }
  }
  return 0;
}



/**
 * @brief this method is needed when |A| size has changed
 */

int build_default_probabilities()  {
  gv_bsens          = vector <double>(gv_align_alphabet_size,0);
  gv_bsens_k        = 0;
  gv_bsel           = vector <double>(gv_align_alphabet_size,0);
  gv_bsel_k         = 0;

  double remaining  = 1.0;

  /* bsel : 0.5,0.25,0.125,0.125,
   * bsens : 0.125,0.125,0.25,0.5
   */

  for(int i = 0 ; i < gv_align_alphabet_size; i++) {
    if ( i < (gv_align_alphabet_size - 1) ) {
      remaining *= .5;
    }

    gv_bsel[i] = remaining;
    gv_bsens[gv_align_alphabet_size-i-1] = remaining;
  }

  return 0;
}

/**
 * @brief computer the weight (gv_bsel_weight) according to background probabilities (gv_bsel)
 * @see gv_bsel_weight
 * @see gv_bsel
 */

void computeWeight() {
  gv_bsel_weight  = vector<double>(gv_seed_alphabet_size);
  gv_bsel_minprob =  1e32;
  gv_bsel_maxprob = -1e32;

  /* compute background probs */
  for(int b = 0; b < gv_seed_alphabet_size ; b++) {
    double pr = 0.0;
    for(int a = 0; a < gv_align_alphabet_size ; a++) {
      if (gv_subsetseed_matching_matrix[a][b]) {
        for (int i = 0; i < power(gv_align_alphabet_size,gv_bsel_k) ; i++)
          pr += gv_bsel[a+i*gv_align_alphabet_size];
      }
    }
    gv_bsel_weight[b] = pr;
    gv_bsel_minprob   = MIN(pr,gv_bsel_minprob);
    gv_bsel_maxprob   = MAX(pr,gv_bsel_maxprob);

  }

  /* take the log_#(b) */
  for(int b = 0; b < gv_seed_alphabet_size ; b++) {
    gv_bsel_weight[b] = log(gv_bsel_weight[b])/log(gv_bsel_minprob) ;
  }

}


/**
 * @brief l (list of seedproperties) defined global to be saved by the "terminate handler" 
 * @see termin_handler
 */

list<seedproperties> l ;


#ifdef SIGNAL

/**
 * @brief this handler writes output file on premature ending
 */

void termin_handler( int signal ) {
  cerr <<"Signal handler..." << endl;

  /* security input */
  if(gv_nb_input_filenames > 0) {
    for(int i=0;i<gv_nb_input_filenames;i++)
      inputPareto(l,gv_input_filenames[i]);
  }

  /* final output */
  cerr << "# seeds\t sel\t sens\t dist"<<  endl;
  if (!gv_output_filename)
    list_and_areaPareto(l);
  else
    outputPareto(l,gv_output_filename);
  exit(0);
}

#endif






/**
 *@brief iedera main program
 *@param argc is the number of arguments (including program name)
 *@param argv is the value of all the arguments (including program name)
 *@return error code
 */

int main(int argc, char * argv[]) {  

  int nbruns = 0;

  l = list<seedproperties>();
  l.push_front(seedproperties(0.0,1.0,1.0,string("_")));
  l.push_back( seedproperties(1.0,0.0,1.0,string("_")));


  build_default_subsetseed_matching_matrix();
  build_default_vectorizedsubsetseed_scoring_matrix();
  build_default_probabilities();

  SCANARG(argc,argv);

  /* randomize */
  srand(GETMSTIME()*GETPID());

  /* initial pareto set */
  if(gv_nb_input_filenames > 0) {
    for(int i=0;i<gv_nb_input_filenames;i++)
      inputPareto(l,gv_input_filenames[i]);
  }

#ifdef SIGNAL
  signal(SIGTERM, &termin_handler );
  signal(SIGSTOP, &termin_handler );
#endif


  /*
   * O) DISPLAY PARAMETERS
   */
  cerr << "* alignment alphabet size |A| : " << gv_align_alphabet_size << " , " << "seed alphabet size |B| : "  << gv_seed_alphabet_size << endl;
  cerr << "* subset seed matching matrix : "; DISPLAYMATRIX(cerr,gv_subsetseed_matching_matrix); cerr << endl;
  if (gv_vectorized_flag) {
    cerr << "* subset seed scoring matrix : "; DISPLAYMATRIX(cerr,gv_vectorizedsubsetseed_scoring_matrix); cerr << endl;
    cerr << "* subset seed scoring threshold : " << gv_vectorizedsubsetseed_scoring_threshold << endl;
  }

  /* compute the weight for each seed element */
  computeWeight();
  cerr << "* seed alphabet weights " ; DISPLAYTABLE(cerr,gv_bsel_weight); cerr <<endl;

  /* build the foreground/background models */
  automaton a_sel;
  if (gv_bsel_k == 0) {
    a_sel.Automaton_Bernoulli(gv_bsel);
    cerr << "* background Bernoulli model "; DISPLAYTABLE(cerr,gv_bsel); cerr << endl;
  } else {
    a_sel.Automaton_Markov(gv_bsel,gv_bsel_k);
    cerr << "* background Markov model "; DISPLAYTABLE(cerr,gv_bsel);
    cerr << " (order : " << (gv_bsel_k)<< ")" << endl;
  }

  automaton a_sens;
  if (!gv_bsens_automaton) {
    if (gv_bsens_k == 0) {
      a_sens.Automaton_Bernoulli(gv_bsens);
      cerr << "* foreground Bernoulli model "; DISPLAYTABLE(cerr,gv_bsens); cerr << endl;
    } else {
      a_sens.Automaton_Markov(gv_bsens,gv_bsens_k);
      cerr << "* foreground Markov model "; DISPLAYTABLE(cerr,gv_bsens);
      cerr << " (order : " << (gv_bsens_k) << ")" << endl;
    }
  } else {
    cerr << "* foreground Ad hoc model " << endl;
    a_sens = *gv_bsens_automaton;
  }



  /*
   * build the homogeneous automaton if requested and
   * compute the foreground probability of homogeneous paths
   */

  double pr_div_homogeneous = 1.00;
  automaton * a_homogeneous = new automaton();
  if (gv_homogeneous_flag) {
    cerr << "* Homogeneous automaton : {";
    for(int a=0; a < gv_align_alphabet_size ; a++) {
      if  (a>0)  cerr << ",";
      cerr << (gv_homogeneous_scores[a]);
    }
    cerr << "}" << endl;

    a_homogeneous->Automaton_Homogeneous(gv_homogeneous_scores, gv_alignment_length);
    if (gv_minimize_flag) {
      automaton * na = a_homogeneous->Hopcroft();
      delete a_homogeneous;
      a_homogeneous = na ;
    }
    automaton * a_product_homogeneous_sens = a_homogeneous->product(a_sens, PRODUCT_BUTNOT_FINAL_LOOP, true, gv_alignment_length);

    if (gv_lossless_flag) {
      pr_div_homogeneous       = a_product_homogeneous_sens->PrLossless(gv_alignment_length,gv_lossless_costs_vector,gv_lossless_cost_threshold);
    } else {
      pr_div_homogeneous       = a_product_homogeneous_sens->PrFinal(gv_alignment_length);
    }

    delete a_product_homogeneous_sens;
    cerr << "- pr_div_homogenous :" << pr_div_homogeneous << endl;
  }


  /*
   * build the excluded seed automaton if requested
   * compute the foreground probability of excluded paths
   */

  double pr_div_excluded = 0.00;
  automaton * a_excluded = NULL;

  if (gv_xnbseeds) {
    cerr << "* Excluded automaton : ";
    for(int i=0; i < gv_xnbseeds ; i++) {
      if  (i>0)  cerr << ",";
      cerr << (gv_xseeds[i])->str();
    }
    cerr << endl;


    /* a) build the excluded seed automaton */
    for(int i=0 ; i < gv_xnbseeds ; i++){

      automaton * a_se = new automaton();
      SEEDAUTOMATON(a_se,gv_xseeds[i],false);

      if (gv_minimize_flag) {
        automaton * na = a_se->Hopcroft();
        delete a_se;
        a_se = na ;
      }

      if (i > 0) {
        automaton * a_excluded_temp = a_excluded;
        a_excluded =  a_excluded->product(*a_se, PRODUCT_UNION_FINAL_LOOP, false, gv_alignment_length);
        if (gv_minimize_flag) {
          automaton * na = a_excluded->Hopcroft();
          delete a_excluded;
          a_excluded = na ;
        }
        delete a_excluded_temp;
        delete a_se;
      } else {
        a_excluded = a_se;
      }
    }// for each seed

    /* b) compute the foreground probability */
    if (gv_homogeneous_flag) {
      automaton * na = a_excluded->product(*a_homogeneous, PRODUCT_UNION_FINAL_LOOP, false, gv_alignment_length);
      delete a_excluded;
      a_excluded = na;
      if (gv_minimize_flag) {
        automaton * na = a_excluded->Hopcroft();
        delete a_excluded;
        a_excluded = na;
      }
    }


    automaton * a_xpr_sens   = a_excluded->product(a_sens, PRODUCT_UNION_NO_FINAL_LOOP, true, gv_alignment_length);
    if (gv_lossless_flag) {
      pr_div_excluded          = a_xpr_sens->PrLossless(gv_alignment_length,gv_lossless_costs_vector,gv_lossless_cost_threshold);
    } else {
      pr_div_excluded          = a_xpr_sens->PrFinal(gv_alignment_length);
    }
    delete      a_xpr_sens;
    if (pr_div_excluded == 1.0) {
      ERROR("EXCLUDED SEEDS"," the seed(s) provided by the \"-mx\" option reaches a full sensitivity of 1.0");
    }

    cerr << "- pr_div_excluded : " << pr_div_excluded << endl;
  }



  /*
   * I) MAIN STATISTICAL PART
   */

			
  if (gv_motif) {

    /*
     * A) check a command line given set of seeds ( -m 1022,20102)
     */

    automaton * a_spr = NULL;
    double      selp = 0.0;

    for(int i=0 ; i < gv_nbseeds ; i++){

      /* seed automaton */
      automaton * a_s = new automaton();

      if (gv_cycles_flag) 
	gv_seeds[i]->setCyclePos(gv_cycles_pos_list[i],gv_cycles[i]);
      
      cerr << "= seed : " << (*gv_seeds[i]) << endl;
      SEEDAUTOMATON(a_s, gv_seeds[i], gv_seeds[i]->cycled() || gv_multihit_flag);
      cerr << " - size  : " << (a_s->size()) << endl;
      if (gv_minimize_flag) {
        automaton * na = a_s->Hopcroft();
        delete a_s;
        a_s = na ;
      }
      cerr << " - reduced size : " << (a_s->size()) << endl;

      /* compute partial selectivity */
      double sel;
      if (gv_lossless_flag) {
	sel = gv_seeds[i]->selectivityFromWeight();
      } else {
	automaton * a_pr_s_sel      = a_s->product(a_sel, PRODUCT_UNION_FINAL_LOOP, true, gv_alignment_length);
	sel                         = a_pr_s_sel->PrFinal(gv_seeds[i]->span());
	delete a_pr_s_sel;
      }
      if (gv_seeds[i]->cycled()) {
	sel *= (double)(gv_seeds[i]->nbpos())/(gv_seeds[i]->maxpos());
      }
      selp += sel;


      /* cycle */
      if (gv_seeds[i]->cycled()) {
	automaton * na = a_s;	
	automaton * a_cycle = new automaton();
	a_cycle->Automaton_Cycle(gv_seeds[i]->maxpos(),gv_seeds[i]->pos(),gv_seeds[i]->nbpos());
	a_s = (a_s)->product(*a_cycle, gv_multihit_flag?PRODUCT_INTERSECTION_NO_FINAL_LOOP:PRODUCT_INTERSECTION_FINAL_LOOP, false, gv_alignment_length);
	delete a_cycle;
	delete na;
	cerr << " - cycled size : " << (a_s->size()) << endl;
	if (gv_minimize_flag) {
	  automaton * na = a_s->Hopcroft();
	  delete a_s;
	  a_s = na ;
	  cerr << " - reduced cycled size : " << (a_s->size()) << endl;
	}
      }



	
      /* (2) product automaton for sensitivity */

      if (i > 0) {
        automaton * a_spr_temp = a_spr;
        a_spr =  a_spr->product(*a_s, gv_multihit_flag?PRODUCT_UNION_NO_FINAL_LOOP_ADD:PRODUCT_UNION_FINAL_LOOP, false, gv_alignment_length);
        cerr << "= product size : " << (a_spr->size()) << endl;
        if (gv_minimize_flag) {
          automaton * na = a_spr->Hopcroft();
          delete a_spr;
          a_spr = na ;
        }
	if (!gv_multihit_flag)
	  cerr << " - reduced product size : " << (a_spr->size()) << endl;
        delete a_spr_temp;
        delete a_s;
      } else {
        a_spr = a_s;
      }
    }// for each seed

    //mhits
    if (gv_multihit_flag) {
      automaton * a_spr_temp = a_spr;
      a_spr = a_spr->mhit(gv_multihit_nb, gv_alignment_length);
      delete a_spr_temp;
      cerr << "= mhits size : " << (a_spr->size()) << endl;
      if (gv_minimize_flag) {
	automaton * na = a_spr->Hopcroft();
	delete a_spr;
	a_spr = na ;
	cerr << " - mhits reduced size : " << (a_spr->size()) << endl;
      }
    }

    //homogeneous
    if (gv_homogeneous_flag) {
      automaton * na = a_spr->product(*a_homogeneous, PRODUCT_INTERSECTION_NO_FINAL_LOOP, false, gv_alignment_length);
      delete a_spr;
      a_spr = na;
      cerr << "= homogeneous product size : " << (a_spr->size()) << endl;
      if (gv_minimize_flag) {
        automaton * na = a_spr->Hopcroft();
        delete a_spr;
        a_spr = na;
	cerr << " - reduced homogeneous product size : " << (a_spr->size()) << endl;
      }
    }

    //mx
    if (gv_xnbseeds) {
      automaton * na = a_spr->product(*a_excluded, PRODUCT_BUTNOT_NO_FINAL_LOOP, false, gv_alignment_length);
      delete a_spr;
      a_spr = na;
      cerr << "= mx product size : " << (a_spr->size()) << endl;
      if (gv_minimize_flag) {
        automaton * na = a_spr->Hopcroft();
        delete a_spr;
        a_spr = na;
	cerr << " - reduced mx product size : " << (a_spr->size()) << endl;
      }
    }

    /* (3) compute sensitivity */
    double sens;
    int lossless = 0;

    if (gv_lossless_flag) {
      automaton * a_pr_sens     = a_spr->product(a_sens, PRODUCT_UNION_NO_FINAL_LOOP, true, gv_alignment_length);
      cerr << "= final lossless product size : " << (a_pr_sens->size()) << endl;
                       sens     = a_pr_sens->PrLossless(gv_alignment_length,gv_lossless_costs_vector,gv_lossless_cost_threshold);
                       lossless = a_pr_sens->Lossless(gv_alignment_length,gv_lossless_costs_vector,gv_lossless_cost_threshold);
      delete      a_pr_sens;
    } else {
      automaton * a_pr_sens     = a_spr->product(a_sens, PRODUCT_UNION_NO_FINAL_LOOP, true, gv_alignment_length);
      cerr << "= final sensitivity product size : " << (a_pr_sens->size()) << endl;
                       sens     = a_pr_sens->PrFinal(gv_alignment_length);
      delete      a_pr_sens;
    }

    delete a_spr;
    sens /= pr_div_homogeneous - pr_div_excluded;

    /* compute selectivity */    
    double sel = 1.0 - (double)MIN(1.0,selp);


    /* (4) build the memorizing structure */
    double distance = dist(sel,sens);
    ostringstream outs;
    for (int i = 0 ; i < gv_nbseeds ; i++) {
      if (i > 0)
        outs << ",";
      outs << (gv_seeds[i])->str();
    }

    seedproperties e = seedproperties(sel,sens,distance,outs.str(),lossless);

    /* (5) insertion inside pareto set */
    l.push_back(e);

  } else {
    bool   hillclimbing_flag     = false;
    /*
     * B) compute for all sets of seeds ...
     */
    double      sel[MAXSEEDS]     = {0.0};
    automaton * a_s[MAXSEEDS]     = {NULL};
    double hillclimbing_threshold = 1e-6;

    /* init the seeds and set the cycles */
    for (int i=0 ; i < gv_nbseeds ; i++) {
      gv_seeds[i] = new seed();
      if (gv_cycles_flag) 
	gv_seeds[i]->setCyclePos(gv_cycles_pos_list[i],gv_cycles[i]);
    }


    double sens_hillclimbing    = 0.0;
    int    seed_to_hillclimbing = 0;
    int    last_seed_to_hillclimbing = 0;

    while (1) { /* for all seeds */
      
      /* 1) build seed automata */
      for (int i=0 ; i < gv_nbseeds ; i++) {
        if (!a_s[i]){
          (a_s[i]) = new automaton();
          SEEDAUTOMATON(a_s[i], gv_seeds[i], gv_seeds[i]->cycled() || gv_multihit_flag);
          if (gv_minimize_flag) {
            automaton * na = (a_s[i])->Hopcroft();
            delete (a_s[i]);
            (a_s[i]) = na ;
          }
	
	  /* compute partial selectivity */
	  if (gv_lossless_flag) {
	    sel[i] = gv_seeds[i]->selectivityFromWeight();
	  } else {
	    automaton * a_pr_s_sel      = (a_s[i])->product(a_sel, PRODUCT_UNION_FINAL_LOOP, true, gv_alignment_length);
	    sel[i]                       = a_pr_s_sel->PrFinal(gv_seeds[i]->span()) ;
	    delete a_pr_s_sel;
	  }
	  if (gv_seeds[i]->cycled()) {
	    sel[i] *= (double)gv_seeds[i]->nbpos()/gv_seeds[i]->maxpos();
	  }

	  /* cycle */
	  if (gv_seeds[i]->cycled()) {
	    automaton * na = a_s[i];	
            automaton * a_cycle = new automaton();
            a_cycle->Automaton_Cycle(gv_seeds[i]->maxpos(),gv_seeds[i]->pos(),gv_seeds[i]->nbpos());
            a_s[i] = (a_s[i])->product(*a_cycle, gv_multihit_flag?PRODUCT_INTERSECTION_NO_FINAL_LOOP:PRODUCT_INTERSECTION_FINAL_LOOP, false, gv_alignment_length);
            delete a_cycle;
	    delete na;
	    if (gv_minimize_flag) {
	      automaton * na = a_s[i]->Hopcroft();
	      delete a_s[i];
	      a_s[i] = na ;
	    }
	  }
        }
      } // for each seed


      /* 2) compute selectivity */
      double      selp = 0.0;
      for(int i=0 ; i < gv_nbseeds ; i++) {	
	selp += sel[i];
      } // for each seed


      /* 3) compute sensitivity (and lossless property when needed ...) */
      double sens = 0.0;
      int lossless =  0;

      automaton * a_spr = NULL;
      for(int i=0 ; i < gv_nbseeds ; i++) {
	if (i != 0) {
	  automaton * a_spr_temp = a_spr;
	  a_spr =  a_spr->product(*(a_s[i]), gv_multihit_flag?PRODUCT_UNION_NO_FINAL_LOOP_ADD:PRODUCT_UNION_FINAL_LOOP, false, gv_alignment_length);
	  if (gv_minimize_flag) {
	    automaton * na = a_spr->Hopcroft();
	    delete a_spr;
	    a_spr = na ;
	  }
	  if (i > 1)
	    delete a_spr_temp;
	} else {
	  a_spr = (a_s[0]);
	}
      } // for each seed


      //mhits
      automaton * a_spr_mhits_res = a_spr;
      if (gv_multihit_flag) {
	a_spr_mhits_res = a_spr->mhit(gv_multihit_nb, gv_alignment_length);
	if (gv_minimize_flag) {
	  automaton * na = a_spr_mhits_res->Hopcroft();
	  delete a_spr_mhits_res;
	  a_spr_mhits_res = na ;
	}
      }

      // homogeneous
      automaton * a_spr_h_res = a_spr_mhits_res;
      if (gv_homogeneous_flag) {
	a_spr_h_res = a_spr_mhits_res->product(*a_homogeneous, PRODUCT_INTERSECTION_FINAL_LOOP, false, gv_alignment_length);
	if (gv_minimize_flag) {
	  automaton * na = a_spr_h_res->Hopcroft();
	  delete a_spr_h_res;
	  a_spr_h_res = na;
	}
      }

      // mx
      automaton * a_spr_mx_res = a_spr_h_res;
      if (gv_xnbseeds) {
	a_spr_mx_res = a_spr_h_res->product(*a_excluded, PRODUCT_BUTNOT_NO_FINAL_LOOP, false, gv_alignment_length);
	if (gv_minimize_flag) {
	  automaton * na = a_spr_mx_res->Hopcroft();
	  delete a_spr_mx_res;
	  a_spr_mx_res = na;
	}
      }

      if (gv_lossless_flag) {
	automaton * a_pr_sens     = a_spr_mx_res->product(a_sens, PRODUCT_UNION_NO_FINAL_LOOP, true, gv_alignment_length);
                         sens     =    a_pr_sens->PrLossless(gv_alignment_length,gv_lossless_costs_vector,gv_lossless_cost_threshold);
                         lossless =    a_pr_sens->Lossless(gv_alignment_length,gv_lossless_costs_vector,gv_lossless_cost_threshold);
	delete      a_pr_sens;
      } else {
	automaton * a_pr_sens     = a_spr_mx_res->product(a_sens, PRODUCT_UNION_NO_FINAL_LOOP, true, gv_alignment_length);
	sens                      =    a_pr_sens->PrFinal(gv_alignment_length);
	delete      a_pr_sens;
      }

      if (gv_nbseeds > 1) {
	delete a_spr;
      }
      if (gv_multihit_flag) {
	delete a_spr_mhits_res;
      }
      if (gv_homogeneous_flag) {
	delete a_spr_h_res;
      }
      if (gv_xnbseeds) {
	delete a_spr_mx_res;
      }
      sens /= pr_div_homogeneous - pr_div_excluded;

      /* compute selectivity */
      double sel = 1.0 - (double)MIN(1.0,selp);


      /* (4) build the memorizing structure */
      double distance = dist(sel,sens);
      ostringstream outs;
      for (int i = 0 ; i < gv_nbseeds ; i++) {
	if (i > 0)
	  outs << ",";	
	outs << (gv_seeds[i])->str();
      }

      seedproperties e = seedproperties(sel,sens,distance,outs.str(),lossless);

      /* (5) insertion inside pareto set */
      double sensitivity_threshold = insert_Pareto(l,e);


      /*
       * (6)  Sleep if requested
       */
      CHECK_PROCESS(CMOD);

      /* (7)
       *  display progress
       */

#ifdef DEBUG_HILLCLIMBING
      {
#else  
      if (!(nbruns % 1000)) {
#endif
	double area = selectPareto(l);
	time_t    t    = time(NULL);
	struct tm * tm = localtime(&t);
	
	cerr << "#runs:"<< nbruns << ",\tset size:" << l.size() << ",\tarea:" << area ;
	cerr << "\t" << (tm->tm_year + 1900) << "-" << setw(2) << setfill('0') << (tm->tm_mon+1) << "-"  << setw(2) << setfill('0') << (tm->tm_mday) << "\t"  << setw(2) << setfill('0') << (tm->tm_hour) << ":"  << setw(2) << setfill('0') << (tm->tm_min) << ":" <<  setw(2) << setfill('0') << (tm->tm_sec);
	cerr << ",\t" << "last motif checked:" << outs.str() ;
	if  (gv_hillclimbing_flag) {
	  cerr << ",\thillclimbing_threshold:"<< hillclimbing_threshold ;
	}
	cerr << endl;
      }

      nbruns++;
      
      /* (8) hill climbing method */
      if (gv_hillclimbing_flag) {
	
	/* (8.1) set hill_climbing_flag */
	if(!hillclimbing_flag) {
	  double threshold  = sensitivity_threshold + hillclimbing_threshold;
	  if (threshold  >  0) {
#ifdef DEBUG_HILLCLIMBING
	    cerr <<  "\t + : " << sens << endl;
#endif
	    hillclimbing_flag = true;
	    hillclimbing_threshold = MAX(1e-6,hillclimbing_threshold - (threshold/sqrt(1000+nbruns))); /* more stringent */
	  } else {
#ifdef DEBUG_HILLCLIMBING
	    cerr <<  "\t - : " << sens << endl;
#endif
	    hillclimbing_threshold = MIN(0.99999,hillclimbing_threshold - (threshold*gv_hillclimbing_alpha/sqrt(1000+nbruns))); /* little less stringent */
	  }
	}

        /* (8.2) launch hill_climbing procedure */
        if (hillclimbing_flag) {
	
          /* hillclimb phenomenom detected */
          if (sens > sens_hillclimbing) {

#ifdef DEBUG_HILLCLIMBING
            cerr << "- improved : ";
            for (int i = 0 ; i < gv_nbseeds ; i++) {
	      cerr << *(gv_seeds[i]) << ",";
	    }
            cerr << " :" << sens << endl;
#endif

            sens_hillclimbing = sens;
            last_seed_to_hillclimbing = seed_to_hillclimbing;	    
            for (int i = 0 ; i < gv_nbseeds ; i++) {
	      (gv_seeds[i])->resetswap();
	    }
	    gv_swap_choice=rand()%2;
            seed_to_hillclimbing ++;
            seed_to_hillclimbing %= gv_nbseeds;
	  }

	next_swap:
	  
          /* next seed in hillclimbing order */
          while ((gv_seeds[seed_to_hillclimbing])->nextswap() == 0 ) {

            /* delete automaton associated with the modified seed */
            if (a_s[seed_to_hillclimbing]) {
              delete (a_s[seed_to_hillclimbing]);
              a_s[seed_to_hillclimbing] = NULL;
            }
            if (seed_to_hillclimbing == last_seed_to_hillclimbing) {
              seed_to_hillclimbing = 0;
              last_seed_to_hillclimbing = 0;
              goto eoswap;
            }
            seed_to_hillclimbing ++;
            seed_to_hillclimbing %= gv_nbseeds;
          }


          /* delete automaton associated with the modified seed */
          if (a_s[seed_to_hillclimbing]) {
            delete (a_s[seed_to_hillclimbing]);
            a_s[seed_to_hillclimbing] = NULL;
          }

	  /* (8.3) : discard twice seeds in the same set */
	  for (int i = 0; i < gv_nbseeds; i++)
	    for (int j = 0; j < i; j++)
	      if (gv_seeds[i]->equal(gv_seeds[j]))
		goto next_swap;

          continue;

        eoswap:
          /* end of hill climbing */
	  hillclimbing_flag = false;
          sens  = sens_hillclimbing;
	  sens_hillclimbing = 0.0;
          {
            for (int i = 0 ; i < gv_nbseeds ; i++) {
              (gv_seeds[seed_to_hillclimbing])->setswap();
            }
          }
	  
	  gv_swap_choice=rand()%2;
	  
#ifdef DEBUG_HILLCLIMBING
          cerr << "# Final (" << nbruns  << "):" ;
          {
            for (int i = 0 ; i < gv_nbseeds ; i++) {
              cerr << *(gv_seeds[i]) << ",";
            }
          }
          cerr << "\tsel:" << sel << "\tsens:" << sens << endl << endl;
#endif

        }
      } /* if (gv_hill_climbing_flag)*/
      

      next:

      /* (9) : enumerate the next seed(s) */
      if (gv_nbruns > 0){

        double first_seed_weight = 0;

        /* (9.1) : random enumeration */
        for (int i = 0 ; i < gv_nbseeds ; i++ ) {

        next_rand:
          (gv_seeds[i])->random();

          /* delete automaton associated with the modified seed */
          if (a_s[i]) {
            delete (a_s[i]);
            a_s[i] = NULL;
          }
	  
	  
          /* jive selection activated of not  ? */
          if (gv_jive > 0){
            if (i > 0) {
              double weight = (gv_seeds[i])->weight();
              if ( ( weight / first_seed_weight < gv_jive ) || ( first_seed_weight / weight < gv_jive ) )
                goto next_rand;
            } else {
              first_seed_weight = (gv_seeds[i])->weight();
              if (first_seed_weight <= 0)
                goto next_rand;
            }
          }
        }
	
        if (nbruns >= gv_nbruns)
          goto label_EndOfLoop;

      } else { // gv_nbruns > 0

        /* (9.2) : complete enumeration */
        int j = 0;

        while ( (gv_seeds[j])->next() == 0 ) {

	  /* delete automaton associated with the modified seed */
	  if (a_s[j]) {
            delete (a_s[j]);
            a_s[j] = NULL;
          }
	  
          j++;
	  
          if (j == gv_nbseeds) 
	    goto label_EndOfLoop;
        } /* while (seeds->next()) */


        /* delete automaton associated with the modified seed (a_s[j]->next == 1) */
        if (j < gv_nbseeds && a_s[j]) {
          delete (a_s[j]);
          a_s[j] = NULL;
        }

        for(int i = 0 ; i < j ; i++) { // reset the span of previous j-th seeds to min
          delete (gv_seeds[i]);
          gv_seeds[i] = new seed();
	  if (gv_cycles_flag)
	    gv_seeds[i]->setCyclePos(gv_cycles_pos_list[i],gv_cycles[i]);
        }

      }/* Random/Complete enum */


      /* (9.3) : discard twice seeds in the same set */
      for (int i = 0; i < gv_nbseeds; i++)
	for (int j = 0; j < i; j++)
	  if (gv_seeds[i]->equal(gv_seeds[j]))
	    goto next;
      
    }/* while (1) */
  label_EndOfLoop:
    ;
  } /* enumerate all seeds /or/ check one seed */




  /*
   * III) Display pareto set and area
   */

  /* update pareto set according to other processes (helpful on parallel computation) */
  if(gv_nb_input_filenames > 0) {
    for(int i=0;i<gv_nb_input_filenames;i++)
      inputPareto(l,gv_input_filenames[i]);
  }

  /* final output */
  cerr << "# seeds\t sel\t sens\t dist"<<  endl;
  if (!gv_output_filename)
    list_and_areaPareto(l);
  else
    outputPareto(l,gv_output_filename);
  return 0;
}

// @} 
