// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/Thrust.hh"
#include "Rivet/Projections/FinalState.hh"
#include "Rivet/Projections/UnstableParticles.hh"

namespace Rivet {


  /// @brief baryon correlations
  class ARGUS_1988_I266892 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(ARGUS_1988_I266892);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // projections
      const FinalState fs;
      declare(fs                 , "FS"    );
      declare(UnstableParticles(), "UFS"   );
      declare(Thrust(fs)         , "Thrust");
      // histos
      for(unsigned int ix=0;ix<3;++ix) {
	for(unsigned int iy=0;iy<3;++iy) {
	  if(!(ix==0&&iy==2)) {
	    book(_nBB2[ix][iy],4+2*ix,1,1+iy);
	    book(_n[ix][iy],"TMP/n_"+toString(ix+1)+"_"+toString(iy+1));
	  }
	  if(iy==2) continue;
	  book(_nBB [ix][iy],3+2*ix,1,1+iy);
	}
	if(ix==2) continue;
	book(_h_pt[ix],1,1,1+ix);
	book(_c[ix],"TMP/c_"+toString(ix+1));
      }
      book(_h_pt[2],2,1,1);
    }
    
    /// Recursively walk the decay tree to find the stable decay products of @a p
    void findDecayProducts(Particle mother, Particles& final,
			   Particles & protons, Particles & lambda, Particles & xi,
			   Particles & l1520) {
      for(const Particle & p: mother.children()) {
	if     (p.abspid()==3122  ) lambda.push_back(p);
	else if(p.abspid()==3312  ) xi    .push_back(p);
	else if(p.abspid()==101234) l1520 .push_back(p);
	if(!p.children().empty())
	  findDecayProducts(p, final, protons, lambda, xi, l1520);
	else {
	  final.push_back(p);
	  if(p.abspid()==2212) protons.push_back(p);
	}
      }
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      // Find the Upsilons among the unstables
      const UnstableParticles& ufs = apply<UnstableParticles>(event, "UFS");
      Particles upsilons = ufs.particles(Cuts::pid==553);
      if (upsilons.empty()) {
	_c[1]->fill();
	Thrust thrust = apply<Thrust>(event, "Thrust");
	Vector3 axis = thrust.thrustAxis();
	Particles protons = apply<FinalState>(event, "FS").particles(Cuts::abspid==2212);
	for(unsigned int ix=0;ix<protons.size();++ix) {
	  double d1   = protons[ix].momentum().p3().dot(axis);
	  Vector3 pt1 = protons[ix].momentum().p3()-d1*axis;
	  for(unsigned int iy=ix+1;iy<protons.size();++iy) {
	    if(protons[ix].pid()*protons[iy].pid()>0) continue;
	    double d2   = protons[iy].momentum().p3().dot(axis);
	    Vector3 pt2 = protons[iy].momentum().p3()-d2*axis;
	    double phi = acos(pt1.unit().dot(pt2.unit()));
	    if(phi<0.) phi +=M_PI;
	    phi *=180./M_PI;
	    if(d1*d2>0) _h_pt[0]->fill(phi);
	    else        _h_pt[2]->fill(phi);
	  }
	}
	Particles lambda = ufs.particles(Cuts::abspid==3122);
	for(unsigned int ix=0;ix<lambda.size();++ix) {
	  _n[0][1]->fill();
	  for(unsigned int iy=ix+1;iy<lambda.size();++iy) {
	    if(lambda[ix].pid()*lambda[iy].pid()<0) {
	      _nBB [0][1]->fill(0.5);
	      _nBB2[0][1]->fill(0.5);
	    }
	  }
	}
	Particles xi = ufs.particles(Cuts::abspid==3312);
	for(unsigned int ix=0;ix<xi.size();++ix) {
	  _n[1][1]->fill();
	  _n[1][2]->fill();
	  for(unsigned int iy=0;iy<lambda.size();++iy) {
	    if(xi[ix].pid()*lambda[iy].pid()<0) {
	      _nBB [1][1]->fill(0.5);
	      _nBB2[1][1]->fill(0.5);
	      _nBB2[1][2]->fill(0.5);
	    }
	  }
	}
	Particles l1520 = ufs.particles(Cuts::abspid==101234);
	for(unsigned int ix=0;ix<l1520.size();++ix) {
	  _n[2][1]->fill();
	  _n[2][2]->fill();
	  for(unsigned int iy=0;iy<lambda.size();++iy) {
	    if(l1520[ix].pid()*lambda[iy].pid()<0) {
	      _nBB [2][1]->fill(0.5);
	      _nBB2[2][1]->fill(0.5);
	      _nBB2[2][2]->fill(0.5);
	    }
	  }
	}
      }
      else {
	for (const Particle& ups : upsilons) {
	  _c[0]->fill();
	  LorentzTransform boost;
	  if (ups.p3().mod() > 1*MeV)
	    boost = LorentzTransform::mkFrameTransformFromBeta(ups.momentum().betaVec());
	  Particles final, protons, lambda, xi, l1520;
	  findDecayProducts(ups, final, protons, lambda, xi, l1520);
	  vector<FourMomentum> mom; mom.reserve(final.size());
	  for(const Particle & p : final)
	    mom.push_back(boost.transform(p.momentum()));
	  Thrust thrust;
	  thrust.calc(mom);
	  Vector3 axis = thrust.thrustAxis();
	  for(unsigned int ix=0;ix<protons.size();++ix) {
	    double d1   = protons[ix].momentum().p3().dot(axis);
	    Vector3 pt1 = protons[ix].momentum().p3()-d1*axis;
	    for(unsigned int iy=ix+1;iy<protons.size();++iy) {
	      if(protons[ix].pid()*protons[iy].pid()>0) continue;
	      double d2   = protons[iy].momentum().p3().dot(axis);
	      Vector3 pt2 = protons[iy].momentum().p3()-d2*axis;
	      double phi = acos(pt1.unit().dot(pt2.unit()));
	      if(phi<0.) phi +=M_PI;
	      phi *=180./M_PI;
	      if(d1*d2>0) _h_pt[1]->fill(phi);
	    }
	  }
	  for(unsigned int ix=0;ix<lambda.size();++ix) {
	    _n[0][0]->fill();
	    for(unsigned int iy=ix+1;iy<lambda.size();++iy) {
	      if(lambda[ix].pid()*lambda[iy].pid()<0) {
		_nBB [0][0]->fill(0.5);
		_nBB2[0][0]->fill(0.5);
	      }
	    }
	  }
	  for(unsigned int ix=0;ix<xi.size();++ix) {
	    _n[1][0]->fill();
	    _n[1][2]->fill();
	    for(unsigned int iy=0;iy<lambda.size();++iy) {
	      if(xi[ix].pid()*lambda[iy].pid()<0) {
		_nBB [1][0]->fill(0.5);
		_nBB2[1][0]->fill(0.5);
		_nBB2[1][2]->fill(0.5);
	      }
	    }
	  }
	  for(unsigned int ix=0;ix<l1520.size();++ix) {
	    _n[2][0]->fill();
	    _n[2][2]->fill();
	    for(unsigned int iy=0;iy<lambda.size();++iy) {
	      if(l1520[ix].pid()*lambda[iy].pid()<0) {
		_nBB [2][0]->fill(0.5);
		_nBB2[2][0]->fill(0.5);
		_nBB2[2][2]->fill(0.5);
	      }
	    }
	  }
	}
      }
      //    n lam lambar
      // BEGIN YODA_SCATTER2D_V2 /REF/ARGUS_1988_I266892/d03-x01-y01
      // BEGIN YODA_SCATTER2D_V2 /REF/ARGUS_1988_I266892/d03-x01-y02
      //  r lam lambar  
      // BEGIN YODA_SCATTER2D_V2 /REF/ARGUS_1988_I266892/d04-x01-y01
      // BEGIN YODA_SCATTER2D_V2 /REF/ARGUS_1988_I266892/d04-x01-y02
      // xi- lambar
      // BEGIN YODA_SCATTER2D_V2 /REF/ARGUS_1988_I266892/d05-x01-y01
      // BEGIN YODA_SCATTER2D_V2 /REF/ARGUS_1988_I266892/d05-x01-y02
      //    xi - lamba/xi-
      // BEGIN YODA_SCATTER2D_V2 /REF/ARGUS_1988_I266892/d06-x01-y01
      // BEGIN YODA_SCATTER2D_V2 /REF/ARGUS_1988_I266892/d06-x01-y02
      // BEGIN YODA_SCATTER2D_V2 /REF/ARGUS_1988_I266892/d06-x01-y03
      // 1520 lam
      // BEGIN YODA_SCATTER2D_V2 /REF/ARGUS_1988_I266892/d07-x01-y01
      // BEGIN YODA_SCATTER2D_V2 /REF/ARGUS_1988_I266892/d07-x01-y02
      // 1520lam/1520
      // BEGIN YODA_SCATTER2D_V2 /REF/ARGUS_1988_I266892/d08-x01-y01
      // BEGIN YODA_SCATTER2D_V2 /REF/ARGUS_1988_I266892/d08-x01-y02
      // BEGIN YODA_SCATTER2D_V2 /REF/ARGUS_1988_I266892/d08-x01-y03
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      for(unsigned int ix=0;ix<3;++ix) {
	normalize(_h_pt[ix],1.,false);
      } 
      for(unsigned int iy=0;iy<3;++iy) {
	for(unsigned int ix=0;ix<3;++ix) {
	  if(ix==0 && iy!=2) scale(_nBB2[ix][iy], 2./ *_n[ix][iy]);
	  else if(ix!=0)     scale(_nBB2[ix][iy], 1./ *_n[ix][iy]);
	  if(iy==2) continue;
	  scale(_nBB [ix][iy], 1./ *_c[iy]);
	}
      }
    }

    /// @}


    /// @name Histograms
    /// @{
    Histo1DPtr _h_pt[3];
    CounterPtr _n[3][3],_c[2];
    Histo1DPtr _nBB [3][2],_nBB2[3][3];
    /// @}


  };


  RIVET_DECLARE_PLUGIN(ARGUS_1988_I266892);

}
