/* $Id: w3mimgdisplay.c,v 1.11 2002/12/25 16:14:45 ukai Exp $ */
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "fb.h"
#include "fb_img.h"

static char *background = NULL;
static int offset_x = 0, offset_y = 0;
static int defined_bg = 0, defined_x = 0, defined_y = 0, defined_test = 0;
static int defined_debug = 0;
static char *defined_size = NULL;

#define MAX_IMAGE 1000
typedef struct {
  FB_ANIMATION *pixmap;
  int width;
  int height;
} Image;
static Image *imageBuf = NULL;
static int maxImage = 0, maxAnim = 100, clearMargin = 0;

static void GetOption(int argc, char **argv);
static void DrawImage(char *buf, int redraw);
void ClearImage(char *buf);
static void TermImage(void);


#ifdef W3MIMGDISPLAY_SETUID
static int
check_tty_console(char *tty)
{
    if (tty == NULL || *tty == '\0')
	return 0;
    if (strncmp(tty, "/dev/", 5) == 0)
	tty += 5;
    if (strncmp(tty, "tty", 3) == 0 && isdigit(*(tty + 3)))
	return 1;
    if (strncmp(tty, "vc/", 3) == 0 && isdigit(*(tty + 3)))
	return 1;
    return 0;
}
#else
#define check_tty_console(tty)	1
#endif

int
main(int argc, char **argv)
{
    int len;
    char buf[1024 + 128];
#ifdef W3MIMGDISPLAY_SETUID
    uid_t runner_uid = getuid();
    uid_t owner_uid = geteuid();

    /* swap real and effective */
    setreuid(owner_uid, runner_uid);
#endif
    GetOption(argc, argv);
    if (!defined_debug)
	freopen("/dev/null", "w", stderr);

#ifdef W3MIMGDISPLAY_SETUID
    /* 
     * back real and effective
     * run w3mimg_open() in setuid privileges
     */
    setreuid(runner_uid, owner_uid);

    /* make sure drop privileges now */
    setreuid(runner_uid, runner_uid);
#endif

    if(fb_open())
      return 1;

    if (defined_test) {
	printf("%d %d\n", fb_width() - offset_x, fb_height() - offset_y);
	exit(0);
    }

    if (defined_size) {
      int w, h;
      if (get_image_size(defined_size, &w, &h) == 0)
	printf("%d %d\n", w, h);
      exit(0);
    }
    
    if(background != NULL){
      int r, g, b;
      if(sscanf(background, "#%02x%02x%02x", &r, &g, &b) == 3){
	fb_image_set_bg(r, g, b);
      }
    }

    while (fgets(buf, sizeof(buf), stdin) != NULL) {
	if (!(isdigit(buf[0]) && buf[1] == ';')) {
	    fputc('\n', stdout);
	    fflush(stdout);
	    continue;
	}
	len = strlen(buf);
	if (buf[len - 1] == '\n') {
	    buf[--len] = '\0';
	    if (buf[len - 1] == '\r')
		buf[--len] = '\0';
	}
	/*
	 * w3mimg protocol
	 *  0  1  2 ....
	 * +--+--+--+--+ ...... +--+--+
	 * |op|; |args             |\n|
	 * +--+--+--+--+ .......+--+--+
	 *
	 * args is separeted by ';'
	 * op	args
	 *  0;  params		draw image
	 *  1;  params		redraw image
	 *  2;  -none-		terminate drawing
	 *  3;  -none-		sync drawing
	 *  4;  -none-		nop, sync communication
	 *			response '\n'
	 *  5;  path		get size of image,
	 *			response "<width> <height>\n"
	 *  6;  params(6)	clear image
	 *
	 * params
	 *	<n>;<x>;<y>;<w>;<h>;<sx>;<sy>;<sw>;<sh>;<path>
	 * params(6)
	 *	<x>;<y>;<w>;<h>
	 *   
	 */
	switch (buf[0]) {
	case '0':
	    DrawImage(&buf[2], 0);
	    break;
	case '1':
	    DrawImage(&buf[2], 1);
	    break;
	case '2':
	    TermImage();
	    break;
	case '3':
	    break;
	case '4':
	    fputs("\n", stdout);
	    fflush(stdout);
	    break;
	case '5':
	  {
	    int w, h;
	    if (get_image_size(&buf[2], &w, &h) == 0) {
	      fprintf(stdout, "%d %d\n", w, h);
	      fflush(stdout);
	    } else {
	      fprintf(stdout, "\n");
	      fflush(stdout);
	    }
	  }
	  break;
	case '6':
	    ClearImage(&buf[2]);
	    break;
	}
    }
    TermImage();
    fb_close();
    exit(0);
}

static void
GetOption(int argc, char **argv)
{
    int i;

    for (i = 1; i < argc; i++) {
	if (!strcmp("-bg", argv[i])) {
	    if (++i >= argc)
		exit(1);
	    background = argv[i];
	    defined_bg = 1;
	}
	else if (!strcmp("-x", argv[i])) {
	    if (++i >= argc)
		exit(1);
	    offset_x = atoi(argv[i]);
	    defined_x = 1;
	}
	else if (!strcmp("-y", argv[i])) {
	    if (++i >= argc)
		exit(1);
	    offset_y = atoi(argv[i]);
	    defined_y = 1;
	}
	else if (!strcmp("-test", argv[i])) {
	    defined_test = 1;
	}
	else if (!strcmp("-size", argv[i])) {
	    if (++i >= argc)
		exit(1);
	    defined_size = argv[i];
	}
	else if (!strcmp("-anim", argv[i])) {
	    if (++i >= argc)
		exit(1);
	    maxAnim = atoi(argv[i]);
	}
	else if (!strcmp("-margin", argv[i])) {
	    if (++i >= argc)
		exit(1);
	    clearMargin = atoi(argv[i]);
	    if (clearMargin < 0)
		clearMargin = 0;
	}
	else if (!strcmp("-debug", argv[i])) {
	    defined_debug = 1;
	}
	else {
	    exit(1);
	}
    }
}

void
DrawImage(char *buf, int redraw)
{
    FB_ANIMATION *animation;
    char *p = buf;
    int n = 0, x = 0, y = 0, w = 0, h = 0, sx = 0, sy = 0, sw = 0, sh = 0;

    if (!p)
	return;
    for (; isdigit(*p); p++)
	n = 10 * n + (*p - '0');
    if (*(p++) != ';')
	return;
    for (; isdigit(*p); p++)
	x = 10 * x + (*p - '0');
    if (*(p++) != ';')
	return;
    for (; isdigit(*p); p++)
	y = 10 * y + (*p - '0');
    if (*(p++) != ';')
	return;
    for (; isdigit(*p); p++)
	w = 10 * w + (*p - '0');
    if (*(p++) != ';')
	return;
    for (; isdigit(*p); p++)
	h = 10 * h + (*p - '0');
    if (*(p++) != ';')
	return;
    for (; isdigit(*p); p++)
	sx = 10 * sx + (*p - '0');
    if (*(p++) != ';')
	return;
    for (; isdigit(*p); p++)
	sy = 10 * sy + (*p - '0');
    if (*(p++) != ';')
	return;
    for (; isdigit(*p); p++)
	sw = 10 * sw + (*p - '0');
    if (*(p++) != ';')
	return;
    for (; isdigit(*p); p++)
	sh = 10 * sh + (*p - '0');
    if (*(p++) != ';')
	return;

    n--;
    if (n < 0 || n >= MAX_IMAGE)
	return;
    if (redraw) {
	if (n >= maxImage || !imageBuf[n].pixmap)
	    return;
	goto draw_image;
    }

    if (n >= maxImage) {
	int i = maxImage;
	maxImage = i ? (i * 2) : 2;
	if (maxImage > MAX_IMAGE)
	    maxImage = MAX_IMAGE;
	else if (n >= maxImage)
	    maxImage = n + 1;
	imageBuf = (Image *) realloc((void *)imageBuf,
				     sizeof(Image) * maxImage);
	for (; i < maxImage; i++)
	    imageBuf[i].pixmap = NULL;
    }
    if (imageBuf[n].pixmap) {
        fb_animation_free(imageBuf[n].pixmap);
	imageBuf[n].pixmap = NULL;
    }

    imageBuf[n].pixmap = fb_image_load(p, w, h, maxAnim);

  draw_image:
    if (imageBuf[n].pixmap == NULL) {
       return;
    }

    #define WAIT_CNT 4
    animation = imageBuf[n].pixmap;
    fb_image_draw(animation->current,
		  x + offset_x, y + offset_y,
		  sx, sy,
		  (sw ? sw : animation->width),
		  (sh ? sh : animation->height));
    if (animation->wait > WAIT_CNT) {
        if (animation->current->next)
	  animation->current = animation->current->next;
	animation->wait = 0;
    }
    switch (animation->iterations) {
    case 0:
      break;
    case 1:
      animation->last->next = NULL;
      break;
    default:
      animation->iterations--;
    }
    animation->wait += 1;
}

void
TermImage(void)
{
    if (imageBuf) {
	int i;
	for (i = 0; i < maxImage; i++) {
	    	fb_animation_free(imageBuf[i].pixmap);
	}
	free(imageBuf);
	imageBuf = NULL;
    }
    maxImage = 0;
}

void
ClearImage(char *buf)
{
    char *p = buf;
    int x = 0, y = 0, w = 0, h = 0;

    if (!p)
	return;
    for (; isdigit(*p); p++)
	x = 10 * x + (*p - '0');
    if (*(p++) != ';')
	return;
    for (; isdigit(*p); p++)
	y = 10 * y + (*p - '0');
    if (*(p++) != ';')
	return;
    for (; isdigit(*p); p++)
	w = 10 * w + (*p - '0');
    if (*(p++) != ';')
	return;
    for (; isdigit(*p); p++)
	h = 10 * h + (*p - '0');

    fb_image_clear(x + offset_x - clearMargin,
		   y + offset_y - clearMargin,
		   w + clearMargin * 2,
		   h + clearMargin * 2);
}
