/*
 * ping.c - send ICMP Echo request packets
 *
 * Copyright (c) 2000, 2001 Tim J. Robbins. 
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: icmp_ping.c,v 1.6 2003/08/24 04:21:26 reimannj Exp $
 */

/* Modified by Nik Reiman, <nik@aboleo.net> */

#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/timeb.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "portmon.h"
#include "config.h"
#include "icmp.h"

int rawfd = -1;

int icmp_ping(struct sockaddr_in addr)
{
 int fd, ret = 0;
 struct timeb start_time, end_time;

 // make sure we're running as root for this
 if(getuid()) {
  snprintf(err_msg, STRLARGE, "Permission denied for ICMP ping\n");
  log_write(err_msg);
  ret = -1;
  return ret;
 }

 ftime(&start_time);
 do {
  if((fd = open("/dev/null", O_RDWR)) < 0) {
   snprintf(err_msg, STRLARGE, "Can't open /dev/null\n");
   log_write(err_msg);
   ret = -1;
  }
 }
 while(fd <= 2);

 if(close(fd) < 0) {
  snprintf(err_msg, STRLARGE, "Error closing file handle\n");
  log_write(err_msg);
  ret = -1;
 }

 if((rawfd = socket(PF_INET, SOCK_RAW, IPPROTO_ICMP)) < 0) {
  snprintf(err_msg, STRLARGE,
           "Can't open up raw socket (need root permissions?)\n");
  log_write(err_msg);
  ret = -1;
 }
 if(rawfd <= 2) {
  snprintf(err_msg, STRLARGE, "Raw socket is bad\n");
  log_write(err_msg);
  close(rawfd);
  ret = -1;
 }

 send_echo(&addr);

 if(wait_icmp(&addr) != 0) {
  snprintf(err_msg, STRLARGE, "%s did not respond to ping\n",
           inet_ntoa(addr.sin_addr));
  log_write(err_msg);
  close(rawfd);
  ret = -1;
 }

 // The host must be alive
 // printf ("%s is alive\n", inet_ntoa (addr.sin_addr));
 close(rawfd);

 ftime(&end_time);
 if(ret >= 0) {
  ret = (((int)end_time.time * 1000) + (int)end_time.millitm) -
   (((int)start_time.time * 1000) + (int)start_time.millitm);
 }

 return ret;
}

// Send an ICMP_ECHO request to `dest'.
int send_echo(struct sockaddr_in *dest)
{
 unsigned char outpacket[sizeof(struct icmp_hdr)];
 struct icmp_hdr *icmp_pkt;
 int nw;

 icmp_pkt = (struct icmp_hdr *)outpacket;
 icmp_pkt->type = ICMP_ECHOREPLY;
 icmp_pkt->code = 0;
 icmp_pkt->cksum = 0;
 icmp_pkt->un.echo.id = getpid() & 0xFFFF;
 icmp_pkt->un.echo.seq = 0;
 icmp_pkt->cksum = in_cksum((u_short *) outpacket, sizeof(outpacket));

 if((nw =
     sendto(rawfd, outpacket, sizeof(outpacket), 0,
            (struct sockaddr *)dest, sizeof(*dest))) < 0) {
  if(verbose) {
   printf("error in sendto\n");
  }
  snprintf(err_msg, STRLARGE, "Error in sending ping request\n");
  log_write(err_msg);
  return (-1);
 }

 if(nw != sizeof(outpacket)) {
  if(verbose) {
   printf("short write\n");
  }
  snprintf(err_msg, STRLARGE, "Short write!\n");
  log_write(err_msg);
  return (-1);
 }
 return (0);
}

// Convert number of milliseconds to timeval.
void ms_to_tv(int ms, struct timeval *tv)
{
 tv->tv_sec = ms / 1000;
 tv->tv_usec = (ms % 1000) * 1000;
}

// Convert timeval to number of milliseconds.
int tv_to_ms(struct timeval *tv)
{
 return tv->tv_sec * 1000 + tv->tv_usec / 1000;
}

// Wait for an ICMP_ECHO from the host.
int wait_icmp(struct sockaddr_in *src)
{
 struct sockaddr_in from;
 socklen_t fromlen = sizeof(from);
 struct ip_hdr *ip_pkt;
 struct icmp_hdr *icmp_pkt;
 unsigned char inpacket[IP_MAXPACKET];
 fd_set fdset;

 struct timeval totval, before, after;
 int nrecv, toremain;

 // timouts here are in milliseconds, not seconds
 toremain = timeout * 1000;

 // Keep reading packets until we find the response to the packet
 // we sent or time out.
 do {
  FD_ZERO(&fdset);
  FD_SET(rawfd, &fdset);

  gettimeofday(&before, NULL);
  ms_to_tv(toremain, &totval);
  if(select(rawfd + 1, &fdset, NULL, NULL, &totval) < 0) {
   if(verbose) {
    printf("error in select\n");
   }
   snprintf(err_msg, STRLARGE, "Error in selecting socket\n");
   log_write(err_msg);
   close(rawfd);
   return (-1);
  }
  gettimeofday(&after, NULL);
  toremain -= tv_to_ms(&after) - tv_to_ms(&before);
  if(toremain < 0) {
   close(rawfd);
   return 1;
  }
  if(!FD_ISSET(rawfd, &fdset)) {
   close(rawfd);
   return 1;
  }
  if((nrecv =
      recvfrom(rawfd, inpacket, sizeof(inpacket), 0,
               (struct sockaddr *)&from, &fromlen)) < 0) {
   if(verbose) {
    printf("error in recv\n");
   }
   snprintf(err_msg, STRLARGE, "Error in receiving ping from host\n");
   log_write(err_msg);
   close(rawfd);
   return (-1);
  }

  // Verify some assumptions we make about the kernel ICMP
  // socket handling.
  if((size_t) nrecv < sizeof(struct ip_hdr)) {
   if(verbose) {
    printf("recieved packet too short\n");
   }
   snprintf(err_msg, STRLARGE, "Received packet too short\n");
   log_write(err_msg);
   close(rawfd);
   return (-1);
  }

  ip_pkt = (struct ip_hdr *)inpacket;
  if(ip_pkt->version != 4) {
   if(verbose) {
    printf("host sent back not-ipv4 packet\n");
   }
   snprintf(err_msg, STRLARGE, "Host sent back non-IPv4 packet\n");
   log_write(err_msg);
   close(rawfd);
   return (-1);
  }
  if(ip_pkt->protocol != IPPROTO_ICMP) {
   if(verbose) {
    printf("host sent back non-icmp packet\n");
   }
   snprintf(err_msg, STRLARGE, "Host sent back non-ICMP packet\n");
   log_write(err_msg);
   close(rawfd);
   return (-1);
  }
  if(ip_pkt->hdrlen * 4 + sizeof(struct icmp_hdr) > (size_t) nrecv) {
   if(verbose) {
    printf("packet too short\n");
   }
   snprintf(err_msg, STRLARGE, "Packet too short\n");
   log_write(err_msg);
   close(rawfd);
   return (-1);
  }
  icmp_pkt = (struct icmp_hdr *)(inpacket + ip_pkt->hdrlen * 4);
 }
 while((from.sin_addr.s_addr != src->sin_addr.s_addr) ||
       (icmp_pkt->type != ICMP_ECHO) ||
       (icmp_pkt->un.echo.id != (getpid() & 0xFFFF)) ||
       (icmp_pkt->un.echo.seq != 0));

 close(rawfd);
 return 0;
}

// Calculate IP checksum.
int in_cksum(unsigned short *buf, int len)
{
 int sum = 0;

 while(len > 1) {
  sum += *buf++;
  len -= 2;
 }

 if(len == 1) {
  union
  {
   unsigned short s;
   unsigned char c[2];
  }
  u;

  u.c[0] = *(unsigned char *)buf;
  u.c[1] = 0;
  sum += u.s;
 }

 sum = (sum >> 16) + (sum & 0xFFFF);
 sum += (sum >> 16);
 return ~sum;
}
