/* unittests.c
  $Id: unittests.c,v 1.2 2001/09/30 01:46:54 gwiley Exp $
  Glen Wiley <gwiley@ieee.org>

  this file contains functions that can be used to run unit tests on various
  functions that are part of the PDV utilities

  some of the code in here is messy - I chose to focus on getting some
  decent ouput and relying on diff to catch failures between runs
  rather than getting pretty code or output

	TODO: be smart about checksum command (tolerate alternatives)
*/

#if HAVE_CONFIG_H
 #include "config.h"
#endif
#include <errno.h>
#if HAVE_FCNTL_H
 #include <fcntl.h>
#endif
#if HAVE_GETOPT_H
 #include <getopt.h>
#endif
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#if HAVE_STRING_H
 #include <string.h>
#else
 #include <strings.h>
#endif
#include <sys/stat.h>
#include <sys/types.h>
#if HAVE_UNISTD_H
 #include <unistd.h>
#endif
#include "pdv.h"

const char *CKSUMCMD = "cksum";    /* command used to invoke checksum */
const char *NOTAFILE = "NOTAFILE"; /* this filename must not exist */
const char *TMPFILE  = "TMPFILE";  /* temp file used for some tests */
const int  CMDBUFSZ  = 1024;       /* command line, command buffer size */

char       *g_prg    = "unittests";

void utst_findpdv(void);
void utst_readMsgFile(void);

#define CKMATCH   0
#define CKNOMATCH 1
#define CKERROR   2

/*---------------------------------------- main
*/
int
main(int argc, char *argv[])
{

	utst_findpdv();
	utst_readMsgFile();

	return 0;
} /* main */

/*---------------------------------------- utst_findpdv
  perform unit tests on findpdv() from mkpkgutils.c file
  print output from each test - rely on the user to 'diff' the output
  from a known good test to determine pass/fail
*/
void
utst_findpdv(void)
{
	/*-------------------- */

	return;
} /* utst_findpdv */

/*---------------------------------------- cmpcksum
  compares the checksum of the string and the contents of the specified
  file

  return CKMATCH if they match
  return CKNOMATCH if they do not match
  return CKERROR for errors during cksum operation 
*/
int
cmpcksum(const char *str, int slen, const char *fn)
{
	int    retval = CKMATCH;
	int    l;
	int    fd;
	int    nbytes;
	char   *p;
	char   cmd[CMDBUFSZ];
	char   buf[CMDBUFSZ];
	char   fntmp[255];
	double cksf;
	double cksm;
	FILE   *fh;
	
	sprintf(fntmp, "%d.tmp", getpid());

	/* get cksum of file  - we only keep the last line, the first "word"
	   on the last line is the actual checksum */

	sprintf(cmd, "%s %s", CKSUMCMD, fn);
	fh = popen(cmd, "r");
	if(fh == NULL)
		retval = CKERROR;
	else
	{
		while(fgets(buf, CMDBUFSZ, fh) != NULL);
		pclose(fh);
		cksf = strtod(buf, NULL);
		if(cksf == 0L)
			retval = CKERROR;
	}

	/* get cksum of the string */

	sprintf(cmd, "%s > %s", CKSUMCMD, fntmp);
	fh = popen(cmd, "w");
	if(fh == NULL)
	{
		retval = CKERROR;
		printf("readMsgFile: unable to execute command \"%s\"\n", cmd);
	}
	else
	{
		/* write the string to the pipe */

		l  = slen;
		p  = (char *)str;
		fd = fileno(fh);
		while(l>0)
		{
			nbytes = write(fd, p, l);
			l -= nbytes;
			p += nbytes;
		}

		pclose(fh);

		/* get the results of the checksum */

		buf[0] = '\0';

		fh = fopen(fntmp, "r");
		while(fgets(buf, CMDBUFSZ, fh) != NULL);
		fclose(fh);
	
		cksm = strtod(buf, NULL);

		if(cksm == 0L)
			retval = CKERROR;

		if(cksm != cksf)
			retval = CKNOMATCH;
	}

	unlink(fntmp);

	return retval;
} /* cmpcksum */

/*---------------------------------------- utst_readMsgFile
  perform unit tests on findpdv() from mkpkgutils.c file
  print output from each test - rely on the user to 'diff' the output
  from a known good test to determine pass/fail
*/
void
utst_readMsgFile(void)
{
	char   *msg = NULL;
	int    retval;
	int    i;
	FILE   *fh;

	/*-------------------- missing file */

	printf("readMsgFile: missing file, start...\n");
	retval = readMsgFile((char *) NOTAFILE, &msg);
	printf("readMsgFile: end, retval=%d\n", retval);

	if(msg)
		free(msg);
	msg = NULL;

	/*-------------------- good file - how do we behave with a good file */

	printf("readMsgFile: good file, start...\n");

	/* create the test data file */

	fh = fopen(TMPFILE, "w+");
	for(i=0; i<50; i++)
	{
		fprintf(fh, "utst_readMsgFile\n");
	}
	fclose(fh);

	retval = readMsgFile((char *) TMPFILE, &msg);
	printf("readMsgFile: retval=%d (expect 0)\n", retval);

	switch(cmpcksum(msg, strlen(msg), TMPFILE))
	{
		case CKMATCH:
			printf("readMsgFile: file cksum == msg cksum, pass\n");
			break;
		case CKNOMATCH:
			printf("readMsgFile: file cksum != msg cksum, fail\n");
			break;
		case CKERROR:
			printf("readMsgFile: error running cksum\n");
			break;
	}

	printf("readMsgFile: good file, end\n");

	if(msg)
		free(msg);
	msg = NULL;

	unlink(TMPFILE);

	/*-------------------- large file */

	printf("readMsgFile: large file, start...\n");

	/* create the test data file */

	fh = fopen(TMPFILE, "w+");
	for(i=0; i<200; i++)
	{
		fprintf(fh, "utst_readMsgFile\n");
	}
	fclose(fh);

	retval = readMsgFile((char *) TMPFILE, &msg);
	printf("readMsgFile: retval=%d (expect 0)\n", retval);

	switch(cmpcksum(msg, strlen(msg), TMPFILE))
	{
		case CKMATCH:
			printf("readMsgFile: file cksum == msg cksum, pass\n");
			break;
		case CKNOMATCH:
			printf("readMsgFile: file cksum != msg cksum, fail\n");
			break;
		case CKERROR:
			printf("readMsgFile: error running cksum\n");
			break;
	}

	unlink(TMPFILE);

	printf("readMsgFile: large file, end\n");

	return;
} /* utst_readMsgFile */

/* unittests.c */
