// Copyright 2018 The Cockroach Authors.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
// implied. See the License for the specific language governing
// permissions and limitations under the License. See the AUTHORS file
// for names of contributors.

package main

import "strings"

var psycopgBlacklists = []struct {
	versionPrefix  string
	blacklistname  string
	blacklist      blacklist
	ignorelistname string
	ignorelist     blacklist
}{
	{"v2.2", "psycopgBlackList19_1", psycopgBlackList19_1, "psycopgIgnoreList19_1", psycopgIgnoreList19_1},
	{"v19.1", "psycopgBlackList19_1", psycopgBlackList19_1, "psycopgIgnoreList19_1", psycopgIgnoreList19_1},
}

// getPsycopgBlacklistForVersion returns the appropriate psycopg blacklist and
// ignorelist based on the cockroach version. This check only looks to ensure
// that the prefix that matches.
func getPsycopgBlacklistForVersion(version string) (string, blacklist, string, blacklist) {
	for _, info := range psycopgBlacklists {
		if strings.HasPrefix(version, info.versionPrefix) {
			return info.blacklistname, info.blacklist, info.ignorelistname, info.ignorelist
		}
	}
	return "", nil, "", nil
}

// These are lists of known psycopg test errors and failures.
// When the psycopg test suite is run, the results are compared to this list.
// Any passed test that is not on this list is reported as PASS - expected
// Any passed test that is on this list is reported as PASS - unexpected
// Any failed test that is on this list is reported as FAIL - expected
// Any failed test that is not on this list is reported as FAIL - unexpected
// Any test on this list that is not run is reported as FAIL - not run
//
// Please keep these lists alphabetized for easy diffing.
// After a failed run, an updated version of this blacklist should be available
// in the test log.
var psycopgBlackList19_1 = blacklist{
	"psycopg2.tests.test_async.AsyncTests.test_async_after_async":                                                 "5807",
	"psycopg2.tests.test_async.AsyncTests.test_async_callproc":                                                    "5807",
	"psycopg2.tests.test_async.AsyncTests.test_async_connection_error_message":                                    "5807",
	"psycopg2.tests.test_async.AsyncTests.test_async_cursor_gone":                                                 "5807",
	"psycopg2.tests.test_async.AsyncTests.test_async_dont_read_all":                                               "5807",
	"psycopg2.tests.test_async.AsyncTests.test_async_executemany":                                                 "5807",
	"psycopg2.tests.test_async.AsyncTests.test_async_fetch_wrong_cursor":                                          "5807",
	"psycopg2.tests.test_async.AsyncTests.test_async_iter":                                                        "5807",
	"psycopg2.tests.test_async.AsyncTests.test_async_named_cursor":                                                "5807",
	"psycopg2.tests.test_async.AsyncTests.test_async_scroll":                                                      "5807",
	"psycopg2.tests.test_async.AsyncTests.test_async_select":                                                      "5807",
	"psycopg2.tests.test_async.AsyncTests.test_async_subclass":                                                    "5807",
	"psycopg2.tests.test_async.AsyncTests.test_commit_while_async":                                                "5807",
	"psycopg2.tests.test_async.AsyncTests.test_connection_setup":                                                  "5807",
	"psycopg2.tests.test_async.AsyncTests.test_copy_no_hang":                                                      "5807",
	"psycopg2.tests.test_async.AsyncTests.test_copy_while_async":                                                  "5807",
	"psycopg2.tests.test_async.AsyncTests.test_error":                                                             "5807",
	"psycopg2.tests.test_async.AsyncTests.test_error_two_cursors":                                                 "5807",
	"psycopg2.tests.test_async.AsyncTests.test_fetch_after_async":                                                 "5807",
	"psycopg2.tests.test_async.AsyncTests.test_flush_on_write":                                                    "5807",
	"psycopg2.tests.test_async.AsyncTests.test_lobject_while_async":                                               "5807",
	"psycopg2.tests.test_async.AsyncTests.test_notices":                                                           "5807",
	"psycopg2.tests.test_async.AsyncTests.test_notify":                                                            "5807",
	"psycopg2.tests.test_async.AsyncTests.test_reset_while_async":                                                 "5807",
	"psycopg2.tests.test_async.AsyncTests.test_rollback_while_async":                                              "5807",
	"psycopg2.tests.test_async.AsyncTests.test_scroll":                                                            "5807",
	"psycopg2.tests.test_async.AsyncTests.test_set_parameters_while_async":                                        "5807",
	"psycopg2.tests.test_async.AsyncTests.test_sync_poll":                                                         "5807",
	"psycopg2.tests.test_async_keyword.AsyncReplicationTest.test_async_replication":                               "5807",
	"psycopg2.tests.test_async_keyword.AsyncTests.test_async_connection_error_message":                            "5807",
	"psycopg2.tests.test_async_keyword.AsyncTests.test_async_subclass":                                            "5807",
	"psycopg2.tests.test_async_keyword.AsyncTests.test_connection_setup":                                          "5807",
	"psycopg2.tests.test_async_keyword.CancelTests.test_async_cancel":                                             "5807",
	"psycopg2.tests.test_async_keyword.CancelTests.test_async_connection_cancel":                                  "5807",
	"psycopg2.tests.test_cancel.CancelTests.test_async_cancel":                                                    "5807",
	"psycopg2.tests.test_cancel.CancelTests.test_async_connection_cancel":                                         "5807",
	"psycopg2.tests.test_cancel.CancelTests.test_cancel":                                                          "5807",
	"psycopg2.tests.test_cancel.CancelTests.test_empty_cancel":                                                    "5807",
	"psycopg2.tests.test_connection.AutocommitTests.test_set_session_autocommit":                                  "unknown",
	"psycopg2.tests.test_connection.ConnectionTests.test_cleanup_on_badconn_close":                                "unknown",
	"psycopg2.tests.test_connection.ConnectionTests.test_encoding_name":                                           "unknown",
	"psycopg2.tests.test_connection.ConnectionTests.test_notices":                                                 "5807",
	"psycopg2.tests.test_connection.ConnectionTests.test_notices_consistent_order":                                "5807",
	"psycopg2.tests.test_connection.ConnectionTests.test_notices_deque":                                           "5807",
	"psycopg2.tests.test_connection.ConnectionTests.test_notices_limited":                                         "5807",
	"psycopg2.tests.test_connection.ConnectionTests.test_notices_noappend":                                        "5807",
	"psycopg2.tests.test_connection.ConnectionTests.test_reset":                                                   "unknown",
	"psycopg2.tests.test_connection.ConnectionTests.test_tpc_unsupported":                                         "unknown",
	"psycopg2.tests.test_connection.ConnectionTwoPhaseTests.test_cancel_fails_prepared":                           "unknown",
	"psycopg2.tests.test_connection.ConnectionTwoPhaseTests.test_recovered_xids":                                  "unknown",
	"psycopg2.tests.test_connection.ConnectionTwoPhaseTests.test_status_after_recover":                            "unknown",
	"psycopg2.tests.test_connection.ConnectionTwoPhaseTests.test_tpc_commit":                                      "unknown",
	"psycopg2.tests.test_connection.ConnectionTwoPhaseTests.test_tpc_commit_one_phase":                            "unknown",
	"psycopg2.tests.test_connection.ConnectionTwoPhaseTests.test_tpc_commit_recovered":                            "unknown",
	"psycopg2.tests.test_connection.ConnectionTwoPhaseTests.test_tpc_recover_non_dbapi_connection":                "unknown",
	"psycopg2.tests.test_connection.ConnectionTwoPhaseTests.test_tpc_rollback":                                    "unknown",
	"psycopg2.tests.test_connection.ConnectionTwoPhaseTests.test_tpc_rollback_one_phase":                          "unknown",
	"psycopg2.tests.test_connection.ConnectionTwoPhaseTests.test_tpc_rollback_recovered":                          "unknown",
	"psycopg2.tests.test_connection.ConnectionTwoPhaseTests.test_unparsed_roundtrip":                              "unknown",
	"psycopg2.tests.test_connection.ConnectionTwoPhaseTests.test_xid_construction":                                "unknown",
	"psycopg2.tests.test_connection.ConnectionTwoPhaseTests.test_xid_encoding":                                    "unknown",
	"psycopg2.tests.test_connection.ConnectionTwoPhaseTests.test_xid_from_string":                                 "unknown",
	"psycopg2.tests.test_connection.ConnectionTwoPhaseTests.test_xid_roundtrip":                                   "unknown",
	"psycopg2.tests.test_connection.ConnectionTwoPhaseTests.test_xid_to_string":                                   "unknown",
	"psycopg2.tests.test_connection.ConnectionTwoPhaseTests.test_xid_unicode":                                     "unknown",
	"psycopg2.tests.test_connection.ConnectionTwoPhaseTests.test_xid_unicode_unparsed":                            "unknown",
	"psycopg2.tests.test_connection.IsolationLevelsTestCase.test_encoding":                                        "unknown",
	"psycopg2.tests.test_connection.IsolationLevelsTestCase.test_isolation_level":                                 "unknown",
	"psycopg2.tests.test_connection.IsolationLevelsTestCase.test_isolation_level_autocommit":                      "unknown",
	"psycopg2.tests.test_connection.IsolationLevelsTestCase.test_isolation_level_closed":                          "unknown",
	"psycopg2.tests.test_connection.IsolationLevelsTestCase.test_isolation_level_read_committed":                  "unknown",
	"psycopg2.tests.test_connection.IsolationLevelsTestCase.test_isolation_level_serializable":                    "unknown",
	"psycopg2.tests.test_connection.IsolationLevelsTestCase.test_set_isolation_level":                             "unknown",
	"psycopg2.tests.test_connection.IsolationLevelsTestCase.test_set_isolation_level_abort":                       "unknown",
	"psycopg2.tests.test_connection.IsolationLevelsTestCase.test_set_isolation_level_autocommit":                  "unknown",
	"psycopg2.tests.test_connection.IsolationLevelsTestCase.test_set_isolation_level_default":                     "unknown",
	"psycopg2.tests.test_connection.IsolationLevelsTestCase.test_setattr_isolation_level_int":                     "unknown",
	"psycopg2.tests.test_connection.IsolationLevelsTestCase.test_setattr_isolation_level_invalid":                 "unknown",
	"psycopg2.tests.test_connection.IsolationLevelsTestCase.test_setattr_isolation_level_str":                     "unknown",
	"psycopg2.tests.test_connection.TransactionControlTests.test_idempotence_check":                               "unknown",
	"psycopg2.tests.test_connection.TransactionControlTests.test_mixing_session_attribs":                          "unknown",
	"psycopg2.tests.test_connection.TransactionControlTests.test_set_deferrable":                                  "unknown",
	"psycopg2.tests.test_connection.TransactionControlTests.test_set_deferrable_error":                            "unknown",
	"psycopg2.tests.test_connection.TransactionControlTests.test_set_isolation_level":                             "unknown",
	"psycopg2.tests.test_connection.TransactionControlTests.test_set_isolation_level_str":                         "unknown",
	"psycopg2.tests.test_connection.TransactionControlTests.test_setattr_deferrable":                              "unknown",
	"psycopg2.tests.test_copy.CopyTests.test_copy_bytes":                                                          "5807",
	"psycopg2.tests.test_copy.CopyTests.test_copy_expert_file_refcount":                                           "5807",
	"psycopg2.tests.test_copy.CopyTests.test_copy_expert_textiobase":                                              "5807",
	"psycopg2.tests.test_copy.CopyTests.test_copy_from":                                                           "5807",
	"psycopg2.tests.test_copy.CopyTests.test_copy_from_cols":                                                      "5807",
	"psycopg2.tests.test_copy.CopyTests.test_copy_from_cols_err":                                                  "5807",
	"psycopg2.tests.test_copy.CopyTests.test_copy_from_insane_size":                                               "5807",
	"psycopg2.tests.test_copy.CopyTests.test_copy_from_propagate_error":                                           "5807",
	"psycopg2.tests.test_copy.CopyTests.test_copy_from_segfault":                                                  "5807",
	"psycopg2.tests.test_copy.CopyTests.test_copy_no_column_limit":                                                "5807",
	"psycopg2.tests.test_copy.CopyTests.test_copy_rowcount":                                                       "5807",
	"psycopg2.tests.test_copy.CopyTests.test_copy_rowcount_error":                                                 "5807",
	"psycopg2.tests.test_copy.CopyTests.test_copy_text":                                                           "5807",
	"psycopg2.tests.test_copy.CopyTests.test_copy_to":                                                             "5807",
	"psycopg2.tests.test_copy.CopyTests.test_copy_to_propagate_error":                                             "5807",
	"psycopg2.tests.test_copy.CopyTests.test_copy_to_segfault":                                                    "5807",
	"psycopg2.tests.test_cursor.CursorTests.test_callproc_dict":                                                   "17511",
	"psycopg2.tests.test_cursor.CursorTests.test_executemany_propagate_exceptions":                                "5807",
	"psycopg2.tests.test_cursor.CursorTests.test_external_close_async":                                            "unknown",
	"psycopg2.tests.test_cursor.CursorTests.test_external_close_sync":                                             "unknown",
	"psycopg2.tests.test_cursor.CursorTests.test_invalid_name":                                                    "5807",
	"psycopg2.tests.test_cursor.CursorTests.test_iter_named_cursor_default_itersize":                              "unknown",
	"psycopg2.tests.test_cursor.CursorTests.test_iter_named_cursor_efficient":                                     "unknown",
	"psycopg2.tests.test_cursor.CursorTests.test_iter_named_cursor_itersize":                                      "unknown",
	"psycopg2.tests.test_cursor.CursorTests.test_iter_named_cursor_rownumber":                                     "unknown",
	"psycopg2.tests.test_cursor.CursorTests.test_named_cursor_stealing":                                           "unknown",
	"psycopg2.tests.test_cursor.CursorTests.test_named_noop_close":                                                "unknown",
	"psycopg2.tests.test_cursor.CursorTests.test_namedtuple_description":                                          "unknown",
	"psycopg2.tests.test_cursor.CursorTests.test_not_scrollable":                                                  "unknown",
	"psycopg2.tests.test_cursor.CursorTests.test_scroll_named":                                                    "unknown",
	"psycopg2.tests.test_cursor.CursorTests.test_scrollable":                                                      "unknown",
	"psycopg2.tests.test_cursor.CursorTests.test_stolen_named_cursor_close":                                       "unknown",
	"psycopg2.tests.test_cursor.CursorTests.test_withhold":                                                        "unknown",
	"psycopg2.tests.test_cursor.CursorTests.test_withhold_autocommit":                                             "unknown",
	"psycopg2.tests.test_cursor.CursorTests.test_withhold_no_begin":                                               "unknown",
	"psycopg2.tests.test_dates.DatetimeTests.test_adapt_datetime":                                                 "unknown",
	"psycopg2.tests.test_dates.DatetimeTests.test_adapt_infinity_tz":                                              "unknown",
	"psycopg2.tests.test_dates.DatetimeTests.test_adapt_negative_timedelta":                                       "unknown",
	"psycopg2.tests.test_dates.DatetimeTests.test_adapt_timedelta":                                                "unknown",
	"psycopg2.tests.test_dates.DatetimeTests.test_interval_iso_8601_not_supported":                                "32562",
	"psycopg2.tests.test_dates.DatetimeTests.test_time_24":                                                        "unknown",
	"psycopg2.tests.test_dates.DatetimeTests.test_type_roundtrip_timetz":                                          "26097",
	"psycopg2.tests.test_extras_dictcursor.ExtrasDictCursorTests.testDictConnCursorArgs":                          "5807",
	"psycopg2.tests.test_extras_dictcursor.ExtrasDictCursorTests.testDictCursorRealWithNamedCursorFetchAll":       "5807",
	"psycopg2.tests.test_extras_dictcursor.ExtrasDictCursorTests.testDictCursorRealWithNamedCursorFetchMany":      "5807",
	"psycopg2.tests.test_extras_dictcursor.ExtrasDictCursorTests.testDictCursorRealWithNamedCursorFetchManyNoarg": "5807",
	"psycopg2.tests.test_extras_dictcursor.ExtrasDictCursorTests.testDictCursorRealWithNamedCursorFetchOne":       "5807",
	"psycopg2.tests.test_extras_dictcursor.ExtrasDictCursorTests.testDictCursorRealWithNamedCursorIter":           "5807",
	"psycopg2.tests.test_extras_dictcursor.ExtrasDictCursorTests.testDictCursorRealWithNamedCursorIterRowNumber":  "5807",
	"psycopg2.tests.test_extras_dictcursor.ExtrasDictCursorTests.testDictCursorRealWithNamedCursorNotGreedy":      "5807",
	"psycopg2.tests.test_extras_dictcursor.ExtrasDictCursorTests.testDictCursorWithNamedCursorFetchAll":           "5807",
	"psycopg2.tests.test_extras_dictcursor.ExtrasDictCursorTests.testDictCursorWithNamedCursorFetchMany":          "5807",
	"psycopg2.tests.test_extras_dictcursor.ExtrasDictCursorTests.testDictCursorWithNamedCursorFetchManyNoarg":     "5807",
	"psycopg2.tests.test_extras_dictcursor.ExtrasDictCursorTests.testDictCursorWithNamedCursorFetchOne":           "5807",
	"psycopg2.tests.test_extras_dictcursor.ExtrasDictCursorTests.testDictCursorWithNamedCursorIter":               "5807",
	"psycopg2.tests.test_extras_dictcursor.ExtrasDictCursorTests.testDictCursorWithNamedCursorIterRowNumber":      "5807",
	"psycopg2.tests.test_extras_dictcursor.ExtrasDictCursorTests.testDictCursorWithNamedCursorNotGreedy":          "5807",
	"psycopg2.tests.test_extras_dictcursor.ExtrasDictCursorTests.testDictCursorWithPlainCursorFetchAll":           "5807",
	"psycopg2.tests.test_extras_dictcursor.ExtrasDictCursorTests.testDictCursorWithPlainCursorFetchMany":          "5807",
	"psycopg2.tests.test_extras_dictcursor.ExtrasDictCursorTests.testDictCursorWithPlainCursorFetchManyNoarg":     "5807",
	"psycopg2.tests.test_extras_dictcursor.ExtrasDictCursorTests.testDictCursorWithPlainCursorFetchOne":           "5807",
	"psycopg2.tests.test_extras_dictcursor.ExtrasDictCursorTests.testDictCursorWithPlainCursorIter":               "5807",
	"psycopg2.tests.test_extras_dictcursor.ExtrasDictCursorTests.testDictCursorWithPlainCursorIterRowNumber":      "5807",
	"psycopg2.tests.test_extras_dictcursor.ExtrasDictCursorTests.testDictCursorWithPlainCursorRealFetchAll":       "5807",
	"psycopg2.tests.test_extras_dictcursor.ExtrasDictCursorTests.testDictCursorWithPlainCursorRealFetchMany":      "5807",
	"psycopg2.tests.test_extras_dictcursor.ExtrasDictCursorTests.testDictCursorWithPlainCursorRealFetchManyNoarg": "5807",
	"psycopg2.tests.test_extras_dictcursor.ExtrasDictCursorTests.testDictCursorWithPlainCursorRealFetchOne":       "5807",
	"psycopg2.tests.test_extras_dictcursor.ExtrasDictCursorTests.testDictCursorWithPlainCursorRealIter":           "5807",
	"psycopg2.tests.test_extras_dictcursor.ExtrasDictCursorTests.testDictCursorWithPlainCursorRealIterRowNumber":  "5807",
	"psycopg2.tests.test_extras_dictcursor.ExtrasDictCursorTests.testPickleDictRow":                               "5807",
	"psycopg2.tests.test_extras_dictcursor.ExtrasDictCursorTests.testPickleRealDictRow":                           "5807",
	"psycopg2.tests.test_extras_dictcursor.ExtrasDictCursorTests.testUpdateRow":                                   "5807",
	"psycopg2.tests.test_extras_dictcursor.NamedTupleCursorTest.test_bad_col_names":                               "5807",
	"psycopg2.tests.test_extras_dictcursor.NamedTupleCursorTest.test_cursor_args":                                 "5807",
	"psycopg2.tests.test_extras_dictcursor.NamedTupleCursorTest.test_error_message":                               "5807",
	"psycopg2.tests.test_extras_dictcursor.NamedTupleCursorTest.test_executemany":                                 "5807",
	"psycopg2.tests.test_extras_dictcursor.NamedTupleCursorTest.test_fetchall":                                    "5807",
	"psycopg2.tests.test_extras_dictcursor.NamedTupleCursorTest.test_fetchmany":                                   "5807",
	"psycopg2.tests.test_extras_dictcursor.NamedTupleCursorTest.test_fetchmany_noarg":                             "5807",
	"psycopg2.tests.test_extras_dictcursor.NamedTupleCursorTest.test_fetchone":                                    "5807",
	"psycopg2.tests.test_extras_dictcursor.NamedTupleCursorTest.test_iter":                                        "5807",
	"psycopg2.tests.test_extras_dictcursor.NamedTupleCursorTest.test_minimal_generation":                          "5807",
	"psycopg2.tests.test_extras_dictcursor.NamedTupleCursorTest.test_named":                                       "5807",
	"psycopg2.tests.test_extras_dictcursor.NamedTupleCursorTest.test_named_fetchall":                              "5807",
	"psycopg2.tests.test_extras_dictcursor.NamedTupleCursorTest.test_named_fetchmany":                             "5807",
	"psycopg2.tests.test_extras_dictcursor.NamedTupleCursorTest.test_named_fetchone":                              "5807",
	"psycopg2.tests.test_extras_dictcursor.NamedTupleCursorTest.test_named_rownumber":                             "5807",
	"psycopg2.tests.test_extras_dictcursor.NamedTupleCursorTest.test_no_result_no_surprise":                       "5807",
	"psycopg2.tests.test_extras_dictcursor.NamedTupleCursorTest.test_nonascii_name":                               "5807",
	"psycopg2.tests.test_extras_dictcursor.NamedTupleCursorTest.test_not_greedy":                                  "5807",
	"psycopg2.tests.test_extras_dictcursor.NamedTupleCursorTest.test_record_updated":                              "5807",
	"psycopg2.tests.test_green.CallbackErrorTestCase.test_errors_named_cursor":                                    "unknown",
	"psycopg2.tests.test_ipaddress.NetworkingTestCase.test_cidr_adapt":                                            "unknown",
	"psycopg2.tests.test_ipaddress.NetworkingTestCase.test_cidr_array_cast":                                       "unknown",
	"psycopg2.tests.test_ipaddress.NetworkingTestCase.test_cidr_cast":                                             "unknown",
	"psycopg2.tests.test_ipaddress.NetworkingTestCase.test_inet_adapt":                                            "unknown",
	"psycopg2.tests.test_ipaddress.NetworkingTestCase.test_inet_array_cast":                                       "unknown",
	"psycopg2.tests.test_ipaddress.NetworkingTestCase.test_inet_cast":                                             "unknown",
	"psycopg2.tests.test_lobject.LargeObject64Tests.test_seek_tell_truncate_greater_than_2gb":                     "unknown",
	"psycopg2.tests.test_lobject.LargeObjectNot64Tests.test_seek_larger_than_2gb":                                 "unknown",
	"psycopg2.tests.test_lobject.LargeObjectNot64Tests.test_truncate_larger_than_2gb":                             "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTests.test_close":                                                     "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTests.test_close_after_commit":                                        "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTests.test_close_connection_gone":                                     "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTests.test_close_twice":                                               "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTests.test_create":                                                    "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTests.test_create_with_existing_oid":                                  "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTests.test_create_with_oid":                                           "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTests.test_export":                                                    "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTests.test_export_after_close":                                        "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTests.test_export_after_commit":                                       "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTests.test_factory":                                                   "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTests.test_import":                                                    "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTests.test_mode_defaults":                                             "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTests.test_open_existing":                                             "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTests.test_open_for_write":                                            "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTests.test_open_mode_n":                                               "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTests.test_open_non_existent":                                         "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTests.test_read":                                                      "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTests.test_read_after_close":                                          "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTests.test_read_after_commit":                                         "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTests.test_read_after_tpc_commit":                                     "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTests.test_read_after_tpc_prepare":                                    "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTests.test_read_binary":                                               "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTests.test_read_large":                                                "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTests.test_read_text":                                                 "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTests.test_seek_after_close":                                          "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTests.test_seek_after_commit":                                         "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTests.test_seek_tell":                                                 "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTests.test_tell_after_close":                                          "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTests.test_tell_after_commit":                                         "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTests.test_unlink":                                                    "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTests.test_unlink_after_close":                                        "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTests.test_unlink_after_commit":                                       "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTests.test_write":                                                     "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTests.test_write_after_close":                                         "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTests.test_write_after_commit":                                        "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTests.test_write_large":                                               "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTruncateTests.test_truncate":                                          "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTruncateTests.test_truncate_after_close":                              "unknown",
	"psycopg2.tests.test_lobject.LargeObjectTruncateTests.test_truncate_after_commit":                             "unknown",
	"psycopg2.tests.test_module.ExceptionsTestCase.test_9_3_diagnostics":                                          "unknown",
	"psycopg2.tests.test_module.ExceptionsTestCase.test_diagnostics_copy":                                         "5807",
	"psycopg2.tests.test_module.ExceptionsTestCase.test_diagnostics_from_commit":                                  "5807",
	"psycopg2.tests.test_notify.NotifiesTests.test_many_notifies":                                                 "unknown",
	"psycopg2.tests.test_notify.NotifiesTests.test_notifies_received_on_execute":                                  "unknown",
	"psycopg2.tests.test_notify.NotifiesTests.test_notifies_received_on_poll":                                     "unknown",
	"psycopg2.tests.test_notify.NotifiesTests.test_notify_attributes":                                             "unknown",
	"psycopg2.tests.test_notify.NotifiesTests.test_notify_deque":                                                  "unknown",
	"psycopg2.tests.test_notify.NotifiesTests.test_notify_noappend":                                               "unknown",
	"psycopg2.tests.test_notify.NotifiesTests.test_notify_object":                                                 "unknown",
	"psycopg2.tests.test_notify.NotifiesTests.test_notify_payload":                                                "unknown",
	"psycopg2.tests.test_psycopg2_dbapi20.Psycopg2TPCTests.test_commit_in_tpc_fails":                              "unknown",
	"psycopg2.tests.test_psycopg2_dbapi20.Psycopg2TPCTests.test_rollback_in_tpc_fails":                            "unknown",
	"psycopg2.tests.test_psycopg2_dbapi20.Psycopg2TPCTests.test_tpc_begin":                                        "unknown",
	"psycopg2.tests.test_psycopg2_dbapi20.Psycopg2TPCTests.test_tpc_begin_in_tpc_transaction_fails":               "unknown",
	"psycopg2.tests.test_psycopg2_dbapi20.Psycopg2TPCTests.test_tpc_begin_in_transaction_fails":                   "unknown",
	"psycopg2.tests.test_psycopg2_dbapi20.Psycopg2TPCTests.test_tpc_commit_with_prepare":                          "unknown",
	"psycopg2.tests.test_psycopg2_dbapi20.Psycopg2TPCTests.test_tpc_commit_without_prepare":                       "unknown",
	"psycopg2.tests.test_psycopg2_dbapi20.Psycopg2TPCTests.test_tpc_rollback_with_prepare":                        "unknown",
	"psycopg2.tests.test_psycopg2_dbapi20.Psycopg2TPCTests.test_tpc_rollback_without_prepare":                     "unknown",
	"psycopg2.tests.test_psycopg2_dbapi20.Psycopg2TPCTests.test_xid":                                              "unknown",
	"psycopg2.tests.test_quote.QuotingTestCase.test_koi8":                                                         "unknown",
	"psycopg2.tests.test_quote.QuotingTestCase.test_latin1":                                                       "unknown",
	"psycopg2.tests.test_quote.TestStringAdapter.test_adapt_bytes":                                                "unknown",
	"psycopg2.tests.test_replication.AsyncReplicationTest.test_async_replication":                                 "unknown",
	"psycopg2.tests.test_replication.ReplicationTest.test_create_replication_slot":                                "unknown",
	"psycopg2.tests.test_replication.ReplicationTest.test_datestyle":                                              "unknown",
	"psycopg2.tests.test_replication.ReplicationTest.test_keepalive":                                              "unknown",
	"psycopg2.tests.test_replication.ReplicationTest.test_logical_replication_connection":                         "unknown",
	"psycopg2.tests.test_replication.ReplicationTest.test_physical_replication_connection":                        "unknown",
	"psycopg2.tests.test_replication.ReplicationTest.test_start_and_recover_from_error":                           "unknown",
	"psycopg2.tests.test_replication.ReplicationTest.test_start_on_missing_replication_slot":                      "unknown",
	"psycopg2.tests.test_replication.ReplicationTest.test_start_replication_expert_sql":                           "unknown",
	"psycopg2.tests.test_replication.ReplicationTest.test_stop_replication":                                       "unknown",
	"psycopg2.tests.test_sql.SqlFormatTests.test_copy":                                                            "unknown",
	"psycopg2.tests.test_transaction.DeadlockSerializationTests.test_deadlock":                                    "unknown",
	"psycopg2.tests.test_transaction.TransactionTests.test_commit":                                                "5807",
	"psycopg2.tests.test_transaction.TransactionTests.test_failed_commit":                                         "5807",
	"psycopg2.tests.test_transaction.TransactionTests.test_rollback":                                              "5807",
	"psycopg2.tests.test_types_basic.AdaptSubclassTest.test_adapt_subtype_3":                                      "unknown",
	"psycopg2.tests.test_types_basic.TypesBasicTests.testArray":                                                   "unknown",
	"psycopg2.tests.test_types_basic.TypesBasicTests.testArrayOfNulls":                                            "unknown",
	"psycopg2.tests.test_types_basic.TypesBasicTests.testEmptyArray":                                              "unknown",
	"psycopg2.tests.test_types_basic.TypesBasicTests.testEmptyArrayRegression":                                    "unknown",
	"psycopg2.tests.test_types_basic.TypesBasicTests.testGenericArrayNull":                                        "unknown",
	"psycopg2.tests.test_types_basic.TypesBasicTests.testNestedArrays":                                            "unknown",
	"psycopg2.tests.test_types_basic.TypesBasicTests.testNestedEmptyArray":                                        "unknown",
	"psycopg2.tests.test_types_basic.TypesBasicTests.testNetworkArray":                                            "18846",
	"psycopg2.tests.test_types_basic.TypesBasicTests.testTypeRoundtripBytes":                                      "unknown",
	"psycopg2.tests.test_types_basic.TypesBasicTests.testTypeRoundtripBytesArray":                                 "unknown",
	"psycopg2.tests.test_types_extras.AdaptTypeTestCase.test_cast_composite":                                      "27793",
	"psycopg2.tests.test_types_extras.AdaptTypeTestCase.test_cast_nested":                                         "27793",
	"psycopg2.tests.test_types_extras.AdaptTypeTestCase.test_composite_array":                                     "27793",
	"psycopg2.tests.test_types_extras.AdaptTypeTestCase.test_composite_namespace":                                 "26443",
	"psycopg2.tests.test_types_extras.AdaptTypeTestCase.test_empty_string":                                        "27793",
	"psycopg2.tests.test_types_extras.AdaptTypeTestCase.test_from_tables":                                         "unknown",
	"psycopg2.tests.test_types_extras.AdaptTypeTestCase.test_non_dbapi_connection":                                "27793",
	"psycopg2.tests.test_types_extras.AdaptTypeTestCase.test_register_globally":                                   "27793",
	"psycopg2.tests.test_types_extras.AdaptTypeTestCase.test_register_on_connection":                              "27793",
	"psycopg2.tests.test_types_extras.AdaptTypeTestCase.test_register_on_cursor":                                  "27793",
	"psycopg2.tests.test_types_extras.AdaptTypeTestCase.test_subclass":                                            "27793",
	"psycopg2.tests.test_types_extras.AdaptTypeTestCase.test_wrong_schema":                                        "27793",
	"psycopg2.tests.test_types_extras.HstoreTestCase.test_adapt_8":                                                "unknown",
	"psycopg2.tests.test_types_extras.HstoreTestCase.test_array_cast":                                             "unknown",
	"psycopg2.tests.test_types_extras.HstoreTestCase.test_array_cast_oid":                                         "unknown",
	"psycopg2.tests.test_types_extras.HstoreTestCase.test_non_dbapi_connection":                                   "unknown",
	"psycopg2.tests.test_types_extras.HstoreTestCase.test_oid":                                                    "unknown",
	"psycopg2.tests.test_types_extras.HstoreTestCase.test_register_conn":                                          "unknown",
	"psycopg2.tests.test_types_extras.HstoreTestCase.test_register_curs":                                          "unknown",
	"psycopg2.tests.test_types_extras.HstoreTestCase.test_register_globally":                                      "unknown",
	"psycopg2.tests.test_types_extras.HstoreTestCase.test_register_unicode":                                       "unknown",
	"psycopg2.tests.test_types_extras.HstoreTestCase.test_roundtrip":                                              "unknown",
	"psycopg2.tests.test_types_extras.HstoreTestCase.test_roundtrip_array":                                        "unknown",
	"psycopg2.tests.test_types_extras.HstoreTestCase.test_roundtrip_unicode":                                      "unknown",
	"psycopg2.tests.test_types_extras.JsonTestCase.test_customizable_with_module_not_available":                   "unknown",
	"psycopg2.tests.test_types_extras.JsonTestCase.test_default_cast":                                             "unknown",
	"psycopg2.tests.test_types_extras.JsonTestCase.test_loads":                                                    "unknown",
	"psycopg2.tests.test_types_extras.JsonTestCase.test_module_not_available":                                     "unknown",
	"psycopg2.tests.test_types_extras.JsonTestCase.test_no_conn_curs":                                             "unknown",
	"psycopg2.tests.test_types_extras.JsonTestCase.test_null":                                                     "unknown",
	"psycopg2.tests.test_types_extras.JsonTestCase.test_register_default":                                         "unknown",
	"psycopg2.tests.test_types_extras.JsonTestCase.test_register_globally":                                        "unknown",
	"psycopg2.tests.test_types_extras.JsonTestCase.test_register_on_connection":                                   "unknown",
	"psycopg2.tests.test_types_extras.JsonTestCase.test_register_on_cursor":                                       "unknown",
	"psycopg2.tests.test_types_extras.JsonTestCase.test_scs":                                                      "unknown",
	"psycopg2.tests.test_types_extras.JsonbTestCase.test_default_cast":                                            "unknown",
	"psycopg2.tests.test_types_extras.JsonbTestCase.test_loads":                                                   "unknown",
	"psycopg2.tests.test_types_extras.JsonbTestCase.test_null":                                                    "unknown",
	"psycopg2.tests.test_types_extras.JsonbTestCase.test_register_default":                                        "unknown",
	"psycopg2.tests.test_types_extras.RangeCasterTestCase.test_adapt_date_range":                                  "unknown",
	"psycopg2.tests.test_types_extras.RangeCasterTestCase.test_adapt_number_range":                                "unknown",
	"psycopg2.tests.test_types_extras.RangeCasterTestCase.test_adapt_numeric_range":                               "unknown",
	"psycopg2.tests.test_types_extras.RangeCasterTestCase.test_cast_date":                                         "unknown",
	"psycopg2.tests.test_types_extras.RangeCasterTestCase.test_cast_empty":                                        "unknown",
	"psycopg2.tests.test_types_extras.RangeCasterTestCase.test_cast_inf":                                          "unknown",
	"psycopg2.tests.test_types_extras.RangeCasterTestCase.test_cast_null":                                         "unknown",
	"psycopg2.tests.test_types_extras.RangeCasterTestCase.test_cast_numbers":                                      "unknown",
	"psycopg2.tests.test_types_extras.RangeCasterTestCase.test_cast_timestamp":                                    "unknown",
	"psycopg2.tests.test_types_extras.RangeCasterTestCase.test_cast_timestamptz":                                  "unknown",
	"psycopg2.tests.test_types_extras.RangeCasterTestCase.test_range_escaping":                                    "27791",
	"psycopg2.tests.test_types_extras.RangeCasterTestCase.test_register_range_adapter":                            "27791",
	"psycopg2.tests.test_types_extras.RangeCasterTestCase.test_schema_range":                                      "26443",
	"psycopg2.tests.test_with.WithCursorTestCase.test_exception_swallow":                                          "unknown",
	"psycopg2.tests.test_with.WithCursorTestCase.test_named_with_noop":                                            "unknown",
}

var psycopgIgnoreList19_1 = blacklist{
	"psycopg2.tests.test_green.GreenTestCase.test_flush_on_write": "unknown",
}
