// Copyright 2019 The Cockroach Authors.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
// implied. See the License for the specific language governing
// permissions and limitations under the License.

package exec

import (
	"fmt"

	"github.com/cockroachdb/cockroach/pkg/sql/distsqlpb"
	"github.com/cockroachdb/cockroach/pkg/sql/exec/coldata"
	"github.com/cockroachdb/cockroach/pkg/sql/exec/types"
)

// NewSortChunks returns a new sort chunks operator, which sorts its input on
// the columns given in orderingCols. The inputTypes must correspond 1-1 with
// the columns in the input operator. The input tuples must be sorted on first
// matchLen columns.
func NewSortChunks(
	input Operator, inputTypes []types.T, orderingCols []distsqlpb.Ordering_Column, matchLen int,
) (Operator, error) {
	if matchLen == len(orderingCols) {
		// input is already ordered on all orderingCols, so there is nothing more
		// to sort.
		return input, nil
	}
	chunker, err := newChunker(input, inputTypes, matchLen)
	if err != nil {
		return nil, err
	}
	sorter, err := newSorter(chunker, inputTypes, orderingCols[matchLen:])
	if err != nil {
		return nil, err
	}
	return &sortChunksOp{input: chunker, sorter: sorter}, nil
}

type sortChunksOp struct {
	input  *chunker
	sorter resettableOperator
}

func (c *sortChunksOp) Init() {
	c.input.init()
	c.sorter.Init()
}

func (c *sortChunksOp) Next() coldata.Batch {
	for {
		batch := c.sorter.Next()
		if batch.Length() == 0 {
			if c.input.done() {
				// We're done, so return a zero-length batch.
				return batch
			}
			// We're not yet done - need to process another chunk, so we empty the
			// chunker's buffer and reset the sorter. Note that we do not want to do
			// the full reset of the chunker because we're in the middle of
			// processing of the input to sortChunksOp.
			c.input.emptyBuffer()
			c.sorter.reset()
		} else {
			return batch
		}
	}
}

// chunkerState represents the state of the chunker spooler.
type chunkerState int

const (
	// chunkerReading is the state of the chunker spooler in which it reads a
	// batch from its input and partitions the batch into chunks. Depending on
	// current state of the chunker's buffer and number of chunks in the batch,
	// chunker might stay in chunkerReading state or switch to either of the
	// emitting states.
	chunkerReading chunkerState = iota
	// chunkerEmittingFromBuffer is the state of the chunker spooler in which it
	// prepares to "emit" tuples that have been buffered. All the tuples belong
	// to the same chunk ("emit" is in quotes because the chunker does not emit
	// batches as usual - it, instead, implements spooler interface, and the
	// batches should be accessed through those methods). The chunker transitions
	// to chunkerEmittingFromBuffer state and indicates that the tuples need to
	// be read from the buffer.
	chunkerEmittingFromBuffer
	// chunkerEmittingFromBatch is the state of the chunker spooler in which it
	// prepares to "emit" all chunks that are fully contained within the last
	// read batch (i.e. all chunks except for the last chunk which might include
	// tuples from the next batch). The last chunk within the batch is buffered,
	// the chunker transitions to chunkerReading state and indicates that the
	// tuples need to be read from s.batch.
	chunkerEmittingFromBatch
)

// chunkerReadingState indicates where the spooler needs to read tuples from
// for emitting.
type chunkerReadingState int

const (
	// chunkerReadFromBuffer indicates that the tuples need to be read from the
	// buffer.
	chunkerReadFromBuffer = iota
	// chunkerReadFromBatch indicates that the tuples need to be read from the
	// last read batch directly. Only tuples that are fully contained within the
	// last read batch are "emitted".
	chunkerReadFromBatch
	// inputDone indicates that the input has been fully consumed and there are
	// no more tuples to read from the chunker.
	inputDone
)

// chunker is a spooler that produces chunks from its input when the tuples
// are already ordered on the first matchLen columns. The chunks are not
// emitted in batches as usual when Next()'ed, but, instead, they should be
// accessed via getValues().
//
// Note 1: the chunker assumes that its input produces batches with no
// selection vector, so it always puts a deselector on top of its input. It
// does the coalescing itself, so it does not use an extra coalescer.
// Note 2: the chunker intentionally does not implement resetter interface (if
// it did, the sorter would reset it, but we don't want that since we're likely
// in the middle of processing the input). Instead, sortChunksOp will empty the
// buffer when appropriate.
type chunker struct {
	input Operator
	// inputTypes contains the types of all of the columns from input.
	inputTypes []types.T
	// inputDone indicates whether input has been fully consumed.
	inputDone bool
	// matchLen indicates the number of first columns on which input is ordered.
	matchLen int

	// batch is the last read batch from input.
	batch coldata.Batch
	// partitioners contains one partitioner for each of matchLen first already
	// ordered columns.
	partitioners []partitioner
	// partitionCol is a bool slice for partitioners' output to be ORed.
	partitionCol []bool

	// chunks contains the indices of the first tuples within different chunks
	// found in the last read batch. Note: the first chunk might be a part of
	// the chunk that is currently being buffered, and similarly the last chunk
	// might include tuples from the batches to be read.
	chunks []uint64
	// chunksProcessedIdx indicates which chunk within s.chunks should be
	// processed next.
	chunksProcessedIdx int
	// chunksStartIdx indicates the index of the chunk within s.chunks that is
	// the first one to be emitted from s.batch directly in when reading from
	// batch.
	chunksStartIdx int

	// buffered indicates the number of currently buffered tuples.
	buffered uint64
	// bufferedColumns is a buffer to store tuples when a chunk is bigger than
	// col.BatchSize or when the chunk is the last in the last read batch (we
	// don't know yet where the end of such chunk is).
	bufferedColumns []coldata.Vec

	readFrom chunkerReadingState
	output   coldata.Batch
	state    chunkerState
}

func newChunker(input Operator, inputTypes []types.T, matchLen int) (*chunker, error) {
	var err error
	partitioners := make([]partitioner, matchLen)
	for col := 0; col < matchLen; col++ {
		partitioners[col], err = newPartitioner(inputTypes[col])
		if err != nil {
			return nil, err
		}
	}
	return &chunker{
		input:        NewDeselectorOp(input, inputTypes),
		inputTypes:   inputTypes,
		matchLen:     matchLen,
		partitioners: partitioners,
		state:        chunkerReading,
	}, nil
}

func (s *chunker) init() {
	s.input.Init()
	s.output = coldata.NewMemBatch(s.inputTypes)
	s.bufferedColumns = make([]coldata.Vec, len(s.inputTypes))
	for i := 0; i < len(s.inputTypes); i++ {
		s.bufferedColumns[i] = coldata.NewMemColumn(s.inputTypes[i], 0)
	}
	s.partitionCol = make([]bool, coldata.BatchSize)
	s.chunks = make([]uint64, 0, 16)
}

// done indicates whether the chunker has fully consumed its input.
func (s *chunker) done() bool {
	return s.readFrom == inputDone
}

// prepareNextChunks prepares the chunks for the chunker spooler.
//
// Note: it does not return the batches directly; instead, the chunker
// remembers where the next chunks to be emitted are actually stored. In order
// to access the chunks, getValues() must be used.
func (s *chunker) prepareNextChunks() chunkerReadingState {
	for {
		switch s.state {
		case chunkerReading:
			s.batch = s.input.Next()
			if s.batch.Length() == 0 {
				s.inputDone = true
				if s.buffered > 0 {
					s.state = chunkerEmittingFromBuffer
				} else {
					s.state = chunkerEmittingFromBatch
				}
				continue
			}
			if s.batch.Selection() != nil {
				// We assume that the input has been deselected, so the batch should
				// never have a selection vector set.
				panic(fmt.Sprintf("unexpected: batch with non-nil selection vector"))
			}

			// First, run the partitioners on our pre-sorted columns to determine the
			// boundaries of the chunks (stored in s.chunks) to sort further.
			copy(s.partitionCol, zeroBoolVec)
			for i, partitioner := range s.partitioners {
				partitioner.partition(s.batch.ColVec(i), s.partitionCol, uint64(s.batch.Length()))
			}
			s.chunks = boolVecToSel64(s.partitionCol, s.chunks[:0])

			if s.buffered == 0 {
				// There are no buffered tuples, so a new chunk starts in the current
				// batch.
				if len(s.chunks) > 1 {
					// There is at least one chunk that is fully contained within
					// s.batch, so we proceed to emitting it.
					s.state = chunkerEmittingFromBatch
					continue
				}
				// All tuples in s.batch belong to the same chunk. Possibly tuples from
				// the next batch will also belong to this chunk, so we buffer the full
				// s.batch.
				s.buffer(0 /* start */, s.batch.Length())
				s.state = chunkerReading
				continue
			} else {
				// There are some buffered tuples, so we need to check whether the
				// first tuple of s.batch belongs to the chunk that is being buffered.
				differ := false
				for i := 0; i < s.matchLen; i++ {
					if err := tuplesDiffer(
						s.inputTypes[i],
						s.bufferedColumns[i],
						0, /*aTupleIdx */
						s.batch.ColVec(i),
						0, /* bTupleIdx */
						&differ,
					); err != nil {
						panic(err)
					}
				}
				if differ {
					// Buffered tuples comprise a full chunk, so we proceed to emitting
					// it.
					s.state = chunkerEmittingFromBuffer
					continue
				}

				// The first tuple of s.batch belongs to the chunk that is being
				// buffered.
				if len(s.chunks) == 1 {
					// All tuples in s.batch belong to the same chunk that is being
					// buffered. Possibly tuples from the next batch will also belong to
					// this chunk, so we buffer the full s.batch.
					s.buffer(0 /* start */, s.batch.Length())
					s.state = chunkerReading
					continue
				}
				// First s.chunks[1] tuples belong to the same chunk that is being
				// buffered, so we buffer them and proceed to emitting all buffered
				// tuples.
				s.buffer(0 /* start */, uint16(s.chunks[1]))
				s.chunksProcessedIdx = 1
				s.state = chunkerEmittingFromBuffer
				continue
			}
		case chunkerEmittingFromBuffer:
			s.state = chunkerEmittingFromBatch
			return chunkerReadFromBuffer
		case chunkerEmittingFromBatch:
			if s.chunksProcessedIdx+1 < len(s.chunks) {
				// There is at least one chunk that is fully contained within s.batch.
				// We don't know yet whether the tuples from the next batch belong to
				// the last chunk of the current batch, so we will buffer those and can
				// only emit "internal" to s.batch chunks. Additionally, if
				// s.chunksProcessedIdx == 1, then the first chunk was already combined
				// with the buffered tuples and emitted.
				s.chunksStartIdx = s.chunksProcessedIdx
				s.chunksProcessedIdx = len(s.chunks) - 1
				return chunkerReadFromBatch
			} else if s.chunksProcessedIdx == len(s.chunks)-1 {
				// Other tuples might belong to this chunk, so we buffer it.
				s.buffer(uint16(s.chunks[s.chunksProcessedIdx]), s.batch.Length())
				// All tuples in s.batch have been processed, so we reset s.chunks and
				// the corresponding variables.
				s.chunks = s.chunks[:0]
				s.chunksProcessedIdx = 0
				s.state = chunkerReading
			} else {
				// All tuples in s.batch have been emitted.
				if s.inputDone {
					return inputDone
				}
				panic(fmt.Sprintf("unexpected: chunkerEmittingFromBatch state" +
					"when s.chunks is fully processed and input is not done"))
			}
		default:
			panic(fmt.Sprintf("invalid chunker spooler state %v", s.state))
		}
	}
}

// buffer appends all tuples in range [start,end) from s.batch to already
// buffered tuples.
func (s *chunker) buffer(start uint16, end uint16) {
	for i := 0; i < len(s.bufferedColumns); i++ {
		s.bufferedColumns[i].AppendSlice(
			s.batch.ColVec(i),
			s.inputTypes[i],
			s.buffered,
			start,
			end,
		)
	}
	s.buffered += uint64(end - start)
}

func (s *chunker) spool() {
	s.readFrom = s.prepareNextChunks()
}

func (s *chunker) getValues(i int) coldata.Vec {
	switch s.readFrom {
	case chunkerReadFromBuffer:
		return s.bufferedColumns[i].Slice(s.inputTypes[i], 0 /* start */, s.buffered)
	case chunkerReadFromBatch:
		return s.batch.ColVec(i).Slice(s.inputTypes[i], s.chunks[s.chunksStartIdx], s.chunks[len(s.chunks)-1])
	default:
		panic(fmt.Sprintf("unexpected chunkerReadingState in getValues: %v", s.state))
	}
}

func (s *chunker) getNumTuples() uint64 {
	switch s.readFrom {
	case chunkerReadFromBuffer:
		return s.buffered
	case chunkerReadFromBatch:
		return s.chunks[len(s.chunks)-1] - s.chunks[s.chunksStartIdx]
	case inputDone:
		return 0
	default:
		panic(fmt.Sprintf("unexpected chunkerReadingState in getNumTuples: %v", s.state))
	}
}

func (s *chunker) getPartitionsCol() []bool {
	switch s.readFrom {
	case chunkerReadFromBuffer:
		// There is a single chunk in the buffer, so, per spooler's contract, we
		// return nil.
		return nil
	case chunkerReadFromBatch:
		if s.chunksStartIdx+1 == len(s.chunks)-1 {
			// There is a single chunk that is fully contained within s.batch, so,
			// per spooler's contract, we return nil.
			return nil
		}
		copy(s.partitionCol, zeroBoolVec)
		for i := s.chunksStartIdx; i < len(s.chunks)-1; i++ {
			// getValues returns a slice starting from s.chunks[s.chunksStartIdx], so
			// we need to account for that by shifting as well.
			s.partitionCol[s.chunks[i]-s.chunks[s.chunksStartIdx]] = true
		}
		return s.partitionCol
	default:
		panic(fmt.Sprintf("unexpected chunkerReadingState in getPartitionsCol: %v", s.state))
	}
}

func (s *chunker) emptyBuffer() {
	// We only need to set s.buffered to 0 to empty the buffer.
	s.buffered = 0
}
