/* $Id$ */
/*
 * makes X11 BDF font bold.
 * by Daisuke NISHIKAWA
 *
 * modified code by Yasuyuki Furukawa <yasu@on.cs.keio.ac.jp>, public domain.
 * perl code by NAGAO, Sadakazu <snagao@cs.titech.ac.jp>
 */
#include <ctype.h>
#include "common.h"
#include "bold.h"
#include "utils.h"

#define BUFLEN 256

static void mkbold(int bdir, int pile, Property *prop, int verbose, char *filename);
static void get_option(int *bdir, int *pile, Property *prop, int *verbose, char **filename, int argc, char *argv[]);
static void usage(FILE *fp, int status);


int main(int argc, char *argv[])
{
  Property prop;
  int bdir = TRUE;  /* left shifted image */
  int pile = FALSE; /* right edge */
  int verbose = FALSE;
  char *filename = NULL;

  prop.weight[0] = "Medium";
  prop.weight[1] = "Bold";
  prop.slant[0] = prop.slant[1] = NULL;
  get_option(&bdir, &pile, &prop, &verbose, &filename, argc, argv);
  mkbold(bdir, pile, &prop, verbose, filename);
  return EXIT_SUCCESS;
}

static void mkbold(int bdir, int pile, Property *prop, int verbose, char *filename)
{
  FILE *fp;
  char buf[BUFLEN], outbuf[BUFLEN], keyword[BUFLEN];
  size_t len, len_max;
  BoldBitmap *d;
  int width, height;
  int bitmap, rc;
  ProgressMeter *meter;
  int max_chars;

  width = height = 0;
  prop->width = &width;
  prop->height = &height;
  prop->recast = recast_dummy;
  fp = file_open(filename);
  max_chars = replace_property(fp, BUFLEN, prop);
  if (max_chars < 0) {
    fclose(fp);
    return;
  }
  meter = NULL;
  if (verbose) {
    meter = progress_meter_new(max_chars);
  }
  len_max = len = (width-1)/8+1;
  d = boldBitmap_new(len_max);
  bitmap = FALSE;
  while (fgets(buf, BUFLEN, fp) != NULL) {
    if (isalpha(buf[0])) {
      sscanf(buf, "%s", keyword);
      if (strcmp(keyword, "BBX") == 0) {
        /* recast the metric information from the bounding box */
        rc = recast_dummy(outbuf, &width, &height, NULL, 0, buf);
        len = (width-1)/8+1;
        if (!rc || (len > len_max)) {
          fclose(fp);
          fprintf(stderr, "Incorrect bounding box data: %s\n", buf);
          exit(EXIT_FAILURE);
        }
        fputs(outbuf, stdout);
        continue;
      }
      if (strcmp(keyword, "BITMAP") == 0) {
        bitmap = TRUE;
        fputs(buf, stdout);
        continue;
      }
      if (strcmp(keyword, "ENDCHAR") == 0) {
        bitmap = FALSE;
        fputs(buf, stdout);
        if (verbose) {
          progress_meter_print(meter);
        }
        continue;
      }
    }

    if (bitmap) {
      boldBitmap_make_bold(d, buf, len, bdir, pile);
      boldBitmap_output(stdout, d, len);
      continue;
    }
    fputs(buf, stdout);
  }
  fclose(fp);
  boldBitmap_delete(d);

  if (verbose) {
    progress_meter_clear(meter);
    progress_meter_delete(meter);
  }
}

static void get_option(int *bdir, int *pile, Property *prop, int *verbose, char **filename, int argc, char *argv[])
{
  int i;

  i = 1;
  while (i < argc) {
    if (*argv[i] == '-') {
      switch (*(argv[i]+1)) {
      case 'r':
        *bdir = FALSE;
        break;
      case 'l':
        *bdir = TRUE;
        break;
      case 'R':
        *pile = FALSE;
        break;
      case 'L':
        *pile = TRUE;
        break;
#if defined DEBUG
      case 'W':
        if (++i < argc) {
          prop->weight[0] = argv[i];
        }
        break;
      case 'w':
        if (++i < argc) {
          prop->weight[1] = argv[i];
        }
        break;
#endif /* DEBUG */
      case 'V':
        *verbose = TRUE;
        break;
      case 'h':
        usage(stdout, EXIT_SUCCESS);
        break;
      }
    } else {
      *filename = argv[i];
      break;
    }
    i++;
  }
}

static void usage(FILE *fp, int status)
{
  fprintf(fp,
          "Usage: mkbold [OPTION] [FILE]\n\n"
          "Write processed X11 BDF file to standard output.\n"
          "With no FILE, read standard input.\n"
          "\nOptions:\n"
          "  -l left shifted image OR original (default)\n"
          "  -r right shifted image OR original\n"
          "  -L reflect left edge\n"
          "  -R reflect right edge (default)\n"
          "  -V verbose output\n"
          "  -h display this help and exit\n"
          );
  exit(status);
}
