/*
 *  tcmplex.c
 *
 *  Copyright (C) Thomas streich - June 2001
 *
 *  This file is part of transcode, a linux video stream processing tool
 *      
 *  transcode is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  transcode is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include "main.h"
#include "bbencode.h"

static int verbose=0;

extern int domplex(int has_video, int has_audio);

#define EXE "tcmplex-panteltje"

/* ------------------------------------------------------------ 
 *
 * print a usage/version message
 *
 * ------------------------------------------------------------*/

void version(char *exe)
{
// print id string to stderr
fprintf(stderr, "%s%s-%s (c) 2001 Thomas streich and Jan Panteltje\n",\
exe, PACKAGE, VERSION);
}

void usage()
{
fprintf(stderr,
"\nUsage: %s [options] [-]\n\
 -i name        video stream filename.\n\
 -0 name        audio stream (track 0) filename.\n\
 -p name        same as -0 name.\n\
 -n name        extra audio stream(s) (track n, n ranges from 1 to 7)\n\
                filename (optional).\n\
 -s name        same as -1 name.\n\
 -o name        multiplexed program/system stream filename.\n\
 -m mode        predefined settings [1].\n\
                1 = mpeg1 vbr, buffer 46kb (*** default XVCD).\n\
                b = mpeg1 vbr, buffer 224kb (experimental).\n\
                2 = mpeg2 vbr.\n\
                d = DVD.\n\
                s = SVCD.\n\
                v = VCD.\n\
 -D v[,a[,a1][,a2][,a3][,a4][,a5][,a6][,a7]]\n\
                sync correction for video, audiox, in ms (positive number).\n\
 -c a-b         multiplex selected time interval in seconds [all].\n\
 -B             generates a profile template on stdout [off].\n\
 -F filename    user profile filename [off].\n\
 -d verbosity   verbosity mode [1].\n\
 -v             print version.\n",\
EXE);

//-N             NTSC mode [PAL].\n\
//-P             set 3:2 pulldown flags [off].\n\

exit(1);
} /* end function usage */


int main(int argc, char **argv)
{
char mux_type      = PRO_MPEG1;
int  has_audio     = 0;
int  has_video     = 0;
int  has_output    = 0;
int  tv_type       = ENCODE_PAL;
int  pulldown      = 0;

long ivideo_delay_ms  = 0;
long iaudio_delay_ms  = 0;
long iaudio1_delay_ms = 0;
long iaudio2_delay_ms = 0;
long iaudio3_delay_ms = 0;
long iaudio4_delay_ms = 0;
long iaudio5_delay_ms = 0;
long iaudio6_delay_ms = 0;
long iaudio7_delay_ms = 0;

long mux_start = 0, mux_stop = 0;
int mux_stop_flag = 0;

char *profile_name = NULL;

int ch;
int error = 0;

while ((ch = getopt(argc, argv, "0:p:1:s:2:3:4:5:6:7:o:i:vBd:m:F:ND:Pc:")) != -1)
  {
  switch (ch)
   {
   case '0': 
   case 'p': /* Original tcmplex compatibility */
      
      if(optarg[0]=='-') usage();
      strcpy(AudioFilename, optarg);
      has_audio = 1;
      break;

   case '1': 
   case 's': /* Original tcmplex compatibility */
      
      if(optarg[0]=='-') usage();
      strcpy(Audio1Filename, optarg);
      has_audio = 1;
      break;

   case '2': 
      
      if(optarg[0]=='-') usage();
      strcpy(Audio2Filename, optarg);
      has_audio = 1;
      break;

   case '3': 
      
      if(optarg[0]=='-') usage();
      strcpy(Audio3Filename, optarg);
      has_audio = 1;
      break;

   case '4': 
      
      if(optarg[0]=='-') usage();
      strcpy(Audio4Filename, optarg);
      has_audio = 1;
      break;

   case '5': 
      
      if(optarg[0]=='-') usage();
      strcpy(Audio5Filename, optarg);
      has_audio = 1;
      break;

   case '6': 
      
      if(optarg[0]=='-') usage();
      strcpy(Audio6Filename, optarg);
      has_audio = 1;
      break;

   case '7': 
      
      if(optarg[0]=='-') usage();
      strcpy(Audio7Filename, optarg);
      has_audio = 1;
      break;

   case 'i': 
      
      if(optarg[0]=='-') usage();
      strcpy(VideoFilename, optarg);
      has_video = 1;
      
      break;

   case 'F': 
      
      if(optarg[0]=='-') usage();
      profile_name = optarg;
      
      break;

   case 'N': 
      
      tv_type = ENCODE_NTSC;
      break;

   case 'o': 
      
      if(optarg[0]=='-') usage();
      strcpy(ProgramFilename, optarg);
      has_output = 1;
      break;
      
   case 'P': 
      
      pulldown = 1;
      break;

   case 'm': 
      
      if(optarg[0]=='-') usage();
      memcpy((char*) &mux_type, optarg, 1);
      
      break;

   case 'd': 
      
      if(optarg[0]=='-') usage();
      verbose = atoi(optarg);
      
      break;

   case 'D': 
      
      if(optarg[0]=='-') usage();

      //overwrite profile defaults

      sscanf(optarg, "%ld,%ld,%ld,%ld,%ld,%ld,%ld,%ld,%ld",\
      &ivideo_delay_ms,
      &iaudio_delay_ms,  &iaudio1_delay_ms, &iaudio2_delay_ms, &iaudio3_delay_ms,
      &iaudio4_delay_ms, &iaudio5_delay_ms, &iaudio6_delay_ms, &iaudio7_delay_ms);
      break;

   case 'c': 
      
      if(optarg[0]=='-') usage();

      //overwrite profile defaults

      sscanf(optarg, "%ld-%ld", &mux_start, &mux_stop);

      break;

        
   case 'B':
      
      bb_gen_profile();
      exit(0);
      break;

   case 'v': 
      
      version(EXE);
      exit(0);
      break;
      
   case '?':
   default:
 
      usage();
      break;
   } /* end switch */
 } /* end while */

version(EXE);

/* sanity check command line parameters */   
    
// need at least a file name
if(argc == 1)
  {
  usage();
  }

if(!has_output)
   {
   fprintf(stderr, "ERROR: output filename option -o required\n");
   error = 1;
   }

if(ivideo_delay_ms < 0)
  {
  fprintf(stderr, "ERROR: Do not use negative video delay, use positive delays in audio.\n");
  error = 1;
  }     

if(
  (iaudio_delay_ms < 0) || (iaudio1_delay_ms < 0) || (iaudio2_delay_ms < 0) || (iaudio3_delay_ms < 0) ||
  (iaudio4_delay_ms < 0) || (iaudio5_delay_ms < 0) || (iaudio6_delay_ms < 0) || (iaudio7_delay_ms < 0)
  )
  {
  fprintf(stderr, "ERROR: Do not use negative audio delays, use positive delay in video.\n");
  error = 1;
  }

if(mux_start < 0)
  {
  fprintf(stderr, "ERROR: start seconds must be <= 0.\n");
  error = 1;
  }

if(mux_stop < 0)
  {
  fprintf(stderr, "ERROR: stop seconds must be >= 0.\n");
  error = 1;
  }
    
if(error)
  {
  usage();
  }

//-- setup parameters --
bb_set_profile(profile_name, mux_type, tv_type, 0, 0, pulldown, 1);

// The cmd line parameters have higher priority

if(ivideo_delay_ms  > 0) video_delay_ms  += ivideo_delay_ms;

if(iaudio_delay_ms  > 0) audio_delay_ms  += iaudio_delay_ms;
if(iaudio1_delay_ms > 0) audio1_delay_ms += iaudio1_delay_ms;
if(iaudio2_delay_ms > 0) audio2_delay_ms += iaudio2_delay_ms;
if(iaudio3_delay_ms > 0) audio3_delay_ms += iaudio3_delay_ms;
if(iaudio4_delay_ms > 0) audio4_delay_ms += iaudio4_delay_ms;
if(iaudio5_delay_ms > 0) audio5_delay_ms += iaudio5_delay_ms;
if(iaudio6_delay_ms > 0) audio6_delay_ms += iaudio6_delay_ms;
if(iaudio7_delay_ms > 0) audio7_delay_ms += iaudio7_delay_ms;

if(mux_start > 0) mux_start_time = mux_start;
if(mux_stop > 0) mux_stop_time = mux_stop;

//-- do the job --
domplex(has_video, has_audio);
   
fprintf(stderr, "\n\n");
   
exit(0);
} /* end function main */



