/* code generator
   Copyright (C) 1998, 1999, 2000, 2001, 2002, 2003
   Wouter van Ooijen

This file is part of jal.

jal is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

jal is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with jal; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include "stdhdr.h"
#include "global.h"
#include "target.h"
#include "errorlh.h"
#include "cstringf.h"
#include "treerep.h"
#include "assemble.h"
#include "stacksg.h"
#include "scanner.h"
#include "codegen.h"
#include "parser.h"
#include "regalloc.h"
#include "treetools.h"
#include "varlist.h"


/* return the page bits for the indicated target */
int page_bits(int address)
{
    stack_guard;                /* when removed: unreachable code??? */

    switch (target_chip) {
    case t_12c508:
    case t_12c509a:
    case t_sx18:
    case t_sx28:
        return (address >> 5);
        break;
    case t_16f88:
    case t_16c84:
    case t_16f84:
    case t_16f628:
    case t_12f629:
    case t_12f675:
    case t_16f877:
    case t_16f876:             /* Added 16f876 & 16f873 by Javi 2003-03-01 */
    case t_16f873:             /* Added 16f876 & 16f873 by Javi 2003-03-01 */
        return (address >> 3);
        break;
    default:
        snark(NULL);
        break;
    }

    /* dummy to suppress warnings */
    return 0;
}

/* return the number of instructions occupied by a page 'instruction' */
int page_instruction_size(int address, boolean sacred)
{
    stack_guard;

    switch (target_chip) {
    case t_12c508:
        return 0;
        break;
    case t_12c509a:
        return 1;
        break;
    case t_16f88:
    case t_sx18:
    case t_sx28:
        return 1;
        break;
    case t_16c84:
    case t_16f84:
    case t_16f628:
    case t_12f629:
    case t_12f675:
    case t_18f242:
    case t_18f252:
    case t_18f442:
    case t_18f452:
        return 0;
        break;
    case t_16f877:
    case t_16f876:             /* Added 16f876 & 16f873 by Javi 2003-03-01 */
    case t_16f873:             /* Added 16f876 & 16f873 by Javi 2003-03-01 */
        if ((!sacred) && (page_bits(address) == 0))
            return 1;
        else
            return 2;
        break;
    default:
        log((m, "target chip = %d", target_chip));
        snark(NULL);
    }

    /* dummy to suppress warnings */
    return 0;
}


/* w value re-use */

tree current_w_value = NULL;
int current_w_offset = 0;

void load_w(tree x)
{
    current_w_value = x;
    current_w_offset = 0;
}

void forget_w(void)
{
    current_w_value = NULL;
}

boolean w_is(tree x)
{
    return ((current_w_offset == 0)
            && (current_w_value == x)
        );
}

/* the pic flags */
tree pic_flags, pic_pclath, pic_flag_zero, pic_flag_carry, pic_flag_digit_carry;

/* report whether t is used in p */
boolean uses(tree t, tree p)
{
    assert_pointer(NULL, t);
    assert_kind(NULL, t, node_var);

    if (p == NULL)
        return false;

    switch (p->kind) {

    case node_w:
    case node_org:
    case node_precall:
    case node_label:
        return false;
        break;
    case node_var:
    case node_const:
        return p == t;
        break;
    case node_assign:
    case node_chain:
    case node_ref:
    case node_op:
        return uses(t, p->first)
            || uses(t, p->next);
        break;
    case node_while:
    case node_for:
    case node_if:
        return uses(t, p->first)
            || uses(t, p->next)
            || uses(t, p->condition)
            || uses(t, p->start)
            || uses(t, p->step)
            || uses(t, p->end);
        break;
    case node_asm:
    case node_call:
        /* too difficult or time-consuming to analyse */
        return true;
        break;
    default:
        snark_node(p->loc, p);
        break;
    }
    snark_node(p->loc, p);
    return true;
}

/* copy master address when needed */
void copy_master_address(tree q)
{
    if ((q != NULL)
        && (q->kind == node_var)
        && (q->fixed)           /* did not work with some 'at' vars ? */
        ) {
        if (q->master1 != NULL) {
            assert_kind(q->loc, q->master1, node_var);
            jal_assert(q->loc, q->master1->address != NULL);
            if (q->master1->address != NULL) {
                assert_kind(q->loc, q->master1->address, node_chain);
                if (q->address == NULL) {
                    q->address = new_chain2(NULL, NULL);
                }
                q->address->first = q->master1->address->first;
            }
        }
        if (q->master2 != NULL) {
            assert_kind(q->loc, q->master2, node_var);
            if (q->master2->address != NULL) {
                assert_kind(q->loc, q->master2->address, node_chain);
                if (q->address == NULL) {
                    q->address = new_chain2(NULL, NULL);
                }
                q->address->next = q->master2->address->next;
            }
        }
    }
}

/* types are compatible ? */
boolean code_type_is(tree p, tree t)
{
    stack_guard;
    assert_pointer(NULL, p->type);
    return (p->type == t)
        || ((t == type_byte)
            && (p->type == type_universal)
        );
}

/* last pages routines gathered during coding */
tree first_page_routines = NULL;

/* transfer administration */
int transfer_count = 0;
#define transfer_size   256
tree transfers[transfer_size];
tree transfer_label;
int transfer_address;

void make_transfer(tree p)
{
    stack_guard;
    if (p->jtaddress != NULL)
        return;
    if (transfer_count >= transfer_size) {
        fatal(NULL, (m, "too many indirections"));
    }
    transfers[transfer_count] = p;
    transfer_count++;
    p->jtaddress = new_value(type_universal, transfer_address & 0xFF);
#ifdef __DEBUG__
    log((m, "make_transfer i=%d node=%d a=%x", transfer_count - 1, p->nr, p->jtaddress->x));
#endif
    if ((target_chip == t_16f877) || (target_chip == t_16f876) || (target_chip == t_16f873))    /* Added 16f876 & 16f873 by Javi 2003-03-01 */
        transfer_address += 6;
    else if ((target_cpu == sx_12)
             || (target_cpu == pic_12)
             || (target_cpu == pic_16)
        ) {
        transfer_address += 4;
    } else {                    /* 16x84, 16f628 */
        transfer_address += 2;
    }
}

tree one_transfer(tree p)
{
    stack_guard;
    if (p == NULL) {
        if ((target_chip == t_16f877) || (target_chip == t_16f876) || (target_chip == t_16f873))        /* Added 16f876 & 16f873 by Javi 2003-03-01 */
            return new_chain3(new_asm(NULL, opcode_sleep, NULL, 0),
                              new_asm(NULL, opcode_sleep, NULL, 0), 
                              new_asm(NULL, opcode_sleep, NULL, 0));
        else if ((target_cpu == sx_12)
                 || (target_cpu == pic_12)
                 || (target_cpu == pic_16)
            ) {
            return new_chain2(new_asm(NULL, opcode_sleep, NULL, 0),
                              new_asm(NULL, opcode_sleep, NULL, 0));
        } else {                /* 16x84, 16f628 */
            return new_asm(NULL, opcode_sleep, NULL, 0);

        }
    } else {
        tree q = p;
        if (q->kind == node_procedure)
            q = q->address;
#ifdef __DEBUG__
        log((m, "one_transfer p=%d n=%s", q->nr, q->name));
        trace_subtree(q);
#endif
        return code_code_page(q, new_asm(NULL, opcode_goto, q, 0));
    }
}

tree make_asm_bit(opcode_t opcode, int file, int bit)
{
    tree p = new_node(NULL, node_asm);
    p->opcode = opcode;
    p->first = new_value(type_universal, file);
    p->next = new_value(type_universal, bit);
    return p;
}

tree code_transfers(tree transfer_label)
{
    int n;
    tree code, p, q;
    stack_guard;

    if (verbose_coder)
        log((m, "code_transfers"));

    if (transfer_count == 0) {
        return NULL;
    }

    /* make the list of transfer vectors */
    code = p = new_chain2(NULL, NULL);
    for (n = 0; n < transfer_count; n++) {
        tree put_label;
#ifdef __DEBUG__
        log((m, "one_transfer i=%d node=%d", n, transfers[n]->nr));
#endif
        put_label = transfers[n]->put;
        if (put_label != NULL)
            put_label = put_label->tlabel;
        q = new_chain2(new_chain2(one_transfer(transfers[n]->get), one_transfer(put_label)), NULL);
        p->next = q;
        p = q;
    }

    /* code to set the code page bits */
    switch (target_chip) {
    case t_12c508:{
            p = NULL;
            break;
        }
    case t_12c509a:{
            p = make_asm_bit(opcode_bcf, 3, 5);
            break;
        }
    case t_sx18:
    case t_sx28:{
            p = new_chain2(make_asm_bit(opcode_bcf, 3, 5), make_asm_bit(opcode_bcf, 3, 6));
            break;
        }
    case t_16f88:
    case t_16f84:
    case t_16c84:
    case t_16f628:
    case t_12f629:
    case t_12f675:
    case t_16f877:
    case t_16f876:             /* Added 16f876 & 16f873 by Javi 2003-03-01 */
    case t_16f873:             /* Added 16f876 & 16f873 by Javi 2003-03-01 */
        {
            p = new_asm(NULL, opcode_clrf, new_value(type_universal, 10), 0);
            break;
        }
    case t_18f242:
    case t_18f252:
    case t_18f442:
    case t_18f452:{
            p = new_chain4(new_asm(NULL, opcode_clrf, new_value(type_universal, 0xFFA), 0),
                           new_asm(NULL, opcode_clrf, new_value(type_universal, 0xFFB), 0),
                           new_asm(NULL, opcode_clrc, NULL, 0), 
                           new_asm(NULL, opcode_rlf, new_value(type_universal, 0xFE8), dest_w));
            break;
        }
    default:{
            cassert(false);
            break;
        }
    }

    /* combine it all label and jump */
    code =
        new_chain4(code, transfer_label, p,
                   new_asm(NULL, opcode_movwf,
                           new_value(type_universal, (target_cpu == pic_16 ? 0xFF9 : 0x002)), 0)
        );

    return code;
}

/* codes generated for a boolean expression depends on:
 *    true_next == the true label is the very next location
 *    true_one  == the true part is just one instruction
 *    idem for false_next and false_one
 * these arguments must be passed on through all calls..
 */
#define formal_args tree p, \
   tree true_label,  boolean true_next,  boolean true_one, \
   tree false_label, boolean false_next, boolean false_one
#define my_args true_label, true_next, true_one, \
   false_label, false_next, false_one

/* forward: the main coder routine */
tree code_node(formal_args);

tree code_store(formal_args)
{
    stack_guard;
    cassert(p->type == type_byte);
    return code_register_bank(p, new_asm(p->loc, opcode_movwf, p, 0));
}

tree code_jump(loc_t loc, formal_args)
{
    stack_guard;
    assert_kind(p->loc, p, node_label);
    return code_code_page(p, new_asm(loc, opcode_goto, p, 0));
}

tree try_clrf(tree p)
{
    stack_guard;
    assert_kind(p->loc, p->first, node_ref);
    assert_pointer(p->loc, p->first->first);
    if ((p->first->type == type_byte)
        && (p->first->first->kind != node_w)
        && (p->next->kind == node_ref)
        && (p->next->first->kind == node_const)
        && (p->next->first->value->x == 0)
        ) {
        tree q = new_node(p->loc, node_asm);
#ifdef __DEBUG__
        trace_subtree(p);
#endif
        q->opcode = opcode_clrf;
        q->first = p->first;
        return code_register_bank(p->first, q);
    } else {
        return NULL;
    }
}

tree try_file_op2(tree p, int op, int val, int opcode1, int opcode2)
{
    stack_guard;
    assert_kind(p->loc, p, node_assign);
    cassert(p->type == type_byte);
#ifdef __DEBUG__
    trace_subtree(p);
#endif

    if (
           /* left hand is a var */
           (p->first->kind == node_ref)
           && (p->first->first->kind == node_var)

           /* right hand is indicated op */
           && (p->next->kind == node_op)
           && (p->next->op == op)

           /* op left side of the op is same var as left hand of assign */
           && (p->next->first->kind == node_ref)
           && (p->next->first->first == p->first->first)

           /* right side of the op is the indicated constant */
           && (p->next->next->kind == node_ref)
           && (p->next->next->first->kind == node_const)
           && (p->next->next->first->value->x == val)
        ) {
        tree q = new_node(p->loc, node_asm);
        q->opcode = opcode2;
        q->first = p->first;
        q->dest = dest_f;
        if (opcode1 != opcode_nop) {
            tree x = new_node(p->loc, node_asm);
            tree a = new_chain(x);
            tree b = new_chain(q);
            x->opcode = opcode1;
            a->next = b;
            return code_register_bank(p->first, a);
        } else {
            return code_register_bank(p->first, q);
        }
    } else {
        return NULL;
    }
}

#define try_file_op( p, op, val, code1, code2 ) { \
   tree q = try_file_op2( p, op, val, code1, code2 ); \
   if( q != NULL ) return q; \
}

tree code_skip(loc_t loc, boolean invert, tree flag)
{
    tree skip;

    /* create the test/skip instruction */
    /* skip = new_asm(
       loc,
       invert ? opcode_btfss : opcode_btfsc,
       new_ref( loc, ref_var, new_chain2( pic_flags, flag ), NULL ),
       0
       ); */

    skip = new_node(loc, node_asm);
    skip->opcode = invert ? opcode_btfss : opcode_btfsc;
    skip->first = pic_flags;
    skip->next = flag;

    return code_register_bank(flag, skip);
}

tree code_boolean(formal_args)
{
    tree q, t1, t2;
    boolean invert;
    tree compare, flag, cond;
    stack_guard;

    compare = t2 = NULL;

    assert_pointer(NULL, p);
    assert_pointer(NULL, true_label);
    assert_pointer(NULL, false_label);
    if (verbose_coder)
        log((m, "code_boolean %d %s", p->nr, node_name[p->kind]));

    switch (p->kind) {

    case node_op:
        break;

    case node_ref:{

            assert_pointer(NULL, p->first);
            if (p->first->kind == node_var) {

                if (false_next) {

                    if (false_one) {
                        q = code_register_bank(p, new_asm(p->loc, opcode_btfss, p, 0));
                    } else {
                        q = new_chain4(code_code_page(true_label, NULL),
                                       code_register_bank(p, NULL), 
                                       new_asm(p->loc, opcode_btfsc, p, 0), 
                                       new_asm(p->loc, opcode_goto, true_label, 0));
                    }

                } else if (true_next) {

                    if (true_one) {
                        q = code_register_bank(p, new_asm(p->loc, opcode_btfsc, p, 0));
                    } else {
                        q = new_chain4(code_code_page(false_label, NULL),
                                       code_register_bank(p, NULL), 
                                       new_asm(p->loc, opcode_btfss, p, 0), 
                                       new_asm(p->loc, opcode_goto, false_label, 0));
                    }

                } else {
                    q = new_chain6(code_code_page(false_label, NULL), code_register_bank(p, NULL),
                                   new_asm(p->loc, opcode_btfss, p, 0), 
                                   new_asm(p->loc, opcode_goto, false_label, 0),
                                   code_code_page(true_label, NULL), 
                                   new_asm(p->loc, opcode_goto, true_label, 0));
                }
                return q;

            }
            if (p->first->kind == node_const) {

                assert_kind(p->loc, p->first->value, node_value);
                if (p->first->value->x != 0) {
                    q = code_code_page(true_label, new_asm(p->loc, opcode_goto, true_label, 0));
                } else {
                    q = code_code_page(false_label, new_asm(p->loc, opcode_goto, false_label, 0));
                }
                return q;

            } else {
                snark_node(p->loc, p->first);
            }
            break;
        }

    case node_chain:{
            return new_chain2(code_node(p->first, my_args), code_boolean(p->next, my_args));
        }

    default:{
            snark_node(p->loc, p);
            break;
        }
    }

    assert_kind(p->loc, p, node_op);

    /* first handle the operators on booleans */
#ifdef __DEBUG__
    trace_subtree(p);
#endif
    t1 = follow(p->first);
    assert_pointer(p->loc, t1);
    t1 = t1->type;
    assert_kind(p->loc, t1, node_type);

    if (!is_monop(p->op)) {
        assert_kind(p->loc, p->next->type, node_type);
        t2 = p->next->type;
    }
    if ((t1 == type_bit)
        && (is_monop(p->op)
            || (t2 == type_bit)
        )
        )
        switch (p->op) {
        case op_mnot:{
                return code_boolean(p->first, false_label, false_next, false_one, true_label,
                                    true_next, true_one);
            }

        case op_and:{
                tree label = new_label(NULL, "e_%d", p->nr, "");
                return
                    new_chain3(code_boolean
                               (p->first, label, true, false, false_label, false, false), label,
                               code_boolean(p->next, true_label, true_next, true_one, false_label,
                                            false_next, false_one)
                    );
            }

        case op_or:{
                tree label = new_label(NULL, "e_%d", p->nr, "");
                return
                    new_chain3(code_boolean(p->first, true_label, false, false, label, true, false), label, 
                               code_boolean(p->next, true_label, true_next, true_one, false_label, false_next, false_one)
                    );
            }

        case op_xor:
        case op_not_equal:{
                tree local_true = new_label(NULL, "ene_1_%d", p->nr, "");
                tree local_false = new_label(NULL, "ene_2_%d", p->nr, "");
                return
                    new_chain5(code_boolean
                               (p->first, local_true, true, false, local_false, false, false),
                               local_true, code_boolean(p->next, false_label, false, false,
                                                        true_label, false, false), local_false,
                               code_boolean(p->next, true_label, true_next, true_one, false_label,
                                            false_next, false_one)
                    );
            }

        case op_equal:{
                tree local_true = new_label(NULL, "eeq_1_%d", p->nr, "");
                tree local_false = new_label(NULL, "eeq_2_%d", p->nr, "");
                return
                    new_chain5(code_boolean
                               (p->first, local_true, true, false, local_false, false, false),
                               local_true, code_boolean(p->next, true_label, false, false,
                                                        false_label, false, false), local_false,
                               code_boolean(p->next, false_label, false_next, false_one, true_label,
                                            true_next, true_one)
                    );
            }

        default:{
                break;
            }
        }

    /* now assume that we have an operator which does not involve boolean operands */

    /* determine flag and polarity */
    flag = NULL;
    invert = false;
    switch (p->op) {

    case op_not_equal: invert = true;
    case op_equal:{
        flag = pic_flag_zero;
        break;
        }

    case op_smaller : {
        swapp( p->first, p->next );
        p->op = op_larger;
        invert = true ;
         flag   = pic_flag_carry;
         break;
         }

    case op_larger_or_equal : {
         swapp( p->first, p->next );
         p->op = op_smaller_or_equal;
         flag   = pic_flag_carry;
         break;
         }

    case op_larger : invert = true ;
    case op_smaller_or_equal : {
         flag   = pic_flag_carry;
         break;
         }


    default:{
            snark_node(p->loc, p);
        }
    }
    assert_pointer(NULL, flag);

    /* determine opcode */
    assert_pointer(NULL, p->next);
    assert_kind(NULL, p->next, node_ref);
    switch (p->next->first->kind) {
    case node_var:{
            compare =
                code_register_bank(p->next->first,
                                   new_asm(p->loc, opcode_subwf, p->next->first, dest_w));
            break;
        }

    case node_const:{
            compare =
                code_register_bank(p->next->first,
                                   new_asm(p->loc, opcode_sublw, p->next->first, 0));
            break;
        }

    default:
        snark_node(p->loc, p->next->first);
    }

    /* special case for x != 0 and x == 0 */
    if ((flag == pic_flag_zero)
        && (p->first->kind == node_ref)
        && (p->next->first->kind == node_const)
        && (p->next->first->value->x == 0)
        ) {
        cond = NULL;
        compare = code_register_bank(p->first, new_asm(p->loc, opcode_movf, p->first, dest_f));
    } else {
        cond = code_node(p->first, my_args);
    }

    if (true_next && true_one) {
        return new_chain3(cond, compare, code_skip(p->loc, invert, flag));
    }

    if (false_next && false_one) {
        return new_chain3(cond, compare, code_skip(p->loc, !invert, flag));
    }

    if (true_next) {
        return new_chain5(cond, compare, code_code_page(false_label, NULL),
                          code_skip(p->loc, !invert, flag), 
                          new_asm(p->loc, opcode_goto, false_label, 0));
    }

    if (false_next) {
        return new_chain5(cond, compare, code_code_page(true_label, NULL),
                          code_skip(p->loc, invert, flag), 
                          new_asm(p->loc, opcode_goto, true_label, 0));
    }

    return new_chain7(cond, compare, code_code_page(true_label, NULL),
                      code_skip(p->loc, invert, flag), new_asm(p->loc, opcode_goto, true_label, 0),
                      code_code_page(false_label, NULL), 
                      new_asm(p->loc, opcode_goto, false_label, 0));
}

tree code_byte_op(formal_args)
{
    tree q1, q2, a1, a2, c1, source = NULL;
    stack_guard;
    if (p == NULL)
        return NULL;

    q1 = new_node(p->loc, node_chain);
    q1->next = q2 = new_node(p->loc, node_chain);
#ifdef __DEBUG__
    log((m, "code_statement resume %d %s", p->nr, node_name[p->kind]));
    log((m, "n=%d op=%s", p->nr, op_name[p->op]));
#endif
    switch (p->op) {

    case op_minus:  /* Activated again by Javi 2003-03-31  */
//           swapp( p->first, p->next ); /* removed, this is now done differently */
    case op_plus:
    case op_and:
    case op_or:
    case op_xor:
    case op_not_equal:{
            q1->first = code_node(p->first, my_args);
            q2->first = a1 = new_node(p->loc, node_asm);
            a1->first = p->next;
#ifdef __DEBUG__
            trace_subtree(p);
            trace_subtree(p->next);
#endif
            if (p->next->kind != node_ref) {
                trace_subtree(p);
            }
            assert_kind(p->loc, p->next, node_ref);
            if (p->next->first->kind == node_const) {
                a1->opcode = opcode_from_const[p->op];
            } else {

                assert_kind(p->loc, p->next->first, node_var);
                source = p->next;
                a1->opcode = opcode_from_store[p->op];
                a1->dest = dest_w;
            }
            if (a1->opcode == opcode_undefined) {
                trace_subtree(p);
            }
            jal_assert(p->loc, a1->opcode != opcode_undefined);
            break;
        }

    case op_shift_left:
    case op_shift_right:{
            assert_kind(p->loc, p->next, node_ref);
            assert_kind(p->loc, p->next->first, node_const);
            assert_kind(p->loc, p->next->first->value, node_value);
            assert_kind(p->loc, p->first, node_ref);
            assert_kind(p->loc, p->first->first, node_var);

            if (p->next->first->value->x == 1) {

                source = p->first;
                q1->first = a1 = new_node(p->loc, node_asm);
                a1->opcode = opcode_clrc;
                q2->first = a2 = new_node(p->loc, node_asm);
                a2->first = p->first;
                a2->opcode = opcode_from_store[p->op];
                cassert(a2->opcode != opcode_undefined);
                a2->dest = dest_w;

            } else if (p->next->first->value->x == 4) {

                source = p->first;
                q1 = new_chain2(new_asm(p->loc, opcode_swapf, p->first, dest_w),
                                new_asm(p->loc, opcode_andlw,
                                new_const(new_value (type_universal,
                                ((p->op == op_shift_left) ? 0xF0 : 0x0F))), 0));
                q2 = q1->next;

            } else {
                snark(NULL);
            }
            break;
        }

    case op_mminus:{
            q1->first = code_node(p->first, my_args);
            q1->next = a1 = new_node(p->loc, node_asm);
            a1->opcode = opcode_sublw;
            a1->first = c1 = new_node(p->loc, node_const);
            c1->type = type_byte;
            c1->value = new_value(type_universal, 1);
            break;
        }

    case op_mnot:{
            q1->first = code_node(p->first, my_args);
            q1->next = a1 = new_node(p->loc, node_asm);
            a1->opcode = opcode_xorlw;
            a1->first = new_value(type_universal, -1);
            break;
        }

    default:{
            log((m, "n=%d op=%s", p->nr, op_name[p->op]));
            snark_node(p->loc, p);
            break;
        }
    }
    if (source != NULL) {
        q1->next = code_register_bank(source, q2);
        return q1;
    } else {
        return q1;
    }
}

tree code_op(formal_args)
{
    stack_guard;
    if (p == NULL)
        return NULL;
    assert_kind(NULL, p, node_op);
    assert_kind(NULL, p->type, node_type);

    if (p->type == type_bit) {
        return code_boolean(p, my_args);

    } else if (code_type_is(p, type_byte)) {
        return code_byte_op(p, my_args);

    }

    snark_node(p->loc, p);
    return NULL;
}


/* generate code for the referenced object */
tree code_ref(formal_args)
{
    stack_guard;
    assert_kind(NULL, p, node_ref);
    assert_pointer(NULL, p->first);
#ifdef __DEBUG__
    trace_subtree(p);
#endif

    if ((p->first->indirect)
        /* 04-25 added */
        || (p->indirect)
        ) {
        /* pass indirection address */
        make_transfer(p->first);
        forget_w();
        return new_asm(p->loc, opcode_movlw, new_ref(NULL, ref_var, p->first->jtaddress, NULL), 0);
    } else if (p->first->type == type_bit) {
        return code_boolean(p->first, my_args);

    } else if (code_type_is(p, type_byte)) {
        switch (p->first->kind) {

        case node_const:{
                return new_asm(p->loc, opcode_movlw, p->first, 0);
                break;
            }

        case node_var:{
                return code_register_bank(p, new_asm(p->loc, opcode_movf, p->first, dest_w));
                break;
            }

        case node_w:{
                return p->first;
            }

        default:{
                snark_node(p->loc, p->first);
                break;
            }
        }
    }

    snark_node(p->loc, p);
    return NULL;
}

tree code_if(formal_args)
{
    tree then_part;
    tree else_part;
    int then_size;
    int else_size;
    tree then_label = new_label(NULL, "if_%d_th", p->nr, "");
    tree else_label = new_label(NULL, "if_%d_el", p->nr, "");
    tree beyond_label = new_label(NULL, "if_%d_by", p->nr, "");
    stack_guard;

    forget_w();
    then_part = code_node(p->first, my_args);
    forget_w();
    else_part = code_node(p->next, my_args);
    then_size = code_size(then_part);
    else_size = code_size(else_part);
    forget_w();


    /* degenerate cases: take only the then or else clause */
    if (node_is_constant(p->condition, 1)) {
        return then_part;
    }
    if (node_is_constant(p->condition, 0)) {
        return else_part;
    }

    /* both parts are empty: don't bother to generate any code */
    if ((then_part == NULL) && (else_part == NULL)) {
        return NULL;
    }

    /* empty else clause */
    if (else_part == NULL) {
        return
            new_chain3(code_boolean
                       (p->condition, then_label, true, (then_size == 1), beyond_label, false,
                        false), new_chain2(then_label, then_part), beyond_label);
    }

    /* empty then clause */
    if (then_part == NULL) {
        return
            new_chain3(code_boolean
                       (p->condition, beyond_label, false, false, else_label, true, false),
                       new_chain2(else_label, else_part), beyond_label);
    }

    /* general case */
    return new_chain4(code_boolean(p->condition, then_label, true, false, else_label, false, false),
                      new_chain3(then_label, then_part, code_jump(NULL, beyond_label, my_args)),
                      new_chain2(else_label, else_part), beyond_label);
}

tree code_while(formal_args)
{
    tree again_label = new_label(p->loc, "w_%d_ag", p->nr, "");
    tree body_label = new_label(NULL, "w_%d_bo", p->nr, "");
    tree beyond_label = new_label(NULL, "w_%d_be", p->nr, "");
    tree beyond_jump_label = new_label(NULL, "w_%d_jb", p->nr, "");
    tree body;
    stack_guard;

    forget_w();
    body = code_node(p->first, my_args);

    /* degenerate cases: never and forever loops */
    if (p->condition->kind == node_const) {

        if (p->condition->value->x == 0) {
            return NULL;

        } else {
            cassert(p->condition->value->x == 1);

            return new_chain3(again_label, body, code_jump(NULL, again_label, my_args));
        }
    }

    /* empty body */
    forget_w();
    if (body == NULL) {
        return new_chain4(again_label,
                          code_boolean(p->condition, body_label, true, (chip_without_code_pages),
                                       beyond_label, false, false), new_chain2(body_label,
                                        code_jump(NULL, again_label, my_args)), beyond_label);
    }

    /* general case */
    forget_w();
    return new_chain5(again_label,
                      code_boolean(p->condition, body_label, false, false, beyond_jump_label, true,
                                   (chip_without_code_pages)), new_chain2(beyond_jump_label,
                                   code_jump(NULL, beyond_label, my_args)),
                      new_chain3(body_label, body, code_jump(NULL, again_label, my_args)),
                      beyond_label);
}

tree code_bit_assign(formal_args)
{
    stack_guard;
    assert_pointer(NULL, p);
    assert_pointer(NULL, p->first);
    assert_pointer(NULL, p->next);
    if (verbose_coder) {
        log((m, "code_bit_assign %d %s", p->nr, node_name[p->kind]));
    }
    assert_kind(NULL, p->first, node_ref);
    assert_kind(NULL, p->first->first, node_var);
    assert_pointer(NULL, p->first->first->address);

    if ((p->next->kind == node_ref)
        && (p->next->first->kind == node_const)
        ) {

        /* plain clear or set */
        return code_register_bank(p->first,
                   new_asm(p->loc, ((p->next->first->value->x == 0) ? 
                                   opcode_bcf : opcode_bsf), p->first, 0));

    } else {

        /* general assignment */
        tree d = follow(p->first);
        assert_kind(p->loc, d, node_ref);
        d = d->first;
        assert_kind(p->loc, d, node_var);

        if ((d->is_volatile)
            || (uses(d, p->next))
            ) {

            /* volatile version: set or clear */
            tree true_label = new_label(NULL, "ass_%d_t", p->nr, "");
            tree false_label = new_label(NULL, "ass_%d_f", p->nr, "");
            tree beyond_label = new_label(NULL, "ass_%d_b", p->nr, "");
            return
                new_chain7(code_boolean
                           (p->next, true_label, true, false, false_label, false, false),
                           true_label, code_register_bank(p->first,
                           new_asm(p->loc, opcode_bsf, p->first, 0)),
                           code_code_page(beyond_label,
                           new_asm(p->loc, opcode_goto, beyond_label, 0)),
                           false_label, code_register_bank(p->first,
                                    new_asm(p->loc, opcode_bcf, p->first, 0)), beyond_label);

        } else {
            tree q;

            /* faster version: clear first, set if needed */
            tree true_label = new_label(NULL, "ass_%d_t", p->nr, "");
            tree false_label = new_label(NULL, "ass_%d_f", p->nr, "");
            q = new_chain5(code_register_bank(p->first, new_asm(p->loc, opcode_bcf, p->first, 0)
                           ), code_boolean(p->next, true_label, true, (chip_without_code_pages),
                             false_label, false, false), true_label,
                           code_register_bank(p->first, new_asm(p->loc, opcode_bsf, p->first, 0)
                           ), false_label);
#ifdef __DEBUG__
            trace_subtree(q);
#endif
            return q;
        }
    }

}

tree code_byte_assign(formal_args)
{
    tree q, store;
    stack_guard;
    if (verbose_coder)
        log((m, "code_byte_assign %d %s", p->nr, node_name[p->kind]));
    assert_kind(p->loc, p->first, node_ref);
    assert_pointer(p->loc, p->first->first);

    /* clear? */
    q = try_clrf(p);
    if (q != NULL)
        return q;

    /* one of the one or two opcode file-to-file operators? */
    try_file_op(p, op_plus, 1, opcode_nop, opcode_incf);
    try_file_op(p, op_minus, -1, opcode_nop, opcode_incf);
    try_file_op(p, op_plus, -1, opcode_nop, opcode_decf);
    try_file_op(p, op_minus, 1, opcode_nop, opcode_decf);
    try_file_op(p, op_shift_right, 1, opcode_clrc, opcode_rrf);
    try_file_op(p, op_shift_left, 1, opcode_clrc, opcode_rlf);

    /* can use one of the x += n type opcodes? */
    if ((p->next->kind == node_op)
        && (p->next->op != op_minus)
        && (p->next->first->kind == node_ref)
        && (p->next->first->first == p->first->first)
        && (p->type == type_byte)
        ) {
        op_t opcode = opcode_from_store[p->next->op];
        if ((opcode != 0)
            && (opcode != opcode_rrf)
            && (opcode != opcode_rlf)
            && (code_has(opcode, field_dest))
            ) {
            q = code_node(p->next->next, my_args);
            store = new_asm(p->loc, opcode, p->first, dest_f);
            return new_chain2(q, code_register_bank(p->first, store));
        }
    }

    /* general assignment */
    q = new_node(p->loc, node_chain);
    q->first = code_node(p->next, my_args);
    if (!(p->first->first->kind == node_w)) {
        assert_kind(p->loc, p->first->first, node_var);
        q->next = code_store(p->first, my_args);
    }
    return q;
}

tree code_assign(formal_args)
{
    stack_guard;
    assert_kind(NULL, p, node_assign);
    assert_kind(NULL, p->first->type, node_type);
    assert_kind(NULL, p->first, node_ref);

    if (p->first->type == type_bit) {
        return code_bit_assign(p, my_args);

    } else if (p->first->type == type_byte) {
        return code_byte_assign(p, my_args);
    }

    /* should never get here */
    snark_node(p->loc, p->first);
    return NULL;
}

#define max_virtual_routines 1000
int n_virtual_routines = 0;
tree virtual_routines[max_virtual_routines];
tree call_vectors = NULL;

tree code_procedure(formal_args)
{
    tree ret;
    stack_guard;
    assert_kind(p->loc, p, node_procedure);
    if (verbose_coder)
        log((m, "code_procedure"));

    forget_w();                 /* ???? */

#ifdef __DEBUG__
    log((m, "t=%d f=%d c=%d cwcp=%d", target_chip, t_16f84, t_16c84, chip_without_code_pages));
#endif
    if (chip_with_call_vectors) {
        call_vectors =
            new_chain3(call_vectors, p->call_label,
                       code_code_page(p->address, new_asm(p->loc, opcode_goto, p->address, 0)));
    }

    if (p == interrupt_service_routine) {
        ret = NULL;             /* provided in another way */
    } else if ((p->is_chained) && (!p->has_return)) {
        ret = NULL;
    } else {
        if (target_cpu == pic_12) {
            ret = new_asm(NULL, opcode_retlw, new_const(new_value(type_universal, 0)), 0);
        } else {
            ret = new_asm(NULL, opcode_return, NULL, 0);
        }
    }

    /* append a return statement */
    p->first =
        new_chain7(p->tlabel, code_node(p->transfer1, my_args),
                   ((chip_with_call_vectors) ? NULL : p->call_label),
                   ((p->last_page) ? new_asm(NULL, opcode_clrf, new_value(type_universal, 10),
                                             0) : NULL), code_node(p->first, my_args),
                   code_node(p->transfer2, my_args), ret);

#ifdef __DEBUG__
    log((m, "n=%s iv=%d ht=%d", p->name, (int) p->is_virtual, (int) p->has_transfer));
#endif
    /* remember all virtual routines */
    if ((p->is_virtual) && (p->has_transfer)) {
        if (n_virtual_routines == max_virtual_routines) {
            fatal(p->loc, (m, "too many virtuals"));
        }
        virtual_routines[n_virtual_routines] = p;
        n_virtual_routines++;
    }

    /* must be on the last page? */
    if (p->last_page) {
        first_page_routines = new_chain2(first_page_routines, p);
        p = NULL;
    }

    return p;
}

tree code_node(formal_args)
{
    stack_guard;
    if (p == NULL)
        return NULL;
    if (verbose_coder)
        log((m, "code_node %d", p->nr));

    switch (p->kind) {

        /* error */
    case node_error:{
            fatal(p->loc, (m, "error pragma encountered during code generation"));
        }

    case node_asm:
        /* special case for f877 */
        /* if( ( p->opcode == opcode_page ) && ( target_chip == t_16f877 ) ) {
           return new_chain2( new_asm( NULL, opcode_nop, NULL, 0),
           new_asm( NULL, opcode_nop, NULL, 0));
           } */
        /* otherwise fallthrough */
        /* do nothing */
    case node_label:
        forget_w();
    case node_test:
    case node_org:
    case node_precall:{
            return p;
            break;
        }

#define code_kind( x, y ) case x : { return y( p, my_args ); break; }
        code_kind(node_op, code_op)
            code_kind(node_ref, code_ref)
            code_kind(node_if, code_if)
            code_kind(node_while, code_while)
            code_kind(node_assign, code_assign)
            code_kind(node_procedure, code_procedure)
#undef code_kind
            /* generate code for the parts, don't change the structure */
    case node_chain:{
            p->first = code_node(p->first, my_args);
            p->next = code_node(p->next, my_args);
            if ((p->first != NULL)
                && (p->first->kind == node_decl)
                && (p->first->first != NULL)
                && (p->first->first->kind == node_procedure)
                ) {
                tree f = p->first;
                tree n = p->next;
                p->first = n;
                p->next = new_chain2(f, NULL);  /* why? */

            }
            return p;
            break;
        }

        /* generate code only for a procedure declaration */
    case node_decl:{
            copy_master_address(p->first);
            if ((p->first != NULL)
                && (p->first->kind == node_procedure)) {
                if (p->first == interrupt_service_routine) {
                    p->first = NULL;
                } else {
                    forget_w();
                    p->first = code_node(p->first, my_args);
                    forget_w();
                }
#ifdef __DEBUG__
                if (p->first == NULL)
                    return NULL;
#endif
            }
            p->next = code_node(p->next, my_args);
            return p;
            break;
        }

    case node_call:{
            opcode_t opcode = (p->is_chained ? opcode_goto : opcode_call);
            if (p->indirect) {
                int n = page_instruction_size(0, true);
                if (target_cpu == pic_16) {
                    n = 1;
                }
                return new_chain2((p->first, ((p->x == 0)
                               ? new_asm(p->loc, opcode_movfw, p->first, 0)
                              : ((n == 0) ? new_asm(p->loc, opcode_incf, p->first, dest_w)
                              : new_chain2(new_asm (p->loc, opcode_movlw,
                                new_const(new_value (type_universal, n + 1)), 0),
                               code_register_bank(p->first,
                               new_asm(p->loc, opcode_addwf, p->first, dest_w)))))),
                        code_code_page(transfer_label, new_asm(p->loc, opcode, transfer_label, 0)));
            } else {
                tree x;
                x = code_code_page(p->first->call_label,
                                   new_asm(p->loc, opcode, p->first->call_label, 0));
                forget_w();
                return x;
            }
            forget_w();
            break;
        }

    case node_return:{
            if (p->is_chained) {
                return NULL;
            } else {
                return code_code_page(p->first->ret,
                                      new_asm(p->loc, opcode_goto, p->first->ret, 0));
            }
            break;
        }

    default:{
            snark_node(p->loc, p);
            break;
        }
    }

    snark(NULL);
    return NULL;
}

tree code_tree(tree p)
{
    return code_node(p, NULL, false, false, NULL, false, false);
}

tree code_interrupt(void)
{
    if (interrupt_service_routine == NULL) {
        return NULL;

    } else {
        return code_procedure(interrupt_service_routine, NULL, false, false, NULL, false, false);
    }
}

#define cv( x ) new_const ( new_value( type_universal, x ) )

tree code_interrupt_header(tree target)
{
    if (interrupt_service_routine == NULL) {
        return NULL;
    }

	if( interrupt_service_routine->is_raw ){
		return new_asm( NULL, opcode_goto,   target, 0 );
	}
 
    if (target_cpu == sx_12) {
        return NULL;
    }
    return new_chain10(new_asm(NULL, opcode_movwf, cv(interrupt_save_w), 0),
                       new_asm(NULL, opcode_swapf, cv(3), dest_w), 
					   new_asm(NULL, opcode_clrf, cv(3), 0), 
					   new_asm(NULL, opcode_movwf, cv (interrupt_save_p), 0),
                       new_asm(NULL, opcode_movfw, cv(10), 0),
					   new_asm(NULL, opcode_movwf, cv(interrupt_save_p + 1), 0),
                       new_asm(NULL, opcode_clrf, cv(10), 0), 
					   new_asm(NULL, opcode_movfw, cv(4), 0),
                       new_asm(NULL, opcode_movwf, cv(interrupt_save_p + 2), 0), 
                       new_asm(NULL, opcode_goto, target, 0));
}

tree code_interrupt_trailer(void)
{
    if (interrupt_service_routine == NULL) {
        return NULL;
    }
    if(interrupt_service_routine->is_raw) {
        return new_asm(NULL, opcode_retfie, NULL, 0);
    }
    if (target_cpu == sx_12) {
        return NULL;
    }
    return new_chain9(new_asm(NULL, opcode_movfw, cv(interrupt_save_p + 2), 0),
                      new_asm(NULL, opcode_movwf, cv(4), 0), 
                      new_asm(NULL, opcode_movfw, cv(interrupt_save_p + 1), 0),
                      new_asm(NULL, opcode_movwf, cv(10), 0), 
                      new_asm(NULL, opcode_swapf, cv(interrupt_save_p), dest_w),
                      new_asm(NULL, opcode_movwf, cv(3), 0), 
                      new_asm(NULL, opcode_swapf, cv(interrupt_save_w), dest_f),
                      new_asm(NULL, opcode_swapf, cv(interrupt_save_w), dest_w), 
                      new_asm(NULL, opcode_retfie, NULL, 0));
}

/* depends on the size of the code_interrupt_header */
int transfer_start(void)
{
    switch (target_chip) {
    case t_16c84:
    case t_16f88:
    case t_16f84:
    case t_16f628:
    case t_12f629:
    case t_12f675:
    case t_16f877:
    case t_16f876:             /* Added 16f876 & 16f873 by Javi 2003-03-01 */
    case t_16f873:             /* Added 16f876 & 16f873 by Javi 2003-03-01 */
		{
            if (interrupt_service_routine == NULL) {
                return target_origin + 4;
            } else {
				if( interrupt_service_routine->is_raw ){
					return target_origin + 5;
				} else {
					return  target_origin + 14;
				}
            }
        }
    case t_18f242:
    case t_18f252:
    case t_18f442:
    case t_18f452:
		{ /* for pic18's the isr vector starts at 8 so we add 4 to the values
			 used for the pic16's */
            if (interrupt_service_routine == NULL) {
                return target_origin + 4;
            } else {
				if( interrupt_service_routine->is_raw ){
					return target_origin + 6;
				} else {
					return  target_origin + 15;
				}
            }
        }

    case t_sx18:
    case t_sx28:{
            if (interrupt_service_routine == NULL) {
                return 8;
            } else {
                snark(NULL);
                return 14;
            }
        }

    case t_12c509a:
    case t_12c508:{
            return target_origin + 4;
        }

    default:{
            snark(NULL);
            break;
        }
    }

    /* dummy to suppress warnings */
    return 0;
}

void code(tree * p)
{
    tree main, interrpt, transfers;
    tree main_label, interrupt_label;
    stack_guard;

    /* must be done before all coding */
    init_variables();
    if (target_cpu != pic_16) {
        pic_pclath = new_value(type_universal, 10);
        pic_flags = new_value(type_universal, 3);
    } else {
        pic_pclath = new_value(type_universal, 0xFFA);
        pic_flags = new_value(type_universal, 0xFD8);
    }
    pic_flags->name = new_string("PIC flags");
    pic_flag_zero = new_value(type_universal, 2);
    pic_flag_carry = new_value(type_universal, 0);
    pic_flag_digit_carry = new_value(type_universal, 1);
    transfer_label = new_label(NULL, "__indirection", 0, "");
    transfer_address = transfer_start();

    /* code the main program */
    forget_w();
    main = code_tree(*p);

    /* append the goto for SX */
    if (target_cpu == sx_12) {
        main =
            new_chain3(main, new_org(target_last_rom),
                       new_asm(NULL, opcode_goto, new_const(new_value(type_universal, 3)), 0));
    }

    /* code the interrupt routine */
    forget_w();
    interrpt = code_interrupt();

    /* code the transfer vectors */
    transfers = code_transfers(transfer_label);

    /* combine it all */
    interrupt_label = new_label(NULL, "__interrupt", 0, "");
    main_label = new_label(NULL, "__main", 0, "");
    (*p) =
        new_chain6(((target_cpu == sx_12) ?
                                new_chain4(new_org(0),
                                new_asm(NULL, opcode_goto, interrupt_label, 0), new_org(1),
                                new_asm(NULL, opcode_goto, main_label, 0)) :
/*                    new_chain4(new_org(target_origin + 0), new_asm(NULL, opcode_goto, main_label, 0), new_org(4),*/
                    new_chain4(new_org(target_origin + 0), new_asm(NULL, opcode_goto, main_label, 0), new_org(target_origin + 4),
                               code_interrupt_header(interrupt_label))),
                   new_chain2(new_org(transfer_start()), transfers), first_page_routines,
                   call_vectors, new_chain3(interrupt_label, interrpt, code_interrupt_trailer()),
                   new_chain2(main_label, main));

    total_code_size = code_size(*p);
}


/********** page optimizations ***********/

#define phase_mark_linear   (-3)
#define phase_mark_targets  (-2)
#define phase_remove        (-1)
#define unknown             (-2)
#define no_bank             (-1)
int current_bank;

int bank_from_argument(tree q)
{
    int x, z;

    x = z = 0;

    while (q->kind == node_ref) {
        q = q->first;
    }
    switch (q->kind) {
    case node_var:{

            if (q->address == NULL) {
                snark_node(q->loc, q);
            }
            assert_pointer(q->loc, q->address);
            assert_pointer(q->loc, q->address->first);
            x = q->address->first->x;
            jal_assert(q->loc, x < 4096);
            break;
        }
    case node_value:{
            x = q->x;
            break;
        }
    case node_const:{
            assert_pointer(NULL, q->value);
            x = q->value->x;
            break;
        }
    default:{
            snark_node(q->loc, q);
        }
    }
    if (target_cpu == pic_14) {
        z = (x >> 7);  
    } else if (target_cpu == pic_16) {
      if ((x > 0x07F) && (x < 0xF80) ) {
         z = ( x >> 8 ) ;
      }
      else
      {
         return -1 ;
      }
    } else {
        if (x < 32) {  /* was 16 */
            return 0;   /* was -1 */
        }
        z = ((x >> 4) & 0x7);
    }
    return z;
}

void opt_bank(tree p, int phase)
{
    stack_guard;
    if (p == NULL)
        return;
    if (p->fizzle)
        return;

    if ((false) | (verbose_optimizer)) {
        log((m, "optimize_bank node nr=%04d kind=%d %s cb=%d nb=%d", p->nr, p->kind,
             node_name[p->kind], current_bank, p->bank));
    }

    switch (p->kind) {

    case node_precall:
    case node_decl:
    case node_test:
    case node_type:
    case node_var:
    case node_const:
    case node_w:
    case node_org:{
            /* 2001-04-07 : do nothing! */
            break;

            if (p->first->kind == node_procedure) {
                opt_bank(p->first, phase);
            }
            /* ignore rest */
            break;
        }

    case node_label:{
            switch (phase) {
            case phase_mark_linear:{
                    p->bank = current_bank;
                    break;
                }
            case phase_mark_targets:{
                    break;
                }
            case phase_remove:{
                    current_bank = p->bank;
                    break;
                }
            }
            break;
        }

    case node_procedure:{
            opt_bank(p->first, phase);
            break;
        }

    case node_chain:{
            opt_bank(p->first, phase);
            opt_bank(p->next, phase);
            break;
        }

    case node_asm:{
            switch (phase) {
            case phase_mark_linear:{
                    if (p->opcode == opcode_bank) {
                        int bank = bank_from_argument(p->first);
                        if ((bank == no_bank)
                            && (!p->sacred)
                            ) {
                            p->fizzle = true;
                        } else {
                            current_bank = bank;
                        }
                    }
                    if (p->opcode == opcode_call) {
                        current_bank = unknown;
                    }
                    p->bank = current_bank;
                    break;
                }
            case phase_mark_targets:{
                    if ((p->opcode == opcode_goto)
                        || (p->opcode == opcode_call)
                        ) {
                        tree g = p->first;
                        while (g->kind == node_ref) {
                            g = g->first;
                        }
                        while (g->kind == node_const) {
                            g = g->value;
                        }
                        if (g->kind == node_procedure) {
                            g = g->call_label;
                        }
                        if (g->kind != node_value) {
                            if (g->kind != node_label) {
                                trace_subtree(p);
                            }
                            assert_kind(NULL, g, node_label);
                            if (g->bank != p->bank) {
                                g->bank = unknown;
                            }
                        }
                    }
                    break;
                }
            case phase_remove:{
                    if (p->opcode == opcode_bank) {
                        if ((p->bank == current_bank)
                            && (p->bank != unknown)
                            && (!p->sacred)
                            ) {
                            p->fizzle = true;
                        }
                    }
#ifdef __DEBUG__
                    log((m, "n=%d b=%d cb=%d fiz=%d", p->nr, p->bank, current_bank,
                         (int) p->fizzle));
#endif
                    current_bank = p->bank;
                    break;
                }
            default:{
                    snark(NULL);
                    break;
                }
            }
            break;
        }

    default:{
            snark_node(p->loc, p);
            break;
        }
    }
}

void optimize_bank(tree * p)
{
    if (optimize_bank_instructions) {
        current_bank = unknown;
        opt_bank(*p, phase_mark_linear);
        current_bank = unknown;
        opt_bank(*p, phase_mark_targets);
        current_bank = unknown;
        opt_bank(*p, phase_remove);
    }
}

/******************************************************************************
*
* page instruction elimination
*
* This phase removes redundant page instructions, and replaces the remaining
* page instructions with the appropriate target-specific instructions.
*
* call:
*    void eliminate_page( tree *p )
*
* Four sweeps are done:
* kill:      All non-sacred page instructions are marked fizzle.
* mark:      The location_counter location for all instructions
*            is is determined and (for labels) stored in p->x.
* restore:   Each page instruction that turns out to be needed
*            (determined from the p->x of the target label)
*            is restored by removing the fizzle.
* replace:   The non-fizzled page instructions are replaced
*            by their implementation on the target chip.
*
* The mark and restore sweeps are repeated untill no more page
* instructions need to be resurrected.
*
* A (not fizzeled) page N is translated to
* the following number of instructions:
* - 0 for 16c84, 16f84, 12c508   (generates no code at all)
* - 1 for sx18, sx28             (native bank instruction)
* - 1 for 12c509                 (bcf/bsf)
* - 1 for 16f877 when N == 0     (clrf)
* - 2 for 16f877 when N != 0     (2 * bcf/bsf)
*
* The restore sweep maintains the current page like this:
* - the current page starts as 0
* - a page instruction sets the current page
* - a conditional skip instruction makes the next instruction conditional
* - an unconditional jump sets the current page <free>
* - a label, with the current page <free>, sets the current page
* - a call sets the current page <unknown>
* A page instruction is restored when it refers to a label that has
* a page that is not equal to the current page.
*
******************************************************************************/

/* the sweeps */
#define page_sweep_kill       (-1)
#define page_sweep_mark       (-2)
#define page_sweep_restore    (-3)
#define page_sweep_replace    (-4)

/* the current setting of the page bits */
#define page_is_free          (-1)
#define page_is_unknown       (-2)
int page_current_page;

int page_location_counter;
int page_instructions_restored;
boolean page_conditional;

tree inval = NULL;



/* return the instruction(s) that implement the page on the target */
tree page_instructions(tree p)
{
    stack_guard;

    assert_kind(NULL, p, node_asm);
    jal_assert(p->loc, p->opcode == opcode_page);

    switch (target_chip) {

    case t_12c508:{
            return NULL;
            break;
        }

    case t_18f242:
    case t_18f252:
    case t_18f442:
    case t_18f452:{
            return NULL;
            break;
        }

    case t_12c509a:{
            return inval;
            break;
        }

    case t_sx18:
    case t_sx28:{
            return p;
            break;
        }

    case t_16c84:
    case t_16f84:
    case t_12f629:
    case t_12f675:
    case t_16f628:{
            return inval;
            break;
        }

    case t_16f877:
    case t_16f876:             /* Added 16f876 & 16f873 by Javi 2003-03-01 */
    case t_16f873:             /* Added 16f876 & 16f873 by Javi 2003-03-01 */
        {
            if (page_bits((int) inval) == 0) {
                return inval;
            } else {
                return inval;
            }
            break;
        }

    default:{
            snark(NULL);
        }
    }

    /* dummy to suppress warnings */
    return 0;
}



/*
 * traverse the entire tree
 * and perform the operations of the indicated sweep
 * as described in the introduction
 *
 * Actually most work is done regardless of the current sweep.
 */
tree eliminate_page_visit_node(tree p, int sweep)
{
    boolean next_conditional;
    stack_guard;

    /* empty nodes need no processing */
    if (p == NULL) {
        return NULL;
    }

    /* logging ... */
    if (verbose_optimizer) {
        log((m, "eliminate_page_visit_node nr=%04d kind=%d %s", p->nr, p->kind,
             node_name[p->kind]));
    }
    switch (p->kind) {

        /* don't look deeper */
    case node_precall:
    case node_type:
    case node_var:
    case node_const:
    case node_w:{
            return p;
            break;
        }

        /* do look deeper */
    case node_procedure:
    case node_decl:{
            p->first = eliminate_page_visit_node(p->first, sweep);
            return p;
            break;
        }
    case node_chain:{
            p->first = eliminate_page_visit_node(p->first, sweep);
            p->next = eliminate_page_visit_node(p->next, sweep);
            return p;
            break;
        }

        /* note the new location counter */
    case node_org:{
            page_location_counter = (int) inval;
            return p;
            break;
        }

        /* maintain the location counter */
    case node_test:{
            page_location_counter++;
            return p;
            break;
        }

    case node_label:{

            /* save the current location counter */
            p->x = page_location_counter;

            /* update the current_page */
            if (page_current_page == page_is_free) {
                page_current_page = page_bits(page_location_counter);
            }

            return p;
            break;
        }

    case node_asm:{

            /* maintain the location counter */
            if (!p->fizzle) {
                if (p->opcode == opcode_bank) {
                    page_location_counter += page_instruction_size((int) p->first, p->fizzle);
                } else {
                    page_location_counter++;
                }
            }

            /* restore when needed */
            if (sweep == page_sweep_restore) {
                switch (p->opcode) {

                    /* check the target */
                case opcode_goto:
                case opcode_call:{
                        if ((p->fizzle)
                            && (page_bits((int) inval)
                                != page_current_page)
                            ) {
                            p->fizzle = false;
                            page_instructions_restored++;
                        }
                        break;
                    }

                    /* no action for other opcodes */
                default:{
                        break;
                    }
                }
            }

            next_conditional = false;
            switch (p->opcode) {

                /* invalidate current_page */
            case opcode_call:{
                    page_current_page = page_is_unknown;
                    break;
                }

                /* set current_page */
            case opcode_page:{
                    if (!p->fizzle) {
                        page_current_page = page_bits((int) inval);
                    }

                    /* in kill sweep: fizzle them all */
                    if (sweep == page_sweep_kill) {
                        if (!p->sacred) {
                            p->fizzle = true;
                        }
                    }
                    break;
                }

                /* when not conditional: current_page is no longer important */
            case opcode_goto:
            case opcode_return:
            case opcode_retfie:
            case opcode_retlw:{
                    if (!page_conditional) {
                        page_current_page = page_is_free;
                    }
                    break;
                }

            case opcode_btfsc:
            case opcode_btfss:
            case opcode_decfsz:
            case opcode_incfsz:{
                    next_conditional = true;
                    break;
                }

                /* no action for other opcodes */
            default:{
                    break;
                }
            }

            page_conditional = next_conditional;
            break;
        }

    default:{
            snark_node(p->loc, p);
            break;
        }
    }

    /* dummy to suppress warnings */
    return NULL;
}

void eliminate_page(tree * p)
{
    stack_guard;

    /* mark all page instructions fizzle */
    (*p) = eliminate_page_visit_node(*p, page_sweep_kill);

    /* loop until no more page instructions are restored */
    page_instructions_restored = 1;
    while (page_instructions_restored > 0) {

        /* mark the location counter locations in the p->x of the labels */
        page_location_counter = 0;
        (*p) = eliminate_page_visit_node(*p, page_sweep_mark);

        /* restore page instructions */
        page_conditional = false;
        page_instructions_restored = 0;
        page_current_page = 0;
        page_location_counter = 0;
        (*p) = eliminate_page_visit_node(*p, page_sweep_restore);
    }

    /* replace the instructions */
    (*p) = eliminate_page_visit_node(*p, page_sweep_replace);
}
