/* tree representation
   Copyright (C) 1998, 1999, 2000, 2001, 2002, 2003
   Wouter van Ooijen

This file is part of jal.

jal is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

jal is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with jal; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include "stdhdr.h"
#include "global.h"
#include "target.h"
#include "errorlh.h"
#include "stacksg.h"
#include "reswords.h"
#include "cstringf.h"
#include "treerep.h"
#include "treetools.h"
#include "parser.h"
#include "regalloc.h"
#include "assemble.h"
#include "scanner.h"
#include "codegen.h"

/* the operation types for node_op */
string op_name[] = { "undefined",
    "&", "|", "^",
    "<", ">", "<=", ">=",
    "==", "!=", "-", "+",
    "<<", ">>",
    "*", "/", "%",
    "!", "-", "+",
    "test end", "test assert", "?"
};
string op_label[] = { "UNDEFINED",
    "and", "or", "xor",
    "smaller", "larger", "not_larger", "not_smaller",
    "equal", "not_equal", "minus", "plus",
    "shift_left", "shift_right",
    "times", "divide", "remainder",
    "not", "m_minus", "m_plus",
    "TEST_END", "TEST_ASSERT", "check"
};
int op_inverted[] = { 0,
    op_and, op_or, op_xor,
/*    op_larger_or_equal, op_smaller_or_equal, op_larger, op_smaller, */
	0,0,0,0,
    op_equal, op_not_equal, 0, op_plus,
    0, 0,
    op_times, 0, 0,
    0, 0, 0
};
boolean op_commutates[] = { false,
    true, true, true,
    false, false, false, false,
    true, true, false, true,
    false, false,
    true, false, false,
    false, false, false,
    false, false, true
};
boolean is_monop(op_t x)
{
    stack_guard;
    return (x == op_mnot) || (x == op_mminus) || (x == op_mplus);
}

int operator_index(string s, boolean monop)
{
    int i = 1;
    stack_guard;
    while (op_name[i][0] != '\0') {
        if ((is_monop(i) == monop)
            && (string_match(s, op_name[i]))
            ) {
            return i;
        }
        i++;
    }
    return 0;
}

/* the assembler codes for a node_asm */
int opcode_value_16[] = { 0,
    0x0F00, 0x0B00, 0xEC00, 0x0004, 0xEF00,
    0x0900, 0x0E00, -1, 0x0010, 0x0C00,
    0x0012, 0x0003, 0x0800, -1, 0x0A00,
    0x2400, 0x1400, 0x6A00, 0x0E00, 0x1C00,
    0x2000, 0xEE00,
    0x0400, 0x2C00, 0x2800, 0x3C00, -1,
    0x1000, 0x5000, 0x6E00, 0x0000,
    0x3400, 0x3000, 0x5C00, 0x3800, 0x1800,
    0x4400, 0x4000, 0x5400, 0x5800,
    0x9000, 0x8000, 0xB000, 0xA000,
    0xA4D8, 0xB4D8, 0x84D8, 0x94D8,
    0xA0D8, 0xB0D8, 0x80D8, 0x90D8,
    0xA2D8, 0xB2D8, 0x82D8, 0x92D8,
    0x0720, -1, -1, -1, -1,
    0x0D00, 0x0200, 0x6200, 0x6400, 0x6000,
    0x4C00, 0x4800,
    0x6C00, 0x6800, 0x6600, 0x7000,
    0x0007, 0x0006, 0x0005, 0x00FF,
    0x0008, 0x0009, 0x000A, 0x000B,
    0x000C, 0x000D, 0x000E, 0x000F,
    0x0100,
    0xE200, 0xE600, 0xE300, 0xE700, 0xE500,
    0xE100, 0xE400, 0xD000, 0xE000, 0xD800,
    0xF000, 0,
    0xF000,
    0, 0
};
int opcode_value_12[] = { 0,
    -1, 0x0E00, 0x0900, 0x0004, 0x0A00,
    0x0D00, 0x0C00, 0x0002, 0x000E, 0x0800,
    0x000C, 0x0003, -1, 0x0000, 0x0F00,
    0x01C0, 0x0140, 0x0060, 0x0040, 0x0260,
    -1 , -1,
    0x00E0, 0x02C0, 0x02A0, 0x03C0, 0x000F,
    0x0100, 0x0200, 0x0020, 0x0000,
    0x0340, 0x0300, 0x0080, 0x0380, 0x0180,
    -1, -1, -1, -1,
    0x0400, 0x0500, 0x0600, 0x0700,
    0x0743, 0x0643, 0x0543, 0x0443,
    0x0703, 0x0603, 0x0503, 0x0403,
    0x0723, 0x0623, 0x0523, 0x0423,
    0x0200, 0x0220, 0x0018, 0x0010, -1,
    -1, -1, -1, -1, -1,
    -1, -1,
    -1, -1, -1, -1,
    -1, -1, -1, -1,
    -1, -1, -1, -1,
    -1, -1, -1, -1,
    -1,
    -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1,
    -1, 0,
    -1,
    0, 0
};
int opcode_value_14[] = { 0,
    0x3E00, 0x3900, 0x2000, 0x0064, 0x2800,
    0x3800, 0x3000, 0x0062, 0x0009, 0x3400,
    0x0008, 0x0063, 0x3C00, 0x0060, 0x3A00,
    0x0700, 0x0500, 0x0180, 0x0103, 0x0900,
    -1 , -1,
    0x0300, 0x0B00, 0x0A00, 0x0F00, -1,
    0x0400, 0x0800, 0x0080, 0x0000,
    0x0D00, 0x0C00, 0x0200, 0x0E00, 0x0600,
    -1, -1, -1, -1,
    0x1000, 0x1400, 0x1800, 0x1c00,
    0x1D03, 0x1903, 0x1503, 0x1103,
    0x1C03, 0x1803, 0x1403, 0x1003,
    0x1C83, 0x1883, 0x1483, 0x1083,
    0x0800, 0x0880, -1, -1, -1,
    -1, -1, -1, -1, -1,
    -1, -1,
    -1, -1, -1, -1,
    -1, -1, -1, -1,
    -1, -1, -1, -1,
    -1, -1, -1, -1,
    -1,
    -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1,
    -1, 0,
    -1,
    0, 0
};
string opcode_name[] = { "undefined",
    "addlw", "andlw", "call", "clrwdt", "goto",
    "iorlw", "movlw", "option", "retfie", "retlw",
    "return", "sleep", "sublw", "tris", "xorlw",
    "addwf", "andwf", "clrf", "clrw", "comf",
    "addwfc", "lfsr",
    "decf", "decfsz", "incf", "incfsz", "retiw",
    "iorwf", "movf", "movwf", "nop",
    "rlf", "rrf", "subwf", "swapf", "xorwf",
    "rlncf", "rrncf", "subfwb","subwfb",
    "bcf", "bsf", "btfsc", "btfss",
    "skpz", "skpnz", "setz", "clrz",
    "skpc", "skpnc", "setc", "clrc",
    "skpdc", "skpndc", "setdc", "clrdc",
    "movfw", "tstf", "bank", "page", "HPAGE",
    "mullw", "mulwf", "cpfseq", "cpfsgt", "cpfslt",
    "dcfsnz", "infsnz",
    "negf", "setf", "tstfsz", "btg",
    "daw", "pop", "push", "reset",
    "tblrd*", "tblrd*+", "tblrd*-", "tblrd+*",
    "tblwt*", "tblwt*+", "tblwt*-", "tblwt+*",
    "movlb",
    "bc", "bn", "bnc", "bnn", "bnov",
    "bnz", "bov", "bra", "bz", "rcall",
    ";a2nd", "",
    ";f2nd",
    "BANKA", "BANKB"
};

/* opcode characterisation */
#define field_file         1
#define field_dest         2
#define field_label        4
#define field_const        8
#define field_tris        16
#define field_bit         32
#define field_fsr         64
#define field_flabel     128
#define field_fd          ( field_file | field_dest )
#define field_fb          ( field_file | field_bit )
#define field_ff          ( field_fsr | field_flabel )


int opcode_flags[] = { 0,
    field_const, field_const, field_label, 0, field_label,
    field_const, field_const, 0, 0, field_const,
    0, 0, field_const, field_tris, field_const,
    field_fd, field_fd, field_file, 0, field_fd,
    field_fd, field_ff,
    field_fd, field_fd, field_fd, field_fd, 0,
    field_fd, field_fd, field_file, 0,
    field_fd, field_fd, field_fd, field_fd, field_fd,
    field_fd, field_fd, field_fd, field_fd,
    field_fb, field_fb, field_fb, field_fb,
    0, 0, 0, 0,
    0, 0, 0, 0,
    0, 0, 0, 0,
    field_file, field_file, field_file, field_label, field_label,
    field_const, field_file, field_file, field_file, field_file,
    field_fd, field_fd,
    field_file, field_file, field_file, field_fb,
    0, 0, 0, 0,
    0, 0, 0, 0,
    0, 0, 0, 0,
    field_const,
    field_label, field_label, field_label, field_label, field_label,
    field_label, field_label, field_label, field_label, field_label,
    field_label, 0,
    field_ff,
    field_file, field_file
};
boolean code_has(int c, int f)
{
    return has(opcode_flags[c], f);
}

#define dest_w 'w'
#define dest_f 'f'

#define mode_none 0
#define mode_in   1
#define mode_out  2

boolean mode_has(int c, int m)
{
    return has(c, m);
}

/* assembler opcodes corresponding to a few high-level operators */
#define opcode_from_const \
   ( ( target_cpu == pic_16 ) ? opcode_from_const_16 : ( \
    ( target_cpu == pic_14 ) ? opcode_from_const_14 : opcode_from_const_12 ))
opcode_t opcode_from_const_12[] = { opcode_undefined,
    opcode_andlw, opcode_iorlw, opcode_xorlw,
    opcode_undefined, opcode_undefined, opcode_undefined,
    opcode_undefined,
    opcode_undefined, opcode_undefined, opcode_undefined,
    opcode_undefined,
    opcode_undefined, opcode_undefined,
    opcode_undefined, opcode_undefined,
    opcode_undefined, opcode_undefined,
    opcode_undefined, opcode_undefined, opcode_undefined,
    opcode_undefined, opcode_undefined, opcode_undefined
};
opcode_t opcode_from_const_14[] = { opcode_undefined,
    opcode_andlw, opcode_iorlw, opcode_xorlw,
    opcode_undefined, opcode_undefined, opcode_undefined,
    opcode_undefined,
    opcode_undefined, opcode_undefined, opcode_sublw, opcode_addlw,
    opcode_undefined, opcode_undefined,
    opcode_undefined, opcode_undefined,
    opcode_undefined, opcode_undefined,
    opcode_undefined, opcode_undefined, opcode_undefined,
    opcode_undefined, opcode_undefined, opcode_undefined
};
opcode_t opcode_from_const_16[] = { opcode_undefined,
    opcode_andlw, opcode_iorlw, opcode_xorlw,
    opcode_undefined, opcode_undefined, opcode_undefined,
    opcode_undefined,
    opcode_undefined, opcode_undefined, opcode_sublw, opcode_addlw,
    opcode_undefined, opcode_undefined,
    opcode_undefined, opcode_undefined,
    opcode_undefined, opcode_undefined,
    opcode_undefined, opcode_undefined, opcode_undefined,
    opcode_undefined, opcode_undefined, opcode_undefined
};
opcode_t opcode_from_store[] = { opcode_undefined,
    opcode_andwf, opcode_iorwf, opcode_xorwf,
    opcode_undefined, opcode_undefined, opcode_undefined,
    opcode_undefined,
    opcode_undefined, opcode_undefined, opcode_subwf, opcode_addwf,
    opcode_rlf, opcode_rrf,
    opcode_undefined, opcode_undefined,
    opcode_undefined, opcode_undefined,
    opcode_undefined, opcode_undefined, opcode_undefined,
    opcode_undefined, opcode_undefined, opcode_undefined
};

/* the node types */
string node_name[] = { "undefined",
    "chain", "decl", "ref", "value",
    s_procedure, s_var, s_type, s_const,
    "assign", s_if, s_while, s_for,
    "op", s_asm, "label", "precall",
    "return", "call", "test", "org",
    "error", "W", ""
};

/* types of references */
string ref_name[] = { "undefined",
    "const", "var", "own", "formal", "actual"
};

/********** node data:
 * all: nr, loc, kind, name
 * node_chain( first->*, next->[chain] )
 * node_assign( first->ref, next->exp, type )
 * node_ref( first->decl )
 * node_if( condition->exp, first->chain, next->chain )
 * node_while( condition->exp, first->chain )
 * node_for( var, start, end, step, first->chain, type )
 * node_op( op, first->exp, next->exp, type )
 * node_asm( opcode, first, next )
 * node_call( first->label )
 * node_label( first )
 * node_ref( first->object )
 * node_type( )
 * node_var( type, value )
 * node_const( type, value )
 * node_procedure( first, next, type )
 * object = type | const | var | procedure | function
 * exp    = chain | op | ref
 *
 * Only during parsing the node_decl( prev_decl ) are a backward tree,
 * last_decl points to the current lowest decl of the current block,
 * and only the first decl of a block has first==true.
 */
int generation = 1;

#define for_each_subtree
#define expand_for_all_subtrees   \
   for_each_subtree( first      ) \
   for_each_subtree( next       ) \
   for_each_subtree( condition  ) \
   for_each_subtree( prev_decl  ) \
   for_each_subtree( type       ) \
   for_each_subtree( address    ) \
   for_each_subtree( ret        ) \
   for_each_subtree( actual     ) \
   for_each_subtree( var        ) \
   for_each_subtree( start      ) \
   for_each_subtree( end        ) \
   for_each_subtree( step       ) \
   for_each_subtree( value      ) \
   for_each_subtree( impl       ) \
   for_each_subtree( result     ) \
   for_each_subtree( get        ) \
   for_each_subtree( put        ) \
   for_each_subtree( transfer1  ) \
   for_each_subtree( transfer2  ) \
   for_each_subtree( tlabel     ) \
   for_each_subtree( call_label ) \
   for_each_subtree( jtaddress  ) \


#define assert_kind( loc, p, x ){ \
   assert_pointer( loc, p ); \
   if( p->kind != x ){ \
      snark_node( loc, p ); \
   } \
} \

/* allocate a new node of the indicated kind */
int node_nr = 0;
tree node_root = NULL;
tree free_nodes = NULL;
tree new_node(loc_t loc, int kind)
{
    tree p;
    stack_guard;
    if (free_nodes != NULL) {
        snark(NULL);
        p = free_nodes;
        free_nodes = p->chain;
    } else {
        p = allocate(sizeof(node));
    }
    p->generation = 0;
    p->nr = node_nr++;
    p->loc = loc;
    p->kind = kind;

    /* NULL all pointers */
#undef for_each_subtree
#define for_each_subtree( t ) p->t = NULL;
    expand_for_all_subtrees p->master1 = NULL;
    p->master2 = NULL;
    p->uncle = NULL;

    /* set all fields to suspicious values */
    p->x = -1;
    p->large = 0;
    p->bits = 8888;
    p->op = op_undefined;
    p->name = empty_string;
    p->label = empty_string;
    p->known = false;
    p->is_volatile = false;
    p->used = 0;
    p->opcode = opcode_undefined;
    p->dest = 'i';
    p->mode = mode_none;
    p->fixed = false;
    p->indirect = false;
    p->ref = ref_var;
    p->last_page = false;
    p->is_virtual = false;
    p->is_argument = false;
    p->is_chained = false;
    p->is_target = false;
    p->is_gathered = false;
    p->has_return = false;
    p->has_transfer = false;
    p->is_interrupt = false;
    p->is_raw = false;
    p->sacred = false;
    p->fizzle = false;
    p->bank = 0;
    p->free_bit = -1;
    p->free_byte = -1;
    p->pass_in_w = false;
    p->read_allowed = false;
    p->write_allowed = false;
    p->generated = false;

    p->chain = node_root;
    p->duplicate = NULL;
    p->done = false;
    node_root = p;
    if (verbose_malloc)
        log((m, "alloc %d", p->nr));
    return p;
}

/* forward */
void show_subtree(tree p);
#define trace_subtree( p ) { \
   log(( m , "tree start" )); \
   show_subtree( p ); \
   log(( m , "tree end" )); \
}

void mark_reachable(tree p)
{
    stack_guard;
    if ((p != NULL) && (!p->done)) {
        p->done = true;

        /* recurse of all subtrees */
#undef for_each_subtree
#define for_each_subtree( t ) mark_reachable( p->t );
    expand_for_all_subtrees}
}

void garbage_collection(tree p)
{
    tree q, last = NULL;
    stack_guard;

    /* mark all nodes unused */
    for (q = node_root; q != NULL; q = q->chain) {
        q->done = false;
    }

    /* mark all reachable nodes */
    mark_reachable(node_root);

    /* dispose all non-reachable nodes */
    for (q = node_root; q != NULL; q = q->chain) {
        if (!q->done) {
            assert_pointer(NULL, last);
            last->chain = q->chain;
            q->chain = free_nodes;
            free_nodes = q;
        }
        last = q;
    }
}

int variable_address(tree p)
{
    stack_guard;
    assert_pointer(NULL, p);
    if (p->kind == node_ref)
        return variable_address(p->first);
    if (p->kind == node_const)
        return p->value->x;
    if (p->kind == node_value)
        return p->x;
    assert_kind(p->loc, p, node_var);
    assert_kind(p->loc, p->address, node_chain);
    assert_kind(p->loc, p->address->first, node_value);
    return p->address->first->x;
}

int constant_value(tree p)
{
    stack_guard;
    assert_pointer(NULL, p);
    if (p->kind == node_ref)
        return constant_value(p->first);
    assert_kind(p->loc, p, node_const);
    assert_pointer(NULL, p->value);
    assert_kind(p->loc, p->value, node_value);
    return p->value->x;
}

boolean node_is_constant(tree p, int x)
{
    stack_guard;
    assert_pointer(NULL, p);
    if (p->kind == node_ref)
        return node_is_constant(p->first, x);
    if (p->kind != node_const)
        return false;
    assert_pointer(NULL, p->value);
    if (p->value->kind != node_value)
        return false;
    if (p->value->x != x)
        return false;
    return true;
}

boolean node_is_some_constant(tree p)
{
    stack_guard;
    assert_pointer(NULL, p);
    if (p->kind == node_ref)
        return node_is_some_constant(p->first);
    if (p->kind != node_const)
        return false;
    assert_pointer(NULL, p->value);
    if (p->value->kind != node_value)
        return false;
    return true;
}

/* global nodes: jal build-in types */
tree type_universal, type_bit, type_byte, type_char, type_word, type_long;

/* the transfer variables */
tree transfer_bit, transfer_byte;

/* global nodes: some constants */
tree const_zero, const_one, const_two;

tree transfer_variable(tree t)
{
    stack_guard;
    if (t == type_bit) {
        return transfer_bit;
    } else if (t == type_byte) {
        return transfer_byte;
    } else {
        snark(NULL);
    }
    return NULL;
}

tree universal_result(op_t op)
{
    stack_guard;
    switch (op) {
    case op_smaller:
    case op_larger:
    case op_smaller_or_equal:
    case op_larger_or_equal:
    case op_equal:
    case op_not_equal:
        return type_bit;
    default:
        return type_universal;
    }
}


/********** allocators **********/

tree new_chain(tree q)
{
    tree p = new_node(NULL, node_chain);
    p->first = q;
    return p;
}

tree new_chain2(tree q1, tree q2)
{
    tree p = new_chain(q1);
    p->next = q2;
    return p;
}

tree new_chain3(tree q1, tree q2, tree q3)
{
    tree p = new_chain(q1);
    p->next = new_chain2(q2, q3);
    return p;
}

tree new_chain4(tree q1, tree q2, tree q3, tree q4)
{
    tree p = new_chain(q1);
    p->next = new_chain3(q2, q3, q4);
    return p;
}

tree new_chain5(tree q1, tree q2, tree q3, tree q4, tree q5)
{
    tree p = new_chain(q1);
    p->next = new_chain4(q2, q3, q4, q5);
    return p;
}

tree new_chain6(tree q1, tree q2, tree q3, tree q4, tree q5, tree q6)
{
    tree p = new_chain(q1);
    p->next = new_chain5(q2, q3, q4, q5, q6);
    return p;
}

tree new_chain7(tree q1, tree q2, tree q3, tree q4, tree q5, tree q6, tree q7)
{
    tree p = new_chain(q1);
    p->next = new_chain6(q2, q3, q4, q5, q6, q7);
    return p;
}

tree new_chain8(tree q1, tree q2, tree q3, tree q4, tree q5, tree q6, tree q7, tree q8)
{
    tree p = new_chain(q1);
    p->next = new_chain7(q2, q3, q4, q5, q6, q7, q8);
    return p;
}

tree new_chain9(tree q1, tree q2, tree q3, tree q4, tree q5, tree q6, tree q7, tree q8, tree q9)
{
    tree p = new_chain(q1);
    p->next = new_chain8(q2, q3, q4, q5, q6, q7, q8, q9);
    return p;
}

tree new_chain10(tree q1, tree q2, tree q3, tree q4, tree q5, tree q6, tree q7, tree q8, tree q9,
                 tree q10)
{
    tree p = new_chain(q1);
    p->next = new_chain9(q2, q3, q4, q5, q6, q7, q8, q9, q10);
    return p;
}

tree new_label(loc_t loc, string m, int n, string o)
{
    string s;
    tree p = new_node(loc, node_label);
    stack_guard;
    sprintf(s, m, n, o);
    p->name = new_string_lowercase(s);
    p->label = new_string(p->name);
    string_to_lowercase(p->label);
    return p;
}

tree new_error(loc_t loc)
{
    tree p = new_node(loc, node_error);
    return p;
}

tree new_value(tree type, int value)
{
    tree p = new_node(NULL, node_value);
    p->type = type;
    p->x = value;
    return p;
}

tree new_org(int value)
{
    tree p = new_node(NULL, node_org);
    p->x = value;
    return p;
}

tree new_const(tree v)
{
    tree p = new_node(NULL, node_const);
    p->type = v->type;
    p->value = v;
    return p;
}

tree new_address_value(int value1, int value2)
{
    return new_chain2(new_value(type_universal, value1), new_value(type_universal, value2)
        );
}

tree new_decl(loc_t loc, tree first)
{
    tree p = new_node(loc, node_decl);
    assert_pointer(NULL, first);
    p->first = first;
    p->type = first->type;
    return p;
}

tree new_return(loc_t loc, tree first)
{
    tree p = new_node(loc, node_return);
    assert_pointer(NULL, first);
    p->first = first;
    return p;
}

tree new_w(loc_t loc, int mode)
{
    tree p = new_node(loc, node_w);
    p->type = type_byte;
    p->write_allowed = true;
    p->read_allowed = true;
    p->mode = mode;
    return p;
}

tree new_call(loc_t loc, tree q)
{
    tree p = new_node(loc, node_call);
    assert_pointer(NULL, q);
    p->first = q;
    return p;
}

tree new_precall(loc_t loc, tree q)
{
    tree p = new_node(loc, node_precall);
    assert_pointer(NULL, q);
    p->first = q;
    return p;
}

#define chip_without_code_pages ( \
   ( target_chip == t_16f84 ) \
   || ( target_chip == t_16c84 ) \
   || ( target_chip == t_12c508 ) \
   || ( target_cpu == pic_16 ))

#define chip_without_memory_banks ( \
   ( target_chip == t_12c508 ) \
   || ( target_chip == t_16c84 ) \  
   || ( target_chip == t_16f84 ))

#define chip_with_call_vectors ( \
   ( target_chip == t_sx18 ) \
   || ( target_chip == t_sx28 ) \
   || ( target_chip == t_12c509a ) \
   || ( target_chip == t_12c508 ) \
   || ( target_chip == t_12ce674 ))

tree new_icall(loc_t loc, tree q, int x)
{
    tree p = new_node(loc, node_call);
    assert_pointer(NULL, q);
    p->first = q;
    p->indirect = true;
    p->x = x;
    if (chip_without_code_pages) {
        p->x = 2 * x;
    }
    return p;
}

tree new_var(loc_t loc, char *name, tree type)
{
    tree p = new_node(loc, node_var);
    p->type = type;
    p->name = new_string(name);
    p->write_allowed = true;
    p->read_allowed = true;
    return p;
}

/* this one is not used! */
tree new_proc(char *name, tree first, tree next)
{
    tree p = new_node(NULL, node_procedure);
    p->name = name;
    p->first = first;
    p->next = next;
    p->call_label = new_label(NULL, "_%d_%s_vector", p->nr, name);
    return p;
}

tree new_ref(loc_t loc, ref_t ref, tree t, tree proc)
{
    tree p = new_node(loc, node_ref);
    p->first = t;
    assert_pointer(loc, t);
    p->type = t->type;
    p->ref = ref;
    if (proc != NULL) {
        assert_kind(loc, proc, node_procedure);
        if ((proc == t->put) || (proc == t->get)) {
            p->ref = ref_own;
        }
    }
    return p;
}

tree new_indirect(loc_t loc, tree t)
{
    tree p = new_node(loc, node_ref);
    p->first = t;
    assert_pointer(NULL, t);
    p->type = t->type;
    p->indirect = true;
    return p;
}

tree new_op(loc_t loc, tree t, op_t op, tree left, tree right)
{
    tree p = new_node(loc, node_op);
    stack_guard;
    assert_kind(loc, t, node_type);
    p->type = t;
    p->op = op;
    assert_pointer(loc, left);
    p->first = left;
    if (is_monop(op)) {
        jal_assert(loc, right == NULL);
    } else {
        assert_pointer(loc, right);
        p->next = right;
    }
    return p;
}

tree new_asm(loc_t loc, opcode_t opcode, tree arg1, int arg2)
{
    tree p = new_node(loc, node_asm);
    stack_guard;
    p->opcode = opcode;

    if (code_has(p->opcode, field_const)
        || code_has(p->opcode, field_file)
        || code_has(p->opcode, field_fsr)
        || code_has(p->opcode, field_label)
        || code_has(p->opcode, field_tris)
        ) {
        assert_pointer(NULL, arg1);
        p->first = arg1;
    } else {
        cassert(arg1 == NULL);
    }

    if (code_has(p->opcode, field_dest)) {
        if (arg2 == dest_f) {
            p->dest = dest_f;
        } else if (arg2 == dest_w) {
            p->dest = dest_w;
        } else	{
            snark_node(p->loc, p);
        }
    }


    if (code_has(p->opcode, field_flabel)) {
        assert_pointer(NULL, arg2);
        p->next = new_const(new_value(type_universal, arg2));
    }

    if (code_has(p->opcode, field_bit)) {
        assert_kind(arg1->loc, arg1, node_ref);
        assert_pointer(arg1->loc, arg1->first);
        if (arg1->first->address == NULL)
            trace_subtree(arg1->first);
        assert_pointer(arg1->loc, arg1->first->address);
        assert_kind(arg1->loc, arg1->first->address, node_chain);
        assert_pointer(NULL, arg1->first->address->next);
        p->next = new_value(type_universal, arg1->first->address->next->x);
    }

    if (target_cpu == pic_16) {
        if ((opcode == opcode_goto) || (opcode == opcode_call))  {
            p = new_chain2(p, new_asm(loc, opcode_a2nd, arg1, arg2));
        }
    }

    if (target_cpu == pic_16) {
        if (opcode == opcode_lfsr)  {
            p = new_chain2(p, new_asm(loc, opcode_f2nd, arg1, arg2));
        }
    }



    return p;
}
