#ifndef RAZERFIRMWARE_h
#define RAZERFIRMWARE_h 1

/*
    razertool - Tool for controlling Razer Copperhead(TM) mice
    Copyright (C) 2006  Christopher Lais

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
*/

#include <usb.h>

#include "razerlib.h"

#define USB_VENDOR_FREESCALE	0x0425	/* Actually, Motorola. */
#define USB_PRODUCT_FREESCALE_JW32_ICP	0xff07

#define RAZER_FIRMWARE_LEN		0x10000	/* from 0x0000 */
#define RAZER_FIRMWARE_BLOCK_LEN	0x40
#define RAZER_FIRMWARE_ERASE_BLOCK_LEN	0x200

#define RAZER_FIRMWARE_FIRST_FLASH	0x7000
#define RAZER_FIRMWARE_LAST_FLASH	0xddff

#define RAZER_FIRMWARE_BLOCKS	(RAZER_FIRMWARE_LEN / RAZER_FIRMWARE_BLOCK_LEN)
#define RAZER_FIRMWARE_BLOCKS_FIRST	(RAZER_FIRMWARE_FIRST_FLASH / RAZER_FIRMWARE_BLOCK_LEN)
#define RAZER_FIRMWARE_BLOCKS_FLASH	(RAZER_FIRMWARE_LAST_FLASH / RAZER_FIRMWARE_BLOCK_LEN + 1)

typedef struct razer_firmware_s razer_firmware_t;
struct razer_firmware_s {
	unsigned char mem[RAZER_FIRMWARE_LEN];

	/* which blocks are actually set */
	unsigned char blocks[0x10000 / RAZER_FIRMWARE_BLOCK_LEN];
};

enum {
	RAZER_PROGRESS_DONE,
	RAZER_PROGRESS_READING,
	RAZER_PROGRESS_VERIFYING,
	RAZER_PROGRESS_VERIFY_ERROR,
	RAZER_PROGRESS_ERASING,
	RAZER_PROGRESS_PROGRAMMING,
};

typedef void (*razer_progress_callback_t)(int action, int start_block, int end_block, double fraction);

struct usb_device *razer_find_flash(struct usb_device *prev); /* prev == NULL for first */

#define razer_firmware_ping razer_usb_ping

int razer_firmware_status(struct usb_dev_handle *udev, int *stat, int timeout);

/* Low-level functions */
int razer_firmware_erase_block(struct usb_dev_handle *udev, int start, int len, int timeout);
int razer_firmware_program_block(struct usb_dev_handle *udev, int start, /* const */ char *data, int len, int timeout);
int razer_firmware_read_block(struct usb_dev_handle *udev, int start, char *data, int len, int timeout);
int razer_firmware_read_block_slow(struct usb_dev_handle *udev, int start, char *data, int len, int timeout);
int razer_firmware_verify_block(struct usb_dev_handle *udev, int start, char *data, int len, int timeout);

/* Mid-level functions */
int razer_firmware_write_flash(struct usb_dev_handle *udev, razer_firmware_t *f, razer_progress_callback_t progress);
int razer_firmware_read_flash(struct usb_dev_handle *udev, razer_firmware_t *f, razer_progress_callback_t progress);
int razer_firmware_read_flash_slow(struct usb_dev_handle *udev, razer_firmware_t *f, razer_progress_callback_t progress);
int razer_firmware_verify_flash(struct usb_dev_handle *udev, razer_firmware_t *f, razer_progress_callback_t progress);

int razer_firmware_parse_srec(int fd, razer_firmware_t *f);
void razer_firmware_dump(razer_firmware_t *f);
void razer_firmware_dump_srecord(razer_firmware_t *f);

/* *Extremely* dangerous functions.  A modified laser can permanently damage your eyes. */
/* 0x00(full-on) to 0xff(almost off); lower is higher powered. */
int razer_firmware_read_laser_power(struct usb_dev_handle *udev, int *power, razer_progress_callback_t progress);
int razer_firmware_write_laser_power(struct usb_dev_handle *udev, int power, razer_progress_callback_t progress);

#endif
