/* #includes *//*{{{C}}}*//*{{{*/
#include "config.h"

#ifdef DMALLOC
#include "dmalloc.h"
#endif

#include <sys/types.h>
#include <assert.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pwd.h>
#include <unistd.h>

#include "misc.h"
/*}}}*/

/* mymemmem    -- strstr for binary data */ /*{{{*/
/* Notes *//*{{{*/
/*

You probably heard about so called "efficient" search algorithms, which
are not of the order O(n*n).  They are a typical example of a theorey
which makes wrong assumptions for practical use, e.g. that you can
neglect any constant factors.  Those fancy algorithms are slow for
small, medium and large n, and n is usually not huge.  No startup costs
and only operations which can be translated directly to native machine
code on most machines make brute force work well here, due to using lots
of it.

*/
/*}}}*/
/*@null@*/ char *mymemmem(const char *hay, size_t haysize, const char *needle, size_t needlesize)
{
  while (haysize)
  {
    while (haysize && *needle!=*hay) { ++hay; --haysize; }
    if (haysize==0) return (char*)0;
    else
    {
      const char *oldhay,*oldneedle;
      size_t oldhaysize,oldneedlesize;
    
      oldhay=hay;
      oldneedle=needle;
      oldhaysize=haysize;
      oldneedlesize=needlesize;
      while (*hay++==*needle++ && --needlesize!=0 && --haysize!=0);
      if (needlesize==0) return (char*)oldhay;
      else if (haysize==0) return (char*)0;
      else
      {
        hay=oldhay+1;
        haysize=oldhaysize-1;
        needle=oldneedle;
        needlesize=oldneedlesize;
        continue;
      }
    }
  }
  return (char*)0;
}
/*}}}*/
/* myrmemmem   -- rstrstr for binary data */ /*{{{*/
/*@null@*/ char *myrmemmem(const char *hay, size_t haysize, const char *needle, size_t needlesize)
{
  const char *rhay,*rneedle;

  rhay=hay+haysize-1;
  rneedle=needle+needlesize-1;
  while (haysize!=0)
  {
    while (haysize!=0 && *rneedle!=*rhay) { --rhay; --haysize; }
    if (haysize==0) return (char*)0;
    else
    {
      const char *oldrhay,*oldrneedle;
      size_t oldhaysize,oldneedlesize;
    
      oldrhay=rhay;
      oldrneedle=rneedle;
      oldhaysize=haysize;
      oldneedlesize=needlesize;
      while (*rhay--==*rneedle-- && --needlesize!=0 && --haysize!=0);
      if (needlesize==0) return (char*)(rhay+1);
      else if (haysize==0) return (char*)0;
      else
      {
        rhay=oldrhay-1;
        haysize=oldhaysize-1;
        rneedle=oldrneedle;
        needlesize=oldneedlesize;
        continue;
      }
    }
  }
  return (char*)0;
}
/*}}}*/
#ifndef HAVE_STRDUP
/* strdup      -- return malloced copy of string */ /*{{{*/
/*@null@*/ char *strdup(const char *str)
{
  char *m;

  if ((m=malloc(strlen(str)+1))==(char*)0) return (char*)0;
  strcpy(m,str);
  return m;
}
/*}}}*/
#endif
#ifndef HAVE_STRNDUP
/* strndup     -- return malloced copy of string */ /*{{{*/
/*@null@*/ char *strndup(const char *str, size_t len)
{
  char *m;
  const char *r,*end;

  for (r=str,end=r+len; *r && r<end; ++r);
  len=r-str;

  if ((m=malloc(len+1))==(char*)0) return (char*)0;
  memcpy(m,str,len);
  *(m+len)='\0';
  return m;
}
/*}}}*/
#endif
/* myrealloc   -- ANSI conforming realloc() */ /*{{{*/
#ifdef BROKEN_REALLOC
#undef realloc
void *myrealloc(void *p, size_t n)
{
  return (p==(void*)0 ? malloc(n) : realloc(p,n));
}
#endif
/*}}}*/
/* getshell    -- get name of shell */ /*{{{*/
/*@null@*/ const char *getshell(void)
{
  static const char *shell=(const char*)0;

  if (shell==(const char*)0)
  {
    if ((shell=getenv("SHELL"))==(const char*)0)
    {
      struct passwd *pwd;

      if ((pwd=getpwuid(getuid()))==(struct passwd*)0)
      {
        shell="/bin/sh";
      }
      else
      {
        shell=strdup(pwd->pw_shell);
      }
    }
  }
  return shell;
}
/*}}}*/
/* mytmpnam    -- tmpnam without race condition */ /*{{{*/
/*@null@*/ char *mytmpnam(char *buf)
{
  const char *tmpdir;  
  pid_t i;
  int fd=-1;

  if ((tmpdir=getenv("TMPDIR"))==(char*)0) tmpdir="/tmp";
  if ((strlen(tmpdir)+1+8+1)>(size_t)L_tmpnam) return (char*)0;
  buf[0]='\0';
  for (i=getpid(); i; ++i)
  {
    snprintf(buf,(size_t)L_tmpnam,"%s/%08x",tmpdir,(unsigned int)i);
    if (((fd=open(buf,O_RDWR|O_CREAT|O_EXCL,0666))!=-1) || errno!=EEXIST) break;
    if (fd!=-1) (void)close(fd);
  }
  if (fd!=-1)
  {
    (void)close(fd);
    return buf;
  }
  else
  {
    fprintf(stderr,"fail '%s' (%s)\n",buf,strerror(errno));
    return (char*)0;
  }
}
/*}}}*/
