/*
   LinuxSPA (Linux Serial Protocol Analyser) Copyright(c)1999,2000,2001,2002,2003
   of Grahame M. Kelly (grahame@wildpossum.com) formely <gmkelly@zip.com.au>
   You may distribute and/or use for any purpose modified or unmodified
   copies of this software if you preserve the copyright notice above.

   THIS SOFTWARE IS PROVIDED AS IS AND COMES WITH NO WARRANTY OF ANY
   KIND, EITHER EXPRESSED OR IMPLIED.  IN NO EVENT WILL THE
   COPYRIGHT HOLDER BE LIABLE FOR ANY DAMAGES RESULTING FROM THE
   USE OF THIS SOFTWARE.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

   LinuxSPA (Linux Serial Protocol Analyser) for reverse engineering the
   Nikon E950 Digitial Camera Protocols so that the GNU GPhoto system may
   be used under Linux OS.


   Revision     Date            Comments
   --------     ----            ---------------------------------------

   07.1		19/07/03	Project Renamed to comply with trademarks
   07		11/04/02	Command Line Switches
   06		02/02/02	Bug Fixes
   05		02/10/99	Modified for User Filter Operation
   ...		...		...
   00           26/09/99        Initial 

*/

#include "LinuxSPA.h"

unsigned int IN_count = 0, OUT_count = 0;
char showchar[7] = "(X), \0\0";
char showcont[7] = "CNT, \0\0";
static char last_direction = '@';
static int LineCount = LineMAX + 1, STOP = FALSE;
static void sig_usr (int);	/* Signal Handler */
pid_t	pid;	

int
main (int argc, char **argv)
{
  int c;
  int fdA, fdB, cC, iresult = 0, presult = 0;
  struct termios oldtioA, newtioA, oldtioB, newtioB;
  char bufferA[BUFFSIZE], bufferB[BUFFSIZE];
  fd_set readfds, problem, speedy1, speedy2;

  char deviceA[LOWSIZE] = DEFAULT_A;
  char deviceB[LOWSIZE] = DEFAULT_B;
  char baudrate[LOWSIZE];
  int  Baudrate = atoi(DEFAULT_BAUD);
  speed_t BaudRate = Baudrate;
  int  Databits = DEFAULT_DATA;
  int  Stopbits = DEFAULT_STOP;
  int  Parity = DEFAULT_PARITY;
  int  Type = DEFAULT_TYPE;
  int  Debug = DEFAULT_DEBUG;
  

  int fdes[2];			/* Pipe */
  int maxfd;
  struct ppack PipePackA, PipePackB;
  struct ppack *PPackA = &PipePackA, *PPackB = &PipePackB;

  for (cC = 0; cC < BUFFSIZE - 1; cC++)
    bufferA[cC] = bufferB[cC] = 0x00;

  for (cC = 0; cC < PipeBuffSize; cC++) {
      PPackA->PBuff[cC] = 0x00;
      PPackB->PBuff[cC] = 0x00;
  }

  if (argc < 2) {
	help();
	exit (0);
  }
  
  while (1) {
         int option_index = 0;
         static struct option long_options[] = {
            {"deviceA", 1, 0, 'A'},
            {"deviceB", 1, 0, 'B'},
            {"baudrate", 1, 0, 'b'},
            {"databits", 1, 0, 'd'},
            {"stopbits", 1, 0, 's'},
            {"parity", 1, 0, 'p'},
            {"Debug", 0, 0, 'x'},
            {"mode", 1, 0, 'm'},
            {"verbose", 0, 0, 'v'},
            {"help", 0, 0, 'h'},
            {0, 0, 0, 0}
         };

         c = getopt_long (argc, argv, "A:B:b:d:s:p:d:m:xvh",
                    long_options, &option_index);

         if (c == -1) {
		break; /* We have finished the command line parse */
 	 }

         switch (c) {

	       case 'A':
		   strcpy (deviceA, optarg);
		   break;

	       case 'B':
		   strcpy (deviceB, optarg);
                   break;

               case 'b':
		   strcpy (baudrate, optarg);
		   Baudrate = atoi(baudrate);
		   BaudRate = what_speed(Baudrate);
                   break;

               case 'x':
		   Debug = TRUE;
                   break;

               case 'h':
	       case '?':
		   help();
		   exit (0);
                   break;

               case 's':
		   if (strlen(optarg) >= 1) {
		   	switch (optarg[0]) {
				case '2':
					Stopbits = 2;
					break;	
				case '1':
					Stopbits = 1;
					break;
				default:
					printf("[ABORTING] Incorrect Stopbit selection `%s'\n\n", optarg);
					exit (1);
					break;
			}
		   }
                   break;	

               case 'd':
                   if (strlen(optarg) >= 1) {
                        switch (optarg[0]) {
				case '5':
					Databits = 5;
					break;
				case '6':
					Databits = 6;
					break;
                                case '7':
                                        Databits = 7;
                                        break;
                                case '8':
                                        Databits = 8;
                                        break;
                                default:
                                        printf("[ABORTING] Incorrect Databit selection `%s'\n\n", optarg);
                                        exit (1);
                                        break;
                        }
                   }
                   break;

               case 'p':
                   if (strlen(optarg) >= 1) {
		   switch (optarg[0]) {
			case 'O': /* Odd */
				Parity = 1;
				break;
			case 'E': /* Even */
				Parity = 2;
				break;
			case 'N': /* None */
				Parity = 0;
				break;
                        default:
                                printf("[ABORTING] Incorrect Parity selection `%s'\n\n", optarg);
                                exit (1);
                                break;
			}
		   }
                   break;

               case 'm':
                   if (strlen(optarg) >= 1) {
                   switch (optarg[0]) {
                        case '1': 
                                Type = 1;
                                break;
                        case '2':
                                Type = 2;
                                break;
                        default:
                                printf("[ABORTING] Incorrect Mode selection `%s'\n\n", optarg);
                                exit (1);
                                break;
                        }
                   }
                   break;

               default:
                   printf ("?? getopt returned character code 0%o ??\n", c);
	      }
	 }	       

         if (optind < argc) {
             printf ("[ABORTING] non-option ARGV-elements: ");
             while (optind < argc)
                 printf ("%s ", argv[optind++]);
             printf (" \n");
	     exit (1);
         }
 
  /* Now report to the user and start program proper */

  printf ("\nLinuxSPA: Version %s.%s\n", REV_Major, REV_Minor);
  printf ("Application Computer sniffed on (Device 1) serial line: %s\n", deviceA);
  printf ("Device Under Test sniffed on (Device 2) serial line: %s\n", deviceB);
  if (Debug) {
	printf ("Baudrate = %d, DataBits = %d, StopBits = %d, Parity = %d, Mode = %d, Debug = %d\n\n",\
	   	 Baudrate, Databits, Stopbits, Parity, Type, Debug);
  	printf("Debugging Active\n\n");
  }else{
	printf("\n\n");
  }


  if(Type == 2) {  /* Test we are able to use the serial ports */
  	fdA = open (deviceB, O_RDWR | O_NOCTTY | O_NONBLOCK | O_SYNC);
  	if (fdA < 0) {
      	   perror ("Type 2 Device 1 - Error: ");
	   fprintf(stderr,"\nDevice %s\n\n", deviceA);
           exit (-2);
        }
  	fdB = open (deviceA, O_RDWR | O_NOCTTY | O_NONBLOCK | O_SYNC);
  	if (fdB < 0) {
           perror ("Type 2 Device 2 - Error ");
	   fprintf(stderr,"\nDevice %s\n\n", deviceB);
           exit (-2);
        }
  }else{
  	fdA = open (deviceB, O_RDONLY | O_NOCTTY);
  	if (fdA < 0) {
           perror ("Type 1 Device 1 - Error ");
           fprintf(stderr,"\nDevice %s\n\n", deviceA);
           exit (-1);
    	}
  	fdB = open (deviceA, O_RDONLY | O_NOCTTY);
  	if (fdB < 0) {
           perror ("Type 1 Device 2 - Error ");
           fprintf(stderr,"\nDevice %s\n\n", deviceB);
      	   exit (-1);
    	}
  }


  tcgetattr (fdA, &oldtioA);	/* save current port settings */
  tcgetattr (fdB, &oldtioB);	/* save current port settings */

  bzero (&newtioA, sizeof (newtioA));
  bzero (&newtioB, sizeof (newtioB));

  switch (Databits) {
	case 5: newtioB.c_cflag = newtioA.c_cflag = CS5 | CLOCAL | CREAD;
		break;
	case 6: newtioB.c_cflag = newtioA.c_cflag = CS6 | CLOCAL | CREAD;
		break;
	case 7: newtioB.c_cflag = newtioA.c_cflag = CS7 | CLOCAL | CREAD;
		break;
  	case 8: 
	default:newtioB.c_cflag = newtioA.c_cflag = CS8 | CLOCAL | CREAD;
		break;
  }

  /* Ignore parity errors even if they occur */
  newtioB.c_iflag = newtioA.c_iflag = IGNPAR;
  newtioB.c_oflag = newtioA.c_oflag = 0;

  /* set input mode (non-canonical, no echo,...) */
  newtioB.c_lflag = newtioA.c_lflag = 0;
  newtioB.c_cc[VTIME] = newtioA.c_cc[VTIME] = 0;
  newtioB.c_cc[VMIN] = newtioA.c_cc[VMIN] = 1;

  /* Set new speeds */
  cfsetispeed (&newtioA, BaudRate);
  cfsetospeed (&newtioA, BaudRate);
  cfsetispeed (&newtioB, BaudRate);
  cfsetospeed (&newtioB, BaudRate);

  tcsetattr (fdA, TCSANOW, &newtioA);
  tcsetattr (fdB, TCSANOW, &newtioB);

  tcflush (fdA, TCIFLUSH);
  tcflush (fdB, TCIFLUSH);


  /* Make the Pipe the primary task to write to, secondary to read from */
  presult = pipe (&fdes[0]);
  if (presult < 0)
    {
      perror ("Pipe Create Error ");
      exit (-1);
    }
  if (Debug)
    fprintf (stderr, "Pipe fd = %d, %d\n", fdes[0], fdes[1]);

  switch (pid = fork ())
    {
    case 0:			/* we are the child */
      close (fdes[1]);		/* Close the Write Pipe */
      piper (fdes[0]);		/* Call the Piper ----  */
      break;

    case -1:
      fprintf (stderr, "fork failed.\n");
      exit (-1);
      break;

    default:			/* we are the parent */
      close (fdes[0]);		/* Close the Read Pipe */
      break;
    }

  if (Debug)
    fprintf (stderr, "PARENT: The child PID %d, Write Pipe %d\n", pid, fdes[1]);

  if (signal (SIGUSR1, sig_usr) == SIG_ERR)
    {
      perror ("\nUser signal received OK");
      kill(pid, SIGKILL);
      STOP = TRUE;
    }

  if (fdA > fdB)
     maxfd = fdA + 1;
  else
     maxfd = fdB + 1;

  FD_ZERO (&readfds);
  FD_ZERO (&problem);
  FD_SET (fdA, &readfds);
  FD_SET (fdA, &problem);
  FD_SET (fdB, &readfds);
  FD_SET (fdB, &problem);

  /* Now make a "master" copy of fds for fast switching */
  speedy1 = readfds; speedy2 = problem;

  while (!STOP)
    {

      readfds = speedy1; problem = speedy2;
      iresult = select (maxfd, &readfds, NULL, &problem, NULL);
      if (iresult < 0)
	{
	  perror ("Select on Readfds ");
	  exit (-1);
	}
      if (FD_ISSET (fdA, &readfds))
	{
	  PPackA->count = read (fdA, &PPackA->PBuff[0], PipeBuffSize - 1);
	  if (PPackA->count < 0)
	    {
	      perror ("READ on fdA ");
	      exit (-1);
	    }
	  PPackA->direction = CAMERA_IN;
	  presult = write (fdes[1], &PipePackA, sizeof (PipePackA));
	  if (presult < 0)
	    {
	      perror ("Pipe Write A ");
	      exit (-1);
	    }
	  if (Type == 2)
	    {
	      /* Send Character back out again || install your adds here */
	      presult = write (fdA, &PPackA->PBuff[0], PPackA->count);
	      if (presult != PPackA->count)
		{
		  perror ("Resend fdA error ");
		  exit (-1);
		}
	    }
	}
      if (FD_ISSET (fdB, &readfds))
	{
	  PPackB->count = read (fdB, &PPackB->PBuff[0], PipeBuffSize - 1);
	  if (PPackB->count < 0)
	    {
	      perror ("READ on fdB ");
	      exit (-1);
	    }
	  PPackB->direction = CAMERA_OUT;
	  presult = write (fdes[1], &PipePackB, sizeof (PipePackB));
	  if (presult < 0)
	    {
	      perror ("Pipe Write B ");
	      exit (-1);
	    }
	  if (Type == 2)
	    {
	      /* Send Character back out again || install your adds here */
	      presult = write (fdB, &PPackB->PBuff[0], PPackB->count);
	      if (presult != PPackB->count)
		{
		  perror ("Resend fdB error ");
		  exit (-1);
		}
	    }
	}
      if (FD_ISSET (fdA, &problem))
	{
	  fprintf (stderr, "\n\aProblem on fdA input\n");
	}
      if (FD_ISSET (fdB, &problem))
	{
	  fprintf (stderr, "\n\aProblem on fdB input\n");
	}
      if (iresult < 1)
	fprintf (stderr, "\nShould NOT get here...\a\n");
  
  }
  tcsetattr (fdA, TCSANOW, &oldtioA);	/* Restore ttyA */
  tcsetattr (fdB, TCSANOW, &oldtioB);	/* Restore ttyB */
  close (fdA);
  close (fdB);
  close (fdes[1]);
  kill(pid, SIGKILL);
  return (0);
}
/* End of Main */


speed_t
what_speed (int speed)
{

 /* Convert int speed to System Defined Baudrate     */
 /* Needed to check that user input is a valid rate  */
 switch (speed) {
	case 50:     return  B50; break;
	case 75:     return  B75; break;
	case 110:    return  B110; break;
	case 134:    return  B134; break;
	case 150:    return  B150; break;
	case 200:    return  B200; break;
	case 300:    return  B300; break;
	case 600:    return  B600; break;
	case 1200:   return  B1200; break;
	case 1800:   return  B1800; break;
	case 2400:   return  B2400; break;
	case 4800:   return  B4800; break;
	case 9600:   return  B9600; break;
	case 19200:  return  B19200; break;
	case 38400:  return  B38400; break;
	case 57600:  return  B57600; break;
	case 115200: return  B115200; break;
	case 230400: return  B230400; break;
	default:
		printf("[ABORTING] Incorrect Baudrate selected = %d\n\n", speed);
		exit (-3);
		break;
	}	
}


static void 
sig_usr (int signo)
{
  /* User can terminate the above process */
  /* by using a Control C at the keyboard */ 
  if (signo == SIGINT)
    STOP = TRUE;
    kill(pid, SIGKILL);
  return;
}



/*------------------- Piper Program ----------------------------*/
/*                                                              */
/* Reads sniffed data packets from the pipe and then formats    */
/* the output in accordance to the following design:            */
/*                                                              */
/*      000000 : <  61, 62, 63, 78, 79,       			*/
/*      000000 : >  41, 42, 43,                       		*/
/*      000009 : <  67, 6b,                           	 	*/
/*      000005 : >  00, 01, 02,                       		*/
/*      000013 : <  0a, 0b,                                	*/
/*                                                              */
/* where ^^^ is the number of bytes for that direction and      */
/* the '<' is input, '>' is output direction.                   */
/*                                                              */
/* Bracketted data is the ASCII equivalent or if it is a        */
/* control charcter its nuemonic is shown instead.              */
/*                                                              */
/*--------------------------------------------------------------*/


int 
user_format (char direction, byte * Obuffer, int bytes_read)
{
  int Gcount, Pcount, Sx;
  static int *pcount;

  Sx = 0;
  while (bytes_read)
    {
      if (direction == '<')
	pcount = &IN_count;
      else
	pcount = &OUT_count;

      if (direction != last_direction)
	{
	  last_direction = direction;
	  fprintf (stdout, "\n%06x : %c  ", *pcount, direction);
	  LineCount = 1;
	}
      if (direction == '>')
	{
	  if (LineCount > LineMAX)
	    {
	      fprintf (stdout, "\n%06x : >  ", *pcount);
	      LineCount = 1;
	    }
	}
      if (direction == '<')
	{
	  if (LineCount > LineMAX)
	    {
	      fprintf (stdout, "\n%06x : <  ", *pcount);
	      LineCount = 1;
	    }
	}

      Pcount = Gcount = bytes_read;
      /* Print the Hex data */
      while (Gcount--)
	{
	  if (LineCount > LineMAX)
	    {
	      fprintf (stdout, "\n%06x : %c  ", *pcount, last_direction);
	      LineCount = 1;
	    }
          /* Should put out characters here I think */
          fprintf(stdout, "%02x, ", Obuffer[Sx++]);
	  *pcount += 1;
	  LineCount++;
	}
      bytes_read = bytes_read - Pcount;
    }
  return 0;
}


int 
piper (int fdD)
{
  int result;
  struct ppack RecPacket, *RPack = &RecPacket;

  fprintf (stderr,\
	"\nCHILD: Piper Formatter Read Pipe %d\n", fdD);


  while (!STOP)
    {
      result = read (fdD, RPack, sizeof (RecPacket));
      if (result != sizeof (RecPacket))
	{
	  perror ("Pipe Read Error - A");
	  STOP = TRUE;
	  exit (-1);
	}
      user_format (RPack->direction, &RPack->PBuff[0], RPack->count);
      fflush(stdout);
    }
  close (fdD);
  exit (0);
}


void
help (void)
{
   printf ("\nLinuxSPA [OPTIONS]\n\n");
   printf ("-A --deviceA  : Application Computer serial port. Default = /dev/ttyS0\n");
   printf ("-B --deviceB  : Device Under Test serial port. Default = /dev/ttyS1\n");
   printf ("-b --baudrate : Serial line baudrate (110,300,600,1200,1800,2400,4800,9600,\n");
   printf ("              : 19200, 38400, 57600, 115200, 230400), Default = 115200\n");
   printf ("-d --databits : Data Bits (5,6,7 or 8), Default = 8\n");
   printf ("-h --help     : Help screen\n");
   printf ("-m --mode     : Mode type (1 or 2): See README, Default = 1\n");
   printf ("-s --stopbits : Number of Stop Bits (1 or 2), Default = 1\n");
   printf ("-p --parity   : Parity setting (Odd, Even, None), Default = N\n");
   printf ("-v --verbose  : Verbose flag On\n");
   printf ("-x --Debug    : Debugger On, Default Off\n");
   printf ("\n\n");
   return;
}

/* End of Program */
