/*
 * main.c: Where the rotoscope program begins execution.
 *
 * 
 * Copyright 2006, James Foster
 *
 * This file is part of rotoscope.
 *
 * Rotoscope is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Rotoscope is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with rotoscope; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <stdlib.h>
#include <gtk/gtk.h>
#include <glade/glade.h>
#include <glib.h>
#include <gdk-pixbuf/gdk-pixbuf.h>

#include "graph.h"
#include "render.h"

/* Structures, types, and constants */
typedef enum { IDLE, DRAWING } MouseState;

const guint LEFT_MOUSE_BUTTON = 1;
const guint RIGHT_MOUSE_BUTTON = 3;

/* UI storage */
GladeXML* xml;
GdkColor edge_color;
GdkGC* edge_gc;
GdkPixbuf* src_image;
GdkPixbuf* render_result;

/* Program state variables */
GList* edge_list = NULL;
MouseState mouse_state = IDLE;
Vertex last_click;

/* Functionality and callback functions */
void clear_edge_list()
{
	GList* i = edge_list;
	while( NULL != i )
	{
		free( i->data );
		i = g_list_next( i );
	}
	g_list_free( edge_list );
	edge_list = NULL;
}

void free_all()
{
	/* Free all image pixbufs */
	g_object_unref( src_image );

	/* Free all Edges in edge_list */
	clear_edge_list();
}

void show_operation_failed_dialog()
{
	gint response;
	GtkWidget* operation_failed_dialog = glade_xml_get_widget( xml, "operation_failed_dialog" );
	gtk_dialog_run( (GtkDialog*) operation_failed_dialog );
	gtk_widget_hide( operation_failed_dialog );
}

void show_save_graph_as_dialog( GtkMenuItem* menu_item, gpointer user_data )
{
	char* filename_selected;
	gint response;
	GtkWidget* save_graph_as_dialog = glade_xml_get_widget( xml, "save_graph_as_dialog" );
	response = gtk_dialog_run( (GtkDialog*) save_graph_as_dialog );
	gtk_widget_hide( save_graph_as_dialog );

	if( GTK_RESPONSE_OK == response )
	{
		filename_selected = gtk_file_chooser_get_filename( GTK_FILE_CHOOSER(save_graph_as_dialog) );
		if( FALSE == save_graph_to_file(edge_list, filename_selected) )
		{
			show_operation_failed_dialog();
		}
	}
}

void show_save_rendering_dialog( GtkWidget* widget, gpointer user_data )
{
	char* filename_selected;
	gint response;
	GtkWidget* save_rendering_dialog = glade_xml_get_widget( xml, "save_rendering_dialog" );
	response = gtk_dialog_run( (GtkDialog*) save_rendering_dialog );
	gtk_widget_hide( save_rendering_dialog );
	
	if( GTK_RESPONSE_OK == response )
	{
		filename_selected = gtk_file_chooser_get_filename( GTK_FILE_CHOOSER(save_rendering_dialog) );
		if( FALSE == gdk_pixbuf_save(render_result, filename_selected, "png", NULL, NULL) )
		{
			show_operation_failed_dialog();
		}
	}
}

void show_render_window( GtkWidget* widget, gpointer user_data )
{
	GtkWidget* render_win;
	GtkWidget* render_drawing_area;
	
	render_win = glade_xml_get_widget( xml, "render_win" );
	render_drawing_area = glade_xml_get_widget( xml, "render_drawing_area" );
	
	render( src_image, edge_list, render_result );
	gtk_widget_show( render_win );
	redraw_render_area( NULL, NULL, NULL );
}

void show_about_dialog( GtkMenuItem* menuitem, gpointer user_data )
{
	GtkWidget* about_dialog = glade_xml_get_widget( xml, "about_dialog" );
	gtk_dialog_run( (GtkDialog*) about_dialog );

	gtk_widget_hide( about_dialog );
}

void show_load_graph_dialog( GtkMenuItem* menuitem, gpointer user_data )
{
	gint response;
	GtkWidget* load_graph_dialog = glade_xml_get_widget( xml, "load_graph_dialog" );
	response = gtk_dialog_run( (GtkDialog*) load_graph_dialog );

	/* hide the dialog */
	gtk_widget_hide( load_graph_dialog );
	
	/* if the OK button was pressed, load the selected graph */
	if( GTK_RESPONSE_OK == response )
	{
	}
}

void draw_pixbuf_to_drawing_area( GdkPixbuf* src, GtkWidget* dest )
{
	gint width = gdk_pixbuf_get_width( src );
	gint height = gdk_pixbuf_get_height( src );
	gtk_widget_set_size_request( dest, width, height );
	gdk_draw_pixbuf( (GdkDrawable*) dest->window,
			 NULL,
			 src,
			 0, 0,
			 0, 0,
			 -1, -1,
			 GDK_RGB_DITHER_NONE, 0, 0 );
}

gboolean redraw_render_area( GtkWidget* widget, GdkEventExpose* event, gpointer user_data )
{
	GtkWidget* render_drawing_area = glade_xml_get_widget( xml, "render_drawing_area" );
	draw_pixbuf_to_drawing_area( render_result, render_drawing_area );
}

void draw_edge_to_drawing_area( Edge* edge, GtkWidget* dest )
{
	gdk_draw_line( (GdkDrawable*) dest->window, edge_gc,
			edge->a.x, edge->a.y,
			edge->b.x, edge->b.y );
}

void draw_edges_to_drawing_area( GList* edges, GtkWidget* dest )
{
	Edge* edge;
	GList* i = edges;
	while( NULL != i )
	{
		edge = (Edge*) i->data;
		draw_edge_to_drawing_area( edge, dest );
		i = g_list_next( i );
	}
}

gboolean redraw_drawing_area( GtkWidget* widget, GdkEventExpose* event, gpointer user_data )
{
	GList* i;
	Edge* edge;
	GtkWidget* drawing_area = glade_xml_get_widget( xml, "drawing_area" );

	/* Draw the src_image pixbuf onto the drawing area */
	draw_pixbuf_to_drawing_area( src_image, drawing_area );

	/* Draw edges */
	draw_edges_to_drawing_area( edge_list, drawing_area );

	return FALSE;
}

void load_image_from_file( const char* filename )
{
	int width, height;
	GdkPixbuf* new_image;
	GdkPixbuf* new_render_result;
	
	/* Load a copy of the image we're loading, and get the width/height */
	new_image = gdk_pixbuf_new_from_file( filename, NULL );
	if( NULL == new_image )
	{
		/* Out of memory or couldn't load the image */
		show_operation_failed_dialog();
		return;
	}
	width = gdk_pixbuf_get_width( new_image );
	height = gdk_pixbuf_get_height( new_image );

	/* Create the new render result pixbuf */
	new_render_result = gdk_pixbuf_new( GDK_COLORSPACE_RGB, FALSE, 8, width, height );
	if( NULL == new_render_result )
	{
		/* Out of memory */
		g_object_unref( new_image );
		show_operation_failed_dialog();
		return;
	}

	/* Free the old image */
	g_object_unref( src_image );
	g_object_unref( render_result );
	src_image = new_image;
	render_result = new_render_result;

	/* Draw the new image */
	redraw_drawing_area( NULL, NULL, NULL );
}

void show_load_image_dialog( GtkMenuItem* menuitem, gpointer user_data )
{
	char* filename_selected;
	gint response;
	GtkWidget* render_win = glade_xml_get_widget( xml, "render_win" );
	GtkWidget* load_image_dialog = glade_xml_get_widget( xml, "load_image_dialog" );
	response = gtk_dialog_run( (GtkDialog*) load_image_dialog );

	/* hide the dialog */
	gtk_widget_hide( load_image_dialog );

	/* if the OK button was pressed, load the selected image */	
	if( GTK_RESPONSE_OK == response )
	{
		clear_edge_list();
		filename_selected = gtk_file_chooser_get_filename( GTK_FILE_CHOOSER(load_image_dialog) );
		gtk_widget_hide( render_win );
		load_image_from_file( filename_selected );
	}
}

void undo( GtkMenuItem* menu_item, gpointer user_data )
{
	GList* first_elem = g_list_first( edge_list );
	if( NULL != first_elem )
	{
		free( first_elem->data );
		edge_list = g_list_remove( edge_list, first_elem->data );
	}

	mouse_state = IDLE;
	redraw_drawing_area( NULL, NULL, NULL );
}

void quit( GtkButton* widget, gpointer user_data )
{
	free_all();
	gtk_exit( 0 );
}

gboolean del_quit( GtkWidget* widget, GdkEvent* event, gpointer user_data )
{
	quit( NULL, NULL );
	return FALSE;
}

gboolean mouse_button_pressed( GtkWidget* widget, GdkEventButton* event, gpointer user_data )
{
	GtkWidget* drawing_area = glade_xml_get_widget( xml, "drawing_area" );
	Edge* edge;
	guint x;
	guint y;
	guint button_pressed = event->button;

	x = event->x;
	y = event->y;

	if( x > gdk_pixbuf_get_width(src_image)-1 )
	{
		x = gdk_pixbuf_get_width( src_image ) - 1;
	}
	if( y > gdk_pixbuf_get_height(src_image)-1 )
	{
		y = gdk_pixbuf_get_height( src_image ) - 1;
	}

	if( LEFT_MOUSE_BUTTON == button_pressed )
	{
		if( IDLE == mouse_state )
		{
			mouse_state = DRAWING;
		}
		else
		{
			/* Put the edge from last_click to the current mouse coords into the edge_list */
			edge = malloc( sizeof(Edge) );
			if( NULL == edge )
			{
				show_operation_failed_dialog();
				return FALSE;
			}
			edge->a = last_click;
			edge->b.x = x;
			edge->b.y = y;
			edge_list = g_list_prepend( edge_list, edge );
			draw_edge_to_drawing_area( edge, drawing_area );
		}
		
		/* Record the coords as the last_click */
		last_click.x = x;
		last_click.y = y;
	}

	if( RIGHT_MOUSE_BUTTON == button_pressed )
	{
		/* Put the mouse into the idle state (cancelling any edge being drawn) */
		mouse_state = IDLE;
	}

	return FALSE;
}

gboolean mouse_motion( GtkWidget* widget, GdkEventMotion* event, gpointer user_data )
{
	guint x = (int) event->x;
	guint y = (int) event->y;
	
	/* TODO: Store these coords so that other parts of the program have access to them */
	/* TODO: If in a state where a line is being drawn, redraw the line to these coords */
	
	return FALSE;
}

void update_color( GtkColorButton* button, gpointer user_data )
{
	GdkColor new_color;
	GtkWidget* drawing_area = glade_xml_get_widget( xml, "drawing_area" );
	
	/* Switch colors */
	gtk_color_button_get_color( button, &new_color );
	gdk_colormap_alloc_color( gdk_colormap_get_system(), &new_color, FALSE, TRUE );
	gdk_gc_set_foreground( edge_gc, &new_color );
	gdk_colormap_free_colors( gdk_colormap_get_system(), &edge_color, 1 );
	edge_color = new_color;

	/* Redraw everything so edges appear in the new color */
	redraw_drawing_area( NULL, NULL, NULL );
}

void connect_signals()
{
	glade_xml_signal_connect( xml, "show_about_dialog",		G_CALLBACK(show_about_dialog) );
	glade_xml_signal_connect( xml, "show_load_graph_dialog",	G_CALLBACK(show_load_graph_dialog) );
	glade_xml_signal_connect( xml, "redraw_drawing_area",		G_CALLBACK(redraw_drawing_area) );
	glade_xml_signal_connect( xml, "redraw_render_area",		G_CALLBACK(redraw_render_area) );
	glade_xml_signal_connect( xml, "show_load_image_dialog",	G_CALLBACK(show_load_image_dialog) );
	glade_xml_signal_connect( xml, "quit",				G_CALLBACK(quit) );
	glade_xml_signal_connect( xml, "del_quit",			G_CALLBACK(del_quit) );
	glade_xml_signal_connect( xml, "mouse_button_pressed",		G_CALLBACK(mouse_button_pressed) );
	glade_xml_signal_connect( xml, "mouse_motion",			G_CALLBACK(mouse_motion) );
	glade_xml_signal_connect( xml, "update_color",			G_CALLBACK(update_color) );
	glade_xml_signal_connect( xml, "show_render_window",		G_CALLBACK(show_render_window) );
	glade_xml_signal_connect( xml, "gtk_widget_hide_on_delete",	G_CALLBACK(gtk_widget_hide_on_delete) );
	glade_xml_signal_connect( xml, "show_save_rendering_dialog",	G_CALLBACK(show_save_rendering_dialog) );
	glade_xml_signal_connect( xml, "show_save_graph_as_dialog",	G_CALLBACK(show_save_graph_as_dialog) );
	glade_xml_signal_connect( xml, "undo",				G_CALLBACK(undo) );
}

int main( int argc, char** argv )
{
	GtkWidget* drawing_area;
	GtkWidget* color_button;

	/* Initialize GTK+ and libglade */
	gtk_init( &argc, &argv );
	xml = glade_xml_new( "rotoscope.glade", NULL, NULL );
	if( NULL == xml )
	{
		xml = glade_xml_new( GLADE_PATH, NULL, NULL );
		if( NULL == xml )
		{
			/* Not enough memory! */
			exit( 9 );
		}
	}
	connect_signals();

	/* Initialize the graphics context used for edges in the drawing area */
	drawing_area = glade_xml_get_widget( xml, "drawing_area" );
	color_button = glade_xml_get_widget( xml, "color_button" );
	edge_gc = gdk_gc_new( (GdkDrawable*) drawing_area->window );
	if( NULL == edge_gc )
	{
		/* Not enough memory! */
		exit( 9 );
	}
	gtk_color_button_get_color( (GtkColorButton*) color_button, &edge_color );
	gdk_colormap_alloc_color( gdk_colormap_get_system(), &edge_color, FALSE, TRUE );
	gdk_gc_set_foreground( edge_gc, &edge_color );
	
	/* Initialize source image GdkPixbufs */
	src_image = gdk_pixbuf_new( GDK_COLORSPACE_RGB, FALSE, 8, 1, 1 );
	if( NULL == src_image )
	{
		/* Not enough memory! */
		exit( 9 );
	}
	render_result = gdk_pixbuf_new( GDK_COLORSPACE_RGB, FALSE, 8, 1, 1 );
	if( NULL == render_result )
	{
		/* Not enough memory */
		exit( 9 );
	}
	if( argc > 1 )
	{
		load_image_from_file( argv[1] );
	}

	/* Let GTK+ take control from here on */
	gtk_main();

	return 0;
}

