/* "NETGEN", a netlist-specification tool for VLSI
   Copyright (C) 1989, 1990   Massimo A. Sivilotti
   Author's address: mass@csvax.cs.caltech.edu;
                     Caltech 256-80, Pasadena CA 91125.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation (any version).

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; see the file copying.  If not, write to
the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */

/* hash.c  -- hash table support functions  */

#include "config.h"

#include <stdio.h>
#ifdef IBMPC
#include <alloc.h>
#endif

#include "netgen.h"
#include "hash.h"
#include "objlist.h"


void InitializeHashTable(struct hashlist **tab, int size)
{
	int i;
	for (i = 0; i < size; i++) tab[i] = NULL;
}

int RecurseHashTable(struct hashlist **hashtab, int hashsize,
	int (*func)(struct hashlist *elem))
/* returns the sum of the return values of (*func) */
{
	int i, sum;
	struct hashlist *p;
	
	sum = 0;
	for (i = 0; i < hashsize; i++)
		for (p = hashtab[i]; p != NULL; p = p->next) 
			sum += (*func)(p);
	return(sum);
}

int CountHashTableEntries(struct hashlist *p)
{
	/* not strictly needed, but stops compiler from bitching */
	return ((p != NULL) ? 1:0);
}

int CountHashTableBinsUsed(struct hashlist *p)
{
	if (p->next == NULL) return (1);
	return(0);
}

INLINE
static int hash(char *s, int hashsize)
{
	int hashval;
	
	for (hashval = 0; *s != '\0'; )
		hashval += *s++;
	return (hashval % hashsize);
}

void *HashLookup(char *s, struct hashlist **hashtab, int hashsize)
/* return the 'ptr' field of the hash table entry, or NULL if not found */
{
  struct hashlist *np;
	
  for (np = hashtab[hash(s,hashsize)]; np != NULL; np = np->next)
    if (match(s, np->name)) return (np->ptr);	/* correct match */
  return (NULL); /* not found */
}

struct hashlist *HashPtrInstall(char *name, void *ptr, 
				struct hashlist **hashtab, int hashsize)
/* return the hashlist entry, after (re)initializing its 'ptr' field */
{
  struct hashlist *np;
  int hashval;
	
  hashval = hash(name,hashsize);
  for (np = hashtab[hashval]; np != NULL; np = np->next)
    if (match(name, np->name)) {
      np->ptr = ptr;
      return (np);		/* match found in hash table */
    }

  /* not in table, so install it */
  if ((np = (struct hashlist *) CALLOC(1,sizeof(struct hashlist))) == NULL)
    return (NULL);
  if ((np->name = strsave(name)) == NULL) return (NULL);
  np->ptr = ptr;
  np->next = hashtab[hashval];
  return(hashtab[hashval] = np);
}


struct hashlist *HashInstall(char *name, 
			     struct hashlist **hashtab, int hashsize)
/* return the hashlist entry */
{
  struct hashlist *np;
  int hashval;
	
  hashval = hash(name,hashsize);
  for (np = hashtab[hashval]; np != NULL; np = np->next)
    if (match(name, np->name)) return (np); /* match found in hash table */

  /* not in table, so install it */
  if ((np = (struct hashlist *) CALLOC(1,sizeof(struct hashlist))) == NULL)
    return (NULL);
  if ((np->name = strsave(name)) == NULL) return (NULL);
  np->ptr = NULL;
  np->next = hashtab[hashval];
  return(hashtab[hashval] = np);
}


void HashDelete(char *name, struct hashlist **hashtab, int hashsize)
/* frees a hash table entry, (but not the 'ptr' field) */
{
  int hashval;
  struct hashlist *np;
  struct hashlist *np2;
  
  hashval = hash(name, hashsize);
  np = hashtab[hashval];

  if (match(name, np->name)) {
    /* it is the first element in the list */
    hashtab[hashval] = np->next;
    FREE(np->name);
    FREE(np);
    return;
  }

  /* else, traverse the list, deleting the appropriate element */
  while (np->next != NULL) {
    if (match(name, np->next->name)) {
      np2 = np->next;
      np->next = np2->next;
      FREE(np2->name);
      FREE(np2);
      return;
    }
    np = np->next;
  }
}


static int hashfirstindex; /* was long */
static struct hashlist *hashfirstptr;

void *HashNext(struct hashlist **hashtab, int hashsize)
/* returns 'ptr' field of next element, NULL when done */
{
  if (hashfirstptr != NULL && hashfirstptr->next != NULL) {
    hashfirstptr = hashfirstptr->next;
    return(hashfirstptr->ptr);
  }
  while (hashfirstindex < hashsize) {
    if ((hashfirstptr = hashtab[hashfirstindex++]) != NULL) {
      return(hashfirstptr->ptr);
    }
  }
  hashfirstindex = 0;
  hashfirstptr = NULL;
  return(NULL);
}

void *HashFirst(struct hashlist **hashtab, int hashsize)
{
  hashfirstindex = 0;
  hashfirstptr = NULL;
  return(HashNext(hashtab,hashsize));
}


