/* Copyright (C) 2005, 2006, 2009 and 2011 Chris Vine

The library comprised in this file or of which this file is part is
distributed by Chris Vine under the GNU Lesser General Public
License as follows:

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License, version 2.1, for more details.

   You should have received a copy of the GNU Lesser General Public
   License, version 2.1, along with this library (see the file LGPL.TXT
   which came with this source code package in the c++-gtk-utils
   sub-directory); if not, write to the Free Software Foundation, Inc.,
   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

*/

#include <c++-gtk-utils/lib_defs.h>

#include <unistd.h>
#include <fcntl.h>

#include <utility>

#include <glib.h>

#include <c++-gtk-utils/notifier.h>
#include <c++-gtk-utils/mutex.h>

bool Cgu::Notifier::initialised = false;
pthread_t Cgu::Notifier::thread_id;
std::unordered_set<Cgu::Notifier*>* Cgu::Notifier::object_set_p;
Cgu::PipeFifo* Cgu::Notifier::pipe_p;
Cgu::Thread::Mutex* Cgu::Notifier::set_mutex_p;
Cgu::Thread::Mutex* Cgu::Notifier::write_mutex_p;

namespace Cgu {

void Notifier::init() {

  if (!initialised) {
    // slots will execute in this thread (the one in which this function
    // is called) - save the thread id so that we can detect if this object 
    // is invoked through operator()() or emit() by the same thread and avoid
    // using the pipe
    thread_id = pthread_self();

    // these are global objects which should be initialised at program
    // start-up and last throughout program execution, so if an
    // allocation fails just let the program terminate - there is no
    // need to do any catches to avoid memory leaks.  They need to be
    // allocated dynamically to avoid unordered initialisation of
    // global static objects between different translation units, so
    // allowing Notifier objects to be global static objects.
    object_set_p = new std::unordered_set<Cgu::Notifier*>;
    pipe_p = new PipeFifo{PipeFifo::block};
    // setting FD_CLOEXEC creates the traditional race if another
    // thread is running in the program and it might exec concurrently
    // with the calling of this method in this thread, but that
    // doesn't really matter - having a child process sharing the
    // descriptor for a static unnamed pipe lasting until the end of
    // the program which contains no meaningful information (namely,
    // only Notifier addresses) isn't really a problem.  This can in
    // any event be circumvented by calling this static method before
    // any other threads are started.
    int read_fd = pipe_p->get_read_fd();
    int write_fd = pipe_p->get_write_fd();
    fcntl(read_fd, F_SETFD, fcntl(read_fd, F_GETFD) | FD_CLOEXEC);
    fcntl(write_fd, F_SETFD, fcntl(write_fd, F_GETFD) | FD_CLOEXEC);

    set_mutex_p = new Thread::Mutex;
    write_mutex_p = new Thread::Mutex;

    start_iowatch(read_fd,
		  Callback::make(&Notifier::read_pipe_cb),
		  G_IO_IN);

    initialised = true;
  }
}

Notifier::Notifier() {
  init();

  // we need to place the address of this object in the object set.  The object
  // set is used in Notifier::read_pipe_cb() to detect a case where between
  // Notifier::emit() being called on an object in one thread and the slot
  // executing (in the slot thread) in that object after propagation through
  // the pipe, the Notifier object ceases to exist
  Thread::Mutex::Lock lock{*set_mutex_p};
  object_set_p->insert(this);
}

Notifier::~Notifier() {

  Thread::Mutex::Lock lock{*set_mutex_p};
  object_set_p->erase(this);
}

Callback::SafeFunctor Notifier::connect(const Callback::SafeFunctor& cb) {

  return emitter.connect(cb);
}

Callback::SafeFunctor Notifier::connect(const Callback::SafeFunctor& cb, Releaser& r) {

  return emitter.connect(cb, r);
}

void Notifier::disconnect(const Callback::SafeFunctor& cb) {

  emitter.disconnect(cb);
}

void Notifier::block(const Callback::SafeFunctor& cb) {

  emitter.block(cb);
}

void Notifier::unblock(const Callback::SafeFunctor& cb) {

  emitter.unblock(cb);
}

void Notifier::emit() {

  // if the same thread is signalling as is connected to the slot, call the
  // functor directly rather than pass it through the pipe
  if (in_main_thread()) {
    emitter();
  }
    
  else {
    // make sure the write is atomic between threads
    Thread::Mutex::Lock lock{*write_mutex_p};
    void* instance_p = this;
    pipe_p->write(reinterpret_cast<char*>(&instance_p), sizeof(void*));
  }
}

void Notifier::read_pipe_cb(bool&) {
  void* instance_p;
  int remaining = sizeof(void*);
  ssize_t result;
  char* temp_p = reinterpret_cast<char*>(&instance_p);
  do {
    result = Notifier::pipe_p->read(temp_p, remaining);
    if (result > 0) {
      temp_p += result;
      remaining -= result;
    }
  } while (remaining             // more to come
	   && result             // not end of file
	   && result != -1);     // no error

  if (result > 0) {
    if (instance_p == 0) g_critical("Null pointer passed in Notifier::read_pipe_cb()\n");
    else {
      Notifier* notifier_p = static_cast<Notifier*>(instance_p);
      // check that the Notifier object concerned still exists
      { // scope block for mutex lock
	Thread::Mutex::Lock lock{*set_mutex_p};
	if (object_set_p->find(notifier_p) == object_set_p->end()) return;
      }
      // we do not want to hold the set mutex lock here as we have no idea what is
      // in the slot connected to this signal and we could be creating deadlocks.
      // Therefore despite the use of a std::set object to check validity of this
      // Notifier object, there is still a potential race condition here if the
      // lifetime of this object is controlled by a thread other than the one in
      // which the main program event loop runs (the object's lifetime could end
      // between the set mutex lock being released above and Notifier::emit()
      // completing below).  Notifier objects are normally constructed and destroyed
      // in the same thread as that in which the main program event loop runs, but
      // where that is not the case users should use their own synchronisation methods
      // for lifetime determination.
      // If a connected callback throws, this will not propagate beyond the iowatch
      // dispatcher because that has a C linkage specification.  It will be caught
      // and reported there
      notifier_p->emitter();
    }
  }

  else { // end of file or error on pipe
    // throwing an exception will not work as they must be caught and reported in
    // the iowatch dispatcher because that has a C linkage specification - so just
    // report the error here
    g_critical("IO error in Notifier::read_pipe_cb()\n");
  }
}

} // namespace Cgu
