#!/bin/sh
# Copyright (c) 1999-2007 Gentoo Foundation
# Copyright (c) 2007-2009 Roy Marples <roy@marples.name>
# All rights reserved. Released under the 2-clause BSD license.

# This basically mounts $RC_SVCDIR as a ramdisk.
# The tricky part is finding something our kernel supports
# tmpfs and ramfs are easy, so force one or the other.
svcdir_restorecon()
{
	local rc=0
	if [ -x /usr/sbin/selinuxenabled -a -c /selinux/null ] &&
	  selinuxenabled; then
		restorecon $RC_SVCDIR
		rc=$?
	fi
	return $rc
}

mount_svcdir()
{
	# mount from fstab if we can
	fstabinfo --mount "$RC_SVCDIR" && return 0

	local fs= fsopts="-o rw,noexec,nodev,nosuid"
	local svcsize=${rc_svcsize:-1024}

	# Some buggy kernels report tmpfs even when not present :(
	if grep -Eq "[[:space:]]+tmpfs$" /proc/filesystems; then
		local tmpfsopts="${fsopts},mode=755,size=${svcsize}k"
		mount -n -t tmpfs $tmpfsopts rc-svcdir "$RC_SVCDIR"
		if [ $? -eq 0 ]; then
			svcdir_restorecon
			[ $? -eq 0 ] && return 0
		fi
	fi

	if grep -Eq "[[:space:]]+ramfs$" /proc/filesystems; then
		fs="ramfs"
		# ramfs has no special options
	elif [ -e /dev/ram0 ] \
		&& grep -Eq "[[:space:]]+ext2$" /proc/filesystems; then
		devdir="/dev/ram0"
		fs="ext2"
		dd if=/dev/zero of="$devdir" bs=1k count="$svcsize"
		mkfs -t "$fs" -i 1024 -vm0 "$devdir" "$svcsize"
	else
		echo
		eerror "OpenRC requires tmpfs, ramfs or a ramdisk + ext2"
		eerror "compiled into the kernel"
		echo
		return 1
	fi

	mount -n -t "$fs" $fsopts rc-svcdir "$RC_SVCDIR"
	if [ $? -eq 0 ]; then
		svcdir_restorecon
		[ $? -eq 0 ] && return 0
	fi
}

. "$RC_LIBEXECDIR"/sh/functions.sh
[ -r /etc/rc.conf ] && . /etc/rc.conf

# By default VServer already has /proc mounted, but OpenVZ does not!
# However, some of our users have an old proc image in /proc
# NFC how they managed that, but the end result means we have to test if
# /proc actually works or not. We do this by comparing two reads of
# /proc/self/environ for which we have set the variable VAR to two
# different values. If the comparison comes back equal, we know that
# /proc is not working.
mountproc=true
f=/proc/self/environ
if [ -e $f ]; then
	if [ "$(VAR=a cat $f)" = "$(VAR=b cat $f)" ]; then
		eerror "You have cruft in /proc that should be deleted"
	else
		einfo "/proc is already mounted, skipping"
		mountproc=false
	fi
fi
unset f

if $mountproc; then
	procfs="proc"
	[ "$RC_UNAME" = "GNU/kFreeBSD" ] && proc="linprocfs"
	ebegin "Mounting /proc"
	if ! fstabinfo --mount /proc; then
		mount -n -t "$procfs" -o noexec,nosuid,nodev proc /proc
	fi
	eend $?
fi

# Mount tmpfs on /run when directory exists.
# /run is a new directory for storing volatile runtime data.
# Read more about /run at https://lwn.net/Articles/436012
if [ -d /run ]; then
	if mountinfo -q /run; then
		einfo "/run is already mounted, skipping"
	else
		ebegin "Mounting /run"
		if ! fstabinfo --mount /run; then
			mount -t tmpfs -o mode=0755,nosuid,nodev tmpfs /run
		fi
		eend $?
	fi
	if [ ! -d /run/lock ]; then
		mkdir /run/lock
	fi
	if [ -d /run/lock ]; then
		chown root:uucp /run/lock
		chmod 0775 /run/lock
	fi
elif [ -e /run ]; then
	einfo "Unable to mount /run since it is not a directory"
fi

# Try to mount xenfs as early as possible, otherwise rc_sys() will always
# return RC_SYS_XENU and will think that we are in a domU while it's not.
if grep -Eq "[[:space:]]+xenfs$" /proc/filesystems; then
	ebegin "Mounting xenfs"
	if ! fstabinfo --mount /proc/xen; then
		mount -n -t xenfs xenfs /proc/xen -o nosuid,nodev,noexec
	fi
	eend $?
fi

. "$RC_LIBEXECDIR"/sh/init-common-post.sh
