// (C) Copyright 2011-2012 Hewlett-Packard Development Company, L.P.
define(['hp/services/REST', 'hp/services/IndexFilter', 'hp/services/Log', 'jquery'],
function(REST, IndexFilter, log) { "use strict";

    var IndexService = (function() {

        function IndexService() {
          
            function convertResource(resource) {
                if (resource.hasOwnProperty('displayName')) {
                    resource.name = resource.displayName;
                    resource.status = resource.healthStatus;
                    resource.state = resource.state;
                    delete resource.displayName;
                    delete resource.healthStatus;
                    delete resource.resourceState;
                }
            }
          
            function convertIndexResources(indexResources) {
                if (indexResources.hasOwnProperty('resources')) {
                    indexResources.members = indexResources.resources;
                    delete indexResources.resources;
                    // convert properties
                    $.each(indexResources.members, function (index, resource) {
                        convertResource(resource);
                    });
                    
                    indexResources.count = indexResources.totalIndexResources;
                    delete indexResources.totalIndexResources;
                }
            }
            
            function indexResourcesHandlers(handlers) {
                return {
                    success: function (data) {
                        convertIndexResources(data);
                        handlers.success(data);
                    },
                    error: handlers.error};
            }
            
            function convertTree(tree) {
                convertResource(tree.resource);
                $.each(tree.parents, function (assoc, parents) {
                    $.each(parents, function (index, parent) {
                        convertTree(parent);
                    });
                });
                $.each(tree.children, function (assoc, children) {
                    $.each(children, function (index, child) {
                        convertTree(child);
                    });
                });
            }
            
            function convertTrees(trees) {
                if (trees.hasOwnProperty('trees')) {
                    $.each(trees.trees, function (index, tree) {
                        convertTree(tree);
                    });
                } else {
                    convertTree(trees);
                }
            }
            
            function treesHandlers(handlers) {
                return {
                    success: function (data) {
                        convertTrees(data);
                        handlers.success(data);
                    },
                    error: handlers.error};
            }
            
            function serializeCategory(category) {
                var result;
                if ('array' === $.type(category)) {
                    result = $.map(category, function (cat, index) {
                        return 'category=' + cat;
                    }).join('&');
                } else {
                    result = 'category=' + category;
                }
                return result;
            }

            // TODO make a new method here that will take the new argument (associationNames)
            this.getIndexResources = function (category, start, count, handlers) {
                
                REST.getURI('/index/rest/index/resources?' +
                    serializeCategory(category) +
                    '&start=' + start + '&count=' + count + '&sort=name:asc',
                    indexResourcesHandlers(handlers));
            };
        
            this.getFilteredIndexResources = function (filter, handlers) {
                var uri = '/index/rest/index/resources?';
                var params = [];
                var filterData = filter.data;
                if (! filterData) {
                    log.warn('Reverting to deprecated filter in IndexService.' +
                        ' Convert caller to use IndexFilter.');
                    filterData = filter;
                }
                $.each(filterData, function (name, value) {
                    if ('properties' === name) {
                        $.each(value, function (property, propertyValue) {
                            if ('array' === $.type(propertyValue)) {
                                $.each(propertyValue, function (index, val) {
                                    params.push('filter=' + property + ':' + val);
                                });
                            } else {
                                params.push('filter=' + property + ':' + propertyValue);
                            }
                        });
                    } else if ('terms' === name) {
                        if (value.length > 0) {
                            var tval=[];
                            // if the value contains white space, add double quotes
                            $.each(value, function (index, val) {
                                tval.push((/\s/g.test(val)) ? '"' + val + '"' : val);
                            });
                            params.push('userQuery=' + tval.join(' '));
                        }
                    } else if ('userQuery' !== name) {
                        if ('array' === $.type(value)) {
                            $.each(value, function (index, val) {
                                params.push(name + '=' + val);
                            });
                        } else {
                            params.push(name + '=' + value);
                        }
                    }
                });
                uri += params.join('&');
                REST.getURI(uri, indexResourcesHandlers(handlers));
            };

            this.getAssociationTree = function (uri, depth, handlers) {
                REST.getURI('/index/rest/index/associationtree?rootUri=' + uri +
                    '&depth=' + depth, handlers);
            };

            this.searchIndexResources = function (filter, start, count, handlers) {
                if ('string' === typeof(filter)) {
                    var realFilter = new IndexFilter();
                    realFilter.setUserQuery(filter);
                    filter = realFilter;
                }
                var query = '';
                if (filter.data.terms) {
                    query = filter.data.terms.join(' ');
                }
                REST.getURI('/index/rest/index/resources?userQuery=' + query +
                    '&start=' + start + '&count=' + count,
                    indexResourcesHandlers(handlers));
            };

            this.suggestions = function(category, query, start, count, handlers) {
                var uri = '/index/rest/index/suggestions?';
                if (category) {
                    uri += serializeCategory(category) + '&';
                }
                uri += 'userQuery=' + query + '&start=' + start + '&count=' + count;
                REST.getURI(uri, handlers);
            };

            this.getAssociationTrees = function (category, depth, associationNames,
                start, count, handlers) {
                REST.getURI('/index/rest/index/associationtrees?'+
                        serializeCategory(category) +
                        '&depth=' + depth+
                        '&associationNames='+associationNames+
                        '&start=' + start+
                        '&count=' + count,
                        handlers);
            };

            this.getIndexForResource = function (uri, handlers) {
                REST.getURI('/index/rest/index/resources?start=0&count=1&filter=_uri:' +
                    uri, handlers);
            };

            /**
             * Retrieves the associations from the index service
             * @param {string} startObjUri - The starting object uri (optionally null)
             * to look at when getting associations.
             * @param {string} endObjUri - The ending object uri (optionally null)
             * to look at when getting associations.
             * @param {string} associationName - One of a set association names
             * (optionally null).
             * @param {string} relationship - Boolean value (optionally null).
             * @param {object} handlers - Success and Error handlers.
             */
            this.getAssociations = function (startObjUri, endObjUri, associationName,
                relationship, handlers) {
                var params = [];
                if (startObjUri) {
                    params.push("startObjUri=" + startObjUri);
                }
                if (endObjUri) {
                    params.push("endObjUri=" + endObjUri);
                }
                if (associationName) {
                    params.push("associationName=" + associationName);
                }
                if (relationship) {
                    params.push("hasARelationship=" + relationship);
                }
                var request = "/index/rest/index/associations?" + params.join('&');
                REST.getURI(request, handlers);
            };

            /**
             * Retrieves the parent and children associations from the index service
             * @param {object} Options object telling parameters to set in REST call.
             * Not used if not set.
             *   Supported params (Pass either uri or catagory, but not both)
             *
             *     uri = Uri of item to retrieve. Retrieves single item specified by Uri.
             *
             *     catagory = What type of object to get. {servers,enclosure,...}.
             *                Fetches a range of objects of this type.
             *       start = Which item to start with (For use with catagory)
             *       count = How many items to fetch (For use with catagory)
             *
             *     childDepth = Descent depth for following child associations. (Always valid)
             *     parentDepth = Descent depth for following parent associations. (Always valid)
             *     handlers = object w/ success/error handlers (Always)
             */
            this.getParentAndChildrenAssociations = function (options) {
                var params = [];
                var handlers = null;
                var uri = "";

                if(arguments.length == 1) {
                    // New version, parsing an Option object
                    uri = ((options.uri) ? options.uri : "");
                    if (options.category) {
                        if ('array' === $.type(options.category)) {
                            $.each(options.category, function (index, cat) {
                                params.push("category=" + cat);
                            });
                        } else {
                            params.push("category=" + options.category);
                        }
                    }
                    if (options.childDepth) {
                        params.push("childDepth=" + options.childDepth);
                    }
                    if (options.parentDepth) {
                        params.push("parentDepth=" + options.parentDepth);
                    }
                    if (options.start) {
                        params.push("start=" + options.start);
                    }
                    if (options.count) {
                        params.push("count=" + options.count);
                    }
                    if (options.sort) {
                        params.push("sort=" + options.sort);
                    }
                    if (options.handlers) {
                        handlers = options.handlers;
                    }
                } else {
                    // Old version which took multiple arguments
                    uri = arguments[0];
                    var childDepth = arguments[1];
                    var parentDepth = arguments[2];
                    handlers = arguments[3];
                
                    if (childDepth) {
                        params.push("childDepth=" + childDepth);
                    }
                    if (parentDepth) {
                        params.push("parentDepth=" + parentDepth);
                    }
                }
                REST.getURI("/index/rest/trees" + uri + "?" + params.join('&'),
                    treesHandlers(handlers));
            };
        }
        return new IndexService();
    }());
    
    return IndexService;
});
