// (C) Copyright 2011-2012 Hewlett-Packard Development Company, L.P.
define(['fs/presenter/users/UsersPresenter',
        'hp/view/FormStateView',
        'hp/core/Localizer',
        'hp/model/Session',
        'hp/core/Notifications',
        'jquery',
        'hp/lib/jquery.hpToggle',
        'lib/jquery.validate' ],
function(presenter, FormStateView, Localizer, session, notifications) { "use strict";

    var UsersAddView = (function() {

        /**
         * Constructor
         */
        function UsersAddView() {

            var FORM = '#cic-user-add-form',
                ADD = '#cic-user-add',
                ADDAGAIN = "#cic-user-add-again",
                NAME = '#cic-user-loginname',
                FULLNAME = '#cic-user-fullname',
                MAX_NAME_LENGTH = 39,
                INITIAL_PASSWD = '#cic-user-initialpasswd',
                CONFIRM_PASSWD = '#cic-user-confirmpasswd',
                MIN_PASSWORD_LENGTH = 8,
                EMAIL = '#cic-user-email',
                OFFICE_PHONE = '#cic-user-officephone',
                MOBILE_PHONE = '#cic-user-mobilephone',
                ROLE_VALIDATOR = '#cic-user-role-validator',
                ROLES_LIST = '#cic-user-roles-list',
                ROLES = ROLES_LIST + ' input[type="checkbox"]',
                ROLE_SWITCH = '#cic-user-role-switch',
                CANCEL = '#cic-user-add-cancel',
                NUMBER_OF_ROLES = 2, //For Pulsar
                ROLE_LIST ='#cic-user-roles',
                ROLE_SELECT_NAME = 'cicuserrolename',
                ROLE_SWITCH_RADIO = 'input[name=cicuserrolename]:checked',
                ROLE_SPECIALIZED_CHECKBOXES = "input[type='checkbox'][id^='cic-user-role']",
                SPECIALIZED_VALUE = 'specialized',
                FULL_VALUE = 'full',
                READONLY_VALUE = 'readonly',
                SPECIALIZED_BUTTONSET ='#cic-user-role-name-specialized';

            var validator;
            // prevent double click
            var isSubmitted;
            var formStateView = new FormStateView();
            var firstRoleTemplate = null,
                roleTemplate = null;
            var isSpecialized;
            var currentUserName;
            var currentUserFullName;
            var addAgain = false;
            var twoRoles = false;
            var taskName;
            var isValidLogin = true;//assume is true i.e loginname is valid
            var numberOfClicks = 0;

            /**
             * @private
             */
            function onSpecializedUserClick() {
                isSpecialized = true;
                $(SPECIALIZED_BUTTONSET).attr("checked", true);
                $(ROLES_LIST).show();
            }

            /**
             * @private
             */
            function reset() {
                $(NAME).val("");
                $(FULLNAME).val("");
                $(INITIAL_PASSWD).val("");
                $(CONFIRM_PASSWD).val("");
                $(EMAIL).val("");
                $(OFFICE_PHONE).val("");
                $(MOBILE_PHONE).val("");
                $(ROLE_SWITCH).attr('checked', true).trigger('change');
                $(ROLES).prop("checked", false);
                validator.resetForm();
                formStateView.reset();
                if (addAgain){
                    onSpecializedUserClick();
                    $(NAME).focus();
                }
                isValidLogin = true;
                numberOfClicks = 0;
                isSubmitted = false;
            }

            /**
             * @private
             */
            function resetCancel() {
                reset();
            }

            /**
             * @private
             */
            function onRolesAddedSuccess(user) {
                var alertMsg = Localizer.getString('fs.users.add.addSuccess', [currentUserName]);
                var userAddedSuccess = {
                    summary : alertMsg,
                    uri : taskName,
                    progress : 100,
                    changing : true,
                    status : "ok"
                };
                notifications.add(userAddedSuccess, true);
                resetCancel();
                formStateView.setMessage({summary:alertMsg, status:'ok', changing : false});
                if (!addAgain) {
                    $(CANCEL).trigger('click');
                }
            }

            /**
             * @private
             */
            function onRolesAddedError(errorMessage) {
                var addMsg = Localizer.getString('fs.users.add.roleerror',[$(NAME).val()]);
                var errorMsg = Localizer.formatRestError(addMsg,
                        errorMessage);
                formStateView.setMessage({summary: addMsg, status: 'error',
                    details: errorMessage});
                presenter.deleteUser(currentUserName);
                var userAddedError = {
                    summary : addMsg,
                    uri : taskName,
                    changing : false,
                    status : "error",
                    details : errorMsg
                };
                notifications.add(userAddedError, true);
                isSubmitted = false;
            }

            /**
             * @private
             */
            function onUserAddedSuccess(user) {
                var username, fullname, alertMsg;
                var roles = [];
                username = user.userName;
                fullname = user.fullName;
                alertMsg = Localizer.getString('fs.users.add.addingMsg', [ username ]);
                var userAddProgress = {
                    summary : alertMsg,
                    uri : taskName,
                    progress : 50,
                    changing : true,
                    status : "info"
                };
                notifications.add(userAddProgress, false);
                formStateView.setMessage({summary:alertMsg, status:'info', changing:true});

                if (twoRoles){
                   roles.push($(ROLE_SWITCH_RADIO).val());
                }
                else if(isSpecialized)
                {
                    $(ROLES).each(function(index, role) {
                        if ($(role).is(':checked')) {
                            roles.push($(role).val());
                        }
                    });
                }
                else if($(ROLE_SWITCH_RADIO).val() === FULL_VALUE){
                    roles.push(Localizer.getString('fs.users.common.infrastructure_administrator'));
                }else if ($(ROLE_SWITCH_RADIO).val() === READONLY_VALUE){
                    roles.push(Localizer.getString('fs.users.common.read_only'));
                }
                currentUserName =username;
                currentUserFullName = fullname;
                presenter.assignUserToRoles(username, roles, {
                    success : onRolesAddedSuccess,
                    error : onRolesAddedError
                });
            }

            /**
             * @private
             */
            function onUserAddedError(errorMessage) {
                var addMsg = Localizer.getString('fs.users.add.error',[$(NAME).val()]);
                var errorMsg = Localizer.formatRestError(addMsg,
                        errorMessage);
                formStateView.setMessage({summary: addMsg, status: 'error',
                    errorMessage: errorMessage});
                var userAddedError = {
                    summary : addMsg,
                    uri : taskName,
                    changing : false,
                    status : "error",
                    details : errorMsg
                };
                notifications.add(userAddedError, true);
                isSubmitted = false;
            }

            /**
             * @private
             */
            function onAddUserClick() {
                if ($(FORM).valid() && isSubmitted == false) {
                    numberOfClicks = numberOfClicks + 1;
                    isSubmitted = true;
                    var notifMsg = Localizer.getString('fs.users.add.addingMsg',[$(NAME).val()]);
                    formStateView.setMessage(notifMsg, 'info', true);
                    if(numberOfClicks == 1){
                        taskName = session.getUser() + '-addUserTask-' + (new Date()).getTime();
                    }
                    var userAddBegin = {
                        summary : notifMsg,
                        uri : taskName,
                        progress : 0,
                        changing : true,
                        status : "info"
                    };
                    notifications.add(userAddBegin, false);
                    presenter.addUser($(NAME).val(), $(FULLNAME).val(),
                            $(INITIAL_PASSWD).val(), $(CONFIRM_PASSWD)
                            .val(), $(EMAIL).val(), $(
                                    OFFICE_PHONE).val(),
                                    $(MOBILE_PHONE).val(), {
                        success : onUserAddedSuccess,
                        error : onUserAddedError
                    });
                }

                return false;
            }


            /**
             * @private
             */
            function onFullOrReadonlyUserClick() {
                isSpecialized = false;
                $(ROLES_LIST).hide();
            }


            /**
             * @private
             */
            function checkRoles() {
                var result = false;
                $(ROLES).each(function(index, role) {
                    if(isSpecialized)
                    {
                        if ($(role).is(':checked')) {
                            result = true;
                        }
                    }
                    else
                    {
                      result = true;
                    }
                });
                return result;
            }

            //to construct the role section in the page
            function updateRoles(){

                var roles = presenter.getRoles(),
                    rolesList = $(ROLES_LIST);

                $(rolesList).empty();

                if (roles.length == NUMBER_OF_ROLES){
                    twoRoles = true;
                    isSpecialized = true;
                    $(ROLE_SWITCH).empty();
                }
                var counter = 0;//counter as in else case case we need to know the right order

                $.each(roles, function(index, role) {//iterating through all available roles
                    var theRoleName =  role.roleName ? role.roleName.replace(/_/, ' ') : 'noname';
                    var correctCaseRole = '';
                    var splitRole = role.roleName.replace(/_/, ' ').split(' ');
                    $.each(splitRole, function(key, word) {
                        if(key !=0){
                            correctCaseRole = correctCaseRole +' '+ word.toLowerCase();
                        }else{
                            correctCaseRole = correctCaseRole + word;
                        }
                    });
                    if(twoRoles){//Pulsar has only 2 roles
                        var radio = $("<input type='radio'>");
                        var id = theRoleName.replace(' ','-');//id is the roles as in DB, space replaced by hyphen
                        radio.attr("name", ROLE_SELECT_NAME)
                             .attr("id", id)
                             .attr("value", theRoleName);
                        var label = $('<label/>');
                        label.attr("for", id);
                        label.html(correctCaseRole);

                        $(ROLE_SWITCH).append(radio);
                        $(ROLE_SWITCH).append(label);
                        if (counter == 0){//roles validation error
                            var valid = $("<input type='hidden'>");
                              valid.attr("name", "role_validator")
                                   .attr("id", "cic-user-role-validator")
                                   .addClass("hp-error");
                              $(ROLE_SWITCH).append(valid);
                         }
                        if (index !== roles.length - 1){
                           $(ROLE_SWITCH).append($('<br/>'));
                        }
                        counter = counter + 1;
                    }else {
                        //because full and read only are in the toggle
                        if (role.roleName !== 'No_Privileges' &&
                            role.roleName !== Localizer.getString('fs.users.common.infrastructure_administrator') &&
                            role.roleName !== Localizer.getString('fs.users.common.read_only')) {

                            var item = (counter === 0) ? firstRoleTemplate.clone() : roleTemplate.clone();
                            var idChkbox = 'cic-user-role-' + counter;
                            var labelChkbox = correctCaseRole;
                            item.attr('id', '');
                            item.show();
                            $('input[type="checkbox"]', item).attr('id', idChkbox)
                                .val(role.roleName)
                                .attr('name', role.roleName);
                            $('label:eq(1)', item).text(labelChkbox)
                                .attr('for', idChkbox);
                            rolesList.append(item);

                            if (counter === 0) {
                                $(ROLE_SPECIALIZED_CHECKBOXES).change(function(){
                                    if(checkRoles()){
                                        $(ROLE_VALIDATOR).next().hide();//the auto generated error label
                                    }
                                });
                                $(ROLE_VALIDATOR).rules("add", { userRoleValidation : ROLE_VALIDATOR });
                            }
                            counter = counter + 1;
                        }
                    }
                });
                if (twoRoles){
                    $(ROLE_LIST).empty(); //dont show the list of checkboxes
                }
                formStateView.reset();
            }

            /**
             * @private
             */
           function checkLoginNameExists(loginName){
                presenter.checkLoginNameExists(loginName,{
                    success : function (doesExist){
                        isValidLogin = !doesExist; //if exists, it is not a valid login
                    },
                    error : function (errorInfo){
                        isValidLogin = true; //if any error, just say its a valid login - will be caught at backend anyway
                    }
                } );

              return isValidLogin;
            }
            /**
             * @private
             */
            function initForm() {

                isSubmitted = false;
                var validRoleValidation = Localizer.getString('fs.users.validations.roleValidation');
                var validMatchPasswd = Localizer.getString('fs.users.add.matchpasswd');
                var validUserNameAlphanum = Localizer.getString('fs.users.validations.usernameValidation');
                var validUserNameFirstletter = Localizer.getString('fs.users.validations.usernameValidation2');
                var validPhoneNum = Localizer.getString('fs.users.validations.phonenumValidation');
                var validFullname = Localizer.getString('fs.users.validations.fullnameValidation');
                var validPassword = Localizer.getString('fs.users.validations.passwordValidation');
                var validLoginNameValidation = Localizer.getString('fs.users.validations.validLoginNameValidation');

                $.validator.addMethod("userRoleValidation", function
                        (value, element, param) {
                    return checkRoles();
                }, validRoleValidation);

                $.validator.addMethod("cicAddUserLoginNameValidation", function
                        (value, element, param) {
                    return checkLoginNameExists(value);
                }, validLoginNameValidation);

                validator = $(FORM).validate({
                    rules : {
                        loginname : {
                            required : true,
                            maxlength : MAX_NAME_LENGTH,
                            cicUsersAddAlphaNumCheck : NAME,
                            cicUsersAddStartsWithLetterCheck : NAME,
                            cicAddUserLoginNameValidation : NAME
                        },
                        fullname : {
                            cicUsersAddFullNameCheck : FULLNAME,
                            maxlength : MAX_NAME_LENGTH
                        },
                        email : {
                            email : EMAIL
                        },
                        initialpasswd : {
                            required : true,
                            cicUsersAddCheckPasswdString : INITIAL_PASSWD,
                            minlength : MIN_PASSWORD_LENGTH
                        },
                        confirmpasswd : {
                            cicUsersAddMatchPasswd : INITIAL_PASSWD
                        },
                        officephone : {
                            cicUsersAddPhoneNumCheck : OFFICE_PHONE
                        },
                        mobilephone : {
                            cicUsersAddPhoneNumCheck : MOBILE_PHONE
                        },
                        cicuserrolename : {//only when all are radio buttons(For Pulsar-2 roles)
                            required : true
                        }
                    },
                    messages : {
                        cicuserrolename : {//only when all are radio buttons(For Pular-2 roles)
                            required : validRoleValidation
                        }
                    },
                    errorPlacement : function(error, element){//only when all are radio buttons(For Pular-2 roles)
                        if (twoRoles && element.attr('type') === 'radio') {
                            //insert the error message after first radio button's label
                            var ff='input[type="radio"][id="' + element.attr('id') + '"]:first';
                            error.insertAfter(element.siblings('label[for="' + $(ff).attr("id") +'" ]'));
                        }else{
                            error.insertAfter(element);
                        }
                    }
                });

            //alphanumeric and - . _
            jQuery.validator.addMethod("cicUsersAddAlphaNumCheck", function(value,
                element, param) {
                return (/^[0-9-_a-zA-Z]+$/.test(value));
                },
            validUserNameAlphanum);

            //No special characters allowed <>;,"'&\/|+.:=
            jQuery.validator.addMethod("cicUsersAddCheckPasswdString", function(
                value, element, param) {
                return (/^[^<>;,"'&\\\/|+:= ]+$/.test(value));
                },
            validPassword);

            //should start with a letter only
            jQuery.validator.addMethod("cicUsersAddStartsWithLetterCheck", function(
                value, element, param) {
                return (/^[a-zA-Z]+.*$/.test(value));
                },
            validUserNameFirstletter);

            // check for  special characters
            jQuery.validator.addMethod("cicUsersAddPhoneNumCheck", function(value,
                element, param) {
                var isValidPh = /^[^"'&=<>]+$/
                  .test(value);
                return (value.length == 0 || isValidPh);
                },
            validPhoneNum);

            // check for fullname contain letters ' . - and space
            jQuery.validator.addMethod("cicUsersAddFullNameCheck",function(value,
                element, param) {
                var isValidFullname = /^[a-zA-Z0-9 ._'\-]+$/.test(value);
                return (isValidFullname || value.length == 0);
                },
            validFullname);

            //match password
            jQuery.validator.addMethod("cicUsersAddMatchPasswd", function(
                value, element, param) {
                // bind to the blur event of the target in order to
                // revalidate whenever the target field is updated
                var target = $(param).unbind(".validate-equalTo")
                  .bind("blur.validate-equalTo", function() {
                    $(element).valid();
                  });
                return value == target.val();
                },
            validMatchPasswd);

        }

            function displayRoles() {
                if ($(ROLE_SWITCH_RADIO).val() === SPECIALIZED_VALUE){
                    onSpecializedUserClick();
                }
                else{
                    onFullOrReadonlyUserClick();
                }
            }

            /**
             * @public
             */
            this.init = function() {
                // Remove the template for a role from the DOM (used for cloning later)
                firstRoleTemplate = $('#cic-user-first-role-template').detach();
                roleTemplate = $('#cic-user-role-template').detach();

                initForm();

                formStateView.init({form: FORM,
                    autoTrack: true});

                resetCancel();

                $(ADDAGAIN).on('click', function(ev) {
                    addAgain = true;
                    onAddUserClick.call(this);
                    ev.preventDefault();
                });
                $(ADD).on('click', function(ev) {
                    addAgain = false;
                    onAddUserClick.call(this);
                    ev.preventDefault();
                });

                $(ROLE_SWITCH).buttonset();
                onSpecializedUserClick();
//                $(ROLE_SWITCH).hpToggle({
//                    checked: Localizer.getString('fs.users.add.specialized'),
//                    unchecked: Localizer.getString('fs.users.add.full')
//                });


                $(ROLE_SWITCH).change(function(e){
                    displayRoles();
                    if(checkRoles()){
                        $(ROLE_VALIDATOR).next().hide();//the auto generated error label
                    }
                });

                $(CANCEL).on('click', function(ev){
                  formStateView.reset();
                });

                presenter.on('rolesChanged', updateRoles);
            };
            /**
             * @public
             */
            this.resume = function() {
                resetCancel();
                onSpecializedUserClick();
                presenter.on('rolesChanged', updateRoles);

            };

            this.pause = function () {
                formStateView.reset();
                presenter.off('rolesChanged', updateRoles);
            };
        }

        return new UsersAddView();
    }());

    return UsersAddView;
});
