// (C) Copyright 2011-2012 Hewlett-Packard Development Company, L.P.
/**
 * @type {MasterListView}
 */
define(['hp/presenter/MasterPanePresenter',
    'hp/view/SearchBoxView',
    'hp/view/MasterErrorView',
    'hp/core/Localizer',
    'hp/core/Style',
    'jquery'],
function (MasterPanePresenter, searchBoxView, errorView, localizer, style) {
"use strict";

    var MasterListView = (function () {

        // consts can go here
        var LIST = '.hp-master-list';
        var ITEM = '.hp-master-list-item';
        var TITLE = '.hp-master-header .hp-master-title';
        var COUNT = '.hp-page-item-count';
        var COLLAPSER = 'header .hp-collapser';
        var FILTERBAR_CONTROL = '.hp-filterbar-control';
        var RESET_FILTERS = '.hp-filter-reset';
        var FILTERBAR = '.hp-master-filter';
        var FILTER = '.hp-filter[data-filter-property]';
        var SORTER = '#hp-master-sort';
        var SELECTED = 'hp-selected';
        var ACTIVE = 'hp-active';
        var EXPANDED = 'hp-expanded';
        var COLLAPSED = 'hp-collapsed'; // for IE8
        var FILTERBAR_OPEN = 'hp-filterbar-open';
        var KEY_DOWN = 40;
        var KEY_UP = 38;

        /**
         * @constructor
         */
        function MasterListView() {

            //
            // private class instance variables
            //
            var presenter = new MasterPanePresenter();
            var page = null;
            var resource = null;
            var renderer = null;
            var multiSelect = false;
            var addHelp = null;
            var pendingItemHeight = 0;
            var manualScrollTo = -1;
            var expandedUris = [];
            
            function fireRelayout() {
                // we delay so any CSS animation can finish
                setTimeout(function () {
                    page.trigger('relayout');
                }, style.animationDelay());
            }

            function clearSelection() {
                $(ITEM, page).removeClass(SELECTED);
            }

            function setSelection(selection) {
                if (selection.uris && selection.uris.length > 0) {
                    $.each($(ITEM), function () {
                        var item = this;
                        var indexResult = $.data(item, 'hpIndexResult');

                        if ((indexResult) &&
                            ($.inArray(indexResult.uri, selection.uris) !== -1)) {
                            $(item).addClass(SELECTED);
                        }
                    });
                }
            }
            
            function scrollToSelection() {
                var scrollTo = manualScrollTo;
                var reset = false;
                
                if (scrollTo === -1) {
                    var item = $(ITEM + '.' + SELECTED).first();
                    if (item.length > 0) {
                        if ($(item).position().top < 0) {
                            scrollTo =  $(LIST, page).scrollTop() + item.position().top;                                                
                        } else if (($(item).position().top + $(item).outerHeight()) >
                            $(LIST, page).innerHeight()) {
                            // make sure we don't scroll up so we can't see the top of the item
                            scrollTo =  Math.min(
                                $(LIST, page).scrollTop() + item.position().top,
                                $(LIST, page).scrollTop() + item.position().top -
                                $(LIST, page).innerHeight() + item.outerHeight()); 
                        }
                    }
                    reset = true;
                }
                
                if (scrollTo !== -1) {
                    $(LIST, page).scrollTop(scrollTo);
                }

                setTimeout(function () {
                    if (reset) {
                        manualScrollTo = -1; // reset since we just scrolled
                    }
                }, 100);
            }
            
            function onItemClick(event) {
                // get index result from object and call presenter
                var item = $(event.target).closest(ITEM);
                var indexResult = $(item).data('hpIndexResult');
                if (multiSelect && (event.ctrlKey || event.metaKey)) {
                    presenter.toggleSelectIndexResult(indexResult);
                } else if (multiSelect && event.shiftKey) {
                    presenter.selectContiguousIndexResults(indexResult);
                } else {
                    presenter.selectIndexResult(indexResult);
                }
                $(LIST, page).focus();
            }
            
            function setEmptyMessage() {
                var splash = $('<li></li>').addClass('hp-master-list-splash');
                var empty = $('<div></div>').addClass('hp-master-list-empty').
                    text(presenter.getEmptyMessage());
                splash.append(empty);
                if (presenter.haveContacted() && ! presenter.haveSome() && addHelp) {
                    splash.append('<div class="hp-help">' + addHelp + '</div>');
                }
                $(LIST, page).append(splash);
            }
            
            function saveExpandedUris() {
                expandedUris = $(ITEM + '.' + EXPANDED, page).map(function () {
                    return $(this).data('hpIndexResult').uri;
                }).get();
            }
            
            function onCollapserClick(event) {
                $(event.currentTarget).toggleClass(ACTIVE);
                $(event.currentTarget).closest(ITEM).
                    toggleClass(EXPANDED).toggleClass(COLLAPSED);
                saveExpandedUris();
                scrollToSelection(); // in case we expanded off the bottom
            }
            
            function alignSelection(item, uri) {
                // set selection state
                if ($.inArray(uri, presenter.getSelection().uris) != -1) {
                    item.addClass(SELECTED);
                } else {
                    item.removeClass(SELECTED);
                }
            }
            
            function alignExpansion(item, uri) {
                 if (-1 !== $.inArray(uri, expandedUris)) {
                      item.addClass(EXPANDED);
                      $(COLLAPSER, item).addClass(ACTIVE);
                  } else {
                      item.addClass(COLLAPSED);
                      $(COLLAPSER, item).removeClass(ACTIVE);
                  }
            }
            
            function createRenderedItem(result) {
                var item = renderer(result);
                item.data('hpIndexResult', result);
                // set expansion state
                alignExpansion(item, result.uri);
                // set selection state
                alignSelection(item, result.uri);
                return item;
            }
            
            function createPendingItem(result, height) {
                var item = $('<li></li>').
                    addClass("hp-pending hp-master-list-item").
                    css('height', (height ? height : pendingItemHeight)).
                    data('hpIndexResult', result);
                // set expansion state
                alignExpansion(item, result.uri);
                // set selection state
                alignSelection(item, result.uri);
                return item;
            }
            
            function replaceItem(item, result) {
                var mismatch = ($(item).data('hpIndexResult').uri !== result.uri);
                var makePending = false;
                if (($(item).position().top + $(item).outerHeight() +
                    (2 * pendingItemHeight)) < 0) {
                    // above region, empty if different
                    if (! $(item).hasClass('hp-pending') || mismatch) {
                        makePending = true;
                    }
                } else if ($(item).position().top >
                    ($(LIST, page).innerHeight() + (2 * pendingItemHeight))) {
                      // below region, empty if not
                    if (! $(item).hasClass('hp-pending') || mismatch) {
                        makePending = true;
                    } else {
                        // already empty, can quit loop
                        return false;
                    }
                } else {
                    // within scroll region, update
                    $(item).replaceWith(createRenderedItem(result));
                }
                
                if (makePending) {
                    if (! $(item).hasClass('hp-pending') || mismatch) {
                        if (mismatch) {
                            $(item).replaceWith(createPendingItem(result));
                        } else {
                            $(item).replaceWith(createPendingItem(result,
                                $(item).height()));
                        }
                    }
                }
                
                return true;
            }
            
            /**
             * @private
             * @param {Object} data the IndexResults
             */
            function onIndexResultsChange(indexResults) {
                var oldItem, newItem, oldItems;
                var overflowed = false;
                
                if (indexResults) {
                    $(COUNT, page).text(indexResults.total);
                }
                
                if (indexResults && renderer) {
                
                    clearSelection();
                    $(LIST + ' .hp-master-list-control', page).remove();
                    $(LIST + ' .hp-master-list-splash', page).remove();
                    oldItems = $(ITEM, page);
                    
                    $(TITLE, page).text(presenter.getHeaderTitle());
                
                    $.each(indexResults.members, function (index, result) {
                        if (index >= oldItems.length) {
                            oldItem = null;
                        } else {
                            oldItem = $(oldItems[index]);
                        }
                        if (oldItem) {
                            if (! replaceItem(oldItem, result) && ! overflowed) {
                                overflowed = true;
                            }
                        } else {
                            if (! overflowed) {
                                newItem = createRenderedItem(result);
                                $(LIST, page).append(newItem);
                                if (! pendingItemHeight) {
                                    pendingItemHeight = newItem.height();
                                } else {
                                    pendingItemHeight = Math.min(pendingItemHeight,
                                        newItem.height());
                                }
                                if (newItem.position().top >
                                    ($(LIST, page).innerHeight() +
                                        (2 * pendingItemHeight))) {
                                    overflowed = true;
                                }
                            } else {
                                newItem = createPendingItem(result);
                                $(LIST, page).append(newItem);
                            }
                        }
                    });
                    
                    // remove anything beyond the results we have
                    $(ITEM, page).slice(indexResults.count).remove();
                    
                    if (indexResults.total > indexResults.count) {
                        $(LIST, page).append(
                            '<li class="hp-master-list-control">' + 
                            '<a id="hp-master-show-more">' +
                            localizer.getString('core.master.showMore') +
                            '</a></li>');
                        $('#hp-master-show-more', page).click(function (event) {
                            $(event.currentTarget).replaceWith('…');
                            presenter.showMore();
                        });
                    } else if (indexResults.total === 0) {
                        setEmptyMessage();
                    }
                    
                    // scroll to selection, if not manually scrolled
                    scrollToSelection();
                }
            }

            // align selection classes with uris
            function onSelectionChange(selectionArg) {
                clearSelection();
                setSelection(selectionArg);
                scrollToSelection();
            }
            
            function closeFilterbar() {          
                $('.hp-sub-nav', page).append(
                    $(FILTERBAR + ' .hp-filter', page).removeClass('hp-active hp-pinned'));
                $(page).removeClass(FILTERBAR_OPEN);
                fireRelayout();
            }

            function openFilterbar() {
                $(page).addClass(FILTERBAR_OPEN);
                $(FILTERBAR, page).append(
                    $('.hp-sub-nav .hp-filter', page).addClass('hp-active hp-pinned'));
                fireRelayout();
            }
            
            function toggleFilterbar() {
                if ($(FILTERBAR, page).length > 0 &&
                    ! $(page).hasClass(FILTERBAR_OPEN)) {
                    openFilterbar();
                } else {
                    closeFilterbar();
                }
            }
            
            function resetFilters() {
                presenter.resetFilters();
            }
            
            function onPropertyFilterChange(event, value) {
                var name = $(event.currentTarget).attr('data-filter-property');
                if (name) {
                    presenter.setFilterValue(name, value);
                }
            }
            
            function onSortChange(event, sort) {
                presenter.setSort(sort.id, sort.direction);
            }
            
            function onFilterChange(filter) {
                var sort = presenter.getSort();
                $(FILTER, page).each(function (index, elem) {
                    var name = $(elem).attr('data-filter-property');
                    var value = presenter.getFilterValue(name);
                    $(elem).hpSelect('set', value);
                });
                if (sort) {
                    $(SORTER, page).hpSelect('set',
                        {ids: [sort.name], direction: sort.direction});
                }
                searchBoxView.setLocalSearchText(presenter.getSearchText());
            }
            
            function onScroll() {
                var result;
                
                // stop auto scrolling if the user scrolls
                manualScrollTo = $(LIST, page).scrollTop();
                
                $(ITEM, page).each(function (index, item) {
                    result = $(item).data('hpIndexResult');
                    return replaceItem(item, result);
                });
            }
            
            function onKeyDown(event) {
                if ($(LIST, page).is(":focus")) {
                    var keyCode = (event.which ? event.which : event.keyCode);
                    if (keyCode === KEY_DOWN) {
                        manualScrollTo = -1;
                        $('.hp-master-list-item.hp-selected').next().trigger('click');
                        event.stopPropagation();
                        return false;
                    } else if (keyCode === KEY_UP) {
                        manualScrollTo = -1;
                        $('.hp-master-list-item.hp-selected').prev().trigger('click');
                        event.stopPropagation();
                        return false;
                    }
                }
            }
            
            /**
             * @public
             * Stop the timer polling of the index service.
             */
            this.pause = function () {
                presenter.pause();
                presenter.off("indexResultsChange", onIndexResultsChange);
                presenter.off("selectionChange", onSelectionChange);
                presenter.off("filterChange", onFilterChange);
                errorView.pause();
            };

            /**
             * @public
             * Resume the timer polling of the index service.
             */
            this.resume = function () {
                manualScrollTo = -1;
                presenter.on("indexResultsChange", onIndexResultsChange);
                presenter.on("selectionChange", onSelectionChange);
                presenter.on("filterChange", onFilterChange);
                presenter.resume();
                errorView.resume();
            };
            
            /**
             * @public
             * Intialize the view.
             */
            this.init = function (args) {
                resource = args.resource;
                page = args.page;
                renderer = args.listItemRenderer;
                multiSelect = args.multiSelect;
                addHelp = args.addHelp;
                
                presenter.init(args.resource, args.routePrefix);
                errorView.init(presenter);
                setEmptyMessage();
                
                $('.hp-sub-nav .hp-select', page).hpSelect();
                $('.hp-master-filter .hp-select', page).hpSelect();
                $(FILTERBAR_CONTROL, page).on('click', toggleFilterbar);
                $(RESET_FILTERS, page).on('click', resetFilters);
                $(FILTER, page).change(onPropertyFilterChange);
                $(SORTER, page).hpSelect({sorting: true});
                $(SORTER, page).change(onSortChange);
                $(LIST, page).scroll(onScroll);
                $(LIST, page).on('click', COLLAPSER, onCollapserClick);
                $(LIST, page).on('click', ITEM, onItemClick);
                $(page).on('focus', LIST, function (event) {
                    if (event.target == $(LIST, page)[0]) {
                        $(document).off('keydown', onKeyDown);  
                        $(document).on('keydown', onKeyDown);
                    }
                });
                $(page).on('blur', LIST, function (event) {
                    if (event.target == $(LIST, page)[0]) {
                        $(document).off('keydown', onKeyDown);  
                    }
                });
                
                var id = $('#hp-master-sort li:not([data-sort-direction]):not(.hp-sub-options).hp-selected').data('id');
                var direction = $('#hp-master-sort li[data-sort-direction].hp-selected').data('sort-direction');
                if (id) {
                    if (!direction) {
                        direction = 'asc';
                    }
                    presenter.setSort(id, direction);
                }
            };
        }

        return MasterListView;

    }());

    return MasterListView;
});
