// (C) Copyright 2011-2013 Hewlett-Packard Development Company, L.P.
define([ 'hp/services/REST','hp/model/DevelopmentSettings','jquery'],
function(REST, settings) {"use strict";

    var NetworkSetupService = (function() {

        var TASK_STATUS_TIMEOUT = 3000, // msec
            REACHABLE_TIMEOUT = 10000, // msec
            iPv6Disabled = false;    // unless explicitly disabled ipv6 should be displayed
        
        /**
         * @constructor Singleton class containing service logic for the initial
         *              network settings page
         * @type {NetworkSetupService}
         */
        function NetworkSetupService() {

            /**
             * Translate the addresses for a single node from UI network object to FTS network DTO.
             *
             * @param {NetworkDTO} networkDTO The network DTO to fill in. 
             * @param {string} dtoTag The node to update in the DTO.
             * @param {object} networkObject The network object to read.
             * @param {string} objTag The node to read in the network object.
             */
            function getFTSNetworkDTONode(networkDTO, dtoTag, networkObject, objTag) {
                networkDTO[dtoTag + 'Ipv4Addr'] = networkObject.addrs[objTag].ipv4;
                networkDTO[dtoTag + 'Ipv6Addr'] = networkObject.addrs[objTag].ipv6;
                if ((dtoTag != "virt") && ("ipv4_2" in networkObject.addrs.node1)) {
                   networkDTO[dtoTag + "Ipv4Alias"]= networkObject.addrs[objTag].ipv4_2;              
                   networkDTO[dtoTag + "Ipv6Alias"]= networkObject.addrs[objTag].ipv6_2;                
                }
            }

            /**
             * Translate the network object used by the UI to FTS's network DTO.
             *
             * @param {networkObject}
             */
            function getFtsNetworkDTO(networkObject) {
                var networkDTO = {
                    ipv4Type : networkObject.ipv4Type,
                    ipv6Type : networkObject.ipv6Type,
                    hostname : networkObject.hostname,
                    dynamicDns : networkObject.dnsFromDHCP,
                    ipv4Subnet: networkObject.ipv4Subnet,
                    ipv4Gateway: networkObject.ipv4Gateway,
                    ipv6Subnet: networkObject.ipv6Subnet,
                    ipv6Gateway: networkObject.ipv6Gateway,    
                    confOneNode: networkObject.confOneNode,
                    activeNode: networkObject.localNodeID,
                    allowTransientValidationErrors:networkObject.allowTransientValidationErrors
                };
                getFTSNetworkDTONode(networkDTO, 'app1', networkObject, 'node1');
                getFTSNetworkDTONode(networkDTO, 'app2', networkObject, 'node2');
                getFTSNetworkDTONode(networkDTO, 'virt', networkObject, 'cluster');
  
                if (!networkObject.dnsFromDHCP) { 
                    networkDTO.domainName=networkObject.domainName;
                    networkDTO.nameServers = [];
                    if (networkObject.priDns) {
                        networkDTO.nameServers.push(networkObject.priDns);
                    }
                    if (networkObject.altDns) {
                        networkDTO.nameServers.push(networkObject.altDns);
                    }
                }
              
                return networkDTO;
            }
            
            /**
             * Translate the data for a single node from FTS network DTO to UI network object.
             * 
             * @param {object} networkObject UI network object to update
             * @param {string} objTag Node to update in the network object
             * @param {NetworkDTO} networkDTO FTS network DTO to read
             * @param {string} dtoTag Node to read from the DTO
             */
            function getUINetworkObjectNode(networkObject, objTag, networkDTO, dtoTag) {
                if (!networkObject.addrs) {
                    networkObject.addrs = {};
                }
                networkObject.addrs[objTag] = {
                    ipv4: networkDTO[dtoTag + "Ipv4Addr"],
                    ipv6: networkDTO[dtoTag + "Ipv6Addr"]
                };
                if ("app1Ipv4Alias" in networkDTO) { // all aliases will be present as properties
                    networkObject.addrs[objTag].ipv4_2 =  networkDTO[dtoTag + "Ipv4Alias"];                 
                    networkObject.addrs[objTag].ipv6_2 =  networkDTO[dtoTag + "Ipv6Alias"];  
                }
            
            }
            
            /**
             * Translate FTS's network DTO to the network object used by the UI.
             * 
             * @param {networkDTO}
             */
            function getUINetworkObject(networkDTO) {
                var networkObject = {
                    ipv4Type : networkDTO.ipv4Type,
                    ipv6Type : networkDTO.ipv6Type,
                    hostname : networkDTO.hostname,
                    domainName: networkDTO.domainName,
                    ipv4Subnet: networkDTO.ipv4Subnet,
                    ipv4Gateway: networkDTO.ipv4Gateway,
                    ipv6Subnet: networkDTO.ipv6Subnet,
                    ipv6Gateway: networkDTO.ipv6Gateway,    
                    confOneNode: networkDTO.confOneNode !== false,
                    localNodeID: networkDTO.activeNode,
                    dnsFromDHCP : networkDTO.dynamicDns
                };
                getUINetworkObjectNode(networkObject, 'node1', networkDTO, 'app1');
                getUINetworkObjectNode(networkObject, 'node2', networkDTO, 'app2');
                getUINetworkObjectNode(networkObject, 'cluster', networkDTO, 'virt');
                if (networkDTO.app2Ipv6Addr && networkDTO.app2Ipv6Addr != '') {
                    networkObject.twoNodeData = true;
                } else {
                    networkObject.twoNodeData = false;
                }
                                
                networkObject.priDns = (networkDTO.nameServers.length > 0) ? networkDTO.nameServers[0] : null;
                networkObject.altDns = (networkDTO.nameServers.length > 1) ? networkDTO.nameServers[1] : null;
                
                // figure out the domain  in case this is the FTS and we want to automatically append the domain name 
                // to the hostname that is  not fully qualified.
                if (!networkDTO.domainName) {
                    if (networkDTO.searchDomains && networkDTO.searchDomains.length !=0) {
                        networkObject.searchDomainName = networkDTO.searchDomains[0];
                    }
                }
                
                return networkObject;
            }
                
            /**
             * @public setInitialNetwork - Send REST request to set the
             *         initial network configuration
             *
             * @param {string} networkObject - The values from the network
             *        settings page
             * @param {Object} handlers - The success and error handler functions
             *
             */
            this.setNetworkConfiguration = function(networkObject,  handlers) {
                var ftsNetworkDTO = getFtsNetworkDTO(networkObject);
                REST.postObject('/rest/appliance/configuration/networkconfig', ftsNetworkDTO, handlers);
            };

            /**
             * @public validateInitialNetwork - Send REST request to validate the
             *         initial network configuration
             *
             * @param {string} networkObject - The values from the network
             *        settings page
             * @param {Object} handlers - The success and error handler functions
             *
             */
            this.validateNetworkConfiguration = function(networkObject,  handlers) {
                var ftsNetworkDTO = getFtsNetworkDTO(networkObject);
                REST.postObject('/rest/appliance/configuration/networkconfig/validator', ftsNetworkDTO, handlers);
            };

            /**
             * Get the current network configuration of the server.
             *
             * @param {{success:function(NetworkDTO):void, error:function(ErrorMessage):void}} handlers
             *     Handler functions for success and error conditions.
             */
            this.getNetworkConfiguration = function(handlers) {
                REST.getURI('/rest/appliance/configuration/networkconfig', {
                    success: function(networkDTO) {
                        handlers.success(getUINetworkObject(networkDTO));
                    },
                    error: handlers.error
                });
            };

            /**
             * Get the status of a task tracker task.
             *
             * @param {string} hostname The appliance hostname to talk to (== null for current)
             * @param {string} uri The uri to use to get the task status
             * @param {{success:function(TaskResourceDTO):void, error:function(ErrorMessage):void}
             *     handlers Handler functions for success and error conditions.
             */
            this.getTaskStatus = function(hostname, uri, handlers) {
                // Get the standard Ajax options for a REST GET call.
            	
                if (hostname) {
                    uri = "https://" + hostname + uri;
                }
                var options = REST.getAjaxOptions(uri, 'GET', handlers);
            	// Set a short timeout on the call.
                options.timeout = TASK_STATUS_TIMEOUT;
                
                // Issue the call.
                $.ajax(options);
            };

            /**
             * Find out whether a host is reachable, by trying to read its controller-state.json file.
             * 
             * @param {string} hostname Hostname to attempt, or null for current
             * @param {object} handlers Success and error handlers
             */
            this.checkReachable = function(hostname, handlers) {  
                
                 jQuery.support.cors = true;
                 var destUrl;
                  
                 if ( $.browser.msie ) {
                   destUrl = (hostname ? 'https://' + hostname : '') + '/controller-state.json';                  
                } else {
                   destUrl = (hostname ? 'http://' + hostname : '') + '/controller-state.json';                    
                }      
                $.ajax({
                    url : destUrl,
                    type : 'GET',
                    success : handlers.success,
                    error : handlers.error,
                    dataType : "json",
                    cache: false,
                    timeout: REACHABLE_TIMEOUT
                });
            };
            
            /**
             * Find out whether the peer node is valid to be used as a standby node in our cluster.
             * 
             * @param {object} handlers The success and error handler functions.
             */
            this.validatePeerNode = function(handlers) {
                handlers.success(true);
            };
            
            /**
             * @private Exposed only for unit tests.
             */
            this._getFtsNetworkDTO = function(networkObject) {
                return getFtsNetworkDTO(networkObject);
            };
             /**
             * @public hook for the partners to disable IPV6
             */
            this.setIPv6Disabled = function(disableIPv6Flag) {
                iPv6Disabled=disableIPv6Flag;
            };
            
             /**
             * @public Exposed only for unit tests.
             */
            this.getIPv6Disabled = function() {
                return (iPv6Disabled ||  settings.disableIPv6());
            };
        }

        return new NetworkSetupService();
    }());

    return NetworkSetupService;
});

