// (C) Copyright 2013 Hewlett-Packard Development Company, L.P.
/**
 * @type {UnitConversion}
 *
 * This is a collection of routines needed to manage the conversion of data
 * that is currently being reported back from CRM, but used in Profile Manager
 * and CRM.  All conversions are done as base 10 numbers versus base 2 numbers.
 * Therefore, 1K in these routines is equal to 1000 and not 1024.
 */
define([], function() { "use strict";

    var UnitConversion = (function () {
      
        var PER_KILO = (1000);
        var PER_MEGA = (1000 * 1000);
        var PER_GIGA = (1000 * 1000 * 1000);
        var PER_TERRA = (1000 * 1000 * 1000 * 1000);
        var PER_PETA = (1000 * 1000 * 1000 * 1000 * 1000);
        
        /**
         * @constructor
         */
        function UnitConversion() {

            /**
             * @public
             *
             * Converts a value from 10^3 to 10^1
             */
            this.convertMbstoGb = function (value) {
                return value / 1000;
            };

            /**
             *
             * @public
             *
             * Converts a value from 10^1 to 10^3
             *
             */
            this.convertGbtoMbs = function (value) {
                return value * 1000;
            };

            /**
             * @public
             *
             * Converts a value from 10^9 to 10^3
             */
            this.convertBitsToMbs = function (value) {
                return value / 1000000;
            };

            /**
             * Rounds a decimal number to a given number of significant figures (or digits).
             * 
             * sigFigs(123.456, 3) = 123
             * sigFigs(123.456, 5) = 123.46
             * 
             * For 3 significant digits combined with getUnit(), these are some example outputs:
             * 
             * 3.73GB
             * 201MB
             * 12.7KB
             * 
             * See also http://blog.magnetiq.com/post/497605344/rounding-to-a-certain-significant-figures-in-javascript
             */
            function sigFigs(value, sig) {
                if (value === 0) {
                    return 0;
                } else if (value < 1) {
                    return value.toFixed(sig);
                }
                var mag = Math.pow(10, sig - Math.floor(Math.log(value) / Math.LN10) - 1);
                // Math.round() is necessary when 1/mag does not return a whole number
                if ((1 / mag) > 1) {
                    return Math.round(value * mag) * Math.round(1 / mag);
                } else {
                    return Math.round(value * mag) / Math.round(mag);
                }
            }

            function getUnit(value) {
                if (value >= PER_PETA) {
                    return {value: value / PER_PETA, units: 'petabytes'};
                } else if (value >= PER_TERRA) {
                    return {value: value / PER_TERRA, units: 'terrabytes'};
                } else if (value >= PER_GIGA) {
                    return {value: value / PER_GIGA, units: 'gigabytes'};
                } else if (value >= PER_MEGA) {
                    return {value: value / PER_MEGA, units: 'megabytes'};
                } else if (value >= PER_KILO) {
                    return {value: value / PER_KILO, units: 'kilobytes'};
                } else {
                    return {value: value, units: 'bytes'};
                }
            }

            this.normalizeBytes = function (value) {
                return getUnit(value);
            };
            
            this.normalizeBytesSigFigs = function (value, sig) {
                return getUnit(sigFigs(value, sig));
            }
        }

        return new UnitConversion();
    }());

    return UnitConversion;

});