// (C) Copyright 2011-2012 Hewlett-Packard Development Company, L.P.

define(['hp/core/Style',
    'text!hpPages/graphics/logical_switch_template.html',
    'jquery',
    'lib/excanvas',
    'hp/lib/jquery.hpStatus'],
function(style, templateHtml) { "use strict";

    var GraphicLogicalSwitch = (function() {
      
        var LOGICAL_SWITCH = '.hp-logical-switch';
        var CANVAS = LOGICAL_SWITCH + ' canvas';
        var SWITCH_ROW = '.hp-physical-switch-row';
        var SWITCH = '.hp-switch';
        var LOGICAL_UPLINK = '.hp-logical-uplink';
        var UP_PORT = '.hp-up-port';
        var HAS_HIGHLIGHT = 'hp-has-highlight';
        var HIGHLIGHT = 'hp-highlight';
        var SELECTED = 'hp-selected';
        var BODY = '#hp-body-div';
        
        /**
         * Constructor
         */
        function GraphicLogicalSwitch() {
          
            var container = null;
            var switchRowTemplate = null;
            var switchTemplate = null;
            var switchPortTemplate = null;
            var uplinkTemplate = null;
            var connections = {}; // logicalUplinkId: [
                // {switchId: S, portIndex: P}, ...]
            var resizeTimer = null;
            // Cache of style.related()
            var relatedStyle;
            
            // temporary state for drawing
            // lots of cross linking to speed things up
            var context;
            var uplinkLayouts = {}; // id: {coords: C2, portLayouts: []}
            var portLayouts = {}; // id: {}
            var uplinkRows = []; //{index: I1, coords: C1, uplinkLayouts: []}, ...
            var switchRows = []; // {index: I2, coords: C3, portLayouts: [
                //{coords: C4, row: SWITCH_ROW}, ...
                //]}
          
            function itemCoords(item) {
                var containerOffset = $(LOGICAL_SWITCH, container).offset();
                var itemOffset = item.offset();
                var left = itemOffset.left - containerOffset.left;
                var top = itemOffset.top - containerOffset.top;
                var bottom = top + item.outerHeight();
                var right = left + item.outerWidth();
                var centerX = Math.ceil(left + ((right - left) / 2));
                var centerY = Math.ceil(top + ((bottom - top) / 2));
                return {left: left, top: top, bottom: bottom, right: right,
                    centerX: centerX, centerY: centerY};
            }
            
            function drawPortConnection(portLayout, top) {
                var portCoords = portLayout.coords;
                
                context.moveTo(portCoords.centerX, portCoords.top);
                context.lineTo(portCoords.centerX, top);
            }
            
            function drawUplinkSwitchRow(index, uplinkSwitchRow) {
                context.moveTo(uplinkSwitchRow.left, uplinkSwitchRow.top);
                context.lineTo(uplinkSwitchRow.right, uplinkSwitchRow.top);
                
                $.each(uplinkSwitchRow.portLayouts, function (index, portLayout) {
                    drawPortConnection(portLayout, uplinkSwitchRow.top);
                });
            }
            
            function drawUplinkConnections(uplinkLayout) {
                var highlight = $('#' + uplinkLayout.id).hasClass(HIGHLIGHT);
                var coords = uplinkLayout.coords;
                
                context.beginPath();
                if (highlight) {
                    context.lineWidth = relatedStyle.highlight.width;
                    context.strokeStyle = relatedStyle.highlight.color;
                } else {
                    context.lineWidth = relatedStyle.secondary.width;
                    context.strokeStyle = relatedStyle.secondary.color;
                }
                context.moveTo(coords.left - 1, coords.top + 1);
                context.lineTo(coords.left - 1, uplinkLayout.bottom - 1);
                context.lineCap = 'square';
                context.stroke();
                
                if (uplinkLayout.hasOwnProperty('farLeft')) {
                    context.moveTo(uplinkLayout.coords.left - 2, uplinkLayout.bottom);
                    context.lineTo(uplinkLayout.farLeft, uplinkLayout.bottom);
                    context.lineTo(uplinkLayout.farLeft, uplinkLayout.farBottom);
                }
                if (uplinkLayout.hasOwnProperty('farRight')) {
                    context.moveTo(uplinkLayout.coords.left - 2, uplinkLayout.bottom);
                    context.lineTo(uplinkLayout.farRight, uplinkLayout.bottom);
                    context.lineTo(uplinkLayout.farRight, uplinkLayout.farBottom);
                }
                
                $.each(uplinkLayout.uplinkSwitchRows, drawUplinkSwitchRow);
                
                context.lineCap = 'square';
                context.stroke();
            }
            
            function createOrUpdateLogicalSwitchRow(uplinkLayout, portLayout) {
                var switchRow = portLayout.switchRow;
                var coords = portLayout.coords;
                var uplinkSwitchRow = null;
                
                $.each(uplinkLayout.uplinkSwitchRows, function (index, row) {
                    if (row.switchRow === switchRow) {
                        uplinkSwitchRow = row;
                        return false;
                    }
                });
                
                if (! uplinkSwitchRow) {
                    uplinkSwitchRow = {
                        left: coords.centerX,
                        right: coords.centerX,
                        switchRow: switchRow,
                        top: (switchRow.coords.top - 50) +
                            ( 5 * switchRow.nextUplinkIndex),
                        portLayouts: [portLayout]
                    };
                    uplinkLayout.uplinkSwitchRows.push(uplinkSwitchRow);
                    switchRow.nextUplinkIndex += 1;
                    //console.log('!!! GLS create uplink switch row',
                    //    uplinkLayout.index, switchRow.index, uplinkSwitchRow.top);
                    
                    if (uplinkLayout.uplinkRow === uplinkRows.slice(-1)[0] &&
                        portLayout.switchRow === switchRows[0]) {
                        // uplink is in last row and switch is in first row,
                        // stretch just to uplink
                        uplinkSwitchRow.left =
                            Math.min(uplinkSwitchRow.left,
                                uplinkLayout.coords.left-4);
                        uplinkSwitchRow.right =
                            Math.max(uplinkSwitchRow.right,
                                uplinkLayout.coords.left-4);
                        uplinkSwitchRow.top = uplinkLayout.bottom;
                    } else {
                        // not directly above/below each other, draw row to the edge
                        if (Math.floor((uplinkLayout.index / 4) % 2) === 0) {
                            if (! uplinkLayout.hasOwnProperty('farLeft')) {
                                uplinkLayout.farLeft = (5 * uplinkLayout.index) + 2;
                            }
                            uplinkSwitchRow.left = uplinkLayout.farLeft;
                        } else {
                            if (! uplinkLayout.hasOwnProperty('farRight')) {
                                uplinkLayout.farRight =
                                    portLayout.switchRow.coords.right + (5 * uplinkLayout.index) + 2;
                            }
                            uplinkSwitchRow.right = uplinkLayout.farRight;
                        }
                        uplinkLayout.farBottom = uplinkSwitchRow.top;
                    }
                } else {
                    uplinkSwitchRow.left =
                        Math.min(uplinkSwitchRow.left, coords.centerX);
                    uplinkSwitchRow.right =
                        Math.max(uplinkSwitchRow.right, coords.centerX);
                    uplinkSwitchRow.portLayouts.push(portLayout);
                }
            }
            
            function createPortLayout(uplinkLayout, connection) {
                var switchElem = $('#' + connection.switchId);
                var portElem = $(UP_PORT + ':eq(' + connection.portIndex + ')',
                    switchElem);
                var coords = itemCoords(switchElem);
                var switchRow, portLayout;
                
                // see if we already have a row for this switch
                $.each(switchRows, function (index, switchRow2) {
                    if (switchRow2.coords.top === coords.top) {
                        switchRow = switchRow2;
                        return false;
                    }
                });
                
                if (! switchRow) {
                    switchRow = {
                        index: switchRows.length,
                        coords: coords,
                        portLayouts: [],
                        nextUplinkIndex: 0
                    };
                    switchRows.push(switchRow);
                    //console.log('!!! GLS create switch row', switchRow.index,
                    //    connection.switchId, connection.portIndex);
                } else {
                    switchRow.coords.right =
                        Math.max(switchRow.coords.right, coords.right);
                }
                
                coords = itemCoords(portElem);
                portLayout = {
                    coords: coords,
                    id: portElem.attr('id'),
                    switchRow: switchRow,
                    uplinkLayout: uplinkLayout,
                    // needed for selection highlighting
                    portElem: portElem,
                    switchElem: switchElem
                };
                switchRow.portLayouts.push(portLayout);
                uplinkLayout.portLayouts.push(portLayout);
                portLayouts[portElem.attr('id')] = portLayout;
                
                // figure out switch row extent
                createOrUpdateLogicalSwitchRow(uplinkLayout, portLayout);
                
                return portLayout;
            }
            
            function createUplinkLayout(index, uplink) {
                var coords = itemCoords(uplink);
                var uplinkRow, uplinkLayout;
                
                if (uplinkRows.length === 0 ||
                    uplinkRows.slice(-1)[0].coords.top !== coords.top) {
                    // add a row
                    uplinkRow = {coords: coords, uplinkLayouts: []};
                    uplinkRows.push(uplinkRow);
                    //console.log('!!! GLS new uplink row for', uplink.attr('id'));
                } else {
                    uplinkRow = uplinkRows.slice(-1)[0];
                }
                
                uplinkLayout = {
                    index: index,
                    id: uplink.attr('id'),
                    coords: coords,
                    // for flagpole line bottom
                    bottom: coords.bottom + 10 +
                        (5 * uplinkRow.uplinkLayouts.length),
                    colorIndex: index,
                    uplinkRow: uplinkRow,
                    portLayouts: [], // for connections
                    uplinkSwitchRows: [],
                    // for port selection highlighting
                    uplinkElem: uplink
                };
                uplinkRow.uplinkLayouts.push(uplinkLayout);
                uplinkLayouts[uplinkLayout.id] = uplinkLayout;
                
                return uplinkLayout;
            }
            
            function alignUplinks() {
                var maxWidth = 0;
                $(LOGICAL_UPLINK, container).each(function (index, uplink) {
                    maxWidth = Math.max(maxWidth, $(uplink).outerWidth());
                });
                $(LOGICAL_UPLINK, container).each(function (index, uplink) {
                    $(uplink).css('width', maxWidth);
                });
            }
            
            function alignSwitches() {
                var maxWidth = 0;
                $(SWITCH, container).each(function (index, switchElem) {
                    maxWidth = Math.max(maxWidth, $(switchElem).outerWidth());
                });
                $(SWITCH, container).each(function (index, switchElem) {
                    $(switchElem).css('width', maxWidth);
                });
            }
            
            function resetDrawing() {
                uplinkRows = [];
                switchRows = [];
                uplinkLayouts = {};
                portLayouts = {};
                //console.log('!!! GLS reset drawing');
            }
          
            function draw(relayout) {
                var logicalSwitch = $(LOGICAL_SWITCH, container);
                var canvas = $(CANVAS, container)[0];
                var uplinkLayout, padding, highlighted = [];
                
                $(CANVAS, container).attr('width', logicalSwitch.outerWidth()-2).
                    attr('height', logicalSwitch.outerHeight()-2);
                if (!canvas.getContext && window.G_vmlCanvasManager) {
                    window.G_vmlCanvasManager.initElement(canvas);
                }
                if (canvas.getContext) {
                    context = canvas.getContext('2d');
                    context.clearRect(0, 0, canvas.width, canvas.height);
                    
                    if (relayout || uplinkRows.length === 0) {
                        resetDrawing();
                        
                        // set left padding to allow for connections
                        padding = (5 * $(LOGICAL_UPLINK, container).length) + 5
                        $(LOGICAL_SWITCH, container).css('padding-left', padding);
                        $(CANVAS, container).css('margin-left', -padding);
                        
                        // make all logical uplinks the same width
                        alignUplinks();
                        // make all switches the same width
                        alignSwitches();
                        
                        // first layout the uplinks
                        $(LOGICAL_UPLINK, container).each(function (index, uplink) {
                            uplinkLayout = createUplinkLayout(index, $(uplink));
                        });
                        
                        // then layout the connection ports
                        $.each(uplinkLayouts, function (index, uplinkLayout) {
                            if (connections[uplinkLayout.id]) {
                                $.each(connections[uplinkLayout.id],
                                    function(index, connection) {
                                        createPortLayout(uplinkLayout, connection);
                                    });
                            }
                        });
                    }
                    
                    // draw connections, first unhighlighted, then highlighted
                    // this is so highlighted ones overlay non highlighted ones
                    $.each(uplinkRows, function (index, uplinkRow) {
                        $.each(uplinkLayouts, function (index, uplinkLayout) {
                            if (! uplinkLayout.highlight) {
                                drawUplinkConnections(uplinkLayout);
                            } else {
                                highlighted.push(uplinkLayout);
                            }
                        });
                    });
                    $.each(highlighted, function (index, uplinkLayout) {
                        drawUplinkConnections(uplinkLayout);
                    });
                }
            }
            
            function dehighlight() {
                $(UP_PORT, container).removeClass(HIGHLIGHT);
                $(SWITCH, container).removeClass(HIGHLIGHT);
                $(LOGICAL_UPLINK, container).removeClass(HIGHLIGHT);
                $(LOGICAL_SWITCH, container).removeClass(HAS_HIGHLIGHT);
                draw(false);
            }
            
            function highlightUplink(uplinkLayout) {
                uplinkLayout.uplinkElem.addClass(HIGHLIGHT);
                uplinkLayout.highlight = true;
                $.each(uplinkLayout.portLayouts, function(index, portLayout) {
                    portLayout.portElem.addClass(HIGHLIGHT);
                    portLayout.switchElem.addClass(HIGHLIGHT);
                });
                $(LOGICAL_SWITCH, container).addClass(HAS_HIGHLIGHT);
            }
            
            function dehighlightUplink(uplinkLayout) {
                uplinkLayout.uplinkElem.removeClass(HIGHLIGHT);
                uplinkLayout.highlight = false;
                $.each(uplinkLayout.portLayouts, function(index, portLayout) {
                    portLayout.portElem.removeClass(HIGHLIGHT);
                    portLayout.switchElem.removeClass(HIGHLIGHT);
                });
                // another uplink might be selected 
                if ($(LOGICAL_UPLINK + '.' + SELECTED, container).length === 0) {
                    $(LOGICAL_SWITCH, container).removeClass(HAS_HIGHLIGHT);
                }
            }
            
            function onEnterUplink(event) {
                var uplink = $(event.currentTarget);
                var id = uplink.attr('id');
                var uplinkLayout = uplinkLayouts[id];
                if (uplinkLayout && ! uplink.hasClass(SELECTED)) {
                    highlightUplink(uplinkLayout);
                    draw(false);
                }
            }
            
            function onLeaveUplink(event) {
                var uplink = $(event.currentTarget);
                var id = uplink.attr('id');
                var uplinkLayout = uplinkLayouts[id];
                if (uplinkLayout && ! uplink.hasClass(SELECTED)) {
                    dehighlightUplink(uplinkLayout);
                    draw(false);
                }
            }
            
            function onEnterPort(event) {
                var port = $(event.currentTarget);
                var id = port.attr('id');
                var uplinkLayout;
                var portLayout = portLayouts[id];
                if (portLayout &&
                    ! portLayout.uplinkLayout.uplinkElem.hasClass(SELECTED)) {
                    highlightUplink(portLayout.uplinkLayout);
                    draw(false);
                }
            }
            
            function onLeavePort(event) {
                var port = $(event.currentTarget);
                var id = port.attr('id');
                var portLayout = portLayouts[id];
                if (portLayout &&
                    ! portLayout.uplinkLayout.uplinkElem.hasClass(SELECTED)) {
                    dehighlightUplink(portLayouts[id].uplinkLayout);
                    draw(false);
                }
            }
            
            var onClickBody = function() {}; // forward reference for Sonar
            
            function deselect() {
                $(LOGICAL_UPLINK, container).removeClass(SELECTED);
                $(UP_PORT, container).removeClass(SELECTED);
                $(BODY).off('click', onClickBody);
            }
            
            function selectUplink(uplink) {
                uplink.addClass(SELECTED);
                // delay to avoid flickering
                setTimeout(function () {$(BODY).on('click', onClickBody);}, 50);
            }
            
            function selectPort(port) {
                var id = port.attr('id');
                // might not have any uplink port
                if (portLayouts[id]) {
                    portLayouts[id].uplinkLayout.uplinkElem.addClass(SELECTED);
                }
                // delay to avoid flickering
                setTimeout(function () {$(BODY).on('click', onClickBody);}, 50);
            }
            
            onClickBody = function () {
                deselect();
                dehighlight();
                draw(true);
            }
            
            function onClickUplink(event) {
                var uplink = $(event.currentTarget);
                deselect();
                selectUplink(uplink);
            }
            
            function onClickPort(event) {
                var port = $(event.currentTarget);
                deselect();
                selectPort(port);
            }
            
            function onResize(event) {
                clearTimeout(resizeTimer);
                resizeTimer = setTimeout(function () {
                    draw(true);
                }, 50);
            }
            
            /**
             * @public
             */
            this.init = function(containerArg) {
                relatedStyle = style.related();
                var template = $(templateHtml);
                switchPortTemplate = $(UP_PORT, template).remove();
                switchTemplate = $(SWITCH, template).remove();
                switchRowTemplate = $(SWITCH_ROW, template).remove();
                uplinkTemplate = $(LOGICAL_UPLINK, template).remove();
                container = containerArg;
                $(container).append(template);
                $(container).on({'mouseenter': onEnterUplink,
                    'mouseleave': onLeaveUplink, 'click': onClickUplink},
                    LOGICAL_UPLINK);
                $(container).on({'mouseenter': onEnterPort,
                    'mouseleave': onLeavePort, 'click': onClickPort},
                    UP_PORT);
            };
            
            this.pause = function () {
                $(window).off('resize', onResize);
            };
            
            this.resume = function () {
                $(window).on('resize', onResize);
                draw(true);
            };
            
            this.addSwitchRow = function (id) {
                var result = switchRowTemplate.clone().attr('id', id);
                $('.hp-physical-switch-rows', container).append(result);
                return result;
            };
            
            this.addSwitch = function (rowElem, id) {
                var result = switchTemplate.clone().attr('id', id);
                $('.hp-physical-switches', rowElem).append(result);
                return result;
            };
            
            this.addSwitchPort = function (switchElem, id) {
                var result = switchPortTemplate.clone().attr('id', id);
                $('.hp-up-ports', switchElem).append(result);
                return result;
            };
            
            this.addLogicalUplink = function (id) {
                var result = uplinkTemplate.clone().attr('id', id);
                $('.hp-logical-uplinks', container).append(result);
                return result;
            };
            
            this.connect = function (logicalUplinkId, switchId, portIndex) {
                var valid = true;
                var switchElem;
                
                if ($('#' + logicalUplinkId, container).length !== 1) {
                    console.warn('No logical uplink with id', logicalUplinkId);
                    valid = false;
                }
                switchElem = $('#' + switchId, container);
                if (switchElem.length !== 1) {
                    console.warn('No switch with id', switchId);
                    valid = false;
                } else if ($(UP_PORT + ':eq(' + portIndex + ')',
                    switchElem).length !== 1) {
                    console.warn('No port with index', portIndex, 'in switch',
                        switchId);
                    valid = false;
                }
                if (valid) {
                    if (! connections.hasOwnProperty(logicalUplinkId)) {
                        connections[logicalUplinkId] = [];
                    }
                    connections[logicalUplinkId].push({
                        switchId: switchId,
                        portIndex: portIndex}
                    );
                }
            };
            
            this.done = function () {
                draw(true);
            };
            
            this.clear = function () {
                $('.hp-physical-switch-rows', container).empty();
                $('.hp-logical-uplinks', container).empty();
                connections = {};
                uplinkRows = [];
                switchRows = [];
            };
        }

        return GraphicLogicalSwitch;
    }());
    
    return GraphicLogicalSwitch;
});
