// (C) Copyright 2011-2012 Hewlett-Packard Development Company, L.P.
define(['hp/core/Localizer',
        'hp/services/UploadService',
        'hp/model/UploaderModel'],
function(localizer, service, Model) {"use strict";

    var UploaderPresenter = ( function() {
        function UploaderPresenter() {
            var model = new Model();
            
            function createAbortHandler(handlers){
                return function() {
                    model.setUploadDone();
                    model.setAborted();
                    model.setStopTime(new Date().getTime());
                    var abortErrorInfo = {
                       errorMessage : localizer.getString('core.file.upload.cancel.canceled'),
                       message : localizer.getString('core.file.upload.cancel.canceled')
                    };
                    model.setError(abortErrorInfo);
                    if(handlers && handlers.onAbort){
                        handlers.onAbort(abortErrorInfo);
                    }
                };
            }
            
            function createProgressHandler(handlers){
                return function(e) {
                    model.updateProgress(e);
                    if(handlers && handlers.progressHandler){
                        handlers.progressHandler(e);
                    }
                };
            }
            
            this.createAbortHandler = createAbortHandler;
            this.createProgressHandler = createProgressHandler;
            
            this.upload = function(uri,formData,handlers) {
                model.setUploading();
                model.setStartTime(new Date().getTime());
                var externalOptions;
                if(handlers.optionsCallback){
                    externalOptions = handlers.optionsCallback();
                }
                
                service.upload(uri,formData, {
                        success : function(data) {
                            model.setStopTime(new Date().getTime());
                            model.setUploadDone();
                            model.setResults(data);
                            if(handlers.success){
                                handlers.success(data);
                            }
                        },
                        error : function(errorInfo) {
                            model.setStopTime(new Date().getTime());
                            model.setUploadDone();
                            model.setError(errorInfo);
                            if(handlers.error){
                                handlers.error(errorInfo);
                            }
                        },
                        beforeSend : function(xhr) {
                            if(handlers.beforeSend){
                                handlers.beforeSend(xhr);
                            }
                        },
                        onAbort : createAbortHandler(handlers),
                        progressHandler : createProgressHandler(handlers) 
                    },
                    externalOptions
                );
            };
            
            this.isPending = function(uri,handlers) {
                service.isPending(uri,{
                    success : function(data) {
                        if(data) {
                            model.setUploadDone();
                            model.setResults(data);
                            model.setPending(true);
                            if(handlers && handlers.success) {
                                handlers.success(data);
                            }
                        }
                    },
                    error : function(errorInfo) {
                        if(handlers && handlers.error) {
                            handlers.error(errorInfo);
                        }
                    }
                });
            };
            
            this.isUpgradeFilePending = function(){
                return model.getPending();
            };

            this.getProgress = function() {
                return model.getProgress();
            };

            this.getIsUploading = function() {
                return model.isUploading();
            };
            
            this.getStartTime = function() {
                return model.getStartTime();
            };
            
            this.getStartTimeStr = function() {
                return (new Date(model.getStartTime())).toISOString();
            };
            
            this.getStopTime = function() {
                return model.getStopTime();
            };
            
            this.getElapsedTime = function() {
                var start = model.getStartTime();
                var stop = model.getStopTime();
                var current = new Date().getTime();
                var result = 0;
                
                if (model.isUploading) {
                    result = current - start;
                } else {
                    result = stop - start;
                    if (result < 0) {
                        result = 0;
                    }
                }
                return result;
            };
            
            this.getElapsedTimeStr = function() {
                return (new Date(this.getElapsedTime())).toISOString();
            };

            this.clear = function() {
                model.clear();
            };
            
            this.getResults = function() {
                return model.getResults();
            };

            this.getError = function() {
                return model.getError();
            };

            this.getAborted = function() {
                return model.getAborted();
            };
            
            this.setOptions = function(opts) {
                model.setOptions(opts);
            };
            this.getOptions = function() {
                return model.getOptions();
            };

            this.setFiles = function(f){
                model.setFiles(f);
            };
            this.getFiles = function(){
                var files = model.getFiles();
                var retObj = [];
                $.each(files,function(index,file){
                    retObj.push({
                        name: file.name,
                        size: file.size
                    });
                });
                return retObj;
            };
            
            // This is just for testing
            this.getModel = function(){
                return model;
            };

            // public
            this.init = function() {
                model.clear();
            };
        }

        return UploaderPresenter;

    }());

    return UploaderPresenter;
});
