// (C) Copyright 2011-2012 Hewlett-Packard Development Company, L.P.
/**
 * @type {DialogView}
 */
define(['hp/core/Localizer', 'text!hpPages/core/dialog_overlay.html',
    'text!hpPages/core/dialog_notification.html',
    'jquery', 'hp/lib/jquery.hpEllipsis'],
function(localizer, dialogOverlayHtml, dialogNotificationHtml) {
"use strict";

    var DialogView = (function() {
      
        var CONTAINER = '.hp-dialog-container';
        var DEPENDENCIES = '.hp-dependencies';
        var VARIABLE_CONTENTS = '.hp-dialog-variable-contents';
        var NOTIFICATION = '.hp-dialog-notification';
        var BODY = '#hp-body-div';
        var OK = '.hp-ok';
        var CANCEL = '.hp-cancel';
        var ACTIVE = 'hp-active';
        var ENTER = '13';
        var ESCAPE = '27';

        /**
         * Constructor
         * Show a dialog
         * @param {args} Object with properties as follows:
         *
         *    contents: contents of dialog
         *    ok: the function called when the user clicks ok.
         *    cancel: the function called when the user clicks cancel.
         */
        function DialogView(args) {
          
            var self;
            var overlay;
            var component;
            var userTextSelecting = false;
            // Forward references so Sonar won't complain
            var cleanup;
            
            function calculateVariableContentsHeight() {
                var windowHeight = $(window).height();
                // use a percentage of the window, if possible
                var percentage = windowHeight * 0.7;
                // otherwise, only use what we have remaining space for
                var available = windowHeight - 55;
                var top;
                if ($(DEPENDENCIES, component).length > 0) {
                    if ($(DEPENDENCIES, component).is(':visible')) {
                        top = $(DEPENDENCIES, component).offset().top;
                    } else {
                        top = $(DEPENDENCIES, component).parent().offset().top +
                            $(DEPENDENCIES, component).parent().outerHeight();
                    }
                    available -= top;
                }
                if ($(VARIABLE_CONTENTS, component).length > 0) {
                    available -= $(VARIABLE_CONTENTS, component).offset().top;
                }
                if ($('.hp-prompt', component).length > 0) {
                    available -= $('.hp-prompt', component).height();
                }
                if ($('footer', component).length > 0) {
                    available -= $('footer', component).height();
                }
                    
                return Math.min(available, percentage);
            }
            
            function calculateMinWidth() {
                var headerWidth = $('header > *', component).outerWidth();
                var footerWidth = Math.max(
                    $('footer .hp-controls', component).outerWidth(),
                    $('footer .hp-form-controls', component).outerWidth());
                if ($('footer #hp-form-state', component).length > 0) {
                    footerWidth += 200; // allow room for minimal form state
                }
                return Math.max(headerWidth, footerWidth);
            }
            
            function onResize() {
                var maxHeight = calculateVariableContentsHeight();
                var minWidth = calculateMinWidth();
                var maxWidth = Math.min(920, $(window).width() - 40); // 40 is for padding
                $(DEPENDENCIES, component).css('max-height', maxHeight);
                $(VARIABLE_CONTENTS, component).css('max-height', maxHeight);
                $('.hp-dialog-notification .hp-contents', component).
                    css('max-height', maxHeight -
                          $('.hp-dialog-notification .hp-full header').outerHeight());
                $(component).css({'min-width': Math.min(minWidth, maxWidth),
                    'max-width': maxWidth});
            }
            
            function onCancel(event) {
                if (args.cancel) {
                    args.cancel();
                }
                cleanup();
                if (event) {
                    event.preventDefault();
                }
            }
            
            function onOk(event) {
                var doCleanup = true;
                if (args.ok) {
                    // Pass the button that was clicked in case
                    // the caller has multiple OK buttons.
                    if (false === args.ok(this)) {
                        doCleanup = false;
                    }
                }
                if (doCleanup) {
                    cleanup();
                }
                if (event) {
                    event.preventDefault();
                }
            }
            
            function onKeyDown(event) {
                var keyCode = '' + (event.which ? event.which : event.keyCode);
                if (keyCode === ENTER &&  
                    !/^(button|reset|submit)$/i.test(event.target.type)) {
                    // ENTER key anywhere in form except the buttons
                    // Trigger the handler depending on which button has hp-primary class
               	    $('footer .hp-form-controls .hp-primary', component).first().trigger('click');
                } else if (keyCode === ESCAPE) {
                    // ESCAPE key anywhere
                    onCancel(event);
                }
            }
            
            function onMouseMove() {
                userTextSelecting = true;
            }
            
            function onMouseUp() {
                $(BODY).off('mousemove', onMouseMove);
                $(BODY).off('mouseup', onMouseUp);
            }
            
            function onMouseDown() {
                $(BODY).on('mousemove', onMouseMove);
                $(BODY).on('mouseup', onMouseUp);
            }
            
            function deactivateNotification() {
                if (! userTextSelecting) {
                    $(BODY).off('click', deactivateNotification);
                    $(NOTIFICATION, component).off('mousedown', onMouseDown);
                    $(NOTIFICATION, component).removeClass(ACTIVE);
                    onResize();
                } else {
                    userTextSelecting = false;
                }
            }
            
            function activateNotification() {
                if (! $(NOTIFICATION, component).hasClass(ACTIVE)) {
                    $(NOTIFICATION, component).addClass(ACTIVE);
                    // delay to avoid flickering
                    setTimeout(function () {
                        $(NOTIFICATION, component).on('mousedown', onMouseDown);
                        $(BODY).on('click', deactivateNotification);}, 50);
                    onResize();
                }
            }
            
            cleanup = function () {
                $(document).off('keydown', onKeyDown);
                $(window).off('resize', onResize);
                $(OK, component).off('click', onOk);
                $(CANCEL, component).off('click', onCancel);
                $(CONTAINER, overlay).removeClass(ACTIVE);
                overlay.removeClass(ACTIVE);
                component.detach();
                overlay.remove();
            };
            
            /**
             * @public
             */
            function initialize() {
                component = $(args.contents);
                localizer.localizeDom(component);
                overlay = $(dialogOverlayHtml);
                localizer.localizeDom(overlay);
                $(CONTAINER, overlay).append(component).addClass(ACTIVE);
                
                overlay.addClass(ACTIVE);
                $('#hp-body-div').append(overlay);
                $('> header h1', component).hpEllipsis();
                                
                $(OK, component).on('click', onOk);
                $(CANCEL, component).on('click', onCancel);
                $(document).on('keydown', onKeyDown);
                $(window).on('resize', onResize);
                setTimeout(onResize, 50);
            }
            
            self = this;
            initialize();
            
            this.cancel = function () {
                onCancel();
            };
            
            /**
             * @public
             * Set the notification message.
             * @param {notif} Object with properties as follows:
             *
             *    status: one of 'ok', 'error', 'warning', 'info'
             *    summary: minimal text summary, should start with an action if possible
             *    details: longer description that can have line breaks and links,
             *              will show up in a tooltip/flyout
             *    actions: Array of action links
             *
             *    The only property that is required is the summary.
             *
             * @param expand Boolean expand any details right away
             */
            this.setMessage = function(notif, expand) {
                var notificationHtml;
                if ($(NOTIFICATION, component).length === 0) {
                    notificationHtml = $(dialogNotificationHtml);
                    $('header', component).after(notificationHtml);
                    $(NOTIFICATION, component).click(activateNotification);
                }
                if (notif.status) {
                    $(NOTIFICATION + ' > .hp-status, ' +
                      NOTIFICATION + ' .hp-full header > .hp-status', component).
                        hpStatus(notif.status, notif.changing);
                } else {
                    $(NOTIFICATION + ' > .hp-status', component).
                        attr('class', 'hp-status').text('');
                }
                // Don't replace if it hasn't changed,
                // allows user selection to persist.
                if ($(NOTIFICATION + ' .hp-message', component).text() !== notif.summary) {
                    $(NOTIFICATION + ' .hp-message', component).text(notif.summary);
                }
                if (notif.details) {
                    // Don't replace if it hasn't changed,
                    // allows user selection to persist.
                    if ($(NOTIFICATION + ' .hp-details', component).html() !==
                        notif.details) {
                        $(NOTIFICATION + ' .hp-details', component).html(notif.details);
                    }
                } else {
                    $(NOTIFICATION + ' .hp-details', component).
                        text('No details').addClass('hp-unset');
                }
                $(NOTIFICATION + ' .hp-actions', component).empty();
                if (notif.actions) {
                    $.each(notif.actions, function (index, action) {
                        $(NOTIFICATION + ' .hp-actions', component).
                            append($(action).wrap('<li></li>').parent());
                    });
                }
                
                if (expand) {
                    activateNotification();
                }
            };
            
            this.clearMessage = function() {
                $(NOTIFICATION, component).remove();
            };
        }

        return DialogView;
    }());

    return DialogView;
});
