// (C) Copyright 2011 Hewlett-Packard Development Company, L.P.
define(['hp/services/REST', 'jquery'],
function(REST) {"use strict";

    var StatusService = ( function() {

        var CONTROLLER_STATE_URL = '/controller-state.json',
            REBOOT_APPLIANCE_URL = '/cgi-bin/status/reboot-appliance.cgi',
            SUPPORT_DUMP_URL = '/rest/appliance/support-dumps';

        /**
         * @class Service to retrieve the controller-state.json file from the server.
         */
        function StatusService() {

            // Public constants for the possible controller states.
            this.OK = 'OK';
            this.ERROR = 'ERROR';
            this.STARTING = 'STARTING';
            this.QUORUM_RECOVERY = 'QUORUM_RECOVERY';
            this.UNREACHABLE = 'UNREACHABLE';
            this.STOPPED = 'STOPPED';
            this.RESTORE = 'RESTORE';
            this.RESTORE_NO_RESET_ERROR = 'RESTORE_NO_RESET_ERROR';
            this.UPGRADE = 'UPGRADE';
            this.FAILOVER = 'FAILOVER';
            this.STANDBY = 'STANDBY';
            this.FACTORY_RESET = 'FACTORY_RESET';

            var SUPPORT_DUMP_ERROR_CODE = 'CI';
            
            var TASK_STATUS_TIMEOUT = 5000; // msec
            
            /*
             * Policy Issues:
             * 1. There is a maximum amount of time we are willing to wait
             *    between a user interacting with the UI and that user being
             *    notified that the appliance is UNREACHABLE.
             * 2. There is, by program agreement, an amount of time that we
             *    allow for a response to a UI request for a response from the
             *    appliance.
             * 3. Between these two numbers, we will make every attempt to
             *    ensure that the appliance has an opportunity to respond with
             *    it's current status in order to avoid transient waiting
             *    states.
             */
            var STATE_UNREACHABLE_TIMEOUT = 10000; // msec
            var STATE_REQUEST_TIMEOUT = 5000; // msec
            var REFRESH_STATE_RETRY_LIMIT = STATE_UNREACHABLE_TIMEOUT / STATE_REQUEST_TIMEOUT;

            // Private variable caching the last state retrieved from the server.
            var stateRetrieved = null;
            
            function ajaxStateRequest (attempt, timeout, callback) {
                var async = $.isFunction(callback);
                $.ajax({
                    url: CONTROLLER_STATE_URL,
                    dataType: 'json',
                    timeout: timeout,
                    cache: false,
                    success: function(state, requestStatus, xhr) {
                        stateRetrieved = state;
                        if (async) {
                            callback(stateRetrieved);
                        }
                    },
                    error: function(xhr, requestStatus) {
                        // can't use "this.UNREACHABLE", because the context is changed
                        stateRetrieved = { state: 'UNREACHABLE' };
                        if (async) {
                            callback(stateRetrieved);
                        }
                    },
                    async: async
                });
                if (!async) {
                    return stateRetrieved;
                }
            }

            /**
             * Return the cached value of the controller state (one of the six constants above).
             *
             * @return {string} The state.
             */
            this.getState = function() {
                return stateRetrieved;
            };

            /**
             * Update the controller state from the server.
             *
             * @param {function(string):void} callback passed new controller state.  If not a function, the
             *        state is updated synchronously.
             * @return {string} The new state, if no callback is specified.
             */
            this.refreshState = function(clientCallback) {
                var async = $.isFunction(clientCallback);
                var attempt = 1;
                
                if (async) {
                    var stateCallback = function (state) {
                        attempt++;
                        if (('UNREACHABLE' == state.state) && (attempt <= REFRESH_STATE_RETRY_LIMIT)) {
                            ajaxStateRequest(attempt, STATE_REQUEST_TIMEOUT, stateCallback);
                        } else {
                            if (async) {
                                clientCallback(stateRetrieved);
                            }
                        }
                    };
                    ajaxStateRequest(attempt, STATE_REQUEST_TIMEOUT, stateCallback);
                } else {
                    var state = ajaxStateRequest(attempt, STATE_REQUEST_TIMEOUT);
                    while (attempt <= REFRESH_STATE_RETRY_LIMIT) {
                        attempt++;
                        if (('UNREACHABLE' == state.state) && (attempt <= REFRESH_STATE_RETRY_LIMIT)) {
                            state = ajaxStateRequest(attempt, STATE_REQUEST_TIMEOUT);
                        }
                    }
                }
                
                if (!async) {
                    return stateRetrieved;
                }
            };

            /**
             * Sends a request to the server to reboot.  The backend only reboots if the controller state is this.ERROR.
             * @param {Object} Callbacks for error and success of the reboot operation.
             */
            this.rebootAppliance = function(handlers) {
                REST.postObject(REBOOT_APPLIANCE_URL, null, handlers);
            };

            /**
             * Sends a request to create support dump with specific error code.
             */
            this.createSupportDump = function(handlers) {
                REST.postObject(SUPPORT_DUMP_URL, {errorCode : SUPPORT_DUMP_ERROR_CODE}, handlers);
            };
            
           /**
             * Get the status of a task tracker task.
             *            
             * @param {string} uri The uri to use to get the task status
             * @param {{success:function(TaskResourceDTO):void, error:function(ErrorMessage):void}
             *     handlers Handler functions for success and error conditions.
             */
            this.getTaskStatus = function( uri, handlers) {
                // Get the standard Ajax options for a REST GET call.
                var options = REST.getAjaxOptions(uri, 'GET', handlers);
                // Set a short timeout on the call.
                options.timeout = TASK_STATUS_TIMEOUT;
                REST.getURI(uri,handlers,options);
                
                // Issue the call.
               // $.ajax(options);
            };
            
            /**
             * Get the Factory Reset task
             *  
             * @param {{success:function(TaskResourceDTO):void, error:function(ErrorMessage):void}
             *     handlers Handler functions for success and error conditions.
             */
            this.getFactoryResetTask = function( handlers) {
                REST.getURI('/cluster/rest/taskt',handlers);
            };
            /**
             * Check the controller state with respect to https and http communication
             * 
             */
            this.checkControllerState = function(ssl, handlers) {
                jQuery.support.cors = true;
                var destUrl;
                var hostname = window.location.href.split('#')[0];                               	
                var uri = (hostname ? hostname.split('//')[1] : null);
                if (ssl ) 
                {
                    destUrl = (uri ? 'https://' + uri : '/') + 'controller-state.json';  
                }
                else
                {
                    destUrl = (uri ? 'http://' + uri : '/') + 'controller-state.json';
                }                
                $.ajax({                	
                    url : destUrl,
                    type : 'GET',
                    success : handlers.success,
                    error : handlers.error,
                    dataType : "json",
                    cache: false,
                    timeout: 3000
                });
            }
        }

        return new StatusService();

    }());

    return StatusService;

});
