// (C) Copyright 2011 Hewlett-Packard Development Company, L.P.

define(['fs/presenter/alerts/AlertsPanelPresenter',
    'hp/core/UrlFragment',
    'hp/core/LinkTargetBuilder',
    'hp/core/StatusState',
    'jquery',
    'hp/lib/jquery.hpPanels',
    'lib/jquery.ThreeDots',
    'lib/jquery.dateFormat',
    'hp/lib/date'],
function(AlertsPanelPresenter, urlFragment, linkTargetBuilder, statusState) { "use strict";

    var AlertsPanelView = (function() {

        var ALERTS_TABLE = '#cic-server-show-alerts-table',
            ALERTS_TABLE_ROW_TEMPLATE = '#tour-overview-alert-template',
            ALERT_COUNT_CRITICAL = '#alertCountCritical',
            ALERT_COUNT_WARNING = '#alertCountWarning',
            ALERT_COUNT_OK = '#alertCountOk',
            ALERT_COUNT_CRITICAL_ICON = '#alertIconCritical',
            ALERT_COUNT_WARNING_ICON = '#alertIconWarning',
            ALERT_COUNT_OK_ICON = '#alertIconOk',
            ALERT_PREFIX = 'alertId',
            ALERT_NOTIFY_BOX = '#cic-alert-panel-notify';


        /**
         * Constructor
         */
        function AlertsPanelView() {

            var lastUri = null;

            var alertsPanelPresenter = new AlertsPanelPresenter();

            function clearAlertTable(){
                $('[id^="' + ALERT_PREFIX + '"]').remove();
                $(ALERT_NOTIFY_BOX).hide();
            }

            function updateAlertCount(idCount, idIcon, count){
                if (count === 0){
                    // No alerts
                    $(idCount).text(count);
                    $(idIcon).addClass('hp-unset');
                }
                else if (count === -1){
                    // Alerts not yet obtained
                    $(idCount).html('&mdash;');
                    $(idIcon).addClass('hp-unset');
                }
                else {
                    // 1 or more alerts
                    $(idCount).text(count);
                    $(idIcon).removeClass('hp-unset');
                }
            }

            function updateAlertStats(){
                var alertObj = alertsPanelPresenter.getAlertData();
                if (alertObj){
                    updateAlertCount(ALERT_COUNT_CRITICAL, ALERT_COUNT_CRITICAL_ICON, alertObj.critCount);
                    updateAlertCount(ALERT_COUNT_WARNING, ALERT_COUNT_WARNING_ICON, alertObj.majorCount);
                    updateAlertCount(ALERT_COUNT_OK, ALERT_COUNT_OK_ICON, alertObj.okCount);

                    var alertRow;
                    clearAlertTable();
                    if (alertObj.alerts){
                        if (alertObj.alerts.length > 0){
                            // Build a new table via cloning
                            $.each(alertObj.alerts, function (idx, singleAlert) {
                                alertRow = $(ALERTS_TABLE_ROW_TEMPLATE).clone();
                                alertRow.attr('id', ALERT_PREFIX + idx);
                                $('.hp-status', alertRow).addClass('hp-status-' +
                                    statusState.getHealthStatus(alertsPanelPresenter.getStatus(singleAlert.severity)));
                                $('.hp-alert-description .ellipsis_text', alertRow).html(
                                        linkTargetBuilder.replaceTokensWithLinks(singleAlert.description));
                                $('.hp-alert-date', alertRow)
                                    .text($.format.date(Date.parseISOString(singleAlert.alertDate), 'M/dd/yy') + " " +
                                        $.format.date(Date.parseISOString(singleAlert.alertDate), 'h:mm a'));
                                alertRow.show();
                                $(ALERTS_TABLE).append(alertRow);
                                $('.hp-alert-description', alertRow).ThreeDots({max_rows: 1});
                            });
                        }
                        else{
                            $(ALERT_NOTIFY_BOX).show();
                        }
                    }
                }
                else {
                    clearAlertTable();
                    updateAlertCount(ALERT_COUNT_CRITICAL, ALERT_COUNT_CRITICAL_ICON, -1);
                    updateAlertCount(ALERT_COUNT_WARNING, ALERT_COUNT_WARNING_ICON, -1);
                    updateAlertCount(ALERT_COUNT_OK, ALERT_COUNT_OK_ICON, -1);
                }
            }

            // public
            this.onIndexResultsChange = function() {
                alertsPanelPresenter.setUri(lastUri);
            };

            // public
            this.onSelectionChange = function(selection) {
                // Set uri for alert panels
                if (selection.uris && (selection.uris.length > 0)){
                    lastUri = selection.uris[0];
                    alertsPanelPresenter.setUri(lastUri);
                }
                else{
                    lastUri = null;
                    alertsPanelPresenter.setUri(lastUri);
                }
            };

            function setupEvents(){
                alertsPanelPresenter.on("alertChange", updateAlertStats);
            }
            function removeEvents(){
                alertsPanelPresenter.off("alertChange", updateAlertStats);
            }

            /**
             * @public
             */
            this.init = function() {
                alertsPanelPresenter.init();
                setupEvents();

            };

            this.pause = function() {
                // clear the old data
                clearAlertTable();
                updateAlertCount(ALERT_COUNT_CRITICAL, ALERT_COUNT_CRITICAL_ICON, -1);
                updateAlertCount(ALERT_COUNT_WARNING, ALERT_COUNT_WARNING_ICON, -1);
                updateAlertCount(ALERT_COUNT_OK, ALERT_COUNT_OK_ICON, -1);

                lastUri = null;
                removeEvents();
            };

            this.resume = function() {
                setupEvents();
            };
        }

        return AlertsPanelView;
    }());

    return new AlertsPanelView();
});
