// (C) Copyright 2012 Hewlett-Packard Development Company, L.P.

define(['fs/presenter/settings/DateTimeSetupPresenter',
        'fs/presenter/common/CommonUsersPresenter',
        'fs/presenter/settings/RestorePresenter',
        'hp/core/Localizer',
        'hp/core/Notifications',
        'hp/view/DialogView',
        'hp/services/Log',
        'hp/model/Session',
        'hp/view/UploaderView',
        'hp/view/FormStateView',
        'hp/core/Router',
        'jquery',
        'lib/jquery.dateFormat',
        'hp/lib/jquery.hpCollapsible', 
        'hp/lib/jquery.hpProgressBar'],
function(dateTimeSetupPresenter, commonUsersPresenter, restorePresenter, 
     localizer, notifications, DialogView, log, session, UploaderView,
     FormStateView, router ) {"use strict";

    var RestoreUploadView = ( function() {

        var uploader = null;
        var restoreDialogContents = null;
        var restoreDialog = null;
        var formStateView = null;
        var hasRestoreWarningBeenShown = false;
        var fileSelected = null;
        var xhrFromUploader = null;
        var otherUsersLoggedInList = [];
        var backupInfo = null;

        // elements in the page
        var INFO_BOX = '#cic-settings-restore-backup-info-box',
            INFO_BOX_BACKUP_FILE_NAME = '#cic-settings-restore-backup-info-box-file-name',
            INFO_BOX_BACKUP_VERSION = '#cic-settings-restore-backup-info-box-version',
            INFO_BOX_BACKUP_DATE = '#cic-settings-restore-backup-info-box-date',
            
            RESTORE_WARNING = '#cic-settings-restore-warning',
            RESTORE_USERS = '#cic-settings-restore-apply-users-list',
            RESTORE_USERS_LOGGED_IN_COLLAPSE = '#cic-settings-restore-apply-loggedInUsers-collapse',
            RESTORE_USERS_LOGGED_IN_COUNT = '#cic-settings-restore-apply-warning-usersCount',
            RESTORE_USERS_LOGGED_IN_LIST = '#cic-settings-restore-apply-warning-usersLoggedInList',
            
            RESTORE_DIALOG = '#cic-settings-restore-dialog',
            RESTORE_FORM = "#cic-settings-restore-form",
            UPLOADER = '#cic-restore-fileupload',
            FILE_CHOOSE = '.hp-upload-file-choose',
            CANCEL_LINK = '.hp-upload-cancel',
            RESTORE_BUTTON = '#cic-settings-restore-restoreFromBackup',
            CANCEL_BUTTON = '#cic-settings-restore-cancel';
        
        /**
         * Constructor
         */
        function RestoreUploadView() {

            function displayError(message, details) {
                formStateView.setMessage({
                    summary : message,
                    details : details,
                    status : "error"
                });
            }

            function error(errorInfo){
                displayError(errorInfo.message, 
                    errorInfo.recommendedActions ? errorInfo.recommendedActions.join(' ') : null);
            }

            function clearError() {
                formStateView.reset();
            }
            
            function clearRestoreWarning() {
                restoreDialog.clearMessage();
            }
            
            function resetFileChooser() {
                $(FILE_CHOOSE).replaceWith($(FILE_CHOOSE).val('').clone(true));
            }

            function reset() {
                clearRestoreWarning();
                clearError();
                uploader.reset();
                resetFileChooser();
                hasRestoreWarningBeenShown = false;
                $(INFO_BOX).hide();
                $(RESTORE_USERS_LOGGED_IN_COUNT).html('');
                $(RESTORE_USERS_LOGGED_IN_LIST).html('');
                $(RESTORE_BUTTON).attr('disabled', 'disabled');
                $(CANCEL_BUTTON).removeAttr('disabled');
                
            }

            function displayRestoreWarning() {
                $(RESTORE_USERS_LOGGED_IN_COUNT).html('');
                $(RESTORE_USERS_LOGGED_IN_LIST).html('');

                var count = otherUsersLoggedInList.length;
                if (count > 1) {
                    for (var i = 0; i < count; i++) {
                        var node = $('<li/>');
                        node.html(otherUsersLoggedInList[i]);
                        $(RESTORE_USERS_LOGGED_IN_LIST).append(node);
                    }
                    $(RESTORE_USERS_LOGGED_IN_COUNT)
                    .html(localizer.getString('fs.settings.restore.applyBoxUsersLoggedIn',
                        [ count ]));
                    $(RESTORE_USERS_LOGGED_IN_COLLAPSE).show();
                }
                else if (count == 1) {
                    $(RESTORE_USERS_LOGGED_IN_COUNT)
                    .html(localizer.getString('fs.settings.restore.applyBoxOneUserLoggedIn', 
                        [ otherUsersLoggedInList[0] ]));
                    $(RESTORE_USERS_LOGGED_IN_COLLAPSE).show();
                }
                
                restoreDialog.setMessage({summary: $(RESTORE_WARNING).html(),
                          status: "warning",
                          details: $(RESTORE_USERS).html()
                          }, true);
                          
                hasRestoreWarningBeenShown = true;
            }
            
            function onGetOtherUsersLoggedInSuccess(otherUsers) {
                otherUsersLoggedInList = otherUsers;
                displayRestoreWarning();
            }
            
            function onGetOtherUsersLoggedInError(errorInfo) {
                reset();
                error(errorInfo);
            }
            
            function getRestoreWarningInfo() {
                commonUsersPresenter.getOtherUsersLoggedIn(
                    {success: onGetOtherUsersLoggedInSuccess, 
                     error: onGetOtherUsersLoggedInError});
            }
            
            function onStartRestoreSuccess(){
                restoreDialog.cancel();
                router.go('/'); 
                window.location.reload();
            }

            function onStartRestoreError(errorInfo){
                reset();
                error(errorInfo);
            }
            
            function showUploadedBackupInfo() {
                clearError();
                clearRestoreWarning();
                var dateVal = dateTimeSetupPresenter.parseDateTime(backupInfo.created);
                dateVal = $.format.date(dateVal, 'MMM d, yyyy hh:mm a');
                $(INFO_BOX_BACKUP_FILE_NAME).text(localizer.getString('fs.settings.restore.fileName', [fileSelected.name]));
                $(INFO_BOX_BACKUP_DATE).text(localizer.getString('fs.settings.restore.createDate', [dateVal]));
                $(INFO_BOX_BACKUP_VERSION).text(localizer.getString('fs.settings.restore.version',
                    [backupInfo.platformType, backupInfo.hardwareModel, 
                    backupInfo.firmwareVersionMajor, backupInfo.firmwareVersionMinor]));
                $(INFO_BOX).show();
                $(RESTORE_BUTTON).removeAttr('disabled');
            }
            
            function validationSuccess(data) {
                backupInfo = data;
                resetFileChooser();
                showUploadedBackupInfo();
            }
            
            function validationFailure(data) {
                reset();
                displayError(data.errorMessage, data.resolutionMessage);
            }
            
            function onFileSelection(files) {
                clearError();
                clearRestoreWarning();
                hasRestoreWarningBeenShown = false;
                $(RESTORE_BUTTON).attr('disabled', 'disabled');
                fileSelected = files[0];
            }
            
            function onBeforeStart(){
                var maxBackupSize = restorePresenter.getMaxBackupSize();
                
                // Don't check file size and allow upload if maximum not set
                if (maxBackupSize == null) {
                    return false;
                }
                
                // Display error and abort upload if backup is too big
                if (fileSelected.size > maxBackupSize) {
                    displayError(
                        localizer.getString('fs.settings.restore.backupTooBigErrMsg'),
                        localizer.getString('fs.settings.restore.backupTooBigResolutionMsg', [maxBackupSize])
                    );
                }
                return fileSelected.size > restorePresenter.getMaxBackupSize();
            }

            function onBeforeSend(xhr){
                xhrFromUploader = xhr;
                // The UX team does not want a cancel link 
                // in the backup upload dialog
                $(CANCEL_LINK).hide();
            }
            
            function onUploadSuccess(data) {
                if (data.status == 'SUCCEEDED') {
                    validationSuccess(data);
                } else {
                    validationFailure(data);
                }
            }

            function onUploadError(errorInfo) {
                reset();
                error(errorInfo);
            }
            
            function createSuccessContent() {
                $(INFO_BOX).show();
                return [$(INFO_BOX)];
            }


            /**
             * Initialization.
             */
            this.init = function() {

                uploader = new UploaderView();

                formStateView = new FormStateView();
                formStateView.init({form: RESTORE_FORM});

                restoreDialogContents = $(RESTORE_DIALOG).detach().show();
                restoreDialog = new DialogView({contents: restoreDialogContents});

                reset();
                
                restorePresenter.init();

                uploader.uploadInit({
                    uploadBaseId : UPLOADER ,
                    uploadUri : '/rest/backups/archive',
                    onFileSelection : onFileSelection,
                    beforeStart : onBeforeStart,
                    beforeSend : onBeforeSend,
                    uploadSuccess : onUploadSuccess,
                    uploadError :  onUploadError,
                    successContent: createSuccessContent
                });
               
                $(CANCEL_BUTTON).on('click', function(){
                   if (xhrFromUploader != null) {
                       xhrFromUploader.abort();
                   }
                   reset();
                   restoreDialog.cancel();
                   router.go('/settings/'); 
                });

                $(RESTORE_BUTTON).off('click');
                $(RESTORE_BUTTON).on('click', function(){
                    if(!hasRestoreWarningBeenShown){
                        getRestoreWarningInfo();
                    }
                    else {
                        clearRestoreWarning();
                        // disable the buttons so that a user cannot click again while the restore is starting
                        $(RESTORE_BUTTON).attr('disabled', 'disabled');
                        $(CANCEL_BUTTON).attr('disabled', 'disabled');
                        // let user know that the restore operation started
                        formStateView.setMessage({
                            summary : localizer.getString('fs.settings.restore.restoreStarted')
                        });
                        
                        restorePresenter.startRestore(
                            {success: onStartRestoreSuccess, error: onStartRestoreError}, 
                            backupInfo.uri);
                    }
                 });
            };
            
            this.pause = function() {
                uploader.pause();
            };

            this.resume = function() {
                uploader.resume();
            };
        }

        return new RestoreUploadView();
    }());

    return RestoreUploadView;
});
