// (C) Copyright 2011-2013 Hewlett-Packard Development Company, L.P.
define(['hp/core/Environment',
        'hp/core/Localizer',
        'fs/presenter/status/StatusPresenter',
        'hp/view/ModalDialogView',
        'hp/presenter/settings/FactoryResetPresenter',
        'hp/core/Router',
        'jquery'],

function(environment, Localizer, statusPresenter, modalDialog,
        factoryResetPresenter, router) {"use strict";

    var StatusErrorView = ( function() {

        var CONSOLE_DETAILS = '#hp-error-console-details';
        var BROADCAST_DETAILS = '#hp-error-broadcast-details';
        var REBOOT_LINK = '#hp-error-reboot-link';
        var FACTORY_RESET_OPTION = '#hp-error-factory-reset-option';
        var FACTORY_RESET_LINK = '#hp-error-factory-reset-link';
        var SUPPORT_DUMP_LINK = '#hp-error-support-dump-link';
        
        /**
         * @class View script for the /status/error page
         */
        function StatusErrorView() {
            
            var container;
            
            /**
             * Creates a modal dialog with only one button.
             * Note: DialogView and ModalDialogView create modal dialogs
             * that contains Ok and Cancel buttons. The requirement is
             * to have a single button to dismiss the dialog, some of our
             * dialogs. Old guidance from UX is 'Ok' - new guidance is 'Close' - 
             * so code and comments changed to be agnostic.
             */
            function showMessageDialog(msg, buttonText){
                
                if (!container) {
                    container = $('<div>').hide().appendTo('body');
                }
                
                $(container).html(msg);
                $(container).dialog({
                    modal:true,
                    position : "center",
                    draggable : false,
                    resizable : false,
                    dialogClass: 'hp-simple-dialog',
                    buttons: [{
                        text: buttonText,
                        click: function(){
                            $(container).remove();
                            container = null;
                    }}]
                });
            }

            /**
             * Callback for when the call to reboot the appliance fails.  Show a dialog with a generic error message.
             */
            function handleRebootError(e) {
                // show the user a dialog with a generic error message
                var dialogContent =
                    Localizer.getString('fs.status.error.reboot.error.message');
                var dialog = $('<div/>').text(dialogContent);
                dialog.dialog({
                    autoOpen : true,
                    modal : true,
                    resizable : false,
                    width : 440,
                    buttons : {
                        OK : function() {
                            dialog.dialog("destroy");
                            dialog.remove();
                        }
                    },
                    title : Localizer.getString('fs.status.error.reboot.error.title'),
                    closeText : ''
                });
            }

            function rebootConfirmed() {
                statusPresenter.rebootAppliance({
                    error: handleRebootError,
                    success: function () {}
                    // nothing needs doing on success
                });
            }


            function onFactoryResetFailure(errorInfo) {
                var msg = Localizer.getString('fs.settings.factoryReset.startFailure');
                var closeButtonText = Localizer.getString('fs.status.reset.closeButton');
                showMessageDialog(msg, closeButtonText);

                /*
                 * TODO: When AD629 (Error starting factory reset not displayed properly in UI) is addressed,
                 * figure out how these failures should be reported.
                 *
                 * AD629 is now addressed and the showMessageDialog() call above is appropriate.
                 * 
                 * When the UX team comes up with new mocks, if those new mocks call for a more complicated error
                 * page that does support some kind of notifications area, then this module should be updated to
                 * depend on NotificationsView and something like the following should be included here.

                var factoryResetError = {
                    summary : Localizer.getString('fs.settings.factoryReset.startFailure'),
                    status : "error"
                };
                notifications.add(factoryResetError, true, true, true);
                */
            }

            function onFactoryResetSuccess(result) {
               statusPresenter.setFactoryReset(true);
               router.go("/status", "waiting for factory reset to finish"); 
            }

            function startFactoryReset() {
                factoryResetPresenter.startFactoryReset('RECOVERY', { 
                    success: function(result) {
                        onFactoryResetSuccess(result);
                    },                
                    error: onFactoryResetFailure
                });                                          
            }

            function createSupportDump() {
                statusPresenter.createSupportDump();
            }

            /**
             * Callback function when the call to create support dump fails. Show a dialog with a generic error message.
             */
            function onCreateSupportDumpError(data) {
                var msg = Localizer.getString('fs.status.supportDump.error.message');
                var closeButtonText = Localizer.getString('fs.status.supportDump.error.closeButton');

                showMessageDialog(msg, closeButtonText);
            }

            /**
             * Callback function when the call to create support dump succeeds.
             * Once support dump is created, automatically download it.
             * @param data
             */
            function onCreateSupportDumpSuccess(data) {

                $('<a>',{
                    id: 'hiddenDownloader',
                    style: {display: 'none'},
                    href:data
                }).appendTo('body');

                $('#hiddenDownloader').click(function(){
                    window.location = $('#hiddenDownloader').attr('href');
                });

                $('#hiddenDownloader').trigger('click');
            }

            function registerEvents() {
                statusPresenter.on("createDumpSuccess", onCreateSupportDumpSuccess);
                statusPresenter.on("createDumpError", onCreateSupportDumpError);
            }
            
            function showFactoryResetOption() {
                // TODO: Isolate infrequent case where isKioskMode() == false on console.
                // alert("environment.isKioskMode() == " + environment.isKioskMode());
                return ( environment.isKioskMode() && !factoryResetPresenter.factoryResetDisabled() );
            }

            this.init = function() {

                statusPresenter.init();
                registerEvents();

                modalDialog.init();
                $(REBOOT_LINK).live('click', function(e) {
                    e.preventDefault();
                    var msg = Localizer.getString('fs.status.waiting.rebootConfirmMsg');
                    var confirmationButtonText = Localizer.getString('fs.status.waiting.confirmButton');
                    var cancelButtonText = Localizer.getString('fs.status.waiting.cancelButton');


                    modalDialog.showDialog({message: msg,
                        confirmButton: confirmationButtonText,
                        confirmFunction: rebootConfirmed,
                        cancelButton:  cancelButtonText});

                });
                
                if (showFactoryResetOption()) {
                    $(document).on('click', FACTORY_RESET_LINK, function(e) {
                        e.preventDefault();
                        var msg = Localizer.getString('fs.status.reset.resetConfirmMsg');
                        var confirmationButtonText = Localizer.getString('fs.status.reset.confirmButton');
                        var cancelButtonText = Localizer.getString('fs.status.reset.cancelButton');
                        
                        modalDialog.showDialog({message: msg,
                            confirmButton: confirmationButtonText,
                            confirmFunction: startFactoryReset,
                            cancelButton:  cancelButtonText});
                    });
                }

                $(document).on('click', SUPPORT_DUMP_LINK, function(e){
                    e.preventDefault();

                    var msg = Localizer.getString('fs.status.waiting.supportDumpConfirmMsg');
                    var okButtonText = Localizer.getString('core.details.ok');

                    showMessageDialog(msg, okButtonText);

                    createSupportDump();

                });

                document.title = Localizer.getString('fs.status.error.title');
                if (showFactoryResetOption()) {
                    $(BROADCAST_DETAILS).hide();
                } else {
                    $(CONSOLE_DETAILS).hide();
                    $(FACTORY_RESET_OPTION).hide();
                }
            };

            this.pause = function() {
                // nothing
            };

            this.resume = function() {
                document.title = Localizer.getString('fs.status.error.title');
            };
        }

        return new StatusErrorView();

    }());

    return StatusErrorView;

});
