// (C) Copyright 2011-2013 Hewlett-Packard Development Company, L.P.
/**
 * @type {MasterTableView}
 */
define(['hp/core/Localizer',
    'hp/core/Style',
    'jquery',
    'lib/jquery.dataTables'],
function (localizer, style) {
"use strict";

    var MasterTableView = (function () {

        var MASTER_PANE = '.hp-master-pane';
        var TABLE = '.hp-master-table';
        var SCROLLER = '.dataTables_scrollBody';
        var TABLE_WRAPPER = '.dataTables_wrapper';
        var HEADER_PARTS = '.dataTables_scrollHead, .dataTables_scrollHeadInner, ' +
            '.dataTables_scrollHeadInner > table';
        var SORT_HEADER_CELLS =
                '.dataTables_scrollHead thead td.sorting_disabled';
        var TABLE_BODY = '> tbody';
        var ROW = TABLE_BODY + ' > tr';
        var ROW_COLLAPSER = '> td > .hp-collapser';
        var SELECTED = 'hp-selected';
        var ACTIVE = 'hp-active';
        var EXPANDED = 'hp-expanded';
        var EXPAND_ROW_ON_CLICK = false;
        var KEY_DOWN = 40;
        var KEY_UP = 38;

        /**
         * @constructor
         */
        function MasterTableView() {

            var presenter = null;
            var page = null;
            var container = null;
            var pane = null;
            var table = null;
            var dataTable = null;
            var scroller = null;
            var sortColumnNames = [];
            var resource = null;
            var detailsRenderer = null;
            var detailsCollapsed = null;
            var multiSelect = false;
            var manualScrollTo = -1;
            var disableSelectionScroll = false;
            var manageWidth = false;
            var layoutTimer = null;
            var relayoutTimer = null;
            var addHelp = null;
            var fnClearRowCallback;

            function fireRelayout() {
                // we delay so any CSS animation can finish
                clearTimeout(relayoutTimer);
                relayoutTimer = setTimeout(function () {
                    page.trigger('relayout');
                }, style.animationDelay());
            }

            function expandRow(row) {
                if (! $(ROW_COLLAPSER, row).hasClass(ACTIVE)) {
                    $(ROW_COLLAPSER, row).addClass(ACTIVE);
                    row.addClass(EXPANDED);
                    var details = detailsRenderer(dataTable.fnGetData(row[0]));
                    var detailsRow = dataTable.fnOpen(row[0], details[0],
                        'hp-row-details-cell');
                    $(detailsRow).addClass('hp-row-details-row');
                    if (row.hasClass(SELECTED)) {
                        $(detailsRow).addClass(SELECTED);
                    }
                    details.show();
                }
            }

            function collapseRow(row) {
                if ($(ROW_COLLAPSER, row).hasClass(ACTIVE)) {
                    $(ROW_COLLAPSER, row).removeClass(ACTIVE);
                    if (detailsCollapsed) {
                        detailsCollapsed(dataTable.fnGetData(row[0]));
                    }
                    row.removeClass(EXPANDED);
                    dataTable.fnClose(row[0]);
                }
            }

            function toggleExpansion(row) {
                if (row.hasClass(EXPANDED)) {
                    collapseRow(row);
                } else {
                    expandRow(row);
                }
            }

            // remove all selection classes
            function clearTableSelection() {
                $(dataTable.fnSettings().aoData).each(function (index, item) {
                    $(item.nTr).removeClass(SELECTED);
                });
            }

            function setTableSelection(uris) {
                if (uris && uris.length > 0) {
                    var rows = $(ROW, table);
                    $.each(rows, function (index, row) {
                        var indexResult = dataTable.fnGetData(row);

                        if ((indexResult) &&
                            ($.inArray(indexResult.uri, uris) !== -1)) {
                            $(row).addClass(SELECTED);
                            if ($(row).hasClass(EXPANDED)) {
                                $(row).next().addClass(SELECTED);
                            }
                        } else {
                            if ($(row).hasClass(EXPANDED)) {
                                $(row).next().removeClass(SELECTED);
                            }
                        }
                    });
                }
            }

            function scrollToRow(row) {
                var current = scroller.scrollTop();
                var rowTop = row.position().top - scroller.position().top;
                var target = -1;

                if (rowTop < 0) {
                    target =  current + rowTop;
                } else if ((rowTop + row.outerHeight()) > scroller.innerHeight()) {
                    // make sure we don't scroll up so we can't see the top of the row
                    target =  Math.min(current + rowTop,
                        current + rowTop - scroller.innerHeight() + row.outerHeight());
                }

                if (target !== -1) {
                    scroller.scrollTop(target);
                }
            }

            function scrollToSelection() {
                var row;

                if (manualScrollTo !== -1) {
                    scroller.scrollTop(manualScrollTo);
                } else if (! disableSelectionScroll) {
                    row = $(ROW + '.' + SELECTED, table).first();
                    if (row.length > 0) {
                        scrollToRow(row);
                    }

                    setTimeout(function () {
                        if (manualScrollTo !== -1) {
                            // reset since we just scrolled
                            manualScrollTo = -1;
                        }
                    }, 50);
                }
            }

            function resetSelection(selection) {
                clearTableSelection();
                setTableSelection(selection.uris);
                scrollToSelection();
            }

            // Moved out of onRowClick for sonar
            function handleRowClickSelection(row, event) {
                // ignore if already selected and clicking collapser
                if (!(row.hasClass(SELECTED) &&
                      $(event.target).hasClass('hp-collapser'))) {
                    var indexResult = dataTable.fnGetData(row[0]);
                    if (multiSelect && (event.ctrlKey || event.metaKey)) {
                        presenter.toggleSelectIndexResult(indexResult);
                    } else if (multiSelect && event.shiftKey) {
                        presenter.selectContiguousIndexResults(indexResult);
                    } else {
                        presenter.selectIndexResult(indexResult);
                    }
                }                
            }

            /**
             * @private
             * Called when a row is clicked on in the table
             * @param {Event} event The javascript click event
             */
            function onRowClick(event) {
                // get index result from object and call presenter
                var row = $(event.currentTarget);
                if (row.hasClass('hp-row-details-row')) {
                    // click in details row, treat as selection of main row
                    row = row.prev();
                } else if (EXPAND_ROW_ON_CLICK) {
                    if (detailsRenderer) {
                        toggleExpansion(row);
                    }
                }
                disableSelectionScroll = true;

                handleRowClickSelection(row, event);

                // Allow for embedding input elements in row details (Activity notes)
                if (! scroller.is(':focus') &&
                    ! $(event.target).is(':focus')) {
                    scroller.focus();
                }
                disableSelectionScroll = false;
                scrollToRow(row);
            }

            function onShowMoreClick() {
                $(this).replaceWith('…');
                presenter.showMore();
            }

            /**
             * Adds the "show more" or "Add ..." row at the end.
             */
            function addControls() {
                if (presenter.haveMore()) {
                    // add "show more" row
                    $(TABLE_BODY, table).append(
                        '<tr class="hp-master-table-control">' +
                        '<td colspan="10">' +
                        '<a class="hp-master-show-more">' +
                        localizer.getString('core.master.showMore') +
                        '</a>' +
                        '</td></tr>');
                    $('.hp-master-show-more', table).on('click', onShowMoreClick);
                } else if (! presenter.haveSome() && addHelp) {
                    $(TABLE_BODY, table).append(
                        '<tr class="hp-master-table-control">' +
                        '<td colspan="10">' +
                        '<div class="hp-help">' + addHelp + '</div>' +
                        '</td></tr>');
                }
            }

            function removeControls() {
                $('.hp-master-show-more', table).off('click', onShowMoreClick);
                $('.hp-master-table-control', table).remove();
            }

            function calcTargetHeight() {
                var result;
                // use all height available in the pane that isn't used by other
                // elements in the pane
                result = pane.height();
                // don't count table headers
                result -= $('thead', table).outerHeight(true);
                result -= $('.dataTables_scrollHead', pane).outerHeight(true);
                $.each(pane.children(), function (index, child) {
                    if ($(child).is(':visible') &&
                        ! $(child).hasClass('hp-master-table') &&
                        ! $(child).hasClass('dataTables_wrapper') &&
                        ! $(child).hasClass('ui-resizable-handle')) {
                        result -= $(child).outerHeight(true);
                    }
                });
                return result;
            }
            
            function calculateRequestedTableWidth(oSettings) {
                // use any widths provided by the caller
                var result = 0;
                $.each(oSettings.aoColumns, function (index, column) {
                    var elem;
                    if (column.sWidth) {
                        result += parseInt(column.sWidth, 10);
                    } else {
                        elem = $('.dataTables_scrollHead td:nth-child(' +
                            (index + 1) + ')').first();
                        if ($('.hp-status', elem).length > 0) {
                            column.sWidth = '20px';
                            result += 20; // for in-progress spinner
                        } else {
                            result += $(elem).outerWidth(true);
                        }
                    }
                });
                // never go below 170
                result = Math.max(170, result);
                return result;
            }
            
            function calculateRequestedHeaderWidth() {
                var result = 0;
                $.each($('.hp-master-header h1', pane).children(), function (index, child) {
                    if ($(child).is(':visible')) {
                       result = Math.max($(child).outerWidth(true), result);
                    }
                });
                return result;
            }
            
            function setWidths(paneWidth, tableWidth, oSettings) {
                var panePadding;
                if (manageWidth) {
                    // precalculate padding to avoid animation effects
                    panePadding = pane.outerWidth() - pane.width();
                    pane.css('width', paneWidth);
                    // master pane width is content-box
                    $('.hp-details-pane', page).css('left', paneWidth + panePadding);
                    oSettings.oScroll.sX = paneWidth;
                } else {
                    if (! pane.hasClass('hp-resized')) {
                        pane.css({'width': ''});
                    }
                }
                
                if (page) {
                    scroller.css({'width': paneWidth});
                    $(HEADER_PARTS, container).css('width', tableWidth);
                    table.css('width', tableWidth);
                } else {
                    table.css('width', '');
                }
                
                dataTable.fnAdjustColumnSizing(false);
            }

            function layout() {
                var oSettings = dataTable.fnSettings();
                var targetHeight = calcTargetHeight();
                var currentHeight = parseInt(scroller.css('height'), 10);
                var targetPaneWidth, targetTableWidth, targetHeaderWidth;
                
                manageWidth = manageWidth && ! pane.hasClass('hp-resized');
                
                if (manageWidth) {
                    // table drives pane
                    targetTableWidth = calculateRequestedTableWidth(oSettings);
                    targetHeaderWidth = calculateRequestedHeaderWidth();
                    targetPaneWidth = targetTableWidth;
                    if (table.height() >= targetHeight) {
                        targetPaneWidth = Math.max(targetHeaderWidth,
                            targetTableWidth + style.scrollBarWidth());
                        if (targetPaneWidth === targetPaneWidth) {
                            targetTableWidth = targetPaneWidth - style.scrollBarWidth();
                        }
                    } else {
                        targetPaneWidth = Math.max(targetHeaderWidth, targetTableWidth);
                        targetTableWidth = targetPaneWidth;
                    }
                } else {
                    // pane drives table
                    if (pane) {
                        targetPaneWidth = pane.width();
                        targetTableWidth = targetPaneWidth;
                        if (table.height() >= targetHeight) {
                            targetTableWidth -= style.scrollBarWidth();
                        }
                    }
                }
                
                //console.log('!!! MTV layout', table.width(),'x',currentHeight,' -> ', targetTableWidth,'x',targetHeight, ' ', pane.width(),'x',targetHeight,' -> ',targetPaneWidth,'x',targetHeight,manageWidth);
                
                if (targetHeight && targetHeight !== currentHeight) {
                    // adjust table scroll
                    oSettings.oScroll.sY = targetHeight;
                    scroller.css({'height': targetHeight});
                }

                if (targetTableWidth && (targetPaneWidth !== pane.width() ||
                    targetTableWidth !== table.width())) {
                    setWidths(targetPaneWidth, targetTableWidth, oSettings);

                    dataTable.fnDraw();
                    
                    // fnDraw removes the controls
                    addControls();

                    if (manageWidth) {
                        fireRelayout();
                    }
                }
            }

            /*
             * Recursively walks the DOM removing children "bottom up"
             *   See http://msdn.microsoft.com/en-us/library/bb250448%28v=vs.85%29.aspx
             *   Specifically: DOM Insertion Order Leak Model
             *
             * We use jQuery's remove() function which is supposed to remove all eventhandlers
             *   registered with jQuery, as well as the DOM element itself.
             *
             * We are recursively removing all children under the <tbody></tbody> element.
             *   We do this from the last child to the first, each child also has its DOM tree walked
             *   in the same manner.  In this way we force clearing of the DOM from the bottom up
             *   as described in the MSDN article.
             */
            function recursiveDomRemoval(domElement) {
                var elementChildren = domElement.children;
                if (typeof elementChildren != "undefined" && elementChildren.length > 0) {
                    for (var i = elementChildren.length; i--;) {
                        recursiveDomRemoval(elementChildren[i]);
                    }
                }
                $(domElement).remove();
            }

            /*
             * Every refresh we need to take care to remove the entire table DOM as well as any
             *   event handlers that were registered
             */
            function clearTableComponents() {
                /* Use the dataTable jQuery object to get all nodes currently associated with the table
                 *   Cache that value in a local variable so IE8 does not have trouble iterating over
                 *   The array.
                 *
                 * Take a look at: http://benhollis.net/experiments/browserdemos/foreach/array-iteration.html
                 *   with an IE8 browser to see why we don't use $.each or another iteration model.
                 */
                var tableNodes = dataTable.fnGetNodes();
                for (var i = tableNodes.length; i--;) {
                    var nRow = tableNodes[i];

                    //If this function is defined, an implementor has chosen to process a function for each row removal
                    if (fnClearRowCallback) {
                        fnClearRowCallback.call(nRow, nRow);  // "this" is nRow in fnClearRowCallback()
                    }

                    recursiveDomRemoval(nRow);
                }

                removeControls();

                //The last stage of DOM removal is to remove the <tbody> section
                var dataTableTBody = dataTable.children('tbody');
                if (dataTableTBody && dataTableTBody[0]) {
                    $(dataTableTBody[0]).remove();
                }

                //Now re-add the <tbody> child and set the reference so dataTable can use it in fnDraw
                dataTable.append("<tbody></tbody>");
                dataTable.fnSettings().nTBody = dataTable.children('tbody')[0];
            }

            function resetSort() {
                // reset sort indicator
                var sort = presenter.getSort();
                if (sort) {
                    $(SORT_HEADER_CELLS, container).removeClass('sort_asc sort_desc');
                    $(SORT_HEADER_CELLS + '[data-name="' + 
                        sort.name + '"]', container).addClass('sort_' + sort.direction);
                }
            }

            /**
             * @private
             * @param {Object} data the IndexResults
             */
            function onIndexResultsChange(indexResults) {
                if (indexResults) {
                    // remember who's expanded
                    var expandedUris = $(ROW + '.' + EXPANDED, table).map(
                        function (index, row) {
                            return dataTable.fnGetData(row).uri;
                        }).get();

                    $(TABLE_WRAPPER, page).removeClass('hp-empty');

                    clearTableComponents();
                    dataTable.fnClearTable(indexResults.count === 0);
                    dataTable.fnAddData(indexResults.members);
                    
                    if (indexResults.count === 0) {
                        var oSettings = dataTable.fnSettings();
                        oSettings.oLanguage.sEmptyTable = presenter.getEmptyMessage();
                    }

                    // pre-expand rows
                    $(ROW, table).each(function (index, row) {
                        var indexResult = dataTable.fnGetData(row);
                        if (indexResult &&
                            ($.inArray(indexResult.uri, expandedUris) !== -1)) {
                            toggleExpansion($(row));
                        }
                    });

                    addControls();
                    resetSort();
                    layout();

                    resetSelection(presenter.getSelection());
                }
            }


            function onIndexResultsError(errorInfo) {
                if (page) {
                    $(TABLE_WRAPPER, page).addClass('hp-empty');
                }
            }

            // align table row selection classes with uris
            function onSelectionChange(selection) {
                resetSelection(selection);
            }

            function onKeyDown(event) {
                if (scroller.is(":focus")) {
                    var keyCode = (event.which ? event.which : event.keyCode);
                    if (keyCode === KEY_DOWN) {
                        manualScrollTo = -1;
                        $('.hp-master-table tr.hp-selected').next().trigger('click');
                        event.stopPropagation();
                        return false;
                    } else if (keyCode === KEY_UP) {
                        manualScrollTo = -1;
                        $('.hp-master-table tr.hp-selected').prev().trigger('click');
                        event.stopPropagation();
                        return false;
                    }
                }
            }

            // called when the user clicks on a table column header
            function sortColumn(event) {
                var cell = $(event.currentTarget);
                var propertyName = cell.attr('data-name');
                if (cell.hasClass('sort_asc')) {
                    $(SORT_HEADER_CELLS, container).removeClass('sort_asc sort_desc');
                    cell.addClass('sort_desc');
                    presenter.setSort(propertyName, 'desc');
                } else {
                    $(SORT_HEADER_CELLS, container).removeClass('sort_asc sort_desc');
                    cell.addClass('sort_asc');
                    presenter.setSort(propertyName, 'asc');
                }
            }

            function onResize(event) {
                // IE8 assigns the onResize event to the document object not
                // the window object so testing for document as well
                if (event.target == window || event.target == document) {
                    clearTimeout(layoutTimer);
                    layoutTimer = setTimeout(layout, 50);
                } else if (event.target === page[0]) {
                    clearTimeout(layoutTimer);
                    layout();
                }
            }

            /**
             * @private
             * Initialize the master table and attach event handlers from the presenter
             * @param {object} optionsArg Any options to pass to jQuery.datatables.
             */
            function masterTableInit(optionsArg) {
              
                table.addClass('hp-selectable');

                if (page && pane) {
                    manageWidth = (pane.hasClass('hp-master-pane') ||
                        pane.parents('.hp-master-pane').length === 1);
                }

                var options = {
                    "bPaginate" : false,
                    "bFilter" : false,
                    "bInfo" : false,
                    "bDeferRender": true,
                    "sScrollY" : calcTargetHeight(),
                    "sScrollX" : (page ? undefined : '100%'),
                    "bAutoWidth": false,
                    "bSort": false,
                    "aaData" : []
                };
                $.extend(options, optionsArg);

                fnClearRowCallback = optionsArg.fnClearRowCallback;
                var oLanguage = localizer.getString('core.dataTables.oLanguage');
                if (oLanguage) {
                    options.oLanguage = oLanguage;
                    oLanguage.sEmptyTable = presenter.getEmptyMessage();
                }

                // Initialize dataTable
                dataTable = table.dataTable(options);

                scroller = $(SCROLLER, container);
                scroller.attr('tabindex', '1');
                scroller.scroll(function () {
                    // stop auto scrolling if the user scrolls
                    manualScrollTo = scroller.scrollTop();
                });

                var sortable = $.map(options.aoColumns, 
                    function (col) {
                        return col.hasOwnProperty('bSortable') ? col.bSortable : true;
                    });

                // We do our own sorting, since we don't give datatables all
                // the data and we use the index service to sort.
                sortColumnNames = $.map(options.aoColumns, function (col) {
                    var match;
                    if (col.mDataProp) {
                        // strip leading "attributes.", if any
                        match = col.mDataProp.match(/^(multiA|a)ttributes\.(.*)/);
                        if (match) {
                            return match[2];
                        } else {
                            return col.mDataProp;
                        }
                    } else {
                        return ''; // if undefined, so be it
                    }
                });

                // Set up the header cells for sorting, store the sort
                // property name as a data- attribute.
                $(SORT_HEADER_CELLS, container).addClass("sort").each(
                    function (index, cell) {
                        if (sortable[index]) {
                            $(cell).click(sortColumn);
                        }
                        $(cell).attr('data-name', sortColumnNames[index]);
                    });

                // do initial sorting, if any
                if (options.aaSorting && options.aaSorting.length > 0) {
                    var sortOpt = options.aaSorting[0];
                    if (sortColumnNames[sortOpt[0]]) {
                        presenter.setSort(sortColumnNames[sortOpt[0]],
                            sortOpt[1]);
                    }
                }

                table.on('click', 'tbody tr', onRowClick);

                if (detailsRenderer && ! EXPAND_ROW_ON_CLICK) {
                    table.on('click', 'tbody tr td .hp-collapser',
                        function (ev) {
                            toggleExpansion($(this).parents('tr'));
                        });
                }
            }

            /**
             * @public
             * Stop the timer polling of the index service.
             */
            this.pause = function () {
                presenter.off("indexResultsChange", onIndexResultsChange);
                presenter.off("indexResultsError", onIndexResultsError);
                presenter.off("selectionChange", onSelectionChange);
                $(window).off('resize', onResize);
                if (page) {
                    page.off('relayout', layout);
                }
            };

            /**
             * @public
             * Resume the timer polling of the index service.
             */
            this.resume = function () {
                manualScrollTo = -1;
                presenter.on("indexResultsChange", onIndexResultsChange);
                presenter.on("indexResultsError", onIndexResultsError);
                presenter.on("selectionChange", onSelectionChange);
                $(window).on('resize', onResize);
                if (page) {
                    page.on('relayout', layout);
                }
                resetSort();
            };

            /**
             * @public
             * Intialize the view.
             */
            this.init = function (presenterArg, args) {
                presenter = presenterArg;
                resource = args.resource;
                if (args.page) {
                    page = args.page;
                } else {
                    page = $('.hp-page');
                }
                if (args.table) {
                    table = $(args.table);
                } else {
                    table = $(TABLE, page);
                }
                container = table.parent();
                if (args.pane) {
                    pane = $(args.pane);
                } else if ($(MASTER_PANE, page).length > 0) {
                    pane = $(MASTER_PANE, page);
                } else {
                    pane = container;
                }
                multiSelect = args.multiSelect;
                detailsRenderer = args.detailsRenderer;
                detailsCollapsed = args.detailsCollapsed;
                addHelp = args.addHelp;

                masterTableInit(args.dataTableOptions);

                scroller.on('focus', function (event) {
                    if (event.target === scroller[0]) {
                        $(document).off('keydown', onKeyDown);
                        $(document).on('keydown', onKeyDown);
                    }
                });
                scroller.on('blur', function (event) {
                    if (event.target === scroller[0]) {
                        $(document).off('keydown', onKeyDown);
                    }
                });
            };
        }

        return MasterTableView;

    }());

    return MasterTableView;
});
