// (C) Copyright 2011-2012 Hewlett-Packard Development Company, L.P.
/**
 * @constructor
 * @type {SearchView}
 */
define(['hp/presenter/SearchPresenter',
    'hp/core/Router',
    'hp/view/SearchBoxView',
    'hp/services/Log',
    'hp/core/LinkTargetBuilder',
    'hp/core/Localizer',
    'jquery', 'hp/lib/jquery.hpStatus'],
function(presenter, router, searchBoxView, log, linkTargetBuilder, localizer) {"use strict";

    var SearchView = ( function() {

        var RESULTS = '#hp-search-results tbody';
        var TEMPLATE = '#hp-search-result-template';
        var SPINNER = '#hp-search-spinner';
        var COUNT = '.hp-page-item-count';
        var TYPE = '.hp-search-type';
        var STATUS = '.hp-status';
        var NAME = '.hp-search-name';
        var EXTENDED = '.hp-search-extended';
        var STATUS_MAP = {ok: 'ok', normal: 'ok',
            warning: 'warning', error: 'error',
            disabled: 'disabled', unknown: 'unknown'};

        /**
         * @constructor
         * @type {SearchView}
         */
        function SearchView() {

            var lastQuery;
            var template;
            var selectSearchText = true;

            function parseParameters(url) {
                // Example url:
                //   something/that/is/a#url?key1=value1&key2=&key3=value3
                //
                // Example return value:
                //   { key1 : "value1", key2 : "", key3 : "value3"}

                var parameters = {};
                url.replace(/[?&]+([^=&]+)=([^&]*)/gi, function(m, key, value) {
                    parameters[key] = value;
                });
                return parameters;
            }

            function updateSummary(count) {
                $(COUNT).text(count);
            }

            function addResult(value) {
                var result = template.clone(),
                    link,
                    name = value.name ? value.name : value.description;

                link = linkTargetBuilder.makeLink(name, value.uri, null, value.category);
                $(TYPE, result).html(value.category);
                $(NAME, result).html(link);
                if (value.status) {
                    var status = STATUS_MAP[value.status.toLowerCase()];
                    $(STATUS, result).hpStatus(status);
                }
                if (value.attributes && value.attributes.model) {
                    $(EXTENDED, result).html(value.attributes.model);
                }

                $(RESULTS).append(result);
            }

            function onSearchResultChange(data) {
                //$(SPINNER).hide();
                var totalDisplay = 0;
                $(RESULTS).empty();
                $.each(data.members, function(index, value) {
                    if (linkTargetBuilder.hasViewablePage(value.category)) {
                        totalDisplay++;
                        addResult(value);
                    }
                });
                // Can't use this because we need to include
                // results that weren't returned.
                // updateSummary(totalDisplay);
                updateSummary(data.total);
                // only select search text when we first come to this page
                selectSearchText = false;
            }

            function toInt(intString, defaultValue) {
                var integer = parseInt(intString, 10);
                return isNaN(integer) ? defaultValue : integer;
            }

            function routingInit() {
                router.watch('search', '/search.*', {
                    parent: '/search',
                    change: function (location) {
                        var parameters = parseParameters(location);
                        lastQuery = parameters.query;
                        searchBoxView.setGlobalSearchText(parameters.query);
                        searchBoxView.activate(selectSearchText);
                        presenter.search(parameters.query,
                            toInt(parameters.start, 0),
                            toInt(parameters.count, 20));
                }});
            }

            /**
             * @public
             */
            this.init = function() {
                $(SPINNER).hide(); // not needed?
                template = $(TEMPLATE);
                $(RESULTS).detach(TEMPLATE);
                
                $(RESULTS).on('click', 'tr', function (event) {
                    if (! $(event.target).is('a')) {
                        $('.hp-search-name a', this).click();
                    }
                });

                presenter.init();
                routingInit();
                presenter.on('searchResultChange', onSearchResultChange);

                this.resume();
            };

            /**
             * @public
             */
            this.pause = function() {
            };

            /**
             * @public
             */
            this.resume = function() {
                selectSearchText = true;
                document.title = localizer.getString('search.title');
            };
        }

        return new SearchView();
    }());

    return SearchView;
});
