// (C) Copyright 2011 Hewlett-Packard Development Company, L.P.
/**
 * @constructor
 * @type {SearchView}
 */
define(['hp/presenter/MapPresenter',
    'hp/services/Log',
    'hp/core/LinkTargetBuilder',
    'hp/core/Router',
    'hp/core/Localizer',
    'jquery',
    'lib/excanvas',
    'hp/lib/jquery.hpStatus',
    'hp/lib/jquery.hpDropMenu',
    'hp/lib/jquery.hpSelect'],
function(presenter, log, linkTargetBuilder, router, localizer) {"use strict";

    var MapView = ( function() {
      
        var PAGE = '#hp-map-page';
        var PANE = '#hp-map-pane';
        var MAP = '.hp-map';
        var SPINNER = PAGE + ' .hp-pane-header .hp-spinner';
        var TITLE = PAGE + ' .hp-pane-header h1';
        var FOCUS_ITEM = '#hp-map-item';
        var CANVAS = MAP + ' canvas';
        var HISTORY = '#hp-map-history';
        var CONNECTION_COLOR = 'rgba(128, 128, 128, 0.5)';
        var SELECTED_CONNECTION_COLOR = 'rgba(66, 166, 212, 0.5)';
        var FILTERBAR_CONTROL = '.hp-filterbar-control';
        var FILTERBAR = '.hp-master-filter';
        var FILTERBAR_OPEN = 'hp-filterbar-open';
        
        // lifted from StatusState in CIC, needs an Atlas home
        var HEALTH_STATUS_MAP = {
            OK             : "ok",
            UNKNOWN        : "unknown",
            WARNING        : "warning",
            MAJOR          : "warning",
            CRITICAL       : "error",
            DISABLED       : "disabled",
            ok             : "ok",
            warning        : "warning",
            error          : "error",
            disabled       : "disabled",
            unknown        : "unknown"
        };

        /**
         * @constructor
         * @type {MapView}
         */
        function MapView() {
          
            var map;
            var canvas;
            var context;
            var focus;
            var template;
            var connections;
            var historyTemplate;
            var placedItems = {}; // uri -> item
            
            function itemLocation(item) {
                var itemPosition = item.position();
                return {left: (itemPosition.left + (item.width() / 2) + 35),
                    top: (itemPosition.top + (item.height() / 2) + 20)};
            }
          
            function connect(item1, item2, color) {
                var position1 = itemLocation(item1);
                var position2 = itemLocation(item2);
                context.beginPath();
                context.lineWidth = 3;
                context.lineCap = 'round';
                context.strokeStyle = color;
                context.moveTo(position1.left, position1.top);
                context.quadraticCurveTo(
                    position1.left + ((position2.left - position1.left) / 2),
                    position1.top,
                    position2.left, position2.top);
                context.stroke();
            }
            
            function fillItem(item, resource) {
                var device = $('.hp-device', item);

                $('.hp-status', item).
                    hpStatus(HEALTH_STATUS_MAP[resource.status]);
                $('.hp-device-name', item).empty().
                    html(linkTargetBuilder.makeLink(resource.name,
                        resource.uri));
            
                if (resource.attributes.model) {
                    $('.hp-device-model', item).
                        text(resource.attributes.model);
                    device.removeClass('hp-logical');
                    device.addClass('hp-device');
                } else {
                    $('.hp-device-model', item).text('');
                    device.removeClass('hp-device');
                    device.addClass('hp-logical');
                }
            }
          
            function addItem(resource, items) {
                var item = placedItems[resource.uri];
                if (! item) {
                    item = template.clone();
                    fillItem(item, resource);
                    if (items) {
                        items.parent().removeClass('hp-empty');
                        items.append(item);
                        item.click(function (ev) {
                            router.go('/map/r' + resource.uri, 'map');
                            ev.preventDefault();
                        });
                    }
                    placedItems[resource.uri] = item;
                }
                return item;
            }
          
            function addParents(childItem, treeNode, level) {
                var parentItem;
                var items = $('#hp-map-level-' + level + ' .hp-map-items');
                $.each(treeNode.parents, function (assocName, parents) {
                    $.each(parents, function (index, parentNode) {
                        parentItem = addItem(parentNode.resource, items);
                        connections.push({parent: parentItem, child: childItem,
                            color: (-1 === level ? SELECTED_CONNECTION_COLOR :
                                CONNECTION_COLOR)});
                        addParents(parentItem, parentNode, level - 1);
                    });
                });
            }
            
            function addChildren(parentItem, treeNode, level) {
                var childItem;
                var items = $('#hp-map-level-' + level + ' .hp-map-items');
                $.each(treeNode.children, function (assocName, children) {
                    $.each(children, function (index, childNode) {
                        childItem = addItem(childNode.resource, items);
                        connections.push({parent: parentItem, child: childItem,
                            color: (1 === level ? SELECTED_CONNECTION_COLOR :
                                CONNECTION_COLOR)});
                        addChildren(childItem, childNode, level + 1);
                    });
                });
            }
            
            function drawConnections() {
                if (context) {
                    $(CANVAS).attr('width', map.parent().prop('scrollWidth')).
                        attr('height', map.parent().prop('scrollHeight'));
                    context.clearRect(0, 0, canvas.width, canvas.height);
                    $.each(connections, function (index, connection) {
                        connect(connection.parent, connection.child, connection.color);
                    });
                }
            }

            function onTreeChange(tree) {
                var resource = tree.resource;
                placedItems = {};
                connections = [];
                
                // empty old items
                $('.hp-map-item:not(#hp-map-item)').remove();
                $('.hp-map-level:not(#hp-map-level-0)').addClass('hp-empty');
                
                $(TITLE).text(resource.name);
                placedItems[resource.uri] = focus;
                fillItem(focus, resource);
                addParents(focus, tree, -1);
                addChildren(focus, tree, 1);
                
                drawConnections();
                
                // refresh history
                $(HISTORY).empty();
                $.each(presenter.getHistory(), function (index, historyResource) {
                    var historyItem = historyTemplate.clone();
                    $('a', historyItem).
                        attr('href', '#/map/r' + historyResource.uri).
                        text(historyResource.name);
                    $(HISTORY).prepend(historyItem);
                });
                
                $(SPINNER).hide();
                $(TITLE).show();
            }
            
            function onTreeChanging() {
                $(TITLE).hide();
                $(SPINNER).show();
            }
            
            function closeFilterbar() {          
                $('.hp-sub-nav').append(
                    $(FILTERBAR + ' .hp-filter').removeClass('hp-active hp-pinned')); 
                $(PAGE).removeClass(FILTERBAR_OPEN);
            }

            function openFilterbar() {
                $(PAGE).addClass(FILTERBAR_OPEN);
                $(FILTERBAR).append(
                    $('.hp-sub-nav .hp-filter').addClass('hp-active hp-pinned'));
            }
            
            function toggleFilterbar() {
                if ($(FILTERBAR).length > 0 &&
                    ! $(PAGE).hasClass(FILTERBAR_OPEN)) {
                    openFilterbar();
                } else {
                    closeFilterbar();
                }
                // wait for animation
                setTimeout(function () {drawConnections();}, 500);
            }
            
            function onOrientationChange() {
                map.toggleClass('hp-vertical');
                // wait for animation
                setTimeout(function () {drawConnections();}, 10);
            }

            /**
             * @public
             */
            this.init = function() {
              
                focus = $(FOCUS_ITEM);
                template = focus.clone().attr('id', '');
                focus.addClass('hp-selected');
                
                map = $(MAP);
                canvas = $(CANVAS)[0];
                historyTemplate = $('#hp-map-history-item-template').
                    remove().attr('id', '');
                
                if (!canvas.getContext && window.G_vmlCanvasManager) {
                    window.G_vmlCanvasManager.initElement(canvas);
                }
                
                if (canvas.getContext) {
                    context = canvas.getContext('2d');
                }
                
                $('#hp-map-history-filter').hpDropMenu();
                $('#hp-map-orientation-filter').hpSelect().
                    change(onOrientationChange);
                $(FILTERBAR_CONTROL).on('click', toggleFilterbar);
                $(PANE).on('scroll', drawConnections);

                presenter.init('^/map');

                this.resume();
            };

            /**
             * @public
             */
            this.pause = function() {
                presenter.off('treeChanging', onTreeChanging);
                presenter.off('treeChange', onTreeChange);
                $(window).off('resize', drawConnections);
            };

            /**
             * @public
             */
            this.resume = function() {
                presenter.on('treeChanging', onTreeChanging);
                presenter.on('treeChange', onTreeChange);
                $(window).on('resize', drawConnections);
                document.title = localizer.getString('core.map.title');
            };
        }

        return new MapView();
    }());

    return MapView;
});
