// (C) Copyright 2013 Hewlett-Packard Development Company, L.P.
/**
 * @type {GroupPresenter}
 */
define(['hpsum/model/group/GroupResource',
    'hpsum/presenter/baseline/BaselinePresenter',
    'hpsum/presenter/node/NodePresenter',
    'hp/services/IndexService',
    'hp/core/Router',
    'hp/core/UrlFragment',
    'hp/core/LinkTargetBuilder',
    'hp/core/Notifications',
    'hp/core/EventDispatcher',
    'hp/services/REST',
    'hp/model/Session',
    'hp/core/Localizer'], 

function (resource, baselinePresenter, nodePresenter, indexService, router, urlFragment,
linkTargetBuilder, notifications, EventDispatcher, restService, session, localizer) {"use strict";

	var GroupPresenter = ( function() {

			/**
			 * @constructor
			 */
			function GroupPresenter() {

				var dispatcher = new EventDispatcher(), groups = [], baselines = [], nodes = [], selectedItem = '', trackUri = null, initDone = false, allGroups = null;

				var task = {
					category : 'task',
					name : 'A New Group Adding',
					uri : '/task/task-' + (new Date()).getTime(),
					status : 'unknown',
					changing : true,
					sourceUri : null,
					sourceName : null,
					date : null,
					state : 'running',
					progress : 10,
					progressStep : 'Initializing',
					owner : session.getUser()
				};

				function getItem() {
					var uris = resource.getSelection().uris;
					if (uris && uris.length > 0) {
						resource.getItem(uris[0], {
							success : function(item) {
								selectedItem = item;
								dispatcher.fire("itemChange", item);
							},
							error : function(errorMessage) {
								dispatcher.fire("itemError", errorMessage);
							}
						});
					}
				}

				function selectionChange(selection) {
					dispatcher.fire("selectionChange", selection);
					if (selection.uris) {
						if (!selection.multiSelectId) {
							getItem();
						} else {
						}
					}
				}-

				function onAddGroupError() {

				}


				this.getAllNodes = function(handlers) {
					var localHandlers = {
						success : function(data) {
							nodes = [];
							var hapiNodes = data.hapi.output_data.nodes.node;
							var idx = 0;
							for ( idx = 0; idx < hapiNodes.length; ++idx) {
								var hapiNode = hapiNodes[idx];
								var node = {
									node_status : hapiNode.node_status,
									state : hapiNode.node_state,
									name : hapiNode.node_name,
									type : hapiNode.type,
									uri : hapiNode.node_uri,
									id : hapiNode.node_id,
									session_id : hapiNode.node_session
								};
								nodes.push(node);
								handlers.success(data);
							}
						},
						error : function(msg) {
							handlers.error(msg);
						}
					};
					nodePresenter.getAllNodes(localHandlers);
				};

				this.getAllBaselines = function(handlers) {
					baselinePresenter.getBaselines({
						success : function(data) {
							baselines = [];
							var i = 0, baselineid;
							var paths = [];
							for ( i = 0; i < data.length; i++) {
								paths = data[i].uri.split("/");
								baselineid = paths[5] + "/" + paths[7];
								var obj = {
									id : baselineid,
									name : data[i].name + ' at ' + data[i].location,
									uri : data[i].uri
								};
								baselines.push(obj);
							}
							if (handlers) {
								handlers.success(baselines);
							}
						},
						error : function(msg) {
							if (handlers) {
								handlers.error(msg);
							}
						}
					});
				};

				this.getBaselineArray = function() {
					return baselines;
				}

				this.getNodeArray = function() {
					return nodes;
					;
				}

				this.getGroupArray = function() {
					return groups;
				}

				this.getData = function(handlers) {
					getItem();
					var uri = '/Session/' + session.getToken() + '/node/group/' + selectedItem.hapi.group_id + '/getdata';
					restService.getURI(uri, {
						success : function(data) {
							if (data.hapi.hcode == 0) {
								handlers.success(data);
							}
						},
						error : function(data) {
							handlers.error(data);
						}
					});
				};

				this.getUpdateList = function(item, handler) {
					var uri = 'Session/' + session.getToken() + '/node/group/' + item.group_id + '/getupdatelist';
					restService.postObject(uri, handlers);
				}				
			
				this.setAttributes = function(item, options, handlers) {
					var uri = '/Session/' + session.getToken() + '/node/group/' + item.group_id + '/setattributes';
					restService.postObject(uri, options, {
						success : function(data) {
							if (data.hapi.hcode == 0) {
								handlers.success(data);
							} else
								handlers.error(data.hapi.hmessage);
						},
						error : function(data) {
							handlers.error(data);
						}
					}, null);
				}; 
			
				this.analysis = function(item, complist, handlers) {
					var uri = '/Session/' + session.getToken() + '/node/group/' + item.group_id + '/analysis';
					restService.postObject(uri, complist, "", handlers, {});
				};
				
				this.deploy = function(item, handlers) {
					var uri = '/Session/' + session.getToken() + '/node/group/' + item.group_id + '/deploy';
					var object = null;
					restService.getURI(uri, {
						success : function(data) {
							if (data.hapi.hcode == 0) {
								handlers.success(data);
								resource.getIndexResults();
							} else
								handlers.error(data.hapi.hmessage);
						},
						error : function(data) {
							handlers.error(data);
						}
					});
				};

				this.abortItem = function(group, handlers) {
					var uri = '/session/' + session.getToken() + '/node/group/' + group.group_id + '/abort';
					restService.postObject(uri, null, {
						success : function(data) {
							if (data.hapi.hcode == 0) {
								handlers.success(data);
							} else
								handlers.error(data.hapi.hmessage);
						},
						error : function(data) {
							handlers.error(data);
						}
					}, null);
				};

				this.deleteItem = function(group, handlers) {
					var uri = '/session/' + session.getToken() + '/node/group/' + group.group_id + '/remove';
					restService.postObject(uri, null, {
						success : function(data) {
							if (data.hapi.hcode == 0) {
								handlers.success(data);
								resource.removeSelectedUri('/session/' + session.getToken() + '/node/group/' + group.group_id + '/getdata');
								resource.getIndexResults();
							} else
								handlers.error(data.hapi.hmessage);
						},
						error : function(data) {
							handlers.error(data);
						}
					}, null);
				};

				this.getBaselineDetailsforId = function(baselineId) {
					var i;
					if (baselineId == '')
						return null;
					for ( i = 0; i < baselines.length; i++) {
						if (baselines[i].id.match(baselineId))
							return baselines[i];
					}
					return null;
				};

				this.getGroupIndex = function(handlers) {
					var groupIndexHandlers = {
						success : function(data) {
							if (handlers) {
								handlers.success(data);
							}
						},
						error : function(msg) {
							if (handlers) {
								handlers.error(msg);
							}
						}
					};
					restService.getURI('/session/' + session.getToken() + '/node/group/index', indexGroups(groupIndexHandlers, groups));
				};

				function indexGroups(handlers, groups) {
					var groupDataHandlers = {
						success : function(data) {
							groups.push(data);
						},
						error : function(errorInfo) {
							if (handlers) {
								handlers.error(errorInfo);
							}
						}
					};
					return {
						success : function(data) {
							var i;
							var uri;
							groups = [];
							for ( i = 0; i < data.hapi.output_data.groups.group.length; ++i) {
								var hapiGroup = data.hapi.output_data.groups.group[i];
								restService.getURI('/session/' + session.getToken() + '/node/group/' + hapiGroup.group_id + '/getdata', groupDataHandlers);
							}
							handlers.success(data);
						},
						error : function(data) {
							handlers.error(data);
						}
					};
				}


				this.haveItems = function() {
					return resource.haveSome();
				};

				this.selectionLabel = function() {
					return selectedItem.hapi.name;
				};

				this.getGroupStatus = function(item, handlers) {
					var uri = '/session/' + session.getToken() + '/node/group/' + item.group_id + '/getstatus';
					var thishandlers = {
						success : function(data) {
							handlers.success(data);
						},
						error : function(errorInfo) {
							handlers.error(errorInfo);
						}
					};
					restService.getURI(uri, thishandlers);
				}

				this.startInventory = function(item, handlers) {
					var uri = '/session/' + session.getToken() + '/node/group/' + item.group_id + '/inventory';
					var object = {
						hapi : {
							baselines : {
								baseline : item.baselines
							}
						}
					};
					restService.postObject(uri, object, {
						success : function(data) {
							if (data.hapi.hcode == 0) {
								handlers.success(data);
								resource.getIndexResults();
							} else
								handlers.error(data.hapi.hmessage);
						},
						error : function(data) {
							handlers.error(data);
						}
					}, null);
				};

				function onAddGroupSuccess(data) {
					task.name = "Added a new Group";
					task.status = 'ok';
					var notif = {
						summary : task.name,
						uri : task.uri,
						status : task.status,
						sourceName : task.sourceName,
						sourceUri : task.sourceUri,
						progress : task.progress,
						step : task.progressStep
					};
					notif.summary = 'Added a new Group';
					notif.status = task.status;
					notifications.add(notif, true);
					if (task.sourceUri === trackUri) {
						dispatcher.fire('notificationChange', notif);
					}
				}
				
				this.addItem = function(item, selectItem, handlers) {
					task.sourceName = item.name;
					task.sourceUri = item.uri;
					task.date = item.date;
					var notif = {
						summary : task.name,
						uri : task.uri,
						status : task.status,
						sourceName : task.sourceName,
						sourceUri : task.sourceUri,
						progress : task.progress,
						step : task.progressStep
					};
					notifications.add(notif, true);
					item.status = 'unknown';

					resource.addGroup(item, {
						success : function(data) {
							if (data.hapi.hmessage.match(/Group name already exists/i)) {
								handlers.success(data);
							} else {
								handlers.success(data);
								onAddGroupSuccess(data);
								if (selectItem) {
									var itemUri = '/session/' + session.getToken() + '/node/group/' + data.hapi.group_id + '/getdata';
									resource.setSelectedUris([itemUri]);
								}
								resource.getIndexResults();
								dispatcher.fire('itemChange', item);
								handlers.success(data);
							}
						},
						error : function(data) {
							onAddGroupError(data);
							handlers.error(data);
						}
					});
					trackUri = item.uri;
					resource.getIndexResults();
					if (item.uri === trackUri) {
						dispatcher.fire('notificationChange', notif);
					}

				};

				this.refreshIndexResults = function() {
					resource.getIndexResults();
				};

				this.editItem = function(groupItem, item, handlers) {
					var uri = '/session/' + session.getToken() + '/node/group/' + groupItem.group_id + '/edit';
					var object = {
						hapi : {
							group_name : item.name,
							group_description : item.description,
							group_nodes : {
								group_node : item.nodes
							}
						}
					};

					restService.postObject(uri, object, {
						success : function(data) {
							if (data.hapi.hcode == 0) {
								handlers.success(data);
								resource.getIndexResults();
							} else
								handlers.error(data.hapi.hmessage);
						},
						error : function(data) {
							handlers.error(data);
						}
					}, null);

				};

				this.removeNodes = function(groupItem, item, handlers) {
					var uri = '/session/' + session.getToken() + '/node/group/' + groupItem.group_id + '/removenodes';
					var object = {
						hapi : {
							group_nodes : {
								group_node : item.nodes
							}
						}
					};

					restService.postObject(uri, object, {
						success : function(data) {
							if (data.hapi.hcode == 0) {
								handlers.success(data);
								resource.getIndexResults();
							} else
								handlers.error(data.hapi.hmessage);
						},
						error : function(data) {
							handlers.error(data);
						}
					}, null);
				};

				/**
				 * @public
				 */
				this.init = function() {
					resource.on("selectionChange", selectionChange);
					initDone = true;
					this.getGroupIndex();
				};

				this.resource = function() {
					return resource;
				}

				this.getSelection = function() {
					return resource.getSelection();
				};

				this.allBaselines = function() {
					return allBaselines;
				}

				this.getAddLink = function() {
					var result = null;
					if (session.canPerformAction('GROUP', 'ADD')) {
						result = linkTargetBuilder.makeLink(localizer.getString('hpsum.group.addGroup'), null, 'add', 'group', 'ADD');
					}
					return result;
				};

				/**
				 * @public
				 * Turn on a specific event
				 *
				 */
				this.on = function(eventName, callback) {
					var uris;
					dispatcher.on(eventName, callback);
					switch (eventName) {
						case 'selectionChange':
							callback(resource.getSelection());
							break;
						case 'itemChange':
							uris = resource.getSelection().uris;
							if (uris && uris.length === 1) {
								resource.getItem(uris[0], {
									success : callback
								});
							}
							break;
						case 'itemError':
							uris = resource.getSelection().uris;
							if (uris && uris.length === 1) {
								resource.getItem(uris[0], {
									error : callback
								});
							}
							break;
					}
				};

				/**
				 * @public
				 * Turn off a specific event
				 */
				this.off = function(eventName, callback) {
					dispatcher.off(eventName, callback);
				};
			}

			return new GroupPresenter();
		}());

	return GroupPresenter;
});

